/* sg_delaunay - delaunay triangulization algorithm for gtk+
 * Copyright 2001  Andy Thaller <thaller@ph.tum.de>
 *
 * This library is g_free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <malloc.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkplotdt.h>
#include "sg_delaunay.h"

#define REAL gdouble
#include "triangle.h"
#undef REAL


static void 	sg_delaunay_class_init 		(GtkPlotDTClass *klass);
static gboolean	sg_delaunay_real_triangulate	(GtkPlotDT *dt);

static GtkPlotDTClass *parent_class = NULL;


GtkType
sg_delaunay_get_type (void)
{
  static GtkType data_type = 0;

  if (!data_type)
    {
      GtkTypeInfo data_info =
      {
        "SGdelaunay",
        sizeof (SGdelaunay),
        sizeof (SGdelaunayClass),
        (GtkClassInitFunc) sg_delaunay_class_init,
        (GtkObjectInitFunc) NULL,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      data_type = gtk_type_unique (gtk_plot_dt_get_type(), &data_info);
    }
  return data_type;
}

static void
sg_delaunay_class_init (GtkPlotDTClass *klass)
{
  GtkObjectClass *object_class;
  GtkPlotDTClass *dt_class;

  parent_class = gtk_type_class (gtk_object_get_type ());

  object_class = (GtkObjectClass *) klass;
  dt_class = (GtkPlotDTClass *) klass;

  dt_class->triangulate = sg_delaunay_real_triangulate;
}


/* increase storage size of nodes
 *
 * num: new number of nodes, must be greater that data->node_max
 * returns 0 on failure, 1 on success
 */
static gint
sg_delaunay_expand(GtkPlotDT *data, gint num)
{
  GtkPlotDTnode *nodes= 0;
#ifdef DELAUNAY_DEBUG
  fprintf(stderr,"delaunay: expanding to %d\n",num);
#endif
  if (!data) return 0;
  if (!num || num<=data->node_max) return 1;
  nodes= (GtkPlotDTnode*) g_malloc(sizeof(GtkPlotDTnode)*num);
  if (!nodes) return 0;
  if (data->nodes && data->node_cnt)
    memcpy(nodes,data->nodes,sizeof(GtkPlotDTnode)*data->node_cnt);
  if (data->nodes)
    g_free(data->nodes);
  data->nodes= nodes;
  data->node_max= num;
  return 1;
}

GtkObject*
sg_delaunay_new ()
{
  GtkObject *object;

  object = gtk_type_new (sg_delaunay_get_type ());

  return (object);
}

static gboolean
sg_delaunay_real_triangulate(GtkPlotDT *data)
{
  struct triangulateio in, semi, mid, *out= 0;
  GtkPlotDTtriangle **triangles = NULL;
  gint i;

  if (!data) return FALSE;
  if (!data || ! data->nodes || data->node_cnt<3) return FALSE;

  if(data->triangles){
    GList *list;
    for (list = data->triangles; list; list = list->next){
      if (list->data) g_free(list->data);
        g_list_free(data->triangles);
      data->triangles= NULL;
    }
  }

  /* initialize structures for triangulate() */
  in.numberofpoints= data->node_cnt;
  in.numberofpointattributes = 1;
  in.numberofsegments = 0;
  in.numberofholes = 0;
  in.numberofregions = 0;
  in.pointlist = (double *) malloc(in.numberofpoints * 2 * sizeof(double));
  in.pointattributelist = (double *) malloc(in.numberofpoints *
					    in.numberofpointattributes *
					    sizeof(double));
  in.pointmarkerlist = (int *) malloc(in.numberofpoints * sizeof(int));
  in.regionlist = (double *) 0;

  for (i= 0; i<data->node_cnt; i++) {
    in.pointlist[i*2]= data->nodes[i].x;
    in.pointlist[i*2+1]= data->nodes[i].y;
    in.pointattributelist[i]= data->nodes[i].z;
    in.pointmarkerlist[i]= 0;
  }

  semi.pointlist = (double *) NULL;
  semi.pointattributelist = (double *) NULL;
  semi.pointmarkerlist = (int *) NULL;
  semi.trianglelist = (int *) NULL;
  semi.triangleattributelist = (double *) NULL;
  semi.trianglearealist = (double *) NULL;
  semi.neighborlist = (int *) NULL;
  semi.segmentlist = (int *) NULL;
  semi.segmentmarkerlist = (int *) NULL;
  semi.edgelist = (int *) NULL;
  semi.edgemarkerlist = (int *) NULL;

  mid.pointlist = (double *) NULL;
  mid.pointattributelist = (double *) NULL;
  mid.pointmarkerlist = (int *) NULL;
  mid.trianglelist = (int *) NULL;
  mid.triangleattributelist = (double *) NULL;
  mid.neighborlist = (int *) NULL;
  mid.segmentlist = (int *) NULL;
  mid.segmentmarkerlist = (int *) NULL;
  mid.edgelist = (int *) NULL;
  mid.edgemarkerlist = (int *) NULL;

  /* Triangulate the points. 
   *   number everything from zero (z)
   *   produce an edge list (e)
   *   and a triangle neighbor list (n).
   */

  triangulate("zAenQ", &in, &semi, (struct triangulateio *) NULL);
  out= &semi;

/*  
  triangulate("rza.8Q", &semi, &mid, (struct triangulateio *) NULL);
  out= &mid;
*/ 

  /* all is done, so save everything in the data structure */

  sg_delaunay_expand(data, out->numberofpoints+2);
  data->node_cnt= out->numberofpoints;


  for (i = 0; i < out->numberofpoints; i++) {
    data->nodes[i].x= out->pointlist[i*2];
    data->nodes[i].y= out->pointlist[i*2+1];
    data->nodes[i].z= out->pointattributelist[i];
  }

  triangles = g_new0(GtkPlotDTtriangle *, out->numberoftriangles);
  for (i = 0; i < out->numberoftriangles; i++) {
    GtkPlotDTtriangle *t = g_new0(GtkPlotDTtriangle, 1);
    t->a = out->trianglelist[i*3];
    t->b = out->trianglelist[i*3+1];
    t->c = out->trianglelist[i*3+2];
  
    t->na = &data->nodes[t->a];
    t->nb = &data->nodes[t->b];
    t->nc = &data->nodes[t->c];
    triangles[i] = t;
  
    data->triangles = g_list_prepend(data->triangles, t);
  }
  /*
  for (i = 0; i < out->numberoftriangles; i++)
    printf("%d %d %d %d\n",i,out->neighborlist[i*3],out->neighborlist[i*3+1],out->neighborlist[i*3+2]);
  */
  for (i = 0; i < out->numberoftriangles; i++) {
    if(out->neighborlist[i*3] >= 0)
      triangles[i]->nn[0] = triangles[out->neighborlist[i*3]];  
    if(out->neighborlist[i*3+1] >= 0)
      triangles[i]->nn[1] = triangles[out->neighborlist[i*3+1]];  
    if(out->neighborlist[i*3+2] >= 0)
      triangles[i]->nn[2] = triangles[out->neighborlist[i*3+2]];  
  } 
  g_free(triangles);

  return out->numberoftriangles;
}


