/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include "sg_dataset_worksheet.h"

#define DATASET_NAME_LENGTH 100
#define NENTRIES 20

typedef struct
{
  SGapplication *app;
  SGdataset *dataset;
  SGpluginStyle *constructor;
  GtkWidget *column_entry[NENTRIES];
  GtkWidget *left_button[NENTRIES];
  GtkWidget *right_button[NENTRIES];
  GtkWidget *entries_box;
  GtkWidget *entries_table;
  gint num_entries;
  GtkWidget *worksheet_combo;
  GtkWidget *columns_list;

  gboolean required[NENTRIES];

  GtkPlotArrayList *arrays;

} SGcolumnsDialog;

static void select_columns			(GtkWidget *widget, 
						 gpointer data);
static void init_columns			(SGcolumnsDialog *dialog);
static gint set_column				(GtkWidget *widget, 
						 gpointer data);


static gint
get_column_from_name(SGcolumnsDialog *dialog, const gchar *name)
{
  SGworksheet *worksheet;
  const gchar *worksheet_name = NULL;
  gint n = 0;

  worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dialog->worksheet_combo)->entry));

/*
{
  GList *list = dialog->app->worksheets->list;
  while(list){
    SGlistChild *child = (SGlistChild*)list->data;
    SGworksheet *w = SG_WORKSHEET(child->object);
    printf("%s %s\n",w->name,child->name);
    list = list->next;
  }
}
*/
  

  worksheet = SG_WORKSHEET(sg_list_get(dialog->app->worksheets, worksheet_name));
  for(n = 0; n <= GTK_SHEET(worksheet)->maxcol; n++){
    if(strcmp(name, GTK_SHEET(worksheet)->column[n].name) == 0) return n;
  }

  return -1;
}

static void
sg_edit_columns_update(SGpropertyDialog *pdialog, gpointer data)
{
  SGworksheet *worksheet;
  const gchar *worksheet_name;
  gint i;
  gboolean incomplete = FALSE;
  GtkArg args[13];
  SGcolumnsDialog *dialog = (SGcolumnsDialog *)data;
  GList *dimensions = dialog->dataset->constructor->arrays->arrays;
                                                      
  for(i = 1; i < 13; i++){
    args[i].type = GTK_TYPE_INT;
    GTK_VALUE_INT(args[i]) = -1;
  }

/* copy arrays from constructor */
  i = 1;
  while(dimensions){
    GtkPlotArray *ref;
    const gchar *column_name;
                                                                                
    column_name = gtk_entry_get_text(GTK_ENTRY(dialog->column_entry[i-1]));
    ref = GTK_PLOT_ARRAY(dimensions->data);

    if(column_name && strlen(column_name) > 0) {
      GTK_VALUE_INT(args[i]) = get_column_from_name(dialog, column_name);
      if(GTK_VALUE_INT(args[i]) == -1 && ref->required){
        incomplete = TRUE;
        break;
      }
    } else {
      if(ref->required){
        incomplete = TRUE;
        break;
      }
    }
    dimensions = dimensions->next;
    i++;
  }

  if(incomplete){
    dialog->dataset = NULL;
    return;
  }

  worksheet_name = gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(dialog->worksheet_combo)->entry));

  worksheet = SG_WORKSHEET(sg_list_get(dialog->app->worksheets, worksheet_name));

  if(!worksheet){
    dialog->dataset = NULL;
    return;
  }

  args[0].type = GTK_TYPE_POINTER;
  args[0].name = "SGdatasetWorksheet::worksheet";
  GTK_VALUE_POINTER(args[0]) = worksheet;

  args[1].name = "SGdatasetWorksheet::col_00";
  args[2].name = "SGdatasetWorksheet::col_01";
  args[3].name = "SGdatasetWorksheet::col_02";
  args[4].name = "SGdatasetWorksheet::col_03";
  args[5].name = "SGdatasetWorksheet::col_04";
  args[6].name = "SGdatasetWorksheet::col_05";
  args[7].name = "SGdatasetWorksheet::col_06";
  args[8].name = "SGdatasetWorksheet::col_07";
  args[9].name = "SGdatasetWorksheet::col_08";
  args[10].name = "SGdatasetWorksheet::col_09";
  args[11].name = "SGdatasetWorksheet::col_10";
  args[12].name = "SGdatasetWorksheet::col_11";

  g_object_set(G_OBJECT(dialog->dataset), args[0].name, worksheet, NULL);
  for(i = 0; i < 13; i++) 
    g_object_set(G_OBJECT(dialog->dataset), args[i].name, GTK_VALUE_INT(args[i]), NULL);
  sg_dataset_refresh(SG_DATASET(dialog->dataset));
}

static void
update_columns(SGcolumnsDialog *dialog)
{
  gint i, nrows;
  SGworksheet *worksheet = NULL;
  GList *list;
  gchar name[DATASET_NAME_LENGTH];
  const gchar *text;
  GtkEntry *entry;

  entry = GTK_ENTRY(GTK_COMBO(dialog->worksheet_combo)->entry);

  nrows = GTK_CLIST(dialog->columns_list)->rows;
  for(i = 0; i <= nrows; i++){
    gtk_clist_remove(GTK_CLIST(dialog->columns_list), 0);
  }

  text = gtk_entry_get_text(entry);
  list = dialog->app->worksheets->list;
  while(list){
    SGlistChild *child = (SGlistChild *)list->data;
    worksheet = SG_WORKSHEET(child->object);
    if(strcmp(text, GTK_SHEET(worksheet)->name) == 0) break;
    list = list->next;
  }

  for(i = 0; i <= GTK_SHEET(worksheet)->maxcol; i++){
     if(!GTK_SHEET(worksheet)->column[i].name)
        sprintf(name,"%d",i);
     else
        sprintf(name,"%s",GTK_SHEET(worksheet)->column[i].name);
     text=g_strdup(name);
     gtk_clist_append(GTK_CLIST(dialog->columns_list), &text);
  };

  for(i = 0; i < dialog->num_entries; i++)
    if(dialog->column_entry[i]) gtk_entry_set_text(GTK_ENTRY(dialog->column_entry[i]), "");

}

SGdataset *
sg_edit_columns_dialog (SGapplication *app, SGdataset *dataset)
{
  GtkWidget *frame;
  GtkWidget *columns_frame;
  GtkWidget *label;
  GtkWidget *window;
  GtkWidget *box, *vbox, *sw;
  GList *list = NULL;
  GtkWidget *item;
  SGworksheet *worksheet;
  SGcolumnsDialog *dialog;
  SGdataset *new_dataset;
  SGdialogButton return_val;
  gint i;

  dialog = g_new0(SGcolumnsDialog, 1);
  dialog->dataset = dataset;
  dialog->arrays = dataset->constructor->arrays;
  dialog->app = app;
  dialog->constructor = dataset->constructor;

  /* Create widgets */

  frame = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, FALSE);
  SG_PROPERTY_DIALOG(frame)->apply = sg_edit_columns_update; 
  SG_PROPERTY_DIALOG(frame)->ok = sg_edit_columns_update; 

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

/*********************************************************************/
  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new( _("Dataset Style:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET(gtk_pixmap_new(dataset->constructor->pixmap->pixmap, NULL)), FALSE, FALSE, 0);

  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new( _("Select Worksheet:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  dialog->worksheet_combo = gtk_combo_new();
  gtk_box_pack_end (GTK_BOX (box), dialog->worksheet_combo, FALSE, FALSE, 0);
  gtk_widget_show(box);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->worksheet_combo)->entry), FALSE);
  list = app->worksheets->list;
  while(list){
     SGlistChild *child = (SGlistChild *)list->data;
     worksheet = SG_WORKSHEET(child->object);
     item = gtk_list_item_new_with_label(worksheet->name);
     gtk_widget_show(item);
     gtk_container_add(GTK_CONTAINER(GTK_COMBO(dialog->worksheet_combo)->list), item);
     list = list->next;
  }

/*********************************************************************/
  columns_frame = gtk_frame_new( _("Columns") );
  gtk_frame_set_shadow_type (GTK_FRAME (columns_frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), columns_frame, FALSE, FALSE, 0);

  dialog->entries_table = NULL;
  dialog->entries_box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->entries_box), 5);
  gtk_container_add (GTK_CONTAINER (columns_frame), dialog->entries_box);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 180, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (dialog->entries_box), sw, FALSE, FALSE, 0);

  dialog->columns_list = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), dialog->columns_list);

  for(i = 0; i < NENTRIES; i++) dialog->column_entry[i] = NULL;

  init_columns(dialog);

/********************************************************************/
  gtk_signal_connect(GTK_OBJECT(GTK_COMBO(dialog->worksheet_combo)->entry), 
                     "changed", GTK_SIGNAL_FUNC(select_columns), dialog);

  window = sg_dialog_new( _("SciGraphica: Edit Columns"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_SPREAD);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
  gtk_widget_show_all(frame);

  sg_dialog_add(window, SG_PROPERTY_DIALOG(frame));
  gtk_widget_show_all(frame);

  if(app)
    return_val = sg_dialog_run(window, GTK_OBJECT(app));
  else
    return_val = sg_dialog_run(window, NULL);

  if(return_val == SG_BUTTON_CANCEL || return_val == SG_BUTTON_CLOSE){
     dialog->dataset = NULL;
  }

  new_dataset = dialog->dataset;
  g_free(dialog);

  return new_dataset;
}

static gint
set_column(GtkWidget *widget, gpointer data)
{
  GList *selection;
  SGcolumnsDialog *dialog;
  GtkWidget *entry = NULL;
  const gchar *text;
  gint row;
  gint i;

  dialog = (SGcolumnsDialog *)data;

  selection = GTK_CLIST(dialog->columns_list)->selection;
  if(!selection) return TRUE;

  for(i = 0; i < dialog->num_entries; i++){
    if(widget == dialog->right_button[i]){
      entry = dialog->column_entry[i];
      break;
    }
  }
  text = gtk_entry_get_text(GTK_ENTRY(entry));
  if(strlen(text) > 0) return TRUE;
  
  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(dialog->columns_list), row, 0, &text);

  gtk_entry_set_text(GTK_ENTRY(entry), text);
  gtk_clist_remove(GTK_CLIST(dialog->columns_list), row);
  return TRUE;
}

static gint
restore_column(GtkWidget *widget, gpointer data)
{
  GtkWidget *entry = NULL;
  SGcolumnsDialog *dialog;
  const gchar *text;
  gint col;
  gint i;

  dialog = (SGcolumnsDialog *)data;

  for(i = 0; i < dialog->num_entries; i++){
    if(widget == dialog->left_button[i]){
      entry = dialog->column_entry[i];
      break;
    }
  }
  text = gtk_entry_get_text(GTK_ENTRY(entry));
  if(strlen(text) == 0) return TRUE;

  col = get_column_from_name(dialog, text);
  gtk_clist_insert(GTK_CLIST(dialog->columns_list), col, &text);

  gtk_entry_set_text(GTK_ENTRY(entry), "");
  return TRUE;
}

static void
init_columns(SGcolumnsDialog *dialog)
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  GList *list;
  gint nrows = 0;
  gchar label[NENTRIES][100];
  GParamSpec **args;
  gint nargs;
  SGworksheet *worksheet = NULL;
  gchar *arg_name[] = {"col_00","col_01","col_02","col_03","col_04","col_05","col_06","col_07","col_08","col_09","col_10","col_11"};
  gint i = 0;

  if(!dialog->dataset) return;

  args = g_object_class_list_properties(G_OBJECT_GET_CLASS(G_OBJECT(dialog->dataset)), &nargs);
  worksheet = SG_DATASET_WORKSHEET(dialog->dataset)->worksheet;

  update_columns(dialog);

  if(dialog->entries_table && GTK_IS_WIDGET(dialog->entries_table))
     gtk_container_remove(GTK_CONTAINER(dialog->entries_box), dialog->entries_table);

  dialog->num_entries = g_list_length(dialog->arrays->arrays);
                                                                                
  dialog->entries_table = gtk_table_new(dialog->num_entries, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->entries_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(dialog->entries_table), 5);
                                                                                
  list = dialog->arrays->arrays;
  nrows = 0;
  while(list){
    GtkPlotArray *dim = GTK_PLOT_ARRAY(list->data);

    dialog->required[nrows] = dim->required;

    sprintf(label[nrows], "%s:", dim->label);
    button_box = gtk_hbox_new(TRUE, 0);
    gtk_table_attach_defaults(GTK_TABLE (dialog->entries_table), button_box,
                              0, 1, nrows, nrows + 1);
    
    dialog->left_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->left_button[nrows]),
                      gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->left_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->left_button[nrows], FALSE, FALSE, 0);
                                                                                
    dialog->right_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->right_button[nrows]),
                      gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->right_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->right_button[nrows], FALSE,
FALSE, 0);
                                                                                
    column_label = gtk_label_new(label[nrows]);
    if(dim->required){
      GtkStyle *style = gtk_style_new();
      style->fg[GTK_STATE_NORMAL].red = 56000;
      style->fg[GTK_STATE_NORMAL].green = 0;
      style->fg[GTK_STATE_NORMAL].blue = 0;
      gtk_widget_set_style(column_label, style);
    }
    gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), column_label,
                              1, 2, nrows, nrows + 1);
    dialog->column_entry[nrows] = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(dialog->column_entry[nrows]), FALSE);
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), dialog->column_entry[nrows],
                              2, 3, nrows, nrows + 1);

    if(worksheet){
      gint arg_value;
      g_object_get(G_OBJECT(dialog->dataset), arg_name[nrows], &arg_value, NULL);
      if(arg_value != -1){
        gint row;
        gchar *text = GTK_SHEET(worksheet)->column[arg_value].name;
        const gchar *ctext;

        gtk_entry_set_text(GTK_ENTRY(dialog->column_entry[nrows]), text);

        for(row = 0; row < GTK_CLIST(dialog->columns_list)->rows; row++){
          gtk_clist_get_text(GTK_CLIST(dialog->columns_list), row, 0, &ctext);
          if(strcmp(text, ctext) == 0)
            gtk_clist_remove(GTK_CLIST(dialog->columns_list), row);
        }
      }
    }

    gtk_signal_connect(GTK_OBJECT(dialog->left_button[nrows]), "clicked",
                       (GtkSignalFunc) restore_column, dialog);
    gtk_signal_connect(GTK_OBJECT(dialog->right_button[nrows]), "clicked",
                       (GtkSignalFunc) set_column, dialog);

    nrows++;
    list = list->next;
  }

  gtk_box_pack_start(GTK_BOX(dialog->entries_box), dialog->entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(dialog->entries_table);

  g_free(args);


  for(i = 0; i < nrows; i++){
    if(dialog->required[i]){
      gtk_clist_select_row(GTK_CLIST(dialog->columns_list), 0, 0);
      set_column(dialog->right_button[i], dialog);
    }
  }

}


static void
select_columns(GtkWidget *widget, gpointer data)
{
  SGcolumnsDialog *dialog;
  GtkWidget *column_label;
  GtkWidget *button_box;
  GList *list = NULL;
  gint nrows = 0;
  gchar label[NENTRIES][100];
  gint i;

  dialog = (SGcolumnsDialog *)data;

  update_columns(dialog);

  if(dialog->entries_table && GTK_IS_WIDGET(dialog->entries_table))
     gtk_container_remove(GTK_CONTAINER(dialog->entries_box), dialog->entries_table);

  dialog->num_entries = g_list_length(dialog->dataset->constructor->arrays->arrays);

  dialog->entries_table = gtk_table_new(dialog->num_entries, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->entries_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(dialog->entries_table), 5);

  list = dialog->dataset->constructor->arrays->arrays;
  nrows = 0;
  while(list){
    GtkPlotArray *dim = GTK_PLOT_ARRAY(list->data);
    sprintf(label[nrows], "%s:", dim->label);
    button_box = gtk_hbox_new(TRUE, 0);
    gtk_table_attach_defaults(GTK_TABLE (dialog->entries_table), button_box,
                              0, 1, nrows, nrows + 1);
                                                                                
    dialog->left_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->left_button[nrows]),
                      gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->left_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->left_button[nrows], FALSE, FALSE, 0);
                                                                                
    dialog->right_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->right_button[nrows]),
                      gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->right_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->right_button[nrows], FALSE,
FALSE, 0);
                                                                                
    column_label = gtk_label_new(label[nrows]);
    if(dim->required){
      GtkStyle *style = gtk_style_new();
      style->fg[GTK_STATE_NORMAL].red = 56000;
      style->fg[GTK_STATE_NORMAL].green = 0;
      style->fg[GTK_STATE_NORMAL].blue = 0;
      gtk_widget_set_style(column_label, style);
    }
    gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), column_label,
                              1, 2, nrows, nrows + 1);
    dialog->column_entry[nrows] = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(dialog->column_entry[nrows]), FALSE);
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), dialog->column_entry[nrows],
                              2, 3, nrows, nrows + 1);
                                                                                
    gtk_signal_connect(GTK_OBJECT(dialog->left_button[nrows]), "clicked",
                       (GtkSignalFunc) restore_column, dialog);
    gtk_signal_connect(GTK_OBJECT(dialog->right_button[nrows]), "clicked",
                       (GtkSignalFunc) set_column, dialog);
                                                                                
    nrows++;
    list = list->next;
  }
                                                                                
  gtk_box_pack_start(GTK_BOX(dialog->entries_box), dialog->entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(dialog->entries_table);

  for(i = 0; i < dialog->num_entries; i++){
    if(dialog->required[i]){
      gtk_clist_select_row(GTK_CLIST(dialog->columns_list), 0, 0);
      set_column(dialog->right_button[i], dialog);
    }
  }
}

