/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include "sg_dataset_worksheet.h"

#define NAME_LEN 100
#define NCOLS 12

#define P_(string) string

static void sg_dataset_worksheet_class_init	(SGdatasetWorksheetClass *klass);
static void sg_dataset_worksheet_init		(SGdatasetWorksheet *dataset);
static void sg_dataset_worksheet_finalize	(GObject *object);
static void sg_dataset_worksheet_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_worksheet_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_worksheet_refresh	(SGdataset *dataset);
static void update_arrays			(SGdataset *dataset);
static gboolean sg_dataset_worksheet_connected	(SGdataset *dataset, 
						 gpointer data);
enum {
  ARG_0,
  ARG_WORKSHEET,
  ARG_COL_00,
  ARG_COL_01,
  ARG_COL_02,
  ARG_COL_03,
  ARG_COL_04,
  ARG_COL_05,
  ARG_COL_06,
  ARG_COL_07,
  ARG_COL_08,
  ARG_COL_09,
  ARG_COL_10,
  ARG_COL_11,
};

static SGdatasetClass *parent_class = NULL;

GType
sg_dataset_worksheet_get_type (void)
{
  static GType dataset_type = 0;
                                                                                
  if (!dataset_type)
    {
      static const GTypeInfo dataset_info =
      {
        sizeof (SGdatasetWorksheetClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_dataset_worksheet_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGdatasetWorksheet),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_dataset_worksheet_init,
        NULL,
      };
                                                                                
      dataset_type = g_type_register_static (GTK_TYPE_SG_DATASET, "SGdatasetWorksheet", &dataset_info, 0);
    }
                                                                                
  return dataset_type;
}

static void
sg_dataset_worksheet_class_init (SGdatasetWorksheetClass *klass)
{
  GObjectClass *object_class;
  SGdatasetClass *dataset_class;

  parent_class = (SGdatasetClass *) g_type_class_peek_parent(klass);
  object_class = (GObjectClass *) klass;
  dataset_class = (SGdatasetClass *) klass;

  object_class->finalize = sg_dataset_worksheet_finalize;
  object_class->set_property = sg_dataset_worksheet_set_property;
  object_class->get_property = sg_dataset_worksheet_get_property;

  g_object_class_install_property (object_class,
                           ARG_WORKSHEET,
  g_param_spec_pointer ("worksheet",
                           P_("Worksheet"),
                           P_("Worksheet"),
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_00,
  g_param_spec_int ("col_00",
                           P_("Column 0"),
                           P_("Column 0"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_01,
  g_param_spec_int ("col_01",
                           P_("Column 1"),
                           P_("Column 1"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_02,
  g_param_spec_int ("col_02",
                           P_("Column 2"),
                           P_("Column 2"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_03,
  g_param_spec_int ("col_03",
                           P_("Column 3"),
                           P_("Column 3"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_04,
  g_param_spec_int ("col_04",
                           P_("Column 4"),
                           P_("Column 4"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_05,
  g_param_spec_int ("col_05",
                           P_("Column 5"),
                           P_("Column 5"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_06,
  g_param_spec_int ("col_06",
                           P_("Column 6"),
                           P_("Column 6"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_07,
  g_param_spec_int ("col_07",
                           P_("Column 7"),
                           P_("Column 7"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_08,
  g_param_spec_int ("col_08",
                           P_("Column 8"),
                           P_("Column 8"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_09,
  g_param_spec_int ("col_09",
                           P_("Column 9"),
                           P_("Column 9"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_10,
  g_param_spec_int ("col_10",
                           P_("Column 10"),
                           P_("Column 10"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_COL_11,
  g_param_spec_int ("col_11",
                           P_("Column 11"),
                           P_("Column 11"),
			   -1,G_MAXINT,-1,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));

  dataset_class->refresh = sg_dataset_worksheet_refresh;
  dataset_class->connected = sg_dataset_worksheet_connected;
}

SGdataset *
sg_dataset_worksheet_new ()
{
  SGdataset *dataset;

  dataset = SG_DATASET(g_object_new(sg_dataset_worksheet_get_type(),NULL));

  return dataset;
}

static void
sg_dataset_worksheet_get_property (GObject      *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetWorksheet *dataset;
                                                                                
  dataset = SG_DATASET_WORKSHEET (object);
                                                                                
  switch(prop_id){
    case ARG_WORKSHEET:
      g_value_set_pointer(value,dataset->worksheet);
      break;
    case ARG_COL_00:
      g_value_set_int(value,dataset->col[0]);
      break;
    case ARG_COL_01:
      g_value_set_int(value,dataset->col[1]);
      break;
    case ARG_COL_02:
      g_value_set_int(value,dataset->col[2]);
      break;
    case ARG_COL_03:
      g_value_set_int(value,dataset->col[3]);
      break;
    case ARG_COL_04:
      g_value_set_int(value,dataset->col[4]);
      break;
    case ARG_COL_05:
      g_value_set_int(value,dataset->col[5]);
      break;
    case ARG_COL_06:
      g_value_set_int(value,dataset->col[6]);
      break;
    case ARG_COL_07:
      g_value_set_int(value,dataset->col[7]);
      break;
    case ARG_COL_08:
      g_value_set_int(value,dataset->col[8]);
      break;
    case ARG_COL_09:
      g_value_set_int(value,dataset->col[9]);
      break;
    case ARG_COL_10:
      g_value_set_int(value,dataset->col[10]);
      break;
    case ARG_COL_11:
      g_value_set_int(value,dataset->col[11]);
      break;
  }
}
                                                                                
static void
sg_dataset_worksheet_set_property (GObject      *object,
                         guint            prop_id,
                         const GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetWorksheet *dataset;
                                                                                
  dataset = SG_DATASET_WORKSHEET (object);
                                                                                
  switch(prop_id){
    case ARG_WORKSHEET:
      dataset->worksheet = SG_WORKSHEET(g_value_get_pointer(value));
      break;
    case ARG_COL_00:
      dataset->col[0] = g_value_get_int(value);
      break;
    case ARG_COL_01:
      dataset->col[1] = g_value_get_int(value);
      break;
    case ARG_COL_02:
      dataset->col[2] = g_value_get_int(value);
      break;
    case ARG_COL_03:
      dataset->col[3] = g_value_get_int(value);
      break;
    case ARG_COL_04:
      dataset->col[4] = g_value_get_int(value);
      break;
    case ARG_COL_05:
      dataset->col[5] = g_value_get_int(value);
      break;
    case ARG_COL_06:
      dataset->col[6] = g_value_get_int(value);
      break;
    case ARG_COL_07:
      dataset->col[7] = g_value_get_int(value);
      break;
    case ARG_COL_08:
      dataset->col[8] = g_value_get_int(value);
      break;
    case ARG_COL_09:
      dataset->col[9] = g_value_get_int(value);
      break;
    case ARG_COL_10:
      dataset->col[10] = g_value_get_int(value);
      break;
    case ARG_COL_11:
      dataset->col[11] = g_value_get_int(value);
      break;
  }
}

static void
sg_dataset_worksheet_init(SGdatasetWorksheet *dataset)
{
  gint i;
  dataset->worksheet = NULL;

  for(i = 0; i < NCOLS; i++) {dataset->col[i] = -1; }
  sg_dataset_set_description(SG_DATASET(dataset), _("Using worksheet columns"));
}

void
sg_dataset_worksheet_set_worksheet  (SGdatasetWorksheet *dataset,
			             SGworksheet *worksheet)
{
  dataset->worksheet = worksheet;
  sg_dataset_refresh(SG_DATASET(dataset));
}

void
sg_dataset_worksheet_set_column  (SGdatasetWorksheet *dataset,
		   	          gint col_index, gint column)
{
  dataset->col[col_index] = column;
  sg_dataset_refresh(SG_DATASET(dataset));
}

static void
sg_dataset_worksheet_refresh(SGdataset *dataset)
{
  gint numrows = 0;
  GList *list;

  update_arrays(dataset);
  list = dataset->arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    if(array->size > numrows) numrows = array->size;
    list = list->next;
  }   
 
  list = dataset->children;
  while(list){
    GtkPlotData *child = GTK_PLOT_DATA(list->data); 
    child->is_iterator = FALSE;
    child->iterator = NULL;
    child->iterator_mask = 0;

    child->num_points = numrows;

    if(GTK_IS_PLOT_SURFACE(child)){ 
      GTK_PLOT_SURFACE(child)->recalc_dt = TRUE;
    }

    gtk_plot_data_update(child); 

    list = list->next;
  }
}

static void
sg_dataset_worksheet_finalize(GObject *object)
{
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
update_arrays   (SGdataset *dataset)
{
  SGdatasetWorksheet *real_data;
  SGworksheet *worksheet;
  gint col;
  GtkPlotArray *array = NULL;
  GtkPlotArray *new_array = NULL;
  GList *list;

  real_data = SG_DATASET_WORKSHEET(dataset);
  worksheet = real_data->worksheet;

  col = 0;
  list = dataset->arrays->arrays;
  while(list){
    array = GTK_PLOT_ARRAY(list->data);

    gtk_plot_array_free(array);

    if(real_data->col[col] != -1){
      new_array = sg_worksheet_get_column_data(worksheet, real_data->col[col], array->type);
      if(new_array){
        array->data = new_array->data;
        array->size = new_array->size;

/*
printf("ARRAY %s %d %d\n",array->name,real_data->col[col],new_array->size);
{gint i;
 for(i = 0; i < new_array->size; i++) printf("ROW %d %f\n",i,array->data.data_double[i]);
}
*/
        new_array->own_data = FALSE;
        g_object_unref(G_OBJECT(new_array));
      }
    }

    col++;
    list = list->next;
  }
}

static gboolean
sg_dataset_worksheet_connected(SGdataset *dataset, gpointer object)
{
  SGdatasetWorksheet *data = SG_DATASET_WORKSHEET(dataset);
 
  if(data->worksheet && data->worksheet == object) return TRUE;

  return FALSE;
}
