/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include "sg_dataset_python.h"
#include "sg_python_expr.h"

#define P_(string) string

static void sg_dataset_python_class_init	(SGdatasetPythonClass *klass);
static void sg_dataset_python_init		(SGdatasetPython *dataset);
static void sg_dataset_python_finalize		(GObject *object);
static void sg_dataset_python_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_python_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_python_refresh		(SGdataset *dataset);
static void sg_dataset_python_set_iterator	(SGdataset *dataset);
static GtkPlotData *sg_dataset_python_new_child	(SGdataset *dataset);

enum {
  ARG_0,
  ARG_X,
  ARG_Y,
  ARG_Z,
  ARG_A,
  ARG_DA,
  ARG_DZ,
  ARG_DX,
  ARG_DY,
  ARG_LABELS,
};

static SGdatasetClass *parent_class = NULL;

GType
sg_dataset_python_get_type (void)
{
  static GType dataset_type = 0;
                                                                                
  if (!dataset_type)
    {
      static const GTypeInfo dataset_info =
      {
        sizeof (SGdatasetPythonClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_dataset_python_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGdatasetPython),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_dataset_python_init,
        NULL,
      };
                                                                                
      dataset_type = g_type_register_static (GTK_TYPE_SG_DATASET, "SGdatasetPython", &dataset_info, 0);
    }
                                                                                
  return dataset_type;
}

static void
sg_dataset_python_class_init (SGdatasetPythonClass *klass)
{
  GObjectClass *object_class;
  SGdatasetClass *dataset_class;

  parent_class = (SGdatasetClass *) g_type_class_peek_parent(klass);
  object_class = (GObjectClass *) klass;
  dataset_class = (SGdatasetClass *) klass;

  object_class->finalize = sg_dataset_python_finalize;
  object_class->set_property = sg_dataset_python_set_property;
  object_class->get_property = sg_dataset_python_get_property;

  g_object_class_install_property (object_class,
                           ARG_X,
  g_param_spec_string ("exp_x",
                           P_("X"),
                           P_("X"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_Y,
  g_param_spec_string ("exp_y",
                           P_("Y"),
                           P_("Y"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_Z,
  g_param_spec_string ("exp_z",
                           P_("Z"),
                           P_("Z"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_A,
  g_param_spec_string ("exp_a",
                           P_("A"),
                           P_("A"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_DX,
  g_param_spec_string ("exp_dx",
                           P_("DX"),
                           P_("DX"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_DY,
  g_param_spec_string ("exp_dy",
                           P_("DY"),
                           P_("DY"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_DZ,
  g_param_spec_string ("exp_dz",
                           P_("DZ"),
                           P_("DZ"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_DA,
  g_param_spec_string ("exp_da",
                           P_("DA"),
                           P_("DA"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (object_class,
                           ARG_LABELS,
  g_param_spec_string ("exp_labels",
                           P_("Labels"),
                           P_("Labels"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
                                                                                
  dataset_class->refresh = sg_dataset_python_refresh;
  dataset_class->set_iterator = sg_dataset_python_set_iterator;
  dataset_class->new_child = sg_dataset_python_new_child;
}

SGdataset *
sg_dataset_python_new ()
{
  SGdataset *dataset;

  dataset = SG_DATASET(g_object_new(sg_dataset_python_get_type(),NULL));

  return dataset;
}

static void
sg_dataset_python_get_property (GObject      *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetPython *dataset;
                                                                                
  dataset = SG_DATASET_PYTHON (object);
                                                                                
  switch(prop_id){
    case ARG_X:
      g_value_set_string(value,dataset->p_exp[0]);
      break;
    case ARG_Y:
      g_value_set_string(value,dataset->p_exp[1]);
      break;
    case ARG_Z:
      g_value_set_string(value,dataset->p_exp[2]);
      break;
    case ARG_A:
      g_value_set_string(value,dataset->p_exp[3]);
      break;
    case ARG_DX:
      g_value_set_string(value,dataset->p_exp[4]);
      break;
    case ARG_DY:
      g_value_set_string(value,dataset->p_exp[5]);
      break;
    case ARG_DZ:
      g_value_set_string(value,dataset->p_exp[6]);
      break;
    case ARG_DA:
      g_value_set_string(value,dataset->p_exp[7]);
      break;
    case ARG_LABELS:
      g_value_set_string(value,dataset->p_exp[8]);
      break;
  }
}
                                                                                
static void
sg_dataset_python_set_property (GObject      *object,
                         guint            prop_id,
                         const GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetPython *dataset;
                                                                                
  dataset = SG_DATASET_PYTHON (object);
                                                                                
  switch(prop_id){
    case ARG_X:
      if(dataset->p_exp[0]) g_free(dataset->p_exp[0]);
      dataset->p_exp[0] = g_strdup(g_value_get_string(value));
      break;
    case ARG_Y:
      if(dataset->p_exp[1]) g_free(dataset->p_exp[1]);
      dataset->p_exp[1] = g_strdup(g_value_get_string(value));
      break;
    case ARG_Z:
      if(dataset->p_exp[2]) g_free(dataset->p_exp[2]);
      dataset->p_exp[2] = g_strdup(g_value_get_string(value));
      break;
    case ARG_A:
      if(dataset->p_exp[3]) g_free(dataset->p_exp[3]);
      dataset->p_exp[3] = g_strdup(g_value_get_string(value));
      break;
    case ARG_DX:
      if(dataset->p_exp[4]) g_free(dataset->p_exp[4]);
      dataset->p_exp[4] = g_strdup(g_value_get_string(value));
      break;
    case ARG_DY:
      if(dataset->p_exp[5]) g_free(dataset->p_exp[5]);
      dataset->p_exp[5] = g_strdup(g_value_get_string(value));
      break;
    case ARG_DZ:
      if(dataset->p_exp[6]) g_free(dataset->p_exp[6]);
      dataset->p_exp[6] = g_strdup(g_value_get_string(value));
      break;
    case ARG_DA:
      if(dataset->p_exp[7]) g_free(dataset->p_exp[7]);
      dataset->p_exp[7] = g_strdup(g_value_get_string(value));
      break;
    case ARG_LABELS:
      if(dataset->p_exp[8]) g_free(dataset->p_exp[8]);
      dataset->p_exp[8] = g_strdup(g_value_get_string(value));
      break;
  }
}

static void
sg_dataset_python_init(SGdatasetPython *dataset)
{
  int i;
  for(i = 0; i < 9; i++) dataset->p_exp[i] = NULL;
}

static GtkPlotData * 
sg_dataset_python_new_child	(SGdataset *dataset)
{
  GtkPlotData *new_data;

  new_data = SG_DATASET_CLASS(parent_class)->new_child(dataset);

  new_data->iterator = NULL;
  new_data->is_iterator = FALSE;

  return new_data;
}


static void
sg_dataset_python_set_iterator(SGdataset *dataset)
{
  GList *list;

  list = dataset->children;
  while(list){
    GtkPlotData *real_data = GTK_PLOT_DATA(list->data);
    real_data->is_iterator = FALSE;
    real_data->iterator = NULL;

    list = list->next;
  }
}

void
sg_dataset_python_set   (SGdatasetPython *dataset,
			 const gchar *p_exp[9])
{
  SGdatasetPython *real_data;
  int i;

  real_data = SG_DATASET_PYTHON(dataset);

  for(i = 0; i < 9; i++){
    if(SG_DATASET_PYTHON(dataset)->p_exp[i]) 
      g_free(SG_DATASET_PYTHON(dataset)->p_exp[i]);
    SG_DATASET_PYTHON(dataset)->p_exp[i] = NULL;

    if(SG_DATASET_PYTHON(real_data)->p_exp[i]) 
      g_free(SG_DATASET_PYTHON(real_data)->p_exp[i]);
    SG_DATASET_PYTHON(real_data)->p_exp[i] = NULL;

    if(p_exp[i]){
      SG_DATASET_PYTHON(dataset)->p_exp[i] = g_strdup(p_exp[i]);
      if(real_data != dataset) 
        SG_DATASET_PYTHON(real_data)->p_exp[i] = g_strdup(p_exp[i]);
    }
  }

  sg_dataset_python_refresh(SG_DATASET(dataset));
}

static void
sg_dataset_python_refresh(SGdataset *dataset)
{
  SGdatasetPython *real_data;
  gdouble *x[8];
  gchar **labels = NULL;
  gint i, dim, num, maxdim = 0, len[4];
  GList *list;

  real_data = SG_DATASET_PYTHON(dataset);
  sg_dataset_free_points(SG_DATASET(dataset));

  len[0] = -1;
  for (i = 0; i < 8; i++)
    {
       if (real_data->p_exp[i] && strlen(real_data->p_exp[i])){
         x[i] = sg_eval_expr_double(real_data->p_exp[i], &len[i], len[0]);
       }else{
         x[i] = NULL;
       }
       if (!len[0]) return;
    }

  if (real_data->p_exp[i] && strlen(real_data->p_exp[i]))
    labels = sg_eval_expr_string(real_data->p_exp[i], &num);


  dim = 0;
  list = dataset->arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data); 

    gtk_plot_array_free(array);
    if(array->type == GTK_TYPE_DOUBLE){
      array->data.data_double = x[dim];
      array->size = len[dim];
    } else {
      array->data.data_string = labels;
      array->size = num;
    }

    array->own_data = TRUE;
    if(array->size > maxdim) maxdim = array->size;
    dim++;
                                                                                
    list = list->next;
  }

  list = dataset->children;
  while(list){
    GtkPlotData *child = GTK_PLOT_DATA(list->data);

    child->is_iterator = FALSE;
    child->iterator = NULL;
    child->iterator_mask = 0;
                                                                                
    child->num_points = maxdim;

    gtk_plot_data_update(child);
    list = list->next;
  }
}

static void
sg_dataset_python_finalize(GObject *object)
{
  int i;

  for(i = 0; i < 9; i++){
    if(SG_DATASET_PYTHON(object)->p_exp[i]) {
      g_free(SG_DATASET_PYTHON(object)->p_exp[i]);
      SG_DATASET_PYTHON(object)->p_exp[i] = NULL;
    }
  }
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

