/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_worksheet_file.h"
#include "pixmaps/sg_small.xpm"

static void write_header_xml			(SGworksheetfile *file); 
static void write_footer_xml			(SGworksheetfile *file); 
static void write_column_title_xml		(SGworksheetfile *file, 
                                                 gint col); 
static void write_cell_xml			(SGworksheetfile *file,  
						 gint row, gint col);

void
SGworksheet_xml_export_init(SGplugin *_plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgw");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
SGworksheet_xml_export      (SGpluginFile *plugin, 
                              const gchar *filename,
			      FILE *opened,
                              GObject **object, gpointer data)
{  
  SGworksheetfile *file;
  SGworksheet *worksheet;
  GtkSheet *sheet;

  worksheet = SG_WORKSHEET(*object);
  sheet = GTK_SHEET(worksheet);
  file = sg_worksheet_file_new(worksheet, filename);

  if(opened)
    file->stream = opened;
  else
    file->stream = sg_file_open(filename, "wb");
    
  if (!file->stream){
       g_warning( _("ERROR: Cannot write to file: %s"), filename);
       sg_worksheet_file_destroy(file);
       return FALSE;
  } 

  if(!opened)  sg_file_printf(file->stream,"<?xml version=\"1.0\"?>\n");

  file->write_header = write_header_xml; 
  file->write_footer = write_footer_xml; 
  file->write_col_title = write_column_title_xml; 
  file->write_cell = write_cell_xml; 

  file->titlecolor = g_strdup("\"BBBBBB\"");
  file->cellcolor = g_strdup("\"FFFFFF\"");

  sg_worksheet_file_export(file, NULL);

  if(!opened) sg_file_close(file->stream);

  sg_worksheet_file_destroy(file);

  return TRUE;
}

static void
write_header_xml(SGworksheetfile *file)
{
  SGworksheet *worksheet = file->worksheet;
  gchar *s;

  if(GTK_IS_SG_MATRIX(worksheet))
    sg_file_printf(file->stream,"<sgw:Matrix xmlns:sgw=\"http://scigraphica.sourceforge.net\">\n");
  else
    sg_file_printf(file->stream,"<sgw:Worksheet xmlns:sgw=\"http://scigraphica.sourceforge.net\">\n");

  sg_file_printf(file->stream, "  <sgw:Summary>\n"); 
  sg_file_printf(file->stream, "    <sgw:Item>\n"); 
  sg_file_printf(file->stream, "      <sgw:name>application</sgw:name>\n"); 
  sg_file_printf(file->stream, "      <sgw:val-string>scigraphica</sgw:val-string>\n"); 
  sg_file_printf(file->stream, "    </sgw:Item>\n"); 
  sg_file_printf(file->stream, "    <sgw:Item>\n"); 
  sg_file_printf(file->stream, "      <sgw:name>author</sgw:name>\n"); 
  sg_file_printf(file->stream, "      <sgw:val-string>%s</sgw:val-string>\n", g_get_real_name()); 
  sg_file_printf(file->stream, "    </sgw:Item>\n"); 
  sg_file_printf(file->stream, "  </sgw:Summary>\n"); 
/* FIXME
  sg_file_printf(file->stream, "  <sgw:Geometry Width=\"%d\" Height=\"%d\"/>\n",
          worksheet->width, worksheet->height);
*/
  sg_file_printf(file->stream, "  <sgw:Name>%s</sgw:Name>\n", s = xml_get_string(worksheet->name)); 
  sg_file_printf(file->stream, "  <sgw:MaxCol>%d</sgw:MaxCol>\n", 
          GTK_SHEET(worksheet)->maxcol); 
  sg_file_printf(file->stream, "  <sgw:MaxRow>%d</sgw:MaxRow>\n", 
          GTK_SHEET(worksheet)->maxrow); 
  sg_file_printf(file->stream, "  <sgw:Begin>%d</sgw:Begin>\n", 
          worksheet->begin); 
  sg_file_printf(file->stream, "  <sgw:End>%d</sgw:End>\n", 
          worksheet->end); 
}

static void
write_footer_xml(SGworksheetfile *file)
{
  if(GTK_IS_SG_MATRIX(file->worksheet))
    sg_file_printf(file->stream,"</sgw:Matrix>\n");
  else
    sg_file_printf(file->stream,"</sgw:Worksheet>\n");
}

static void
write_column_title_xml(SGworksheetfile *file, gint col) 
{
  GtkSheetColumn *column;
 
  column = &GTK_SHEET(file->worksheet)->column[col];

  sg_file_printf(file->stream, "  <sgw:Column No=\"%d\" Width=\"%d\" Title=\"%s\" Type=\"%d\" Format=\"%d\" Internal=\"%d\" Precision=\"%d\">\n", 
          col, column->width, column->name, 
          file->worksheet->column[col].type,
          file->worksheet->column[col].format,
          file->worksheet->column[col].internal,
          file->worksheet->column[col].precision);

  if(file->worksheet->column[col].exp) {
    gchar *s;
    sg_file_printf(file->stream, "    <sgw:ColFormula>%s</sgw:ColFormula>\n", 
                       s = xml_get_string(file->worksheet->column[col].exp));
  }
        
  sg_file_printf(file->stream, "  </sgw:Column>\n"); 
}

static void
write_cell_xml(SGworksheetfile *file, gint row, gint col)
{
  gchar *text, *formula;

  formula = sg_worksheet_cell_get_formula(file->worksheet, row, col);

  text = sg_worksheet_cell_get_text(file->worksheet, row, col);


  if((!text || strlen(text) == 0) && (!formula || strlen(formula) == 0))
      return;

  sg_file_printf(file->stream, "  <sgw:Cell Row=\"%d\" Col=\"%d\">\n", row, col);

  if (text && strlen(text) > 0) {
    gchar *s;
    sg_file_printf(file->stream, "    <sgw:Content>%s</sgw:Content>\n", s = xml_get_string(text));
  }

  if (formula && strlen(formula) > 0) {
    gchar *s;
    sg_file_printf(file->stream, "    <sgw:Formula>%s</sgw:Formula>\n",  s = xml_get_string(formula));
  }   

  sg_file_printf(file->stream, "  </sgw:Cell>\n");
}
