/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include <libxml/xmlreader.h>
#include "sg_object_file_reader.h"
#include "sg_types.h"

typedef struct
{
  gchar *last_node;
  GtkPlotData *dataset;
  GtkPlotArray *dim;
} parser_state;

void
sg_style_file_process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){

    if(strcmp(name,"sgp:Object") == 0){ 
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Type") == 0) {
          if(strcmp(value, "GtkPlotArray") == 0){
            state->dim = GTK_PLOT_ARRAY(g_object_new(gtk_plot_array_get_type(), NULL));
            sg_object_file_read_xml(NULL, G_OBJECT(state->dim), reader);
	    gtk_plot_array_list_add(state->dataset->data, state->dim);
            xmlFree(child);
            xmlFree(value);
            break;
          } else  if(strcmp(value, "GtkPlot") != 0 && state->dataset) {
            sg_object_file_read_xml(NULL, G_OBJECT(state->dataset), reader);
                                                                                
            xmlFree(child);
            xmlFree(value);
            break;
          }
        }
        xmlFree(child);
        xmlFree(value);
      }

    }
/***********************************************************
 * ARRAYS 
 ***********************************************************/
    if(strcmp(name, "sgp:Dimension") == 0){
      gchar *name = NULL;
      GtkType type = _GTK_TYPE_DOUBLE;
      gint size = 0;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Name") == 0) name = g_strdup(value);
        if(strcmp(child, "Size") == 0) size = atoi(value);
        if(strcmp(child, "Type") == 0) type = (GtkType)atoi(value);

        xmlFree(child);
        xmlFree(value);
      }

      if(name){
	GtkPlotArray *dim = NULL;
	if(type == GTK_TYPE_STRING || type == _GTK_TYPE_STRING)
          dim = GTK_PLOT_ARRAY(gtk_plot_array_new(name, g_new0(gchar *, size), size, GTK_TYPE_STRING, TRUE));
	if(type == GTK_TYPE_DOUBLE || type == _GTK_TYPE_DOUBLE)
          dim = GTK_PLOT_ARRAY(gtk_plot_array_new(name, g_new0(gdouble, size), size, GTK_TYPE_DOUBLE, TRUE));
	gtk_plot_array_list_add(state->dataset->data, dim);
	state->dim = dim;
        g_free(name);	
      }
    }
/***********************************************************
 * DIMENSION POINTS 
 ***********************************************************/
    if(strcmp(name, "sgp:Point") == 0){
      GtkPlotArray *dim = state->dim;
      gint index = 0;
      if(dim->type == GTK_TYPE_DOUBLE || dim->type == _GTK_TYPE_DOUBLE) {
        gdouble val_double = 0;

        while(xmlTextReaderMoveToNextAttribute(reader)){
          xmlChar *child = xmlTextReaderName(reader);
          xmlChar *value = xmlTextReaderValue(reader);

          if(strcmp(child, "Index") == 0) index = atoi(value);
          if(strcmp(child, "Value") == 0) val_double = atof(value);

          xmlFree(child);
          xmlFree(value);
        }
	dim->data.data_double[index] = val_double;
      } else if(dim->type == GTK_TYPE_STRING || dim->type == _GTK_TYPE_STRING){
        gchar *val_string = NULL;

        while(xmlTextReaderMoveToNextAttribute(reader)){
          xmlChar *child = xmlTextReaderName(reader);
          xmlChar *value = xmlTextReaderValue(reader);

          if(strcmp(child, "Index") == 0) index = atoi(value);
          if(strcmp(child, "Value") == 0) val_string = g_strdup(value);

          xmlFree(child);
          xmlFree(value);
        }
	dim->data.data_string[index] = val_string;
      }
    }


    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  xmlFree(name);
}

gboolean 
sg_style_file_read_xml   (xmlTextReaderPtr reader, GtkPlotData *data)
{  
  gint ret_val;
  parser_state *state;

  state = g_new0(parser_state, 1);
  state->dataset = data;
  state->last_node = NULL;

  if(!reader) return FALSE;

  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    sg_style_file_process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Style") == 0){
      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      g_free(state);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(state->last_node) g_free(state->last_node);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


