/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "pixmaps/sg_small.xpm"
#include "sg_object_file_reader.h"
#include <libxml/xmlreader.h>

enum {
  MODE_IMPORT,
  MODE_OPEN,
};

typedef struct
{
  SGplot *plot;
  FILE *file;

  gint mode;
  gboolean in_child;
  gchar *last_node;
  gchar layer_type[200];
  SGlayer *layer;
} parser_state;

void
SGplot_xml_open_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgp");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

void
SGplot_xml_import_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_IMPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgp");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){

    if(strcmp(name,"sgp:Layer") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Plugin") == 0) {
          SGpluginFile *plugin;
          SGpluginLayer *layer_plugin;

          layer_plugin = sg_plugin_layer_get(value);
          plugin = sg_plugin_file_get("xml", value, SG_PLUGIN_FILE_OPEN);
          if(layer_plugin && plugin) {
            GObject *object;
            GList *list;

            state->layer = sg_layer_new(layer_plugin, .5, .5);
            sg_plot_add_layer(state->plot, state->layer, 0., 0.);
            object = G_OBJECT(state->layer);
            plugin->action(plugin, NULL, state->file, &object, reader);

            list = GTK_PLOT_CANVAS(state->plot)->childs; 
            while(list){
              if(GTK_IS_PLOT_CANVAS_PLOT(list->data) &&
                 GTK_PLOT_CANVAS_PLOT(list->data)->plot == GTK_PLOT_CANVAS_PLOT(state->layer)->plot){
                 GtkPlot *real_plot = GTK_PLOT_CANVAS_PLOT(state->layer)->plot;
                 gtk_plot_canvas_child_move_resize(GTK_PLOT_CANVAS(state->plot),
                                             GTK_PLOT_CANVAS_CHILD(list->data),
                                             real_plot->x,
                                             real_plot->y,
                                             real_plot->x + real_plot->width,
                                             real_plot->y + real_plot->height);
                 break;
              }
              list = list->next;
            }
            
          }
          xmlFree(child);
          xmlFree(value);
          break;
        }

        xmlFree(child);
        xmlFree(value);
      }
    }

    if(strcmp(name,"sgp:Child") == 0) state->in_child = TRUE;

    if(strcmp(name,"sgp:Object") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
        GObject *child_object = NULL;
         
        if(strcmp(child, "Type") == 0) {
          if(state->mode == MODE_OPEN && strcmp(value, "SGplot") == 0) {
            sg_object_file_read_xml(state->file, G_OBJECT(state->plot), reader);
            xmlFree(child);
            xmlFree(value);
            break;
          }
          /* CANVAS CHILD */
          if(state->in_child){
            child_object = g_object_new(gtk_type_from_name(value), NULL);
            if(child_object) {
              GtkPlotCanvasChild *child = GTK_PLOT_CANVAS_CHILD(child_object);
              sg_object_file_read_xml(state->file, child_object, reader); 
              gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(state->plot), 
	  		                child, 
				        child->rx1, child->ry1,
				        child->rx2, child->ry2);
            } 
            state->in_child = FALSE;
          }
        } 
          
        xmlFree(child);
        xmlFree(value);
      }
    }

  }

  if(xmlTextReaderNodeType(reader) == 15){
    if(strcmp(name,"sgp:Child") == 0) state->in_child = FALSE;
  }

  xmlFree(name);
}

gboolean
xml_open   	(parser_state *state, const gchar* filename,
                 GObject **object, gpointer data)
{  
  xmlTextReaderPtr reader;
  gint ret_val;

  if(data)
    reader = (xmlTextReaderPtr)data;
  else
    reader = xmlNewTextReaderFilename(filename);

  if(!reader) return FALSE;

  if(!*object) {
    state->plot = sg_plot_new("");
    *object = G_OBJECT(state->plot);
  } else {
    state->plot = SG_PLOT(*object);
    if(state->mode == MODE_OPEN){
      sg_plot_clear(state->plot);
    }
  }
  state->in_child = FALSE;

  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Plot") == 0) {
      if(state->mode == MODE_OPEN)
        sg_plot_set_size(state->plot, 
  		         state->plot->page_size,
		         state->plot->page_width,
		         state->plot->page_height,
		         state->plot->orientation);

/*
		         state->plot->orientation == GTK_PLOT_PORTRAIT ? state->plot->page_width : state->plot->page_height,
		         state->plot->orientation == GTK_PLOT_PORTRAIT ? state->plot->page_height : state->plot->page_width,
*/

      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(!data) xmlFreeTextReader(reader);
  if(state->last_node) g_free(state->last_node);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


gboolean
SGplot_xml_import   	(SGpluginFile *plugin, 
			 const gchar *filename, 
			 FILE *stream,
                         GObject **object, gpointer data)
{  
  parser_state state;

  state.file = stream;
  state.last_node = NULL;
  state.mode = MODE_IMPORT;

  return(xml_open(&state, filename, object, data));
}

gboolean
SGplot_xml_open   	(SGpluginFile *plugin, 
			 const gchar *filename, 
			 FILE *stream,
                         GObject **object, gpointer data)
{  
  parser_state state;

  state.file = stream; 
  state.last_node = NULL;
  state.mode = MODE_OPEN;

  return(xml_open(&state, filename, object, data));
}

