/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_object_file_reader.h"
#include "sg_types.h"
#include <libxml/xmlreader.h>

enum
{
  ARG_UNKNOWN,
  ARG_COLOR,
  ARG_LINE,
  ARG_VECTOR,
  ARG_PIXMAP,
  ARG_BITMAP,
  ARG_TEXT,
  ARG_ARRAY,
  ARG_LAST,
};

typedef struct
{
  GObject *object;

  gchar *last_node;
  FILE *stream;
  gint id;
  gchar *name;

  GtkPlotLine line;
  GtkPlotText text;
  GtkPlotVector vector;
  GtkPlotArrayArg array;
  GdkColor color;

  GdkWindow *window;
  GdkImage *image;
  GdkBitmap *bitmap;
  GdkPixmap *pixmap;
  GdkGC *gc;
  gulong *pixels;
  gint ncolors;
  gint width;
  gint height;
  gint px;
  gint py;

  GtkArg arg[1];
  gint arg_type;
  gchar *arg_name;
  gchar *arg_value;
} parser_state;

static void
process_pixmap_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);

  /* Start Element */
  if(xmlTextReaderNodeType(reader) == 1){
    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  /* Text node */

  if(state->last_node && xmlTextReaderNodeType(reader) == 3){
    xmlChar *value = xmlTextReaderValue(reader);
    if(!value) return;

    if(strcmp(state->last_node, "sgp:XPMColor") == 0){
      GdkColor color;
      gchar color_str[17];
                                                                                
      g_snprintf(color_str, 17, "#%s", value);
                                                                                
      gdk_color_parse(color_str, &color);
      gdk_color_alloc(gdk_colormap_get_system(), &color);

      state->pixels = (gulong *)g_realloc(state->pixels, (state->ncolors+1)*sizeof(gulong));
      state->pixels[state->ncolors] = color.pixel;
                                                                                
      state->ncolors++;
    }

    if(strcmp(state->last_node, "sgp:Pixels") == 0 && state->pixmap){
      GdkColor color;
      gint index = 0;
      gchar *s;
      gint powers[4] = {1, 16, 256, 4096};
      gint n = 0;
      gint val = 0;
      state->px = state->py = 0;

      for(s = value; s && *s != '\0'; s++){

        if(*s < '0' || *s > 'F' || (*s > '9' && *s < 'A')){
          continue;
        }

        if(*s <= '9')
          val = *s - '0';
        else
          val = *s - 'A' + 10;
        index += powers[3 - n++] * val;

        if(n == 4) {
          color.pixel = state->pixels[index];
          gdk_image_put_pixel(state->image, state->px, state->py, color.pixel);                                                                                
          state->px++;
          if(state->px == state->width){
            state->py++;
            state->px = 0;
          }
          index = 0;
          n = 0;
        } 
      }
    }
    if(strcmp(state->last_node, "sgp:Pixels") == 0 && state->bitmap){
      gchar *s;
      state->px = state->py = 0;

      for(s = value; s && *s != '\0'; s++){
        gulong pixel = *s - '0';
        gdk_image_put_pixel(state->image, state->px, state->py, pixel);                                                                                
        state->px++;
        if(state->px == state->width){
          state->py++;
          state->px = 0;
        }
      }
    }
    xmlFree(value);
  }
  if(state->last_node && xmlTextReaderNodeType(reader) == 15){
    g_free(state->last_node);
    state->last_node = NULL;
  }
    
                                                                                
}

static gboolean 
pixmap_read(xmlTextReaderPtr reader, parser_state *state)
{
  gint ret_val;
  state->pixels = NULL;
  state->ncolors = 0;
  state->px = state->py = 0;
  
  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_pixmap_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Pixmap") == 0){
      if(name) xmlFree(name);
      if(state->pixels) g_free(state->pixels);
      state->pixels = NULL;
      return TRUE;
    }
    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Bitmap") == 0){
      if(name) xmlFree(name);
      if(state->pixels) g_free(state->pixels);
      state->pixels = NULL;
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }

  if(state->pixels) g_free(state->pixels);
  state->pixels = NULL;
  return ret_val;
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);
  GObject *object = NULL;

  /* Start Element */
  if(xmlTextReaderNodeType(reader) == 1){
    if(strcmp(name,"sgp:Arg") == 0){
      guint type = 0;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Type") == 0) type = _gtk_fundamental_type[atoi(value)];

        if(strcmp(child, "Value") == 0) {
          state->arg_value = g_strdup(value);
          if(strcmp(state->arg_value, "GdkColor") == 0){
            state->arg_type = ARG_COLOR;
          }
          if(strcmp(state->arg_value, "GtkPlotVector") == 0){
            state->arg_type = ARG_VECTOR;
          }
          if(strcmp(state->arg_value, "GtkPlotLine") == 0){
            state->arg_type = ARG_LINE;
          }
          if(strcmp(state->arg_value, "GtkPlotText") == 0){
            state->arg_type = ARG_TEXT;
          }
          if(strcmp(state->arg_value, "GdkPixmap") == 0){
            state->arg_type = ARG_PIXMAP;
            state->ncolors = 0;
            state->pixels = g_new0(gulong, 1);
          }
          if(strcmp(state->arg_value, "GdkBitmap") == 0){
            state->arg_type = ARG_BITMAP;
          }
          if(strcmp(state->arg_value, "Array") == 0){
            GtkPlotArray *array = GTK_PLOT_ARRAY(state->object);
            switch(array->type){
              case GTK_TYPE_DOUBLE:
              case _GTK_TYPE_DOUBLE:
		array->type = GTK_TYPE_DOUBLE;
                array->data.data_double = g_new0(gdouble, array->size);
                break; 
              case GTK_TYPE_FLOAT:
              case _GTK_TYPE_FLOAT:
		array->type = GTK_TYPE_FLOAT;
                array->data.data_float = g_new0(gfloat, array->size);
                break; 
              case GTK_TYPE_INT:
              case _GTK_TYPE_INT:
		array->type = GTK_TYPE_INT;
                array->data.data_int = g_new0(gint, array->size);
                break; 
              case GTK_TYPE_STRING:
              case _GTK_TYPE_STRING:
		array->type = GTK_TYPE_STRING;
                array->data.data_string = g_new0(gchar *, array->size);
                break; 
            }
            state->array = array->data;
            state->arg_type = ARG_ARRAY;
          }
        }
        if(strcmp(child, "Name") == 0) state->arg_name = g_strdup(value);

        if(strcmp(child, "R") == 0) {
          if(state->arg_type == ARG_COLOR) state->color.red = atoi(value);
          if(state->arg_type == ARG_LINE) state->line.color.red = atoi(value);
        }
        if(strcmp(child, "G") == 0) {
          if(state->arg_type == ARG_COLOR) state->color.green = atoi(value);
          if(state->arg_type == ARG_LINE) state->line.color.green = atoi(value);
        }
        if(strcmp(child, "B") == 0) {
          if(state->arg_type == ARG_COLOR) state->color.blue = atoi(value);
          if(state->arg_type == ARG_LINE) state->line.color.blue = atoi(value);
        }
        if(strcmp(child, "X") == 0) {
           if(state->arg_type == ARG_VECTOR) state->vector.x = atof(value);
           if(state->arg_type == ARG_TEXT) state->text.x = atof(value);
        }
        if(strcmp(child, "Y") == 0) {
           if(state->arg_type == ARG_VECTOR) state->vector.y = atof(value);
           if(state->arg_type == ARG_TEXT) state->text.y = atof(value);
        }
        if(strcmp(child, "Z") == 0) {
           if(state->arg_type == ARG_VECTOR) state->vector.z = atof(value);
        }
        if(strcmp(child, "Angle") == 0) state->text.angle = atoi(value);
        if(strcmp(child, "Justification") == 0) state->text.justification = atoi(value);
        if(strcmp(child, "Transparent") == 0) state->text.transparent = atoi(value);
        if(strcmp(child, "Style") == 0) state->line.line_style = atoi(value);
        if(strcmp(child, "Width") == 0) {
          if(state->arg_type == ARG_LINE) state->line.line_width = atof(value);
          if(state->arg_type == ARG_PIXMAP) state->width = atoi(value);
          if(state->arg_type == ARG_BITMAP) state->width = atoi(value);
        };
        if(strcmp(child, "Height") == 0) {
          if(state->arg_type == ARG_PIXMAP) state->height = atoi(value);
          if(state->arg_type == ARG_BITMAP) state->height = atoi(value);
        }

        xmlFree(child);
        xmlFree(value);
      }
                                                                                
      state->arg[0].type = (GtkType)type;
      state->arg[0].name = state->arg_name;

      switch(type){
        case GTK_TYPE_UINT:
        case _GTK_TYPE_UINT:
        case GTK_TYPE_ENUM:
        case _GTK_TYPE_ENUM:
        case GTK_TYPE_INT:
        case _GTK_TYPE_INT:
        case GTK_TYPE_BOOL:
        case _GTK_TYPE_BOOL:
          g_object_set(G_OBJECT(state->object), state->arg_name, atoi(state->arg_value), NULL);
          if(state->arg_name) g_free(state->arg_name);
          state->arg_name = NULL;
          break;
        case GTK_TYPE_DOUBLE:
        case _GTK_TYPE_DOUBLE:
        case GTK_TYPE_FLOAT:
        case _GTK_TYPE_FLOAT:
          g_object_set(G_OBJECT(state->object), state->arg_name, atof(state->arg_value), NULL);
          if(state->arg_name) g_free(state->arg_name);
          state->arg_name = NULL;
          break;
        case GTK_TYPE_STRING:
        case _GTK_TYPE_STRING:
          g_object_set(G_OBJECT(state->object), state->arg_name, state->arg_value, NULL);
          if(state->arg_name) g_free(state->arg_name);
          state->arg_name = NULL;
          break;
        case G_TYPE_OBJECT:
        case _GTK_TYPE_OBJECT:
          g_object_get(state->object, state->arg_name, &object, NULL);
          if(!object) {
            object = g_object_new(gtk_type_from_name(state->arg_value), NULL);
            sg_object_file_read_xml(state->stream, object, reader);
            g_object_set(G_OBJECT(state->object), state->arg_name, object, NULL);
          } else {
            sg_object_file_read_xml(state->stream, object, reader);
          }
          break;
        case GTK_TYPE_POINTER:
        case _GTK_TYPE_POINTER:
          switch(state->arg_type){
            case ARG_COLOR:
              gdk_color_alloc(gdk_colormap_get_system(), &state->color);
              g_object_set(G_OBJECT(state->object), state->arg_name, &state->color, NULL);
              break;
            case ARG_LINE:
              gdk_color_alloc(gdk_colormap_get_system(), &state->line.color);
              g_object_set(G_OBJECT(state->object), state->arg_name, &state->line, NULL);
              break;
            case ARG_VECTOR:
              g_object_set(G_OBJECT(state->object), state->arg_name, &state->vector, NULL);
              break;
            case ARG_PIXMAP:
              if(state->bitmap){
                gdk_bitmap_unref(state->bitmap);
                state->bitmap = NULL;
              }
              if(state->pixmap){
                gdk_pixmap_unref(state->bitmap);
                state->pixmap = NULL;
              }
              state->pixmap = gdk_pixmap_new(state->window, state->width, state->height, -1);
              state->image = gdk_image_get(state->pixmap, 0, 0,
                                           state->width, state->height);
              pixmap_read(reader, state);
              break;
            case ARG_BITMAP:
              if(state->bitmap){
                gdk_bitmap_unref(state->bitmap);
                state->bitmap = NULL;
              }
              if(state->pixmap){
                gdk_pixmap_unref(state->bitmap);
                state->pixmap = NULL;
              }
              state->bitmap = gdk_pixmap_new(state->window, state->width, state->height, 1);
              state->image = gdk_image_get(state->bitmap, 0, 0,
                                           state->width, state->height);
              pixmap_read(reader, state);
              break;
            default:
              break;
          } 
          break;
        default:
          break;
      }
    }

    if(strcmp(name,"sgp:Border") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
        if(strcmp(child, "Style") == 0) state->text.border = atoi(value);
        if(strcmp(child, "Space") == 0) state->text.border_space = atoi(value);
        if(strcmp(child, "Width") == 0) state->text.border_width = atoi(value);
        if(strcmp(child, "Shadow") == 0) state->text.shadow_width = atoi(value);
      }
    }

    if(strcmp(name,"sgp:Foreground") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
        if(strcmp(child, "R") == 0) state->text.fg.red = atoi(value);
        if(strcmp(child, "G") == 0) state->text.fg.green = atoi(value);
        if(strcmp(child, "B") == 0) state->text.fg.blue = atoi(value);
      }
    }
    if(strcmp(name,"sgp:Background") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
        if(strcmp(child, "R") == 0) state->text.bg.red = atoi(value);
        if(strcmp(child, "G") == 0) state->text.bg.green = atoi(value);
        if(strcmp(child, "B") == 0) state->text.bg.blue = atoi(value);
      }
    }
    if(strcmp(name,"sgp:Font") == 0){
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
        if(strcmp(child, "Name") == 0) state->text.font = g_strdup(value);
        if(strcmp(child, "Height") == 0) state->text.height = atoi(value);
      }
    }
    if(strcmp(name,"sgp:Point") == 0){
      guint index = 0;
                                                                                
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Index") == 0) index = atoi(value);
        if(strcmp(child, "Value") == 0) {
          switch(GTK_PLOT_ARRAY(state->object)->type){
            case GTK_TYPE_DOUBLE:
            case _GTK_TYPE_DOUBLE:
              state->array.data_double[index] = atof(value);
              break;    
            case GTK_TYPE_FLOAT:
            case _GTK_TYPE_FLOAT:
              state->array.data_float[index] = atof(value);
              break;    
            case GTK_TYPE_INT:
            case _GTK_TYPE_INT:
              state->array.data_int[index] = atoi(value);
              break;    
            case GTK_TYPE_STRING:
            case _GTK_TYPE_STRING:
              state->array.data_string[index] = g_strdup(value);
              break;    
          }
        }
      }
    }
    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);

  }

  /* Text node */

  if(state->last_node && xmlTextReaderNodeType(reader) == 3){
    xmlChar *value = xmlTextReaderValue(reader);
    if(!value) return;

    if(strcmp(state->last_node, "sgp:String") == 0){
      state->text.text = g_strdup(value);
    }

    xmlFree(value);
  }

  /* End Element */

  if(xmlTextReaderNodeType(reader) == 15){
    if(strcmp(name, "sgp:Arg") == 0){
      if(state->arg[0].type == GTK_TYPE_POINTER || state->arg[0].type == _GTK_TYPE_POINTER){
        switch(state->arg_type){
/*
          case ARG_COLOR:
            gdk_color_alloc(gdk_colormap_get_system(), &state->color);
            GTK_VALUE_POINTER(state->arg[0]) = &state->color;
            g_object_set(G_OBJECT(state->object), 1, state->arg);
            break;
          case ARG_LINE:
            gdk_color_alloc(gdk_colormap_get_system(), &state->line.color);
            GTK_VALUE_POINTER(state->arg[0]) = &state->line;
            g_object_set(G_OBJECT(state->object), 1, state->arg);
            break;
          case ARG_VECTOR:
            GTK_VALUE_POINTER(state->arg[0]) = &state->vector;
            g_object_set(G_OBJECT(state->object), 1, state->arg);
            break;
*/
          case ARG_TEXT:
            gdk_color_alloc(gdk_colormap_get_system(), &state->text.fg);
            gdk_color_alloc(gdk_colormap_get_system(), &state->text.bg);
            g_object_set(G_OBJECT(state->object), state->arg_name, &state->text, NULL);
            if(state->text.text) g_free(state->text.text);
            state->text.text = NULL;
            if(state->text.font) g_free(state->text.font);
            state->text.font = NULL;
            break;
          case ARG_PIXMAP:
            if(state->image){
              gdk_draw_image(state->pixmap, state->gc, state->image, 0, 0, 0, 0,
                             state->width, state->height);
              gdk_image_destroy(state->image);
              state->image = NULL;
              g_object_set(G_OBJECT(state->object), state->arg_name, state->pixmap, NULL);
              if(state->pixmap) gdk_pixmap_unref(state->pixmap);
              state->pixmap = NULL;
            }
            break;
          case ARG_BITMAP:
            if(state->image){
              gdk_draw_image(state->bitmap, state->gc, state->image, 0, 0, 0, 0,
                             state->width, state->height);
              gdk_image_destroy(state->image);
              state->image = NULL;
              g_object_set(G_OBJECT(state->object), state->arg_name, state->bitmap, NULL);
              if(state->bitmap) gdk_bitmap_unref(state->bitmap);
              state->bitmap = NULL;
            }
            break;
          default:
            break;
        }
      }
      if(state->arg_name) g_free(state->arg_name);
      state->arg_name = NULL;
      if(state->arg_value) g_free(state->arg_value);
      state->arg_value = NULL;
    }
    if(state->last_node) g_free(state->last_node);
    state->last_node = NULL;
  }

  xmlFree(name);
}

gboolean 
sg_object_file_read_xml   (FILE *stream, GObject *object, xmlTextReaderPtr reader)
{  
  gint ret_val;
  parser_state *state;
  GdkWindowAttr attributes;
  gint attributes_mask;

  state = g_new0(parser_state, 1);
  state->object = object;
  state->last_node = NULL;
  state->stream = stream;
  state->id = -1;
  state->name = NULL;

  state->text.font = NULL;
  state->text.text = NULL;

  state->arg_type = ARG_UNKNOWN;
  state->arg_name = NULL;
  state->arg_value = NULL;

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.title = NULL;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gdk_visual_get_system();
  attributes.colormap = gdk_colormap_get_system();
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_VISUAL | GDK_WA_COLORMAP;
                                                                                
  state->window = gdk_window_new (NULL, &attributes, attributes_mask);
                                                                                
  state->gc = gdk_gc_new(state->window);
                                                                                
  state->ncolors = 0;
  state->pixels = g_new0(gulong, 1);
  state->pixmap = state->bitmap = NULL;

  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Object") == 0){
      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      if(state->text.font) g_free(state->text.font);
      if(state->text.text) g_free(state->text.text);
      if(state->pixels) g_free(state->pixels);
      if(state->gc) gdk_gc_unref(state->gc);
      if(state->window) gdk_window_unref(state->window);
      g_free(state);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(state->last_node) g_free(state->last_node);
  if(state->text.font) g_free(state->text.font);
  if(state->text.text) g_free(state->text.text);
  if(state->pixels) g_free(state->pixels);
  if(state->gc) gdk_gc_unref(state->gc);
  if(state->window) gdk_window_unref(state->window);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


