/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "pixmaps/sg_small.xpm"
#include "sg_object_file_reader.h"
#include <libxml/xmlreader.h>

typedef struct
{
  FILE *file;
  SGlayer *layer;
  SGplot *plot;
  SGapplication *app;
  SGlist *datasets;

  gchar *last_node;
} parser_state;

static void	init_default		(SGpluginFile *plugin);
static gboolean xml_read_default   	(SGpluginFile *plugin, 
			     		 const gchar *filename, 
			     		 FILE *stream,
                             		 GObject **object, gpointer data);
  

void
layer_2d_xml_open_init(SGplugin *_plugin)
{
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
  init_default(plugin);
}

gboolean
layer_2d_xml_open		(SGpluginFile *plugin,
			     		 const gchar *filename, 
			     		 FILE *stream,
                             		 GObject **object, gpointer data)
{
  return xml_read_default(plugin, filename, stream, object, data);
}

void
layer_3d_xml_open_init(SGplugin *_plugin)
{
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
  init_default(plugin);
}

gboolean
layer_3d_xml_open		(SGpluginFile *plugin,
			     		 const gchar *filename, 
			     		 FILE *stream,
                             		 GObject **object, gpointer data)
{
  return xml_read_default(plugin, filename, stream, object, data);
}

void
layer_polar_xml_open_init(SGplugin *_plugin)
{
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
  init_default(plugin);
}

gboolean
layer_polar_xml_open		(SGpluginFile *plugin,
			     		 const gchar *filename, 
			     		 FILE *stream,
                             		 GObject **object, gpointer data)
{
  return xml_read_default(plugin, filename, stream, object, data);
}

static void 
init_default(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){
    if(strcmp(name,"sgp:Object") == 0){
                                                                                
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);
                                                                                
        if(strcmp(child, "Type") == 0) {
          if(strcmp(value, "SGlayer") == 0 && state->layer) {
            sg_object_file_read_xml(state->file, G_OBJECT(state->layer), reader);
            xmlFree(child);
            xmlFree(value);
            break;
          }
        }
        if(strcmp(child, "Type") == 0) {
          if(strcmp(value, "GtkPlot") == 0 && state->layer) {
            sg_object_file_read_xml(state->file, G_OBJECT(GTK_PLOT_CANVAS_PLOT(state->layer)->plot), reader);
            xmlFree(child);
            xmlFree(value);
            break;
          }
        }
        xmlFree(child);
        xmlFree(value);
      }
    }

    if(strcmp(name, "sgp:Dataset") == 0) {
      SGpluginFile *plugin = sg_plugin_file_get("xml", "sg_dataset", SG_PLUGIN_FILE_OPEN);
      GObject *object = NULL;
      plugin->action(plugin, NULL, state->file, &object, reader);
      if(object) {
        SGdataset *real_data = SG_DATASET(object);
        if(real_data->id >= 0) {
          GObject *parent = NULL;
          GObject *child = NULL;
          parent = sg_list_get_by_id(state->datasets, real_data->id);
          child = G_OBJECT(real_data->children->data);
          if(!parent){
            gboolean visible = GTK_WIDGET_VISIBLE(GTK_WIDGET(child));
            sg_list_add(state->datasets,
                        G_OBJECT(real_data), real_data->name);
            sg_dataset_add_child(real_data, GTK_PLOT_DATA(child));
            sg_layer_add_dataset_child(state->layer, GTK_PLOT_DATA(child));
            if(!visible) gtk_widget_hide(GTK_WIDGET(child));
          } else {
            gboolean visible = GTK_WIDGET_VISIBLE(GTK_WIDGET(child));
            sg_dataset_add_child(SG_DATASET(parent), GTK_PLOT_DATA(child));
            sg_layer_add_dataset_child(state->layer, GTK_PLOT_DATA(child));
            if(!visible) gtk_widget_hide(GTK_WIDGET(child));
            g_object_unref(G_OBJECT(real_data));
          }
        }
      }
    }

    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  xmlFree(name);
}

static gboolean 
xml_read_default   		(SGpluginFile *plugin, 
			     	 const gchar *filename, 
			     	 FILE *stream,
                             	 GObject **object, gpointer data)
{  
  xmlTextReaderPtr reader;
  gint ret_val;
  parser_state *state;
  gpointer _data;
  gpointer *user_data;

  state = g_new0(parser_state, 1);
  state->last_node = NULL;
  state->file = stream;

  if(data)
    reader = (xmlTextReaderPtr)data;
  else
    reader = xmlNewTextReaderFilename(filename);

  if(!reader) return FALSE;

  state->layer = SG_LAYER(*object);
  state->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(state->layer)->parent);
  _data = g_object_get_data(G_OBJECT(state->plot), "application");
  if(_data) {
    state->app = SG_APPLICATION(_data); 
    state->datasets = state->app->datasets;
  } else {
    state->app = NULL;
    state->datasets = state->plot->datasets;
  }

  ret_val = xmlTextReaderRead(reader);
  
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Layer") == 0){
      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      g_free(state);

      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(!data) xmlFreeTextReader(reader);
  if(state->last_node) g_free(state->last_node);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


