/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>

#define DATASET_NAME_LENGTH 100
#define NENTRIES 20

typedef struct
{
  SGdataset *dataset;
  SGpluginStyle *constructor;
  GtkWidget *column_entry[NENTRIES];
  GtkWidget *left_button[NENTRIES];
  GtkWidget *right_button[NENTRIES];
  GtkWidget *entries_box;
  GtkWidget *entries_table;
  gchar *layer_type;
  gint num_entries;
  GtkWidget *columns_list;

  gboolean return_value;

  GtkPlotArrayList *arrays;
} SGcolumnsDialog;

static void init_columns			(SGcolumnsDialog *dialog);

static void
sg_edit_file_cancel(SGpropertyDialog *pdialog, gpointer data)
{
  SGcolumnsDialog *dialog = (SGcolumnsDialog *)data;
  dialog->return_value = FALSE;
}

static void
sg_edit_file_update(SGpropertyDialog *pdialog, gpointer data)
{
  gint i;
  gboolean incomplete = FALSE;
  SGcolumnsDialog *dialog = (SGcolumnsDialog *)data;
  GtkPlotArrayList *arrays;
  GList *dimensions = dialog->dataset->constructor->arrays->arrays;

  arrays = GTK_PLOT_ARRAY_LIST(gtk_plot_array_list_new());

  i = 0;
  while(dimensions){
    GtkPlotArray *ref;
    const gchar *column_name;

    column_name = gtk_entry_get_text(GTK_ENTRY(dialog->column_entry[i]));
    ref = GTK_PLOT_ARRAY(dimensions->data);
    if(column_name && strlen(column_name) > 0) {
      GtkPlotArray *dim;
      dim = gtk_plot_array_list_get(dialog->arrays, column_name);
      if(dim){
        GtkPlotArray *new_dim;
        if(ref->type != GTK_TYPE_STRING){
          gdouble *new_array = g_malloc(dim->size * sizeof(gdouble));
          int n;
          for(n = 0; n < dim->size; n++){ 
            new_array[n] = atof(dim->data.data_string[n]);
          }
          new_dim = GTK_PLOT_ARRAY(gtk_plot_array_new(ref->name, new_array, dim->size, GTK_TYPE_DOUBLE, TRUE));
          gtk_plot_array_set_label(new_dim, dim->name);
          gtk_plot_array_set_description(new_dim, ref->description);
          gtk_plot_array_set_required(new_dim, ref->required);
        } else {
          new_dim = GTK_PLOT_ARRAY(gtk_plot_array_new(ref->name, dim->data.data_string, dim->size, GTK_TYPE_STRING, TRUE));
          gtk_plot_array_set_label(new_dim, dim->name);
          gtk_plot_array_set_description(new_dim, ref->description);
          gtk_plot_array_set_required(new_dim, ref->required);
          dim->data.data_string = NULL;
        }
        gtk_plot_array_list_add(arrays, new_dim);
        g_object_unref(G_OBJECT(new_dim));
      } else {
        GList *list = dialog->arrays->arrays;
        printf("*** WARNING: Could not find %s\n", column_name);
        while(list){
          printf("%s\n", gtk_plot_array_get_name(GTK_PLOT_ARRAY(list->data)));
          list = list->next;
        }
      }
    } else {
      if(ref->required) {
         incomplete = TRUE;
         break;
      }
    }
    i++;
    dimensions = dimensions->next;
  }
  if(incomplete){
    g_object_unref(G_OBJECT(arrays));
    dialog->return_value = FALSE;
    return;
  }

  dialog->return_value = TRUE;
  sg_dataset_set_arrays(dialog->dataset, arrays);
  g_object_unref(G_OBJECT(arrays));
}

gboolean
sg_edit_file_dialog (SGdataset *dataset, const gchar *type, GtkPlotArrayList *arrays)
{
  GtkWidget *frame;
  GtkWidget *columns_frame;
  GtkWidget *label;
  GtkWidget *window;
  GtkWidget *box, *vbox, *sw;
  SGcolumnsDialog *dialog;
  gchar *layer_type;
  gint i;
  gboolean return_value = FALSE;

  dialog = g_new0(SGcolumnsDialog, 1);
  dialog->dataset = dataset;
  dialog->arrays = arrays;
  dialog->constructor = dataset->constructor;
  dialog->layer_type = layer_type = g_strdup(type);
  dialog->return_value = FALSE;

  /* Create widgets */

  frame = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, FALSE);
  SG_PROPERTY_DIALOG(frame)->apply = sg_edit_file_update; 
  SG_PROPERTY_DIALOG(frame)->ok = sg_edit_file_update; 
  SG_PROPERTY_DIALOG(frame)->cancel = sg_edit_file_cancel; 

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);

/*********************************************************************/
  box = gtk_hbox_new(TRUE, 5);
  gtk_box_pack_start (GTK_BOX (vbox), box, FALSE, FALSE, 0);

  label = gtk_label_new( _("Dataset Style:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);
  gtk_box_pack_start (GTK_BOX (box), GTK_WIDGET(gtk_pixmap_new(dataset->constructor->pixmap->pixmap, NULL)), FALSE, FALSE, 0);
  label = gtk_label_new( SG_PLUGIN(dialog->constructor)->description );
  gtk_misc_set_alignment(GTK_MISC(label), 0., 0.);
  gtk_box_pack_start (GTK_BOX (box), label, FALSE, FALSE, 0);

/*********************************************************************/
  columns_frame = gtk_frame_new( _("Columns") );
  gtk_frame_set_shadow_type (GTK_FRAME (columns_frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), columns_frame, FALSE, FALSE, 0);

  dialog->entries_table = NULL;
  dialog->entries_box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->entries_box), 5);
  gtk_container_add (GTK_CONTAINER (columns_frame), dialog->entries_box);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 180, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (dialog->entries_box), sw, FALSE, FALSE, 0);

  dialog->columns_list = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), dialog->columns_list);

  for(i = 0; i < NENTRIES; i++) dialog->column_entry[i] = NULL;

  init_columns(dialog);

/********************************************************************/
  window = sg_dialog_new( _("SciGraphica: Edit Columns"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_SPREAD);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
  gtk_window_position (GTK_WINDOW (window), GTK_WIN_POS_CENTER);
  gtk_widget_show_all(frame);

  sg_dialog_add(window, SG_PROPERTY_DIALOG(frame));
  gtk_widget_show_all(frame);
  sg_dialog_run(window, NULL);

  return_value = dialog->return_value;
  g_free(dialog);
  g_free(layer_type);
  return return_value;
}

static gint
set_column(GtkWidget *widget, gpointer data)
{
  GList *selection;
  SGcolumnsDialog *dialog;
  GtkWidget *entry = NULL;
  gchar *text;
  const gchar *aux_text;
  gint row;
  gint i;

  dialog = (SGcolumnsDialog *)data;

  selection = GTK_CLIST(dialog->columns_list)->selection;
  if(!selection) return TRUE;

  for(i = 0; i < dialog->num_entries; i++){
    if(widget == dialog->right_button[i]){
      entry = dialog->column_entry[i];
      break;
    }
  }
  if(!entry) return TRUE;
  aux_text = gtk_entry_get_text(GTK_ENTRY(entry));
  if(strlen(aux_text) > 0) return TRUE;
  
  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(dialog->columns_list), row, 0, &text);

  gtk_entry_set_text(GTK_ENTRY(entry), text);
  gtk_clist_remove(GTK_CLIST(dialog->columns_list), row);
  return TRUE;
}

static gint
restore_column(GtkWidget *widget, gpointer data)
{
  GtkWidget *entry = NULL;
  SGcolumnsDialog *dialog;
  gchar *text;
  const gchar *aux_text;
  gint col;
  gint i;
  GList *list = NULL;

  dialog = (SGcolumnsDialog *)data;

  for(i = 0; i < dialog->num_entries; i++){
    if(widget == dialog->left_button[i]){
      entry = dialog->column_entry[i];
      break;
    }
  }
  if(!entry) return TRUE;
  aux_text = gtk_entry_get_text(GTK_ENTRY(entry));
  if(strlen(aux_text) == 0) return TRUE;

  col = 0;
  list = dialog->arrays->arrays;
  while(list){
    if(strcmp(gtk_plot_array_get_name(GTK_PLOT_ARRAY(list->data)), aux_text) == 0) break;
    col++;
    list = list->next;
  }

  text = (gchar *)aux_text;
  gtk_clist_insert(GTK_CLIST(dialog->columns_list), col, &text);
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  return TRUE;
}

static void
init_columns(SGcolumnsDialog *dialog)
{
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint nrows = 0;
  gint i = 0;
  gchar label[NENTRIES][100];
  GList *list = NULL;
  gboolean required[NENTRIES];

  if(!dialog->dataset) return;

  list = dialog->arrays->arrays;
  while(list){
    GtkPlotArray *dim = GTK_PLOT_ARRAY(list->data);
    gchar *name = g_strdup(gtk_plot_array_get_name(dim));
    gtk_clist_append(GTK_CLIST(dialog->columns_list), &name);
    g_free(name);
    list = list->next;
  }

  if(dialog->entries_table && GTK_IS_WIDGET(dialog->entries_table))
     gtk_container_remove(GTK_CONTAINER(dialog->entries_box), dialog->entries_table);

  dialog->num_entries = g_list_length(dialog->dataset->constructor->arrays->arrays);

  dialog->entries_table = gtk_table_new(dialog->num_entries, 3, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(dialog->entries_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->entries_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(dialog->entries_table), 5);

  list = dialog->dataset->constructor->arrays->arrays;
  nrows = 0;
  while(list){
    GtkPlotArray *dim = GTK_PLOT_ARRAY(list->data);

    required[nrows] = dim->required;

    sprintf(label[nrows], "%s:", dim->label);
    button_box = gtk_hbox_new(TRUE, 0);
    gtk_table_attach_defaults(GTK_TABLE (dialog->entries_table), button_box, 
                              0, 1, nrows, nrows + 1);

    dialog->left_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->left_button[nrows]), 
                      gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->left_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->left_button[nrows], FALSE, FALSE, 0);

    dialog->right_button[nrows] = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(dialog->right_button[nrows]), 
                      gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
    gtk_widget_set_usize(dialog->right_button[nrows], 20, 20);
    gtk_box_pack_start(GTK_BOX(button_box), dialog->right_button[nrows], FALSE, FALSE, 0);

    column_label = gtk_label_new(label[nrows]);
    gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
    if(dim->required){
      GtkStyle *style = gtk_style_new();
      style->fg[GTK_STATE_NORMAL].red = 56000;
      style->fg[GTK_STATE_NORMAL].green = 0;
      style->fg[GTK_STATE_NORMAL].blue = 0;
      gtk_widget_set_style(column_label, style);
    }
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), column_label, 
                              1, 2, nrows, nrows + 1);    
    dialog->column_entry[nrows] = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(dialog->column_entry[nrows]), FALSE);
    gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), dialog->column_entry[nrows], 
                              2, 3, nrows, nrows + 1);    

    gtk_signal_connect(GTK_OBJECT(dialog->left_button[nrows]), "clicked",
                       (GtkSignalFunc) restore_column, dialog);
    gtk_signal_connect(GTK_OBJECT(dialog->right_button[nrows]), "clicked",
                       (GtkSignalFunc) set_column, dialog);

    nrows++;
    list = list->next;
  }

  gtk_box_pack_start(GTK_BOX(dialog->entries_box), dialog->entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(dialog->entries_table);

  for(i = 0; i < nrows; i++){
    if(required[i]){
      gtk_clist_select_row(GTK_CLIST(dialog->columns_list), 0, 0);
      set_column(dialog->right_button[i], dialog);
    }    
  }

}

