/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "sg.h"
#include "sg_plugin_function.h"
#include "sg_python_expr.h"

void
linear_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 2;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("a");
  plugin->param[1] = g_strdup("b");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->function = g_strdup("a+b*x");
}

void
quadratic_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a");
  plugin->param[2] = g_strdup("b");
  plugin->param[3] = g_strdup("c");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->function = g_strdup("a+b*(x-x0)+c*(x-x0)**2");
}

void
gaussian_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 3;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a");
  plugin->param[2] = g_strdup("w");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->function = g_strdup("a*exp(-2*(x-x0)**2/w**2)");
}

void
lorentz_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 3;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a");
  plugin->param[2] = g_strdup("w");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->function = g_strdup("2*a*w/(4*(x-x0)**2+w**2)");
}

void
boltzman_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a1");
  plugin->param[2] = g_strdup("a2");
  plugin->param[3] = g_strdup("dx");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->function = g_strdup("(a1 - a2)/(1 + exp((x - x0)/dx)) + a2");
}

void
logistic_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a1");
  plugin->param[2] = g_strdup("a2");
  plugin->param[3] = g_strdup("p");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->function = g_strdup("(a1 - a2)/(1 + (x/x0)**p) + a2");
}

void
expdecay_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("y0");
  plugin->param[2] = g_strdup("a");
  plugin->param[3] = g_strdup("t");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->function = g_strdup("y0 + a * exp(-(x - x0)/t)");
}

void
expgrow_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("y0");
  plugin->param[2] = g_strdup("a");
  plugin->param[3] = g_strdup("t");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->function = g_strdup("y0 + a * exp((x - x0)/t)");
}

void
expassoc_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 5;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("y0");
  plugin->param[1] = g_strdup("a1");
  plugin->param[2] = g_strdup("t1");
  plugin->param[3] = g_strdup("a2");
  plugin->param[4] = g_strdup("t2");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->par_value[4] = 1.0;
  plugin->function = g_strdup("y0 + a1 * (1 + exp(-x/t1)) + a2 * (1 + exp(-x/t2))");
}

void
hyperbl_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 2;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("p1");
  plugin->param[1] = g_strdup("p2");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->function = g_strdup("p1 * x/ (p2 + x)");
}

void
pulse_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 5;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("y0");
  plugin->param[2] = g_strdup("a");
  plugin->param[3] = g_strdup("t1");
  plugin->param[4] = g_strdup("t2");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 0.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->par_value[4] = 1.0;
  plugin->function = g_strdup("y0 + a * (1 + exp(-(x - x0)/t1)) * exp(-(x - x0)/t2)");
}

void
rational0_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 3;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("a");
  plugin->param[1] = g_strdup("b");
  plugin->param[2] = g_strdup("c");
  plugin->par_value[0] = 1.0;
  plugin->par_value[1] = 0.0;
  plugin->par_value[2] = 1.0;
  plugin->function = g_strdup("(b + c*x)/(1 + a*x)");
}

void
sine_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 3;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("a");
  plugin->param[2] = g_strdup("w");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 1.0;
  plugin->par_value[2] = 1.0;
  plugin->function = g_strdup("a * sin(pi*(x - x0)/w)");
}

void
gaussamp_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 4;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("x0");
  plugin->param[1] = g_strdup("y0");
  plugin->param[2] = g_strdup("a");
  plugin->param[3] = g_strdup("w");
  plugin->par_value[0] = 0.0;
  plugin->par_value[1] = 0.0;
  plugin->par_value[2] = 1.0;
  plugin->par_value[3] = 1.0;
  plugin->function = g_strdup("y0 + a * exp(-(x - x0)**2/(2*w**2))");
}

void
allometric_init(SGplugin *_plugin)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(_plugin);

  plugin->nparam = 2;
  plugin->ivar = g_strdup("x");
  plugin->param[0] = g_strdup("a");
  plugin->param[1] = g_strdup("b");
  plugin->par_value[0] = 1.0;
  plugin->par_value[1] = 2.0;
  plugin->function = g_strdup("a * x**b");
}

