/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg_plot.h"
#include "sg_ellipse_dialog.h"
#include "sg_rectangle_dialog.h"
#include "sg_text_dialog.h"
#include "sg_line_dialog.h"
#include "sg_misc_dialogs.h"
#include "sg_layer_control.h"
#include "sg_layer.h"
#include "gtkplotart.h"
#include "sg_plot_clipboard.h"

#define P_(string) string

#define EVENT_METHOD(i, x) GTK_WIDGET_CLASS(GTK_OBJECT_GET_CLASS(GTK_OBJECT(i)))->x

enum {
       REMOVE_LAYER,
       ADD_LAYER,
       ACTIVATE_LAYER,
       LAST_SIGNAL,
};

enum {
	ARG_0,
	ARG_NAME,
	ARG_ORIENTATION,
	ARG_PAGE_SIZE,
	ARG_PAGE_WIDTH,
	ARG_PAGE_HEIGHT,
	ARG_PAGE_UNITS,
	ARG_SCALE,
	ARG_ANTIALIAS,
};

static const gdouble unit_pt[]={ 1.0, 2.83, 28.35, 72.0, (72.0/SG_DPI)};


static void 	sg_plot_class_init 			(SGplotClass *klass);
static void 	sg_plot_init				(SGplot *plot);
static void	sg_plot_destroy				(GtkObject *object); 
static void 	sg_plot_realize				(GtkWidget *widget);
static void     reorder_buttons				(SGplot *plot);
static void     button_toggled				(GtkWidget *widget, 
							 gpointer data);
static gint     button_clicked				(GtkWidget *widget, 
						         GdkEventButton *event,
							 gpointer data);

static gint 	edit_text				(GtkWidget *widget, 
							 GdkEventButton *event,
							 gpointer data);
static void	open_text_dialog			(GtkPlotText *text,
							 SGplot *plot); 
static gint 	apply_dialog_text			(GtkWidget *widget,
							 gpointer data);
static gint 	click_on_item				(GtkPlotCanvas *canvas,
							 GdkEvent *event,
		 					 GtkPlotCanvasChild *item,
							 gpointer data);
static void 	activate_layer				(GtkPlotCanvas *canvas,
							 gpointer data);
static gboolean resize_item				(GtkPlotCanvas *canvas,
							 GtkPlotCanvasChild *item,
							 gdouble width, 
							 gdouble height, 
							 gpointer data);
static gboolean move_item				(GtkPlotCanvas *canvas,
							 GtkPlotCanvasChild *item,
							 gdouble x, gdouble y, 
							 gpointer data);

static gboolean key_press				(GtkWidget *widget, 
							 GdkEventKey *key,
							 gpointer data);
static void sg_plot_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_plot_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);

static guint plot_signals[LAST_SIGNAL] = {0};
static GtkPlotCanvasClass *parent_class = NULL;

GtkType
sg_plot_get_type (void)
{
  static GtkType sg_plot_type = 0;

  if (!sg_plot_type)
    {
      GtkTypeInfo sg_plot_info =
      {
        "SGplot",
        sizeof (SGplot),
        sizeof (SGplotClass),
        (GtkClassInitFunc) sg_plot_class_init,
        (GtkObjectInitFunc) sg_plot_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_plot_type = gtk_type_unique (gtk_plot_canvas_get_type(), &sg_plot_info);
    }

  return sg_plot_type;
}

static void
sg_plot_class_init (SGplotClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;
  GObjectClass *gobject_class;
  SGplotClass *plot_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;
  gobject_class = (GObjectClass*) klass;
  plot_class = (SGplotClass*) klass;

  parent_class = (GtkPlotCanvasClass *)gtk_type_class (gtk_plot_canvas_get_type ());

  widget_class->realize = sg_plot_realize;
  object_class->destroy = sg_plot_destroy;

  plot_signals[ADD_LAYER] =
    gtk_signal_new ("add_layer",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGplotClass, add_layer),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_SG_LAYER);
  plot_signals[REMOVE_LAYER] =
    gtk_signal_new ("remove_layer",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGplotClass, remove_layer),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_SG_LAYER);
  plot_signals[ACTIVATE_LAYER] =
    gtk_signal_new ("activate_layer",
                    GTK_RUN_LAST,
                    GTK_CLASS_TYPE(object_class),
                    GTK_SIGNAL_OFFSET (SGplotClass, activate_layer),
                    gtk_marshal_VOID__OBJECT,
                    GTK_TYPE_NONE, 1, GTK_TYPE_SG_LAYER);

  plot_class->add_layer = NULL;
  plot_class->remove_layer = NULL;

  gobject_class->set_property = sg_plot_set_property;
  gobject_class->get_property = sg_plot_get_property;

  g_object_class_install_property (gobject_class,
                           ARG_NAME,
  g_param_spec_string ("name",
                           P_("Name"),
                           P_("Name"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_ORIENTATION,
  g_param_spec_enum ("orientation",
                           P_("Orientation"),
                           P_("Orientation"),
                           GTK_TYPE_ORIENTATION,0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_PAGE_SIZE,
  g_param_spec_int ("page_size",
                           P_("Page size"),
                           P_("Page size"),
                           0,G_MAXINT,0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_PAGE_WIDTH,
  g_param_spec_int ("page_width",
                           P_("Page width"),
                           P_("Page width"),
                           0,G_MAXINT,0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_PAGE_HEIGHT,
  g_param_spec_int ("page_height",
                           P_("Page height"),
                           P_("Page height"),
                           0,G_MAXINT,0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_PAGE_UNITS,
  g_param_spec_int ("page_units",
                           P_("Page units"),
                           P_("Page units"),
                           0,G_MAXINT,0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_SCALE,
  g_param_spec_double ("scale",
                           P_("Scale"),
                           P_("Scale"),
                           0.0,G_MAXDOUBLE,1.0,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));
  g_object_class_install_property (gobject_class,
                           ARG_ANTIALIAS,
  g_param_spec_boolean ("antialias",
                           P_("Antialias"),
                           P_("Antialias"),
                           FALSE,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));

}

static void
sg_plot_set_property (GObject      *object,
                         guint            prop_id,
                         const GValue          *value,
                         GParamSpec      *pspec)
{
  SGplot *plot;
                                                                                
  plot = SG_PLOT (object);
                                                                                
  switch(prop_id){
    case ARG_NAME:
      sg_plot_rename(plot, (gchar *)g_value_get_string(value));
      break;
    case ARG_ORIENTATION:
      plot->orientation = g_value_get_enum(value);
      break;
    case ARG_PAGE_SIZE:
      plot->page_size = g_value_get_int(value);
      break;
    case ARG_PAGE_WIDTH:
      plot->page_width = g_value_get_int(value);
      break;
    case ARG_PAGE_HEIGHT:
      plot->page_height = g_value_get_int(value);
      break;
    case ARG_PAGE_UNITS:
      plot->page_units = g_value_get_int(value);
      break;
    case ARG_SCALE:
      sg_plot_rescale(plot, g_value_get_double(value));
      break;
    case ARG_ANTIALIAS:
      sg_plot_antialias(plot, g_value_get_boolean(value));
      break;
  }
}

static void
sg_plot_get_property (GObject      *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
  SGplot *plot;
                                                                                
  plot = SG_PLOT (object);
                                                                                
  switch(prop_id){
    case ARG_NAME:
      g_value_set_string(value,plot->name);
      break;
    case ARG_ORIENTATION:
      g_value_set_enum(value,plot->orientation);
      break;
    case ARG_PAGE_SIZE:
      g_value_set_int(value,plot->page_size);
      break;
    case ARG_PAGE_WIDTH:
      g_value_set_int(value,plot->page_width);
      break;
    case ARG_PAGE_HEIGHT:
      g_value_set_int(value, plot->page_height);
      break;
    case ARG_PAGE_UNITS:
      g_value_set_int(value,plot->page_units);
      break;
    case ARG_SCALE:
      g_value_set_double(value,plot->scale);
      break;
    case ARG_ANTIALIAS:
      g_value_set_boolean(value,plot->antialias);
      break;
  }
}

static void
sg_plot_init (SGplot *plot)
{
   plot->layer_control = NULL;
   plot->tool = SG_TOOL_NONE;
   plot->orientation = GTK_PLOT_PORTRAIT;
   plot->page_size = GTK_PLOT_LETTER;
   plot->scale = .65;
   plot->page_width = GTK_PLOT_LETTER_W;
   plot->page_height = GTK_PLOT_LETTER_H;
   plot->page_units = SG_UNIT_MM;

   plot->layers = NULL;
   plot->nlayers = 0;
   plot->active_layer = NULL;

   sg_plot_rescale(plot, .65);

   plot->antialias = FALSE;

   plot->datasets = sg_list_new();
   g_object_ref(G_OBJECT(plot->datasets));
   plot->clipboard = G_OBJECT(sg_plot_clipboard_new());
   g_object_ref(plot->clipboard);

   GTK_PLOT_CANVAS_SET_FLAGS(GTK_PLOT_CANVAS(plot), GTK_PLOT_CANVAS_DND_FLAGS);
}


static void
sg_plot_realize(GtkWidget *widget)
{
 SGplot *plot;

 plot = SG_PLOT(widget);

 GTK_WIDGET_CLASS(parent_class)->realize(widget);

 /* Signals */
 gtk_signal_connect(GTK_OBJECT(GTK_PLOT_CANVAS(plot)), "select_item",
                    (GtkSignalFunc)click_on_item, NULL);

 gtk_signal_connect(GTK_OBJECT(GTK_PLOT_CANVAS(plot)), "button_press_event",
                    (GtkSignalFunc)edit_text, plot);

 gtk_signal_connect(GTK_OBJECT(GTK_PLOT_CANVAS(plot)), "key_press_event",
                    (GtkSignalFunc)key_press, plot);

 gtk_signal_connect(GTK_OBJECT(GTK_PLOT_CANVAS(plot)), "move_item",
                    (GtkSignalFunc)move_item, plot);

 gtk_signal_connect(GTK_OBJECT(GTK_PLOT_CANVAS(plot)), "resize_item",
                    (GtkSignalFunc)resize_item, plot);

}

SGplot *
sg_plot_new(gchar *name)
{
   SGplot *plot;

   plot = SG_PLOT(gtk_widget_new(sg_plot_get_type(), NULL));
   gtk_plot_canvas_construct(GTK_PLOT_CANVAS(plot), GTK_PLOT_LETTER_W, GTK_PLOT_LETTER_H, plot->scale);

   plot->name = g_strdup(name);
   
   return plot;
}   

static void
refresh_layer_control(SGplot *plot, gpointer data)
{
   if(plot->layer_control){
     gchar *path = g_strdup(SG_LAYER_CONTROL(plot->layer_control)->path);
     sg_layer_control_refresh(plot->layer_control, path);
     g_free(path);
   }
}

SGplot *
sg_plot_new_with_layer(SGpluginLayer *plugin, gchar *name)
{
   SGplot *plot;
   SGlayer *default_layer;

   plot = sg_plot_new(name);

   default_layer = sg_layer_new(plugin, .65, .45);
   sg_plot_add_layer(plot, default_layer, .175, .15);

   return plot;
}

static void
button_set_label(GtkWidget *button, gint num)
{
  gchar title[10];
 
  sprintf(title,"%i\n",num);

  gtk_misc_set_alignment(GTK_MISC(GTK_BIN(button)->child), .5, .0);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(button)->child), title);
}

void
sg_plot_add_layer(SGplot *plot, SGlayer *layer, gdouble x, gdouble y)
{
   gint nlayers;
   GtkWidget *button;
   GtkPlotCanvasChild *child;
   gchar path[1000];
   GtkRequisition req;
   gint size;

   nlayers = plot->nlayers;

   button = gtk_toggle_button_new_with_label(" ");
   gtk_widget_size_request(button, &req);
   size = MAX(req.width,req.height);
   gtk_widget_set_usize(button, size, size);
   g_object_set_data(G_OBJECT(layer), "button", button);
   gtk_fixed_put(GTK_FIXED(GTK_PLOT_CANVAS(plot)), 
                 button,
                 nlayers * size, 0);
   gtk_widget_show(button);
   button_set_label(button, nlayers + 1);

   child = GTK_PLOT_CANVAS_CHILD(layer);
   gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(plot), 
                             child, x, y, 
                             x + GTK_PLOT_CANVAS_PLOT(layer)->plot->width,
                             y + GTK_PLOT_CANVAS_PLOT(layer)->plot->height);

   gtk_widget_show(GTK_WIDGET(GTK_PLOT_CANVAS_PLOT(layer)->plot));
   plot->layers = g_list_append(plot->layers, layer);

   gtk_signal_connect(GTK_OBJECT(button), "toggled",
                      GTK_SIGNAL_FUNC(button_toggled), layer);
   gtk_signal_connect(GTK_OBJECT(button), "button_press_event",
                      GTK_SIGNAL_FUNC(button_clicked), layer);

   gtk_signal_connect_object(GTK_OBJECT(layer), "add_dataset",
                      GTK_SIGNAL_FUNC(refresh_layer_control), GTK_OBJECT(plot));
   gtk_signal_connect_object(GTK_OBJECT(layer), "remove_dataset",
                      GTK_SIGNAL_FUNC(refresh_layer_control), GTK_OBJECT(plot));
   
   plot->nlayers++;


/*
   sprintf(path, "%s:%d:", plot->name, nlayers + 1);
   sg_plot_layer_control(plot, path);
*/

   gtk_signal_emit(GTK_OBJECT(plot), plot_signals[ADD_LAYER], layer);
   sg_plot_set_active_layer(plot, layer);
   refresh_layer_control(plot, NULL);
}

void
sg_plot_move_layer(SGplot *plot, SGlayer *layer, gdouble x, gdouble y)
{
   gtk_plot_canvas_child_move(GTK_PLOT_CANVAS(plot), GTK_PLOT_CANVAS_CHILD(layer), x, y);

   gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
   gtk_widget_queue_draw(GTK_WIDGET(plot));
   refresh_layer_control(plot, NULL);
}

void
sg_plot_resize_layer(SGplot *plot, SGlayer *layer, gdouble w, gdouble h)
{
   gtk_plot_canvas_child_move_resize(GTK_PLOT_CANVAS(plot),
                                    GTK_PLOT_CANVAS_CHILD(layer), 
                                    GTK_PLOT_CANVAS_CHILD(layer)->rx1, 
                                    GTK_PLOT_CANVAS_CHILD(layer)->ry1, 
                                    GTK_PLOT_CANVAS_CHILD(layer)->rx1 + w, 
                                    GTK_PLOT_CANVAS_CHILD(layer)->ry1 + h);

   gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
   gtk_widget_queue_draw(GTK_WIDGET(plot));
   refresh_layer_control(plot, NULL);
}

void
sg_plot_remove_layer(SGplot *plot, SGlayer *layer)
{
  GList *layers = NULL;
  GtkPlotCanvasChild *child = NULL;

  if(plot->nlayers == 0) return;

  if(plot->active_layer == layer) plot->active_layer = NULL;

  layers = plot->layers;
  while(layers) {
     if(SG_LAYER(layers->data) == layer){
         if(layer && GTK_IS_OBJECT(layer)) {
           GtkWidget *button;
           GList *childs = GTK_PLOT_CANVAS(plot)->childs;
           while(childs){
             child = GTK_PLOT_CANVAS_CHILD(childs->data);
             if(GTK_IS_PLOT_CANVAS_PLOT(child) && GTK_PLOT_CANVAS_PLOT(child)->plot == GTK_PLOT_CANVAS_PLOT(layer)->plot) break;
             childs = childs->next;
           }

           if(GTK_IS_PLOT_CANVAS_PLOT(child) && GTK_PLOT_CANVAS_PLOT(child)->plot == GTK_PLOT_CANVAS_PLOT(layer)->plot) {
             gtk_signal_emit(GTK_OBJECT(plot), plot_signals[REMOVE_LAYER], layer);

             button = GTK_WIDGET(g_object_get_data(G_OBJECT(layer), "button"));
             gtk_object_ref(GTK_OBJECT(child));
             gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(plot), child); 
             gtk_container_remove(GTK_CONTAINER(GTK_PLOT_CANVAS(plot)), 
                                  button); 
           }
         }
         plot->layers = g_list_remove_link(plot->layers, layers);
         g_list_free_1(layers);
         plot->nlayers--;
         break;
     }
     layers = layers->next;
  }

  if(plot->nlayers > 0) {
      activate_layer(GTK_PLOT_CANVAS(plot), NULL); 
      reorder_buttons(plot);
  } else{
      plot->layers = NULL;
      plot->active_layer = NULL;
  }
  refresh_layer_control(plot, NULL);
}

static void
reorder_buttons(SGplot *plot)
{
  GList *layers = NULL;
  SGlayer *layer;
  gint nlayers = 0;

  layers = plot->layers;

  while(layers) {
     GtkWidget *button;
     layer = SG_LAYER(layers->data);

     button = GTK_WIDGET(g_object_get_data(G_OBJECT(layer), "button"));
     gtk_fixed_move(GTK_FIXED(GTK_PLOT_CANVAS(plot)), 
                    button,
                    nlayers * button->allocation.width, 0);
     gtk_widget_show(button);
     button_set_label(button, nlayers + 1);

     nlayers++;
     layers = layers->next;
  }

}


static void
button_toggled(GtkWidget *widget, gpointer data)
{
  SGlayer *layer;

  layer = SG_LAYER(data);
  sg_plot_set_active_layer(SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent), layer);
}

static gboolean 
mw_destroy(GtkWidget *widget, gpointer data)
{
  if(data)
    gtk_signal_disconnect_by_func(GTK_OBJECT(data),
                                  GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                  GTK_OBJECT(widget));

  SG_PLOT(data)->layer_control = NULL; 
  gtk_main_quit();
  return TRUE;
}

static gint
button_clicked(GtkWidget* widget, GdkEventButton *event, gpointer data)
{
  SGlayer *layer;
  SGplot *plot;
  GList *llink;
  GdkModifierType mods;
  gchar path[1000];
  GtkObject *dialog;

  gdk_window_get_pointer(widget->window, NULL, NULL, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;
  if(event->type != GDK_2BUTTON_PRESS) return FALSE;

  layer = SG_LAYER(data);
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  llink = g_list_find(plot->layers, layer);
  sprintf(path, "%s:%d:dataset", plot->name,
          g_list_position(plot->layers, llink)+1);

  gtk_grab_remove(widget);

  sg_plot_layer_control(plot, path);
  return FALSE;
}

void
sg_plot_set_active_layer(SGplot *plot, SGlayer *layer)
{
  SGlayer *child = NULL;
  GList *list;
  GtkWidget *button = NULL;

/*
  if(layer == plot->active_layer) return;
  if(!layer) return;
*/
  plot->active_layer = layer;

  list = plot->layers;
  while(list){
    child = SG_LAYER(list->data);
   
    button = GTK_WIDGET(g_object_get_data(G_OBJECT(child), "button"));
    gtk_signal_handler_block_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), child);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), child);

    list = list->next;
  }

  button = GTK_WIDGET(g_object_get_data(G_OBJECT(layer), "button"));
  gtk_signal_handler_block_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), layer);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
  gtk_signal_handler_unblock_by_func(GTK_OBJECT(button), GTK_SIGNAL_FUNC(button_toggled), layer);

  gtk_signal_emit(GTK_OBJECT(plot), plot_signals[ACTIVATE_LAYER], layer);
}

gint
sg_plot_rename(SGplot *plot, gchar *name)
{
  if(plot->name){
     g_free(plot->name);
     plot->name = NULL;
  }
 
  plot->name = g_strdup(name);

  return TRUE;
}

void
sg_plot_rescale(SGplot *plot, gdouble scale)
{
  if(scale < .15) return;

  plot->scale = scale;
  gtk_plot_canvas_set_magnification(GTK_PLOT_CANVAS(plot),
                                    plot->scale);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}

void
sg_plot_set_size(SGplot *plot, gint page_size,
                 gint width, gint height, gint orientation)
{
  gint real_width, real_height;

  real_width = width;
  real_height = height;

  if(orientation == GTK_PLOT_PORTRAIT){
    real_width = width;
    real_height = height;
  }else{
    real_width = height;
    real_height = width;
  }

  plot->page_size = page_size;
  plot->page_width = width;
  plot->page_height = height;
  plot->orientation = orientation;

  GTK_PLOT_CANVAS(plot)->pixmap_width = real_width;
  GTK_PLOT_CANVAS(plot)->pixmap_height = real_height;
  gtk_plot_canvas_set_size(GTK_PLOT_CANVAS(plot),
                           real_width, real_height);
  refresh_layer_control(plot, NULL);
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}

void
sg_plot_clear(SGplot *plot)
{
  GList *list;

  list = plot->layers;
  while(list){
    SGlayer *layer;
 
    layer = SG_LAYER(list->data); 
    sg_plot_remove_layer(plot, layer);

    list = plot->layers;
  }  

  list = GTK_PLOT_CANVAS(plot)->childs;
  while(list){
    GtkPlotCanvasChild *child;
 
    child = GTK_PLOT_CANVAS_CHILD(list->data); 
    gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(plot), child);

    list = GTK_PLOT_CANVAS(plot)->childs;
  }  
  
  plot->layers = NULL;  
}

static void
sg_plot_destroy(GtkObject *object)
{
  SGplot *plot;

  plot = SG_PLOT(object);

  sg_plot_clear(plot); 

  if(plot->name) g_free(plot->name);
  plot->name = NULL;

  if(plot->clipboard) g_object_unref(plot->clipboard);
  plot->clipboard = NULL;

  if(plot->datasets) g_object_unref(G_OBJECT(plot->datasets));
  plot->datasets = NULL;

  if(plot->layer_control && GTK_IS_WIDGET(plot->layer_control))
    gtk_widget_unref(plot->layer_control);
  plot->layer_control = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

/* Events */

static gint
layer_control_destroy (GtkWidget *widget, gpointer data)
{
  SGplot *plot = SG_PLOT(data);
  plot->layer_control = NULL;
  return FALSE;
}

void
sg_plot_layer_control(SGplot *plot, const gchar *path)
{
  gpointer data;
  SGapplication *app = NULL;
  gboolean init = FALSE;
  GtkWidget *layer_control = NULL;

  data = g_object_get_data(G_OBJECT(plot), "application");
  if(data) app = SG_APPLICATION(data);
  if(app){
    if(!app->layer_control) {
      app->layer_control = sg_layer_control_new_application(app);
      init = TRUE;
    }
    layer_control = app->layer_control;
  } else {
    layer_control = plot->layer_control = sg_layer_control_new(plot);
    init = TRUE;
  }

  gtk_widget_show_all(layer_control);
  sg_layer_control_open(layer_control, path);

  if(init){
    gtk_signal_connect(GTK_OBJECT(layer_control), "destroy",
                       GTK_SIGNAL_FUNC(layer_control_destroy), plot);
  }
}

static gint
click_on_item(GtkPlotCanvas *canvas, GdkEvent *event, 
              GtkPlotCanvasChild *item, gpointer data)
{
  GtkPlotCanvasChild *child = NULL;
  SGplot *plot;
  GtkPlot *real_plot = NULL;
  GList *llink;
  gchar path[255];
  gboolean double_click;
  gboolean return_value = TRUE;
  SGlayer *layer;
  GtkObject *dialog = NULL;

  double_click = (event->type == GDK_2BUTTON_PRESS);

  plot = SG_PLOT(canvas);

  if(plot->tool == SG_TOOL_ZOOM ||
     plot->tool == SG_TOOL_LINE ||
     plot->tool == SG_TOOL_LINE_ARROW ||
     plot->tool == SG_TOOL_RECTANGLE ||
     plot->tool == SG_TOOL_ELLIPSE)
    return TRUE;

  if(GTK_IS_PLOT_CANVAS_PLOT(item)){
    real_plot = GTK_PLOT_CANVAS_PLOT(item)->plot;
    layer = SG_LAYER(g_object_get_data(G_OBJECT(real_plot), "layer"));
    activate_layer(canvas, layer);
    llink = g_list_find(plot->layers, plot->active_layer);
    switch(GTK_PLOT_CANVAS_PLOT(item)->pos){
      case GTK_PLOT_CANVAS_PLOT_IN_PLOT:
         sprintf(path, "%s:%d", plot->name,
                 g_list_position(plot->layers, llink)+1);
         if(double_click && plot->tool == SG_TOOL_ARROW){
            sg_plot_layer_control(plot, path);
            return_value = FALSE;
         }
         break;
      case GTK_PLOT_CANVAS_PLOT_IN_TITLE:
         sprintf(path, "%s:%d:axis:title", plot->name,
                 g_list_position(plot->layers, llink)+1);
         if(double_click && plot->tool == SG_TOOL_ARROW){
            sg_plot_layer_control(plot, path);
            return_value = FALSE;
         }
         break;
      case GTK_PLOT_CANVAS_PLOT_IN_DATA:
         sprintf(path, "%s:%d:dataset:style", plot->name,
                 g_list_position(plot->layers, llink)+1);
         if(double_click && plot->tool == SG_TOOL_ARROW){
            sg_plot_layer_control(plot, path);
            return_value = FALSE;
         }
         if(plot->tool == SG_TOOL_MARKERS){
            GtkPlotData *active_data;
            GList *markers;
            gint nmarkers = 0;
            
            active_data = GTK_PLOT_CANVAS_PLOT(plot->active_layer)->data;
            if(active_data){
              markers = active_data->markers;
              while(markers){
                nmarkers++;
                markers = markers->next;
              }

              if(nmarkers == 2) sg_layer_remove_markers(plot->active_layer);
 
              if(nmarkers <= 2){ 
                gtk_plot_data_add_marker(active_data,
	  	         GTK_PLOT_CANVAS_PLOT(plot->active_layer)->datapoint);
                gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
                gtk_widget_queue_draw(GTK_WIDGET(canvas));
                sg_layer_set_active_dataset(plot->active_layer,
                                            active_data); 
              }
            }
            return_value = FALSE;
         }
         break;
      case GTK_PLOT_CANVAS_PLOT_IN_AXIS:
         sprintf(path, "%s:%d:axis", plot->name,
                 g_list_position(plot->layers, llink)+1);
         if(double_click && plot->tool == SG_TOOL_ARROW){
            sg_plot_layer_control(plot, path);
            return_value = FALSE;
         }
         break;
      case GTK_PLOT_CANVAS_PLOT_IN_LEGENDS:
         sprintf(path, "%s:%d:legends", plot->name,
                 g_list_position(plot->layers, llink)+1);
         if(double_click && plot->tool == SG_TOOL_ARROW){
            sg_plot_layer_control(plot, path);
            return_value = FALSE;
         }
         break;
      case GTK_PLOT_CANVAS_PLOT_IN_MARKER:
      case GTK_PLOT_CANVAS_PLOT_IN_GRADIENT:
      default:
         return_value = TRUE;
         break;
      
    }
  }
  if(GTK_IS_PLOT_CANVAS_TEXT(item)){
    if(double_click && plot->tool == SG_TOOL_ARROW){
      GtkPlotText *real_text;
 
      real_text = &GTK_PLOT_CANVAS_TEXT(item)->text;

      open_text_dialog(real_text, plot);

      if(!real_text || !real_text->text || strlen(real_text->text) == 0)
        gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(canvas), item);

      gtk_plot_canvas_paint(GTK_PLOT_CANVAS(canvas));
      gtk_widget_queue_draw(GTK_WIDGET(canvas));
      return_value = FALSE;
    }
  }
  if(GTK_IS_PLOT_CANVAS_LINE(item)){
    if(double_click && plot->tool == SG_TOOL_ARROW){
      sg_line_dialog(canvas, GTK_PLOT_CANVAS_LINE(item));
      return_value = FALSE;
    }
  }
  if(GTK_IS_PLOT_CANVAS_RECTANGLE(item)){
    if(double_click && plot->tool == SG_TOOL_ARROW){
      sg_rectangle_dialog(canvas, GTK_PLOT_CANVAS_RECTANGLE(item));
      return_value = FALSE;
    }
  }
  if(GTK_IS_PLOT_CANVAS_ELLIPSE(item)){
    if(double_click && plot->tool == SG_TOOL_ARROW){
      sg_ellipse_dialog(canvas, GTK_PLOT_CANVAS_ELLIPSE(item));
      return_value = FALSE;
    }
  }
  return return_value;
}

static gint 
edit_text(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
  GtkPlotCanvasChild *child;
  GdkModifierType mods;
  GtkPlotText text;
  SGplot *plot;
  GtkPlotText *real_text;
  gint x, y;

  plot = SG_PLOT(data);
  if(plot->tool != SG_TOOL_TEXT) return FALSE;

  gdk_window_get_pointer(widget->window, &x, &y, &mods);
  if(!(mods & GDK_BUTTON1_MASK)) return FALSE;

  text.text = g_strdup("");
  gdk_color_black(gdk_colormap_get_system(), &text.fg);
  gdk_color_white(gdk_colormap_get_system(), &text.bg);
  text.angle = 0;
  text.height = 16;
  text.transparent = TRUE;
  text.border = (GtkPlotBorderStyle)0;
  text.border_width = 0;
  text.border_space = 2;
  text.shadow_width = 3;
  text.justification = GTK_JUSTIFY_LEFT;
  text.font = g_strdup("Helvetica");

  gtk_plot_canvas_get_position(GTK_PLOT_CANVAS(widget), x, y,
                               &text.x, &text.y);

  child = gtk_plot_canvas_text_new(text.font, text.height,
                                   text.angle,
                                   &text.fg, &text.bg,
                                   text.transparent,
                                   text.justification,
                                   text.text);
  gtk_plot_canvas_put_child(GTK_PLOT_CANVAS(plot), child, text.x, text.y, 0., text.y + text.height);

  real_text = &GTK_PLOT_CANVAS_TEXT(child)->text; 
  real_text->border_space = 2;
  real_text->shadow_width = 3;

  open_text_dialog(real_text, plot);

  if(!real_text || !real_text->text || strlen(real_text->text) == 0){
    gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(plot), child);
  } else {
    gint width, height, a, d;
    gtk_plot_text_get_size(real_text->text, real_text->angle, real_text->font, real_text->height, &width, &height, &a, &d);
    child->rx2 = child->rx1 + (gdouble)width/(gdouble)GTK_PLOT_CANVAS(plot)->pixmap_width;
    child->ry2 = child->ry1 + (gdouble)height/(gdouble)GTK_PLOT_CANVAS(plot)->pixmap_height;
    GTK_PLOT_CANVAS_CHILD_CLASS(GTK_OBJECT_GET_CLASS(GTK_OBJECT(child)))->size_allocate(GTK_PLOT_CANVAS(plot), child);
  }

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_widget_queue_draw(GTK_WIDGET(plot));

  return TRUE;
}

static gint
apply_dialog_text(GtkWidget *widget, gpointer data)
{
  SGplot *plot = SG_PLOT(data);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_widget_queue_draw(GTK_WIDGET(plot));

  return TRUE;
}

static void 
open_text_dialog(GtkPlotText *text, SGplot *plot)
{
  GtkWidget *window;

  window = sg_text_dialog_new(text);

  gtk_signal_connect_object(GTK_OBJECT(plot), "destroy",
	             GTK_SIGNAL_FUNC(gtk_widget_destroy),
		     GTK_OBJECT(window));
  gtk_signal_connect(GTK_OBJECT(window), "destroy",
	             GTK_SIGNAL_FUNC(mw_destroy), plot);


  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(window)->ok_button), 
                      "clicked",
                      GTK_SIGNAL_FUNC (gtk_widget_destroy), 
                      GTK_OBJECT(window));
  gtk_signal_connect (GTK_OBJECT(SG_TEXT_DIALOG(window)->apply_button), 
                      "clicked",
                      GTK_SIGNAL_FUNC (apply_dialog_text), 
                      plot);

  gtk_widget_show(window);
  gtk_window_set_modal(GTK_WINDOW(window), TRUE);
  gtk_main();
}

static void
activate_layer(GtkPlotCanvas *canvas, gpointer data)
{
  SGplot *plot;

  if(!canvas) return;
  plot = SG_PLOT(canvas);

  if(!data) 
    sg_plot_set_active_layer(plot, SG_LAYER(plot->layers->data));
  else
    sg_plot_set_active_layer(plot, SG_LAYER(data));

}

static gboolean
resize_item(GtkPlotCanvas *canvas, GtkPlotCanvasChild *item,
            gdouble new_width, gdouble new_height, gpointer data)
{
  return TRUE;
}

static gboolean
move_item(GtkPlotCanvas *canvas, GtkPlotCanvasChild *item,
          gdouble x, gdouble y, gpointer data)
{
  return TRUE;
}


static gboolean
key_press(GtkWidget *widget, GdkEventKey *key, gpointer data)
{
  GtkPlotCanvas *canvas = GTK_PLOT_CANVAS(data);
  GtkPlotCanvasChild *child = NULL;
  GtkPlotAxis *axis = NULL;
  SGplot *plot = SG_PLOT(data);
  gint i = 0;
  gint the_axis = -1;


  switch(key->keyval){
    case GDK_Escape:
      if(canvas->action != GTK_PLOT_CANVAS_ACTION_INACTIVE)
         gtk_plot_canvas_cancel_action(canvas);
      break;
    case GDK_Delete: case GDK_KP_Delete:
      if(canvas->state == GTK_STATE_SELECTED){
        if(GTK_IS_PLOT_CANVAS_PLOT(canvas->active_item)){
          switch(GTK_PLOT_CANVAS_PLOT(canvas->active_item)->pos){
            case GTK_PLOT_CANVAS_PLOT_IN_PLOT:
              if(sg_accept_dialog("Remove layer?", 1) == SG_BUTTON_YES)
                 sg_plot_remove_layer(plot, plot->active_layer);
              gtk_plot_canvas_cancel_action(canvas);
              return TRUE;
            case GTK_PLOT_CANVAS_PLOT_IN_LEGENDS:
              gtk_plot_canvas_cancel_action(canvas);
              gtk_plot_hide_legends(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot);
              gtk_plot_canvas_paint(canvas);
              gtk_widget_queue_draw(GTK_WIDGET(canvas));
              return TRUE;
              break;
            case GTK_PLOT_CANVAS_PLOT_IN_TITLE:
              for(i = 0; i < 4; i++){
                 axis = gtk_plot_get_axis(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->plot, (GtkPlotAxisPos)i);
                 if(axis == GTK_PLOT_CANVAS_PLOT(canvas->active_item)->axis){
                    the_axis = i;
                    break;
                 }
              }

              gtk_plot_canvas_cancel_action(canvas);
              if(the_axis != -1)
                  gtk_plot_axis_hide_title(axis);
              
              gtk_plot_canvas_paint(canvas);
              gtk_widget_queue_draw(GTK_WIDGET(canvas));
              return TRUE;
            case GTK_PLOT_CANVAS_PLOT_IN_GRADIENT:
              gtk_plot_canvas_cancel_action(canvas);
              gtk_plot_data_gradient_set_visible(GTK_PLOT_CANVAS_PLOT(plot->active_layer)->data, FALSE);
              gtk_plot_canvas_paint(canvas);
              gtk_widget_queue_draw(GTK_WIDGET(canvas));
              return TRUE;
              break;
            case GTK_PLOT_CANVAS_PLOT_IN_AXIS:
              return TRUE;
            default:
              child = canvas->active_item;
              gtk_plot_canvas_cancel_action(canvas);
              gtk_plot_canvas_remove_child(canvas, child); 
              gtk_plot_canvas_paint(canvas);
              gtk_widget_queue_draw(GTK_WIDGET(canvas));
              return TRUE;
          }
        } else {
          child = canvas->active_item;
          gtk_plot_canvas_cancel_action(canvas);
          gtk_plot_canvas_remove_child(GTK_PLOT_CANVAS(canvas), child);
          gtk_plot_canvas_paint(canvas);
          gtk_widget_queue_draw(GTK_WIDGET(canvas));
        }
      }
  }

  return FALSE;
}

void
sg_plot_put_pixmap(SGplot *plot, GdkPixmap *pixmap)
{
  GtkPlotCanvas *canvas;
  GtkPlotCanvasChild *child;
  gint width, height;

  canvas = GTK_PLOT_CANVAS(plot);

  gdk_window_get_size(pixmap, &width, &height);

  child = gtk_plot_canvas_pixmap_new(pixmap, NULL);
  gtk_plot_canvas_put_child(canvas, child, 0., 0., 
                            (gdouble)width/(gdouble)canvas->pixmap_width,
                            (gdouble)height/(gdouble)canvas->pixmap_height);

  gtk_widget_queue_draw(GTK_WIDGET(canvas));
}

void
sg_plot_antialias(SGplot *plot, gboolean do_art)
{
  if(do_art && !plot->antialias){
    plot->antialias = TRUE;
#ifdef WITH_LIBART
    gtk_plot_canvas_set_pc(GTK_PLOT_CANVAS(plot), 
                           GTK_PLOT_PC(gtk_plot_art_new(GTK_WIDGET(plot))));
#endif /* WITH_LIBART */
  }
  if(!do_art && plot->antialias){
    plot->antialias = FALSE;
    gtk_plot_canvas_set_pc(GTK_PLOT_CANVAS(plot), 
                           GTK_PLOT_PC(gtk_plot_gdk_new(NULL)));
  }
}

void
sg_plot_refresh_datasets(SGplot *plot)
{
  GList *aux_layers;

  aux_layers = plot->layers;
  while(aux_layers){
      SGlayer *layer;

      layer = (SGlayer *)aux_layers->data;

      sg_layer_refresh_datasets(layer);
      aux_layers = aux_layers->next;
  }
}

void
sg_plot_copy(SGplot *plot)
{
  sg_plot_clipboard_copy(SG_PLOT_CLIPBOARD(plot->clipboard), plot);
}

void
sg_plot_paste(SGplot *plot, gdouble x, gdouble y)
{
  sg_plot_clipboard_paste(SG_PLOT_CLIPBOARD(plot->clipboard), plot, x, y);
}
