/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg.h"
#include "sg_matrix.h"
#include "sg_python_expr.h"
#include "sg_python_worksheet.h"
#define PY_ARRAY_UNIQUE_SYMBOL PyArrayXXX
#define NO_IMPORT_ARRAY
#include <arrayobject.h>

#define DEFAULT_PRECISION 3
#define DEFAULT_ROWS 20 
#define DEFAULT_COLUMNS 5 

static void     sg_matrix_class_init            (SGmatrixClass *klass);
static void     sg_matrix_init                  (SGmatrix *matrix);
static gboolean sg_matrix_update                (SGworksheet *worksheet,
						 gint from_col, gint to_col,
						 gint from_row, gint to_row);
static gboolean sg_matrix_cell_update_format	(SGworksheet *worksheet,
                             			 gint row, gint col);

static void	sg_matrix_destroy		(GtkObject *object); 

static SGworksheetClass *parent_class = NULL;

GtkType
sg_matrix_get_type (void)
{
  static GtkType sg_matrix_type = 0;

  if (!sg_matrix_type)
    {
      GtkTypeInfo sg_matrix_info =
      {
        "SGmatrix",
        sizeof (SGmatrix),
        sizeof (SGmatrixClass),
        (GtkClassInitFunc) sg_matrix_class_init,
        (GtkObjectInitFunc) sg_matrix_init,
        /* reserved_1 */ NULL,
        /* reserved_2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      sg_matrix_type = gtk_type_unique (sg_worksheet_get_type(), &sg_matrix_info);
    }

  return sg_matrix_type;
}

static void
sg_matrix_class_init (SGmatrixClass *klass)
{
  GtkWidgetClass *widget_class;
  GtkObjectClass *object_class;
  SGworksheetClass *worksheet_class;

  widget_class = (GtkWidgetClass*) klass;
  object_class = (GtkObjectClass*) klass;
  worksheet_class = (SGworksheetClass*) klass;

  parent_class = (SGworksheetClass *)gtk_type_class (sg_worksheet_get_type ());

  object_class->destroy = sg_matrix_destroy;
  worksheet_class->update_exp = sg_matrix_update;
  worksheet_class->update_cell_format = sg_matrix_cell_update_format;
}

static void
sg_matrix_init (SGmatrix *matrix)
{
  sg_matrix_set_format(matrix,
                       SG_TYPE_NUMBER,
                       SG_FORMAT_DECIMAL,
                       SG_INTERNAL_DOUBLE,
                       DEFAULT_PRECISION);

  matrix->format.exp = NULL;

  matrix->xmin = matrix->ymin = 0.;
  matrix->xmax = matrix->ymax = 10.;

  matrix->x_values = matrix->y_values = NULL;
  matrix->nx = matrix->ny = 0;
}   

SGmatrix *
sg_matrix_new(gchar *name, gint nrows, gint ncols)
{
  SGmatrix *matrix;

  matrix = SG_MATRIX(gtk_widget_new(sg_matrix_get_type(), NULL));

  if(name){
    SG_WORKSHEET(matrix)->name = g_strdup(name);
    gtk_sheet_set_title(GTK_SHEET(SG_WORKSHEET(matrix)), (const gchar *)name);
  }

  sg_worksheet_add_rows(SG_WORKSHEET(matrix), nrows - DEFAULT_ROWS);
  sg_worksheet_add_columns(SG_WORKSHEET(matrix), ncols - DEFAULT_COLUMNS);

  return matrix;
}  

void
sg_matrix_set_format(SGmatrix *matrix,
                     SGcolumntype type,
                     SGcolumnformat format,
                     SGcolumninternal internal,
                     gint precision)
{
  GtkSheet *sheet;
  gint row,col;

  sheet = GTK_SHEET(SG_WORKSHEET(matrix));

  matrix->format.type = type; 
  matrix->format.format = format; 
  matrix->format.internal = internal; 
  matrix->format.precision = precision;

  switch(matrix->format.type){
    case SG_TYPE_NUMBER:
      for(col = 0; col <= sheet->maxalloccol; col++)
       gtk_sheet_column_set_justification(sheet, col, GTK_JUSTIFY_RIGHT);
       break;
    case SG_TYPE_DATE:
    case SG_TYPE_TIME:
      for(col = 0; col <= sheet->maxalloccol; col++)
       gtk_sheet_column_set_justification(sheet, col, GTK_JUSTIFY_CENTER);
       break;
    case SG_TYPE_TEXT:
    default:
      for(col = 0; col <= sheet->maxalloccol; col++)
       gtk_sheet_column_set_justification(sheet, col, GTK_JUSTIFY_LEFT);
  }

  gtk_sheet_freeze(sheet);
  for(col = 0; col <= sheet->maxalloccol; col++)
    for(row = 0; row <= sheet->maxallocrow; row++)
      sg_worksheet_cell_update_format(SG_WORKSHEET(matrix), row, col);
  gtk_sheet_thaw(sheet);
}

void
sg_matrix_destroy(GtkObject *object)
{
  SGmatrix *matrix;

  matrix = SG_MATRIX(object);

  if(matrix->x_values) g_free(matrix->x_values);
  matrix->x_values = NULL;
  if(matrix->y_values) g_free(matrix->y_values);
  matrix->x_values = NULL;
  if(matrix->format.exp) g_free(matrix->format.exp);
  matrix->format.exp = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

void
sg_matrix_set_exp(SGmatrix *matrix, gchar *exp)
{
  if(matrix->format.exp)
     g_free(matrix->format.exp);

  matrix->format.exp = NULL;

  if(exp && strlen(exp) > 0)
     matrix->format.exp = g_strdup(exp);
}

static gboolean 
sg_matrix_update (SGworksheet *worksheet,
                  gint from_row, gint to_row,
                  gint from_col, gint to_col)
{
  PyObject *object;
  gint i,j;
  gdouble x, y, dx, dy, z;
  gboolean TwoD=FALSE, OneD=FALSE;
  SGmatrix *matrix = SG_MATRIX(worksheet);

  if(!matrix->format.exp) return FALSE;

  dx=(matrix->xmax-matrix->xmin)/gtk_sheet_get_columns_count(GTK_SHEET(worksheet));
  dy=(matrix->ymax-matrix->ymin)/gtk_sheet_get_rows_count(GTK_SHEET(worksheet));
  gtk_sheet_freeze(GTK_SHEET(worksheet));

  for (j=from_col;j<=to_col;j++) /* inclusive */
  { x=(gdouble)j*dx + matrix->xmin;
    OneD=FALSE;
     for (i=from_row;i<=to_row;i++) /* inclusive */
     { y= matrix->ymin+i*dy;
       object = sg_eval_func_xy(matrix->format.exp, x, y, &z);
       if (object)
       {   if (PyArray_Check(object))
           { PyArrayObject *array;
             array=(PyArrayObject *)object;
             if (array->nd>2)   /*No more than 2 dimensions */
             {
                gtk_sheet_thaw(GTK_SHEET(worksheet));
                return FALSE;
             }
             if (array->nd==2) TwoD=TRUE;
             if (array->nd==1) OneD=TRUE;
             python_array(worksheet,i,j,(PyArrayObject *)array,
                          GTK_ORIENTATION_VERTICAL,TRUE);
           }
           else if (PySequence_Check(object))
           {
             python_sequence(worksheet,0,0,object,
                             GTK_ORIENTATION_VERTICAL,TRUE,FALSE);
             OneD=TRUE;
           }
           else if (object!=Py_None)
              python_singleton(worksheet,i,j,object,FALSE,TRUE);
          else
            sg_worksheet_cell_set(worksheet, i, j, matrix->format.exp, TRUE, TRUE);
           Py_XDECREF(object);
        }
        if (OneD || TwoD) break;
     }
     if (TwoD) break;
  }
  gtk_sheet_thaw(GTK_SHEET(worksheet));
  return TRUE;
}

static gboolean
sg_matrix_cell_update_format(SGworksheet *worksheet,
                             gint row, gint col)
{
  GtkSheet *sheet;
  gchar *string = NULL, fpnum[40], pspec[20];
  SGhiddencell *hidden;
  guint type, format, internal, precision;

  sheet = GTK_SHEET(worksheet);
  hidden = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);
  if (!hidden) return FALSE;
  fpnum[0]='\0';
  pspec[0]='\0';

  type = SG_MATRIX(worksheet)->format.type;
  format = SG_MATRIX(worksheet)->format.format;
  internal = SG_MATRIX(worksheet)->format.internal;
  precision = SG_MATRIX(worksheet)->format.precision;

/* Convert to new format */
  if(type == SG_TYPE_NUMBER && hidden->type == SG_TYPE_NUMBER) {
    if(hidden->internal != internal){
      if(internal != SG_INTERNAL_INTEGER){
        glong v = hidden->value.val_long;
        hidden->value.val_double = v;
      } else {
        gdouble v = hidden->value.val_double;
        hidden->value.val_long = v;
      }    
    }
  } else if(type == SG_TYPE_NUMBER && hidden->type != SG_TYPE_NUMBER) {
    gchar *s = hidden->value.val_char;
    hidden->value.val_char = NULL;
    if(internal == SG_INTERNAL_INTEGER)
        hidden->value.val_long = atoi(s);
    else
        hidden->value.val_double = atof(s);
    g_free(s);
  } else if(type != SG_TYPE_NUMBER && hidden->type == SG_TYPE_NUMBER) {
    if(hidden->formula)
      hidden->value.val_char = g_strdup(hidden->formula);
    else
      hidden->value.val_char = g_strdup(gtk_sheet_cell_get_text(sheet,row,col));  }
/* Set string */

  hidden->type = type;
  hidden->format = format;
  hidden->internal = internal;
  hidden->precision = precision;

  if(hidden->type == SG_TYPE_NUMBER){
    if(hidden->internal == SG_INTERNAL_INTEGER){
      switch(hidden->format){
        case SG_FORMAT_DECIMAL:
          g_snprintf(pspec,20,"%%ld");
          g_snprintf(fpnum,40,pspec,hidden->value.val_long);
          string=fpnum;
          break;
        case SG_FORMAT_SCIENTIFIC:
          g_snprintf(pspec,20,"%%1.%de",hidden->precision);
          g_snprintf(fpnum,40,pspec,(double)hidden->value.val_long);
          string=fpnum;
          break;
      }
    } else {
      switch(hidden->format){
        case SG_FORMAT_DECIMAL:
          g_snprintf(pspec,20,"%%1.%df",hidden->precision);
          g_snprintf(fpnum,40,pspec,hidden->value.val_double);
          string=fpnum;
          break;
        case SG_FORMAT_SCIENTIFIC:
          g_snprintf(pspec,20,"%%1.%de",hidden->precision);
          g_snprintf(fpnum,40,pspec,hidden->value.val_double);
          string=fpnum;
          break;
      }
    }
  } else {
      string=hidden->value.val_char;
  }
  sg_worksheet_cell_set_text(worksheet,row, col, string);


  return TRUE;
}

void
sg_matrix_set_xrange(SGmatrix *matrix, gdouble xmin, gdouble xmax)
{
  matrix->xmin = xmin;
  matrix->xmax = xmax;
}

void
sg_matrix_set_yrange(SGmatrix *matrix, gdouble ymin, gdouble ymax)
{
  matrix->ymin = ymin;
  matrix->ymax = ymax;
}

void
sg_matrix_set_x_values(SGmatrix *matrix, gdouble *x, gint nx)
{
  if(matrix->x_values) g_free(matrix->x_values);
  matrix->x_values = NULL;
  matrix->nx = 0;

  if(x && nx > 0){
    gint i;
    matrix->x_values = g_new(gdouble, nx);
    for(i = 0; i < nx; i++) matrix->x_values[i] = x[i];
    matrix->nx = nx;
  }
}

void
sg_matrix_set_y_values(SGmatrix *matrix, gdouble *y, gint ny)
{
  if(matrix->y_values) g_free(matrix->y_values);
  matrix->y_values = NULL;
  matrix->ny = 0;

  if(y && ny > 0){
    gint i;
    matrix->y_values = g_new(gdouble, ny);
    for(i = 0; i < ny; i++) matrix->y_values[i] = y[i];
    matrix->ny = ny;
  }
}
