/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <gdk/gdkkeysyms.h>
#include "sg_entry.h"
#include "sg_locale.h"
#include "config.h"

static char * bold_xpm[] = {
"16 16 2 1",
" 	c None",
".	c #000000000000",
"                ",
"  .........     ",
"   ...   ...    ",
"   ...    ...   ",
"   ...    ...   ",
"   ...    ...   ",
"   ...   ...    ",
"   ........     ",
"   ...    ...   ",
"   ...     ...  ",
"   ...     ...  ",
"   ...     ...  ",
"   ...     ...  ",
"   ...    ...   ",
"  .........     ",
"                "};


/* XPM */
static char * italic_xpm[] = {
"16 16 2 1",
" 	c None",
".	c #000000000000",
"                ",
"        .....   ",
"         ...    ",
"         ...    ",
"        ...     ",
"        ...     ",
"       ...      ",
"       ...      ",
"      ...       ",
"      ...       ",
"     ...        ",
"     ...        ",
"    ...         ",
"    ...         ",
"   .....        ",
"                "};

static void         sg_entry_class_init      	(SGentryClass *klass);
static void         sg_entry_init            	(SGentry      *entry);
static void         sg_entry_destroy         	(GtkObject     *entry);
static void         edit_text			(GtkWidget *text_entry, 
						 gchar *text);
static void         edit_normal			(GtkWidget *widget, 
						 gpointer data);
static void         edit_bold			(GtkWidget *widget, 
						 gpointer data);
static void         edit_italic			(GtkWidget *widget,
						 gpointer data);
static void         edit_plus			(GtkWidget *widget, 
						 gpointer data);
static void         edit_minus			(GtkWidget *widget,
						 gpointer data);
static void         edit_sub			(GtkWidget *widget,
						 gpointer data);
static void         edit_super			(GtkWidget *widget,
						 gpointer data);
static void         edit_greek			(GtkWidget *widget,
						 gpointer data);
static void         new_font			(GtkFontCombo *font_combo,
						 gpointer data);
static void         sg_char_dialog		(SGentry *entry_widget);


static GtkFrameClass *parent_class = NULL;

static gint return_value=-1;


static void
sg_entry_class_init (SGentryClass * klass)
{
  GtkObjectClass *object_class;
  GtkWidgetClass *widget_class;

  parent_class = (GtkFrameClass *) gtk_type_class (gtk_frame_get_type ());
  object_class = (GtkObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;

  object_class->destroy = sg_entry_destroy;
}

static void
sg_entry_destroy (GtkObject * entry)
{

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (entry);
}


static void
sg_entry_init (SGentry * entry)
{
  GtkWidget *widget;
  GtkWidget *child;
  GtkWidget *vbox, *table, *label_frame;
  GtkFrame *frame;
  GdkPixmap *pixmap;
  GtkWidget *font_pixmap;
  GdkBitmap *mask;

  widget=GTK_WIDGET(entry);

  frame = GTK_FRAME(entry);

  vbox = gtk_vbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(frame), vbox);
  gtk_container_set_border_width(GTK_CONTAINER(vbox), 10);

  entry->font_combo = gtk_font_combo_new();

  child = GTK_FONT_COMBO(entry->font_combo)->bold_button;
  gtk_container_remove(GTK_CONTAINER(entry->font_combo), child);
  child = GTK_FONT_COMBO(entry->font_combo)->italic_button;
  gtk_container_remove(GTK_CONTAINER(entry->font_combo), child);

  gtk_box_pack_start(GTK_BOX(vbox), entry->font_combo, FALSE, FALSE, 0);

  GTK_FONT_COMBO(entry->font_combo)->bold_button = entry->bold_button = gtk_button_new();
  GTK_FONT_COMBO(entry->font_combo)->italic_button = entry->italic_button = gtk_button_new();

  gtk_container_add(GTK_CONTAINER(entry->font_combo), entry->bold_button);
  gtk_container_add(GTK_CONTAINER(entry->font_combo), entry->italic_button);
  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               bold_xpm);
  font_pixmap=gtk_pixmap_new(pixmap, mask);
  gtk_container_add(GTK_CONTAINER(entry->bold_button), font_pixmap);

  pixmap=gdk_pixmap_colormap_create_from_xpm_d(NULL,
                                               gdk_colormap_get_system(),
                                               &mask, NULL,
                                               italic_xpm);

  font_pixmap=gtk_pixmap_new(pixmap, mask);
  gtk_container_add(GTK_CONTAINER(entry->italic_button), font_pixmap);
  gtk_widget_show_all(GTK_WIDGET(frame));

  table = gtk_table_new(3,2, FALSE);
  gtk_container_add(GTK_CONTAINER(entry->font_combo), table);

  entry->subs_button = gtk_button_new_with_label( _("s") );
  entry->super_button = gtk_button_new_with_label( _("S") );
  entry->plus_button = gtk_button_new_with_label( _("+") );
  entry->minus_button = gtk_button_new_with_label( _("-") );
  entry->greek_button = gtk_button_new_with_label( _("Greek") );
  entry->normal_button = gtk_button_new_with_label( _("Normal") );
  entry->spec_button = gtk_button_new_with_label( _("Special") );
  gtk_table_attach_defaults(GTK_TABLE(table), entry->super_button, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), entry->subs_button, 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), entry->plus_button, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), entry->minus_button, 1, 2, 1, 2);

  gtk_table_attach_defaults(GTK_TABLE(table), entry->greek_button, 2, 3, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), entry->spec_button, 2, 3, 1, 2);

  gtk_container_add(GTK_CONTAINER(entry->font_combo), entry->normal_button);

  label_frame = gtk_frame_new( _("Label") );
  gtk_box_pack_start(GTK_BOX(vbox), label_frame, FALSE, FALSE, 0);
  entry->text_entry = gtk_entry_new();
  gtk_container_add(GTK_CONTAINER(label_frame),entry->text_entry);

  /* Signals */
  gtk_signal_connect_object (GTK_OBJECT (entry->bold_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_bold),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->italic_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_italic),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->subs_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_sub),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->super_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_super),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->plus_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_plus),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->minus_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_minus),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->greek_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_greek),
                             GTK_OBJECT(entry->text_entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->spec_button), "clicked",
                             GTK_SIGNAL_FUNC (sg_char_dialog),
                             GTK_OBJECT(entry));
  gtk_signal_connect_object (GTK_OBJECT (entry->normal_button), "clicked",
                             GTK_SIGNAL_FUNC (edit_normal),
                             GTK_OBJECT(entry->text_entry));

  gtk_signal_connect (GTK_OBJECT (entry->font_combo), "changed",
                      GTK_SIGNAL_FUNC (new_font), entry->text_entry);

  gtk_widget_show_all(GTK_WIDGET(frame));
}

guint
sg_entry_get_type ()
{
  static guint entry_type = 0;

  if (!entry_type)
    {
      GtkTypeInfo entry_info =
      {
	"SGentry",
	sizeof (SGentry),
	sizeof (SGentryClass),
	(GtkClassInitFunc) sg_entry_class_init,
	(GtkObjectInitFunc) sg_entry_init,
	NULL,
	NULL,
	(GtkClassInitFunc) NULL,
      };
      entry_type = gtk_type_unique (gtk_frame_get_type (), &entry_info);
    }
  return entry_type;
}

GtkWidget *
sg_entry_new ()
{
  GtkFontCombo *entry;

  entry = (GtkFontCombo *)gtk_type_new (sg_entry_get_type ());

  return(GTK_WIDGET(entry));
}

static void
edit_text(GtkWidget *text_entry, gchar *text)
{
  gint position;

  position = gtk_editable_get_position(GTK_EDITABLE(text_entry));
  gtk_editable_insert_text(GTK_EDITABLE(text_entry), text, strlen(text), &position);

  gtk_editable_set_position(GTK_EDITABLE(text_entry), position);
  gtk_widget_grab_focus(text_entry);
}


static void
edit_normal(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\N");
}

static void
edit_bold(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\B");
}

static void
edit_italic(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\i");
}

static void
edit_plus(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\+");
}

static void
edit_minus(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\-");
}

static void
edit_sub(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\s");
}

static void
edit_super(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\S");
}

static void
edit_greek(GtkWidget *widget, gpointer data)
{
  edit_text(widget, "\\g");
}

static void
new_font(GtkFontCombo *font_combo, gpointer data)
{
/*
  GtkWidget *text_entry;
  GtkStyle *style;

  text_entry = GTK_WIDGET(data);

  style = gtk_style_copy (text_entry->style);
  gdk_font_unref (style->font);
  style->font = font_combo->font;
  gdk_font_ref (style->font);

  gtk_widget_set_style (text_entry, style);
  gtk_style_unref(style);
*/
}

static gboolean
ok_clicked(GtkWidget *widget)
{
  return_value = 0;
  gtk_main_quit();
  return FALSE;
}

static gboolean
cancel_clicked(GtkWidget *widget)
{
  return_value = 1;
  gtk_main_quit();
  return FALSE;
}


static void
sg_char_dialog(SGentry *entry_widget)
{
  GtkWidget *dialog;
  GtkCharSelection *charsel;
  gint retval;
  gchar mes[8];
  GtkStyle *style;

  dialog=gtk_char_selection_new ();
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);

  gtk_window_set_modal (GTK_WINDOW(dialog),TRUE);

  charsel = GTK_CHAR_SELECTION(dialog);

/*  sg_dialog_new(dialog);*/

  gtk_window_set_title (GTK_WINDOW(dialog), _("SciGraphica: Special character") );

  /* connect signals */
  gtk_signal_connect_object (GTK_OBJECT (charsel->ok_button), "clicked",
                             GTK_SIGNAL_FUNC (ok_clicked),
                             GTK_OBJECT (dialog));

  gtk_signal_connect_object (GTK_OBJECT (charsel->cancel_button), "clicked",
                               GTK_SIGNAL_FUNC (cancel_clicked),
                               GTK_OBJECT (dialog));


  style = dialog->style;

  gtk_font_combo_select(charsel->font_combo,
                        "Symbol",
                        FALSE,
                        FALSE,
                        12);
  
  gtk_widget_hide(charsel->font_combo->size_combo);
  gtk_widget_hide(charsel->font_combo->bold_button);  
  gtk_widget_hide(charsel->font_combo->italic_button);

  gtk_widget_show(dialog);
  /* wait until dialog get destroyed */
  return_value = -1;
  gtk_main();
  retval=return_value; 

  if(charsel->selection >= 0){
    GtkWidget *child;
    gint position;

    child = (GtkWidget *)GTK_LIST(GTK_COMBO(charsel->font_combo->name_combo)->list)->selection->data;
    position = gtk_list_child_position(GTK_LIST(GTK_COMBO(charsel->font_combo->name_combo)->list),child);

    g_snprintf(mes,8,"\\%d\\x%03d",position,charsel->selection);
    edit_text(entry_widget->text_entry, strdup(mes));
  }
  if(GTK_IS_WIDGET(dialog)) gtk_widget_destroy(dialog);   
}
