/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_clipboard.h"

static void sg_clipboard_class_init   (SGclipboardClass *klass);
static void sg_clipboard_init         (SGclipboard *cb);
static void sg_clipboard_finalize      (GObject *o);

static GObjectClass *parent_class = NULL;

GType
sg_clipboard_get_type (void)
{
  static GType sg_clipboard_type = 0;
                                                                                
  if (!sg_clipboard_type)
    {
      static const GTypeInfo sg_clipboard_info =
      {
        sizeof (SGclipboardClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_clipboard_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGclipboard),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_clipboard_init,
	NULL
      };
                                                                                
      sg_clipboard_type = g_type_register_static (G_TYPE_OBJECT, "SGclipboard",
                                               &sg_clipboard_info, 0);
    }
                                                                                
  return sg_clipboard_type;
}

GObject *
sg_clipboard_new(void)
{
  return G_OBJECT(g_object_new(sg_clipboard_get_type(), NULL));
}

static void
sg_clipboard_class_init(SGclipboardClass *klass)
{
  GObjectClass *object_class;
  SGclipboardClass *cb_class;

  object_class = (GObjectClass*) klass;
  cb_class = (SGclipboardClass*) klass;

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = sg_clipboard_finalize;
}

static void
sg_clipboard_init(SGclipboard *clipboard)
{
  clipboard->selection = NULL;
}

static void
sg_clipboard_finalize(GObject *o)
{
  SGclipboard *clipboard = SG_CLIPBOARD(o);
  sg_clipboard_clear(clipboard);
}


void
sg_clipboard_copy(SGclipboard *cb, SGworksheet *worksheet, gboolean clear)
{
  gint row, col;
  gchar *ftext;
  const gchar *stext;
  const gchar *text = NULL;
  GtkSheetCell *cell=NULL;
  SGhiddencell *link=NULL;
  GtkSheet *sheet=GTK_SHEET(worksheet);
  gint srow,scol,num_selected=0;

  sg_clipboard_clear(cb);

  gtk_sheet_freeze(sheet);
  gtk_sheet_get_active_cell(sheet,&srow,&scol);  

  num_selected=((sheet->range.rowi-sheet->range.row0)+1)*
               ((sheet->range.coli-sheet->range.col0)+1);
  for(row = sheet->range.row0; row <= sheet->range.rowi; row++){
    for(col = sheet->range.col0; col <= sheet->range.coli; col++){
       ftext=NULL;
       stext=NULL;
       text = sg_worksheet_cell_get_text(worksheet, row, col);
       link = (SGhiddencell *)gtk_sheet_get_link(sheet, row, col);

       if (link) ftext = link->formula;

       if (row==srow && col==scol && num_selected==1)
        stext=gtk_entry_get_text(GTK_ENTRY(sheet->sheet_entry));

       if (stext && strlen(stext)>0) text=stext;
/*
       else if (ftext && strlen(ftext)>0) text=ftext;
*/
              
       if(text && strlen(text)>0){
           SGhiddencell *hidden;

           cell = g_new0(GtkSheetCell, 1);
           cell->text = g_strdup(text);
           cell->row = row - sheet->range.row0;
           cell->col = col - sheet->range.col0;

          /* This way we tell the paste function we are sending a formula: */

           hidden = g_new0(SGhiddencell, 1); 
           *hidden = *((SGhiddencell *)link);
           if(hidden->formula) hidden->formula = g_strdup(hidden->formula);
           if(hidden->type == SG_TYPE_TEXT && hidden->value.val_char)
             hidden->value.val_char = g_strdup(hidden->value.val_char);

           cell->link = hidden;
           cb->selection = g_list_append(cb->selection, cell);
       }
    }
  }

  if (clear)
   for (row = sheet->range.row0; row <= sheet->range.rowi; row++)
    for (col = sheet->range.col0; col <= sheet->range.coli; col++)
      { 
        gtk_sheet_remove_link(sheet, row, col);
        gtk_sheet_cell_delete(sheet, row, col);
        if (row==srow && col==scol)
            gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(""));
      }      

  gtk_sheet_thaw(sheet);
}

void
sg_clipboard_paste(SGclipboard *cb, SGworksheet *worksheet)
{
  GtkSheetCell *cell;
  GList *list;
  GtkSheet *sheet=GTK_SHEET(worksheet);
  gint row,col,srow,scol;
  gint nrows, ncols;

  ncols = gtk_sheet_get_columns_count(sheet);
  nrows = gtk_sheet_get_rows_count(sheet);

  gtk_sheet_freeze(sheet);

  list = cb->selection;
  gtk_sheet_get_active_cell(sheet,&srow,&scol);
  while(list){
   cell = (GtkSheetCell *)list->data;
   row=sheet->range.row0 + cell->row;
   col=sheet->range.col0 + cell->col;

   if(row > nrows-1) {
        sg_worksheet_add_rows(worksheet, row-nrows+1);
        nrows = gtk_sheet_get_rows_count(sheet);
   }
   if(col > ncols-1) {
        sg_worksheet_add_columns(worksheet, row-nrows+1);
        ncols = gtk_sheet_get_columns_count(sheet);
   }

   if (cell->link){ /* We were sent a hidden cell, we clone it */
    SGhiddencell *hidden;
    sg_worksheet_hidden_cell_clear(worksheet, row, col);
   
    hidden = g_new0(SGhiddencell, 1); 
    *hidden = *((SGhiddencell *)cell->link);
    if(hidden->formula) hidden->formula = g_strdup(hidden->formula);
    if(hidden->type == SG_TYPE_TEXT && hidden->value.val_char)
          hidden->value.val_char = g_strdup(hidden->value.val_char);
    gtk_sheet_link_cell(sheet, row, col, hidden);
 
    sg_worksheet_cell_set_text(worksheet, row, col, cell->text);
   }else{ /* we were sent plain text */
    sg_worksheet_hidden_cell_clear(worksheet, row, col);
    sg_worksheet_cell_set(worksheet, row, col, 
                          cell->text, FALSE, FALSE);
   }

   /* We are pasting into a selected cell, put text in the entry widget */
/*
   if (row==srow && col==scol)
    { 
      if (cell->link)
         gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(cell->text));
      else
         gtk_entry_set_text(GTK_ENTRY(sheet->sheet_entry),g_strdup(""));
    }
*/
    list = list->next;
  }
  gtk_sheet_thaw(sheet);
}

void
sg_clipboard_clear(SGclipboard *cb)
{
  GtkSheetCell *cell;
  GList *list;

  list = cb->selection;
  while(list){
    cell = (GtkSheetCell *)list->data;
    if(cell->link){
        SGhiddencell *hidden = (SGhiddencell *)cell->link;
        if (hidden->formula){
          g_free(hidden->formula);
          hidden->formula = NULL;
        }
        switch(hidden->type){
          case SG_TYPE_DATE:
          case SG_TYPE_TIME:
          case SG_TYPE_TEXT:
            if (hidden->value.val_char){
              g_free(hidden->value.val_char);
              hidden->value.val_char = NULL;
            }
            break;
          default:
            break;
        }
    }
    g_free(cell->text);
    g_free(cell);

    cb->selection = g_list_remove_link(cb->selection, list);
    g_list_free_1(list);
    list = cb->selection;
  }

  sg_clipboard_init(cb);
}

