/* gtkplotart - wrapper for libart
 * Copyright 1999-2001  Adrian E. Feiguin <feiguin@ifir.edu.ar>
 */

#ifdef WITH_LIBART
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <gtk/gtk.h>

#include <gtkextra/gtkplotpc.h>
#include <gtkextra/gtkplot.h>
#include <gtkextra/gtkpsfont.h>
#include "gtkplotart.h"

static void gtk_plot_art_init                       (GtkPlotArt *pc);
static void gtk_plot_art_class_init                 (GtkPlotArtClass *klass);
static void gtk_plot_art_finalize                   (GObject *object);
static void gtk_plot_art_set_drawable               (GtkPlotArt *pc,
                                                     GdkDrawable *drawable);
static gboolean gtk_plot_art_real_init              (GtkPlotPC *pc);
static void gtk_plot_art_set_viewport               (GtkPlotPC *pc,
						     gdouble w, gdouble h);
static void gtk_plot_art_leave                      (GtkPlotPC *pc);
static void gtk_plot_art_gsave                      (GtkPlotPC *pc);
static void gtk_plot_art_grestore                   (GtkPlotPC *pc);
static void gtk_plot_art_clip                       (GtkPlotPC *pc,
                                                     const GdkRectangle *area);
static void gtk_plot_art_clip_mask                  (GtkPlotPC *pc,
						     gdouble x,
						     gdouble y,
                                                     const GdkBitmap *mask);
static void gtk_plot_art_set_color                   (GtkPlotPC *pc,
                                                     const GdkColor *color);
static void gtk_plot_art_set_lineattr           (GtkPlotPC *pc,
                                                 gfloat line_width,
                                                 GdkLineStyle line_style,
                                                 GdkCapStyle cap_style,
                                                 GdkJoinStyle join_style);
static void gtk_plot_art_set_dash                    (GtkPlotPC *pc,
                                                     gdouble offset_,
                                                     gdouble *values,
                                                     gint num_values);
static void gtk_plot_art_draw_point                  (GtkPlotPC *pc,
                                                     gdouble x, gdouble y);
static void gtk_plot_art_draw_line                   (GtkPlotPC *pc,
                                                     gdouble x1, gdouble y1,
                                                     gdouble x2, gdouble y2);
static void gtk_plot_art_draw_lines                  (GtkPlotPC *pc,
                                                     GtkPlotPoint *points,
                                                     gint numpoints);
static void gtk_plot_art_draw_rectangle              (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble width, 
                                                     gdouble height);
static void gtk_plot_art_draw_polygon                (GtkPlotPC *pc,
                                                     gint filled,
                                                     GtkPlotPoint *points,
                                                     gint numpoints);
static void gtk_plot_art_draw_circle                 (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble size);
static void gtk_plot_art_draw_ellipse                (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble width, 
                                                     gdouble height);
static void gtk_plot_art_set_font                    (GtkPlotPC *pc,
						     GtkPSFont *psfont,
                                                     gint height);
static void gtk_plot_art_draw_string                (GtkPlotPC *pc,
                                                     gint x, gint y,
                                                     gint angle,
                                                     const GdkColor *fg,
                                                     const GdkColor *bg,
                                                     gboolean transparent,
                                                     gint border,
                                                     gint border_space,
                                                     gint border_width,
                                                     gint shadow_width,
                                                     const gchar *font,
                                                     gint height,
                                                     GtkJustification just,
                                                     const gchar *text);
static void gtk_plot_art_draw_pixmap                (GtkPlotPC *pc,
                                                     GdkPixmap *pixmap,
                                                     GdkBitmap *mask,
                                                     gint xsrc, gint ysrc,
                                                     gint xdest, gint ydest,
                                                     gint width, gint height,
                                                     gdouble scale_x, 
                                                     gdouble scale_y);
static ArtSVP * gtk_plot_art_stroke		    (GtkPlotArt *art,
						     ArtVpath *path);

static GtkPlotGdkClass *parent_class = NULL;

GtkType
gtk_plot_art_get_type (void)
{
  static GtkType pc_type = 0;

  if (!pc_type)
    {
      GtkTypeInfo pc_info =
      {
        "GtkPlotArt",
        sizeof (GtkPlotArt),
        sizeof (GtkPlotArtClass),
        (GtkClassInitFunc) gtk_plot_art_class_init,
        (GtkObjectInitFunc) gtk_plot_art_init,
        /* reserved 1*/ NULL,
        /* reserved 2 */ NULL,
        (GtkClassInitFunc) NULL,
      };

      pc_type = gtk_type_unique (GTK_TYPE_PLOT_GDK, &pc_info);
    }
  return pc_type;
}

static void
gtk_plot_art_init (GtkPlotArt *pc)
{
  GdkWindowAttr attributes;
  gint attributes_mask;

  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.title = NULL;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.visual = gdk_visual_get_system ();
  attributes.colormap = gdk_colormap_get_system ();
  attributes.event_mask = 0;
  attributes_mask = GDK_WA_VISUAL | GDK_WA_COLORMAP;

  GTK_PLOT_GDK(pc)->drawable = NULL;
/*
  GTK_PLOT_GDK(pc)->window = gdk_window_new (NULL, &attributes, attributes_mask);
  GTK_PLOT_GDK(pc)->gc = gdk_gc_new(GTK_PLOT_GDK(pc)->window);
*/

  pc->line_width = 0.25;
  pc->line_style = GDK_LINE_SOLID;
  pc->cap_style = 0;
  pc->join_style = 0;
  pc->dash.offset = 0;
  pc->dash.n_dash = 0;
  pc->dash.dash = NULL;
}


static void
gtk_plot_art_class_init (GtkPlotArtClass *klass)
{
  GtkObjectClass *object_class;
  GObjectClass *gobject_class;
  GtkPlotPCClass *pc_class;

  parent_class = gtk_type_class (gtk_plot_gdk_get_type ());

  object_class = (GtkObjectClass *) klass;
  gobject_class = (GObjectClass *) klass;
  pc_class = (GtkPlotPCClass *) klass;

  gobject_class->finalize = gtk_plot_art_finalize;

  pc_class->init = gtk_plot_art_real_init;
  pc_class->leave = gtk_plot_art_leave;
  pc_class->set_viewport = gtk_plot_art_set_viewport;
  pc_class->gsave = gtk_plot_art_gsave;
  pc_class->grestore = gtk_plot_art_grestore;
  pc_class->clip = gtk_plot_art_clip;
  pc_class->clip_mask = gtk_plot_art_clip_mask;
  pc_class->set_color = gtk_plot_art_set_color;
  pc_class->set_dash = gtk_plot_art_set_dash;
  pc_class->set_lineattr = gtk_plot_art_set_lineattr;
  pc_class->draw_point = gtk_plot_art_draw_point;
  pc_class->draw_line = gtk_plot_art_draw_line;
  pc_class->draw_lines = gtk_plot_art_draw_lines;
  pc_class->draw_rectangle = gtk_plot_art_draw_rectangle;
  pc_class->draw_polygon = gtk_plot_art_draw_polygon;
  pc_class->draw_circle = gtk_plot_art_draw_circle;
  pc_class->draw_ellipse = gtk_plot_art_draw_ellipse;
  pc_class->set_font = gtk_plot_art_set_font;
  pc_class->draw_string = gtk_plot_art_draw_string;
  pc_class->draw_pixmap = gtk_plot_art_draw_pixmap;
}


GtkObject *
gtk_plot_art_new (GtkWidget *widget)
{
  GtkObject *object;

  object = gtk_type_new(gtk_plot_art_get_type());

  gdk_rgb_init();

  gtk_plot_gdk_construct(GTK_PLOT_GDK(object), widget);

  return (object);
}


static void
gtk_plot_art_finalize (GObject *object)
{
  gdk_window_unref(GTK_PLOT_GDK(object)->window);
}

static void
gtk_plot_art_set_drawable(GtkPlotArt *pc, GdkDrawable *drawable)
{
  GTK_PLOT_GDK(pc)->drawable = drawable;
}

static gboolean 
gtk_plot_art_real_init (GtkPlotPC *pc)
{
  return TRUE;
}

static void
gtk_plot_art_leave (GtkPlotPC *pc)
{
  GdkGC *gc;

  gc = gdk_gc_new(GTK_PLOT_GDK(pc)->drawable);
  gdk_gc_set_foreground(gc, &pc->color);

  gdk_draw_rgb_image(GTK_PLOT_GDK(pc)->drawable, gc, 
                     0, 0, pc->width, pc->height,
		     GDK_RGB_DITHER_NONE,
		     (guchar *)GTK_PLOT_ART(pc)->buf, pc->width * 3); 

  gdk_gc_unref(gc);
}

static void 
gtk_plot_art_set_viewport               (GtkPlotPC *pc, gdouble w, gdouble h)
{
  GtkPlotArt *art;

  art = GTK_PLOT_ART(pc);

  if(art->buf)
    art_free(art->buf);

  art->buf = art_new(art_u8, w*h*3);
  art_rgb_run_alpha(art->buf, 0xFF, 0xFF, 0xFF, 0xFF, w * h);
}

static void 
gtk_plot_art_gsave                                  (GtkPlotPC *pc)
{
}

static void 
gtk_plot_art_grestore                                  (GtkPlotPC *pc)
{
}

static void 
gtk_plot_art_clip                                   (GtkPlotPC *pc,
                                                     const GdkRectangle *area)
{
/*
  g_warning ("Clipping has not been implemented with antialiasing.");
*/
}

static void 
gtk_plot_art_clip_mask                              (GtkPlotPC *pc,
						     gdouble x,
						     gdouble y,
                                                     const GdkBitmap *mask)
{
}

static void 
gtk_plot_art_set_color                               (GtkPlotPC *pc,
                                                     const GdkColor *color)
{
  GdkColor new_color;

  new_color = *color;
  gdk_color_alloc(gdk_colormap_get_system(), &new_color);

  pc->color = new_color;
}

static void 
gtk_plot_art_set_dash                               (GtkPlotPC *pc,
                                                    gdouble offset,
                                                    gdouble *values,
                                                    gint num_values)
{
  GtkPlotArt *art = GTK_PLOT_ART(pc);
  gint i;

  if(num_values == 0){
    if(art->dash.dash) g_free(art->dash.dash);
    art->dash.n_dash = 0;
    art->dash.dash = NULL;
    return;
  }

  art->dash.offset = 0;
  art->dash.n_dash = num_values;
  if(art->dash.dash) g_free(art->dash.dash);
  art->dash.dash = g_new0(gdouble, num_values);

  for(i = 0; i < num_values; i++){
     art->dash.dash[i] = values[i];
  }  

}

static void gtk_plot_art_set_lineattr           (GtkPlotPC *pc,
                                                 gfloat line_width,
                                                 GdkLineStyle line_style,
                                                 GdkCapStyle cap_style,
                                                 GdkJoinStyle join_style)
{
  gint art_join;
  gint art_cap;

  switch (join_style) {
    case GDK_JOIN_MITER:
          art_join = ART_PATH_STROKE_JOIN_MITER;
	  break;
    case GDK_JOIN_ROUND:
          art_join = ART_PATH_STROKE_JOIN_ROUND;
	  break;
    case GDK_JOIN_BEVEL:
          art_join = ART_PATH_STROKE_JOIN_BEVEL;
	  break;
    default:
          art_join = ART_PATH_STROKE_JOIN_MITER; 
	  break;
  }

  switch (cap_style) {
    case GDK_CAP_BUTT:
    case GDK_CAP_NOT_LAST:
          art_cap = ART_PATH_STROKE_CAP_BUTT;
	  break;
    case GDK_CAP_ROUND:
          art_cap = ART_PATH_STROKE_CAP_ROUND;
	  break;
    case GDK_CAP_PROJECTING:
          art_cap = ART_PATH_STROKE_CAP_SQUARE;
	  break;
    default:
          art_cap = ART_PATH_STROKE_CAP_BUTT; /* shut up the compiler */
	  break;
  }

  GTK_PLOT_ART(pc)->line_style = line_style;
  GTK_PLOT_ART(pc)->join_style = art_join;
  GTK_PLOT_ART(pc)->cap_style = art_cap;
  GTK_PLOT_ART(pc)->line_width = MAX(line_width, 0.25);

  if(line_style == GDK_LINE_SOLID)
     gtk_plot_art_set_dash(pc, 0, NULL, 0);

}

static void 
gtk_plot_art_draw_point                              (GtkPlotPC *pc,
                                                     gdouble x, gdouble y)
{
  gtk_plot_art_draw_line(pc, x, y, x, y);
}

static void 
gtk_plot_art_draw_line                               (GtkPlotPC *pc,
                                                     gdouble x1, gdouble y1,
                                                     gdouble x2, gdouble y2)
{
  GtkPlotArt *art = GTK_PLOT_ART(pc);
  ArtVpath *vec = NULL;
  ArtSVP *svp = NULL;
  art_u32 color;

  color = (pc->color.red/256 << 16) | (pc->color.green/256 << 8) | (pc->color.blue/256);

  vec = art_new(ArtVpath, 3);
  vec[0].code = ART_MOVETO;
  vec[0].x = x1;
  vec[0].y = y1;
  vec[1].code = ART_LINETO;
  vec[1].x = x2;
  vec[1].y = y2;
  vec[2].code = ART_END;
  vec[2].x = x2;
  vec[2].y = y2;

  svp = gtk_plot_art_stroke(art, vec);

  art_rgb_svp_alpha(svp, 0, 0, pc->width - 1, pc->height - 1, 
                    (color << 8) + 0xFF,  
                    GTK_PLOT_ART(pc)->buf, pc->width * 3, NULL);

}

static void 
gtk_plot_art_draw_lines                              (GtkPlotPC *pc,
                                                     GtkPlotPoint *points,
                                                     gint numpoints)
{
  GtkPlotArt *art = GTK_PLOT_ART(pc);
  ArtVpath *vec = NULL;
  ArtSVP *svp = NULL;
  gint i;
  art_u32 color;

  color = (pc->color.red/256 << 16) | (pc->color.green/256 << 8) | (pc->color.blue/256);

  vec = art_new(ArtVpath, numpoints + 1);
  vec[0].code = ART_MOVETO;
  vec[0].x = points[0].x;
  vec[0].y = points[0].y;

  for(i = 1; i < numpoints; i++){
    vec[i].code = ART_LINETO;
    vec[i].x = points[i].x;
    vec[i].y = points[i].y;
  }

  vec[numpoints].code = ART_END;
  vec[numpoints].x = points[numpoints - 1].x;
  vec[numpoints].y = points[numpoints - 1].y;

  svp = gtk_plot_art_stroke(art, vec);

  art_rgb_svp_alpha(svp, 0, 0, pc->width - 1, pc->height - 1, 
                    (color << 8) + 0xFF,  
                    GTK_PLOT_ART(pc)->buf, pc->width * 3, NULL);
}

static void 
gtk_plot_art_draw_rectangle                          (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble width, gdouble height)
{
  GtkPlotArt *art = GTK_PLOT_ART(pc);
  ArtVpath *vec = NULL;
  ArtSVP *svp = NULL;
  art_u32 color;

  color = (pc->color.red/256 << 16) | (pc->color.green/256 << 8) | (pc->color.blue/256);

  vec = art_new(ArtVpath, 6);
  vec[0].code = ART_MOVETO;
  vec[0].x = x;
  vec[0].y = y;
  vec[1].code = ART_LINETO;
  vec[1].x = x+width;
  vec[1].y = y;
  vec[2].code = ART_LINETO;
  vec[2].x = x+width;
  vec[2].y = y+height;
  vec[3].code = ART_LINETO;
  vec[3].x = x;
  vec[3].y = y+height;
  vec[4].code = ART_LINETO;
  vec[4].x = x;
  vec[4].y = y;
  vec[5].code = ART_END;


  if(filled)
    svp = art_svp_from_vpath(vec);
  else
    svp = gtk_plot_art_stroke(art, vec);

  art_rgb_svp_alpha(svp, 0, 0, pc->width - 1, pc->height - 1, 
                    (color << 8) + 0xFF,  
                    GTK_PLOT_ART(pc)->buf, pc->width * 3, NULL);
}

static void 
gtk_plot_art_draw_polygon                            (GtkPlotPC *pc,
                                                     gint filled,
                                                     GtkPlotPoint *points,
                                                     gint numpoints)
{
  GtkPlotArt *art = GTK_PLOT_ART(pc);
  ArtVpath *vec = NULL;
  ArtSVP *svp = NULL;
  gint i;
  art_u32 color;

  color = (pc->color.red/256 << 16) | (pc->color.green/256 << 8) | (pc->color.blue/256);

  vec = art_new(ArtVpath, numpoints + 2);
  vec[0].code = ART_MOVETO;
  vec[0].x = points[0].x;
  vec[0].y = points[0].y;

  for(i = 1; i < numpoints; i++){
    vec[i].code = ART_LINETO;
    vec[i].x = points[i].x;
    vec[i].y = points[i].y;
  }

  vec[numpoints].code = ART_LINETO;
  vec[numpoints].x = points[0].x;
  vec[numpoints].y = points[0].y;
  vec[numpoints+1].code = ART_END;

  if(filled)
    svp = art_svp_from_vpath(vec);
  else
    svp = gtk_plot_art_stroke(art, vec);

  art_rgb_svp_alpha(svp, 0, 0, pc->width - 1, pc->height - 1, 
                    (color << 8) + 0xFF,  
                    GTK_PLOT_ART(pc)->buf, pc->width * 3, NULL);
}

static void 
gtk_plot_art_draw_circle                             (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble size)
{
  GtkPlotPoint *points;
  gint i, npoints = 126;

  points = g_new0(GtkPlotPoint, npoints);

  for (i = 0; i < npoints; i++) {
    double th;

    th = (2 * 3.141592654 * i) / npoints;
    points[i].x = x + size / 2. * cos (th);
    points[i].y = y + size / 2.* sin (th);
  }

  gtk_plot_art_draw_polygon(pc, filled, points, npoints);
  g_free(points);
}

static void 
gtk_plot_art_draw_ellipse                            (GtkPlotPC *pc,
                                                     gint filled,
                                                     gdouble x, gdouble y,
                                                     gdouble width, gdouble height)
{
  GtkPlotPoint *points;
  gint i, npoints = 126;

  points = g_new0(GtkPlotPoint, npoints);

  for (i = 0; i < npoints; i++) {
    double th;

    th = (2 * 3.141592654 * i) / npoints;
    points[i].x = x + width / 2. + width / 2. * cos (th);
    points[i].y = y + height / 2. + height / 2. * sin (th);
  }

  gtk_plot_art_draw_polygon(pc, filled, points, npoints);
  g_free(points);
}

static void 
gtk_plot_art_set_font                                (GtkPlotPC *pc,
						     GtkPSFont *psfont,
                                                     gint height)
{

}

/* subfunction of gtk_plot_art_draw_string(). */
static gint
drawstring(GtkPlotPC *pc,
	   GdkBitmap *dest,
	   GdkGC *gc,
	   GdkColor *black, GdkColor *white,
	   gint dx, gint dy,
	   GtkPSFont *psfont,
	   GdkFont *font,
	   GdkFont *latin_font,
	   GdkWChar wc)
{
  GdkBitmap *tmp;
  GdkFont *dfont;
  GdkImage *image;
  gint w, h, a, d, x, y, d2;
  guint32 pixel;

  if (psfont->i18n_latinfamily && psfont->vertical && (0 > wc || wc > 0x7f)) {
    /* vertical-writing CJK postscript fonts. */
    dfont = font;

    w = gdk_char_width_wc(dfont, wc);
    a = dfont->ascent;
    d = dfont->descent;
    h = a + d;
    d2 = w * d / h;

    tmp = gdk_pixmap_new(GTK_PLOT_GDK(pc)->window, w, h, 1);

    gdk_gc_set_foreground(gc, white);
    gdk_draw_rectangle(tmp, gc, TRUE, 0, 0, -1, -1);
    gdk_gc_set_foreground(gc, black);

    gdk_draw_text_wc(tmp, dfont, gc, 0, a, &wc, 1);

    image = gdk_image_get(tmp, 0, 0, w, h);

    for (y = 0; y < h; y++) {
      for (x = 0; x < w; x++) {
	pixel = gdk_image_get_pixel(image, x, y);
	if (pixel == black->pixel)
	  gdk_draw_point(dest, gc, dx + y, dy + d2 - x);
      }
    }

    gdk_image_destroy(image);
    gdk_drawable_unref(tmp);

    return h;
  } else {
    /* horizontal writing */
    if (psfont->i18n_latinfamily && 0 <= wc && wc <= 0x7f)
      dfont = latin_font;
    else
      dfont = font;

    gdk_draw_text_wc(dest, dfont, gc, dx, dy, &wc, 1);
    w = gdk_char_width_wc(dfont, wc);
    
    return w;
  }
}

static void 
gtk_plot_art_draw_string                        (GtkPlotPC *pc,
                                                gint tx, gint ty,
                                                gint angle,
                                                const GdkColor *fg,
                                                const GdkColor *bg,
                                                gboolean transparent,
                                                gint border,
                                                gint border_space,
                                                gint border_width,
                                                gint shadow_width,
                                                const gchar *font_name,
                                                gint font_height,
                                                GtkJustification just,
                                                const gchar *text)
{
/*
  GTK_PLOT_PC_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(pc)))->draw_string(pc, tx, ty, angle, fg, bg, transparent, border, border_space, border_width, shadow_width, font_name, font_height, just, text);
*/
  GTK_PLOT_PC_CLASS(parent_class)->draw_string(pc, tx, ty, angle, fg, bg, transparent, border, border_space, border_width, shadow_width, font_name, font_height, just, text);
}

static void gtk_plot_art_draw_pixmap                (GtkPlotPC *pc,
                                                     GdkPixmap *pixmap,
                                                     GdkBitmap *mask,
                                                     gint xsrc, gint ysrc,
                                                     gint xdest, gint ydest,
                                                     gint width,
                                                     gint height,
                                                     gdouble scale_x, 
                                                     gdouble scale_y)
{
 /* Use parent class -- gdk-pixbuf */
/*
  GTK_PLOT_PC_CLASS(G_OBJECT_GET_CLASS(G_OBJECT(pc)))->draw_pixmap(pc, pixmap, mask, xsrc, ysrc, xdest, ydest, width, height, scale_x, scale_y);
*/
  GTK_PLOT_PC_CLASS(parent_class)->draw_pixmap(pc, pixmap, mask, xsrc, ysrc, xdest, ydest, width, height, scale_x, scale_y);
}


static ArtSVP * 
gtk_plot_art_stroke		    (GtkPlotArt *art, ArtVpath *path)
{
  ArtSVP *svp = NULL;

  if(art->dash.n_dash != 0){
    ArtVpath *dash_path = NULL;

    dash_path = art_vpath_dash(path, &art->dash);

    art_free(path);
    path = dash_path;
  } 

  svp = art_svp_vpath_stroke (path,
                              art->join_style,
                              art->cap_style,
                              art->line_width,
                              4,
                              0.25);
 
  return (svp);
}

#endif /* WITH_LIBART */
