/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* This is the interface to the python interpreter
   Originally from the now defunct gnocci project.
   C. Steenberg 15 May 2000
*/

#include <stdio.h>
#include <string.h>

#include <glib.h>
#include <gdk/gdk.h>
#include <gdk/gdkkeysyms.h>
#include <gtkextra/gtkextra.h>

#include "sg_python.h"
#include "sg_config.h"
#include "sg_python_worksheet.h"

#include "dialogs/sg_misc_dialogs.h"

gboolean sg_report_python_error;
PyObject *main_dict, *sys_dict, *sg_o, *sg_dict, *config_dict;

/* Flags for module loading */
enum {SG_MODULE_LOAD_MAIN = 1 << 0,      /* Load module dict into main dictionary */
      SG_MODULE_LOAD_SYMBOLS = 1 << 1,   /* Load module's symbols into main dictionary */
      SG_MODULE_LOAD_REQ = 1 << 2 }; /* This module is required for normal operation of SG */

typedef struct _module_opts module_opts;

struct _module_opts
{  
   gchar *name;
   gint flags; 
};

static module_opts comp_imports[]={
 {"os",SG_MODULE_LOAD_MAIN|SG_MODULE_LOAD_REQ},
 {"sys",SG_MODULE_LOAD_MAIN|SG_MODULE_LOAD_REQ},
 {"numarray",SG_MODULE_LOAD_SYMBOLS|SG_MODULE_LOAD_REQ},
 {"pickle",SG_MODULE_LOAD_MAIN|SG_MODULE_LOAD_REQ},
 {NULL,0}
};

static PyObject *default_paths = NULL,
		*builtin_dict = NULL,
		*builtin_o = NULL,
		*os_dict = NULL,
		*os_o = NULL,
		*main_o = NULL,
		*sys_o = NULL;

gint 
sg_python_init(const gchar *progname)
{   
    PyObject *full, *string,*pickle;
    gchar *_progname = g_strdup(progname);

    Py_SetProgramName(_progname);
    Py_Initialize();
    PySys_SetArgv(0, &"");
    import_array();
//    import_libnumeric();
    sg_report_python_error=TRUE;
    g_free(_progname);
    
    main_o=PyImport_ImportModule("__main__");
    builtin_o=PyImport_AddModule("__builtin__");
    sys_o=PyImport_ImportModule ("sys");
    sys_o=PyImport_AddModule ("sys");    
    os_o=PyImport_ImportModule ("os");
    os_o=PyImport_AddModule ("os");    

    main_dict=PyModule_GetDict(main_o);
    sys_dict=PyModule_GetDict(sys_o);
    os_dict=PyModule_GetDict(os_o);

    builtin_dict=PyModule_GetDict(builtin_o);

    PyDict_SetItemString (main_dict, "os", os_o);
    PyDict_SetItemString (main_dict, "sys", sys_o);


/* sg module init */

    sg_python_worksheet_init();
    sg_o=PyImport_AddModule("sg");
    sg_dict=PyModule_GetDict(sg_o);
    PyDict_SetItemString (main_dict, "sg", sg_o);  

/* gtk.py requires this*/
    full=PyList_New (1);    
    string=PyString_FromString (progname);
    PyList_SET_ITEM (full, 0, string);
    PyDict_SetItemString (sys_dict, "argv",full);
    default_paths=PyDict_GetItemString (sys_dict, "path");
    pickle=PyImport_ImportModule("pickle");
    pickle=PyImport_AddModule("pickle");
    PyDict_SetItemString (main_dict, "pickle", pickle);
    if (!pickle) return FALSE;
    return TRUE;
}


static gint 
pickle_config_file(gchar *fname, PyObject *globals, PyObject *locals)
{ 
  FILE *fp;
  PyObject *object, *fo;
  gchar temp[200];

  if (!(fp=fopen(fname,"w"))) return FALSE;
  fo=PyFile_FromFile (fp, "pickle_file", "w", NULL);
  PyDict_SetItemString (sg_dict, "pickle_file", fo);
  g_snprintf(temp,200,"pickler=pickle.Pickler(pickle_file)");
  object=PyRun_String (temp, Py_file_input, globals, locals);
  if (sg_python_error_report(object))
     return FALSE;

  object=PyRun_String ("pickler.dump(sg.config)", Py_file_input, globals, locals);
  sg_python_error_report(object);

  object=PyRun_String ("del pickle_file,pickler", Py_file_input, globals, locals);
  if (sg_python_error_report(object))
   return (FALSE);
  fclose(fp);
  return TRUE; 
}


gint 
sg_python_config_save(void)
{ 
  gchar dir[]  = ".scigraphica";
  gchar file[] = "config";

  gint size;
  gchar *rcdir,*rcfile;

  size = strlen(g_get_home_dir())+1 + strlen(dir)+1;
  rcdir = g_new(gchar,size);
  g_snprintf(rcdir,size,"%s/%s",g_get_home_dir(),dir);

  size = strlen(rcdir)+1 + strlen(file)+1;
  rcfile = g_new(gchar,size);
  g_snprintf(rcfile,size,"%s/%s",rcdir,file);

  if ( pickle_config_file(rcfile, main_dict, sg_dict) == FALSE )
     {/* create scigraphica directory and try again */
      gchar *create_dir;

      size = strlen(rcdir) + 10;
      create_dir = g_new(gchar,size);
      g_snprintf(create_dir,size,"mkdir -p %s",rcdir);
      system(create_dir);
      g_free(create_dir);

      pickle_config_file(rcfile, main_dict, sg_dict);
    }

  g_free(rcdir);
  g_free(rcfile);

  return (TRUE);
}


static gint 
init_modules(SGconfig *config)
{
  gint i;
  PyObject *dict,*items;
  
  dict=PyDict_New();
  
  for (i=0;comp_imports[i].name!=NULL;i++){
/*
    printf("Loading Python module: %s\n",comp_imports[i].name);
*/
    PyDict_SetItemString(dict,comp_imports[i].name,
    PyInt_FromLong ((long) comp_imports[i].flags));
  }
  items=PyDict_Items(dict);
  sg_config_set_value(config->name, config->group, dict, FALSE);
  return TRUE;
}

static gint 
commit_modules(SGconfig *config)
{
    PyObject *keys, *key_name, *module_object, *object, *full, *string, *dict;
    int i=0;
    gchar temp[200];

    dict=sg_config_get_value(config->name, config->group);
    full=PyList_New (1);    
    string=PyString_FromString ("*");
    PyList_SET_ITEM (full, 0, string);

    keys=PyDict_Keys(dict);
    for (i=0;i<PyList_GET_SIZE (keys);i++)
    { 
      gchar *name;
      gint flag;
      key_name=PyList_GetItem (keys,i);
      name=PyString_AsString (key_name);
      
      flag=(gint)PyInt_AsLong(PyDict_GetItemString (dict,name));
      module_object=PyImport_ImportModuleEx (name, main_dict, sg_dict, full);
      printf("Importing python module: %s\n",name);

      /*python_error_report(module_object);*/
      if (!module_object) {
          
          snprintf(temp,200,"Couldn't import module '%s'\nView verbose error log?",name);
          sg_python_error_report_verbose(module_object,FALSE,temp,1);
          continue;
      }
      
      Py_INCREF(module_object);

      if (flag&SG_MODULE_LOAD_SYMBOLS){
          g_snprintf(temp,200,"from %s import *",name);
          object=PyRun_String (temp, Py_single_input, main_dict, sg_dict);
          sg_python_error_report(object);
          if (module_object)
            PyDict_SetItemString (main_dict, name, module_object);
      }
      else if (comp_imports[i].flags&SG_MODULE_LOAD_MAIN && module_object){
          PyDict_SetItemString (main_dict, name, module_object);
      }
    }
    Py_XDECREF(full);
    return i;
}

static gint 
init_module_paths(SGconfig *config)
{
     int j;
     PyObject *items;

     items=PyDict_New();
     for (j=0;j<PyList_GET_SIZE (default_paths);j++){
       PyDict_SetItem(items, PyList_GetItem (default_paths,j),Py_None);
     }
     if (!PyDict_GetItemString(items, PYTHON_MODULE_PATH))
         PyDict_SetItemString(items, PYTHON_MODULE_PATH,Py_None);
     sg_config_set_value(config->name, config->group,
                        items,FALSE);
     return TRUE;
}

static gint commit_module_paths(SGconfig *config){
    PyObject *value;

    value=sg_config_get_value(config->name, config->group);
    PyDict_SetItemString (sys_dict, "path", PyDict_Keys(value));
  return TRUE;

}

static gint 
unpickle_config_file(gchar *fname, PyObject *globals, PyObject *locals)
{
  FILE *fp;
  PyObject *object, *fo, *dict = NULL;
  gchar temp[200];

  if (!(fp=fopen(fname,"r"))) return FALSE;
  fo=PyFile_FromFile (fp, "pickle_file", "r", NULL);
  PyDict_SetItemString (sg_dict, "pickle_file", fo);
  g_snprintf(temp,200,"unpickler=pickle.Unpickler(pickle_file)");
  object=PyRun_String (temp, Py_file_input, globals, locals);
  if (!object){
      sg_python_error_report_verbose(object,FALSE,"Cannot load preferences - Restoring defaults.\n\n"
                                  "View verbose error log?",1);
      Py_XDECREF(object);
      fclose(fp);
      Py_XDECREF(fo);
      return 0;
  }

  g_snprintf(temp,200,"sg.config=unpickler.load()");
  object=PyRun_String (temp, Py_file_input, globals, locals);
  if (!object){
      sg_python_error_report_verbose(object,FALSE,"Cannot load preferences - Restoring defaults.\n\n"
                                  "View verbose error log?",1);
      Py_XDECREF(object);
      fclose(fp);
      Py_XDECREF(fo);
      return 0;
  }

  g_snprintf(temp,200,"del pickle_file,unpickler");
  object=PyRun_String (temp, Py_file_input, globals, locals);
  sg_config_dict=PyDict_GetItemString(sg_dict,"config");
  
  fclose(fp);
  Py_XDECREF(object);
  Py_XDECREF(fo);
  if (dict) return 1;
  else return 0;

}

gint 
sg_python_config_init(void)
{ 
  gchar dir[]  = ".scigraphica";
  gchar file[] = "config";
 
  gchar *rcdir,*rcfile;
  gint size;

  size = strlen(g_get_home_dir())+1 + strlen(dir)+1;
  rcdir = g_new(gchar,size);
  g_snprintf(rcdir,size,"%s/%s",g_get_home_dir(),dir);

  size = strlen(rcdir)+1 + strlen(file)+1;
  rcfile=g_new(gchar,size);
  g_snprintf(rcfile,size,"%s/%s",rcdir,file);

  if ( unpickle_config_file(rcfile, main_dict, sg_dict) == FALSE )
     {/* create scigraphica directory and try again */
      gchar *create_dir;

      size = strlen(rcdir) + 10;
      create_dir=g_new(gchar,size);
      g_snprintf(create_dir,size,"mkdir -p %s", rcdir);
      system(create_dir);
      g_free(create_dir);

      /* try again */
      unpickle_config_file(rcfile, main_dict, sg_dict);
    }

  g_free(rcdir);
  g_free(rcfile);

  sg_config_init();
  sg_config_new_c("startup", "module", init_modules, commit_modules);
  sg_config_new_c("path", "module", init_module_paths, commit_module_paths);

/* Make sure the standard modules are loaded */
  sg_config_exec_commit(sg_config_get_config("path", "module"));
  return (TRUE);
}


gint 
sg_python_error_report(PyObject *object)
{ 
    if (!object)
      { if (sg_report_python_error)
        PyErr_Print();
        if (Py_FlushLine())
            PyErr_Clear(); 
        return TRUE;
      }
    return FALSE;
}

gint 
sg_python_error_report_verbose(PyObject *object,gboolean report, gchar *message, gint type)
{
    if (!object){
        if (report || sg_accept_dialog(message,type) == SG_BUTTON_YES ){
            PyErr_Print();
        }
        if (Py_FlushLine())
            PyErr_Clear();
        return TRUE;
    }
    return FALSE;
}

