/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_plugin.h"
#include "sg_plugin_function.h"
#include "sg_python_expr.h"

static void sg_plugin_function_class_init             (SGpluginFunctionClass *klass);
static void sg_plugin_function_init                   (SGpluginFunction *plugin);
static void sg_plugin_function_finalize                (GObject *object);
static gdouble function_default_action                (SGpluginFunction *plugin,
                                                       gdouble *par_val,
                                                       gdouble ival,
                                                       gboolean *error,
                                                       gpointer data);


extern PyObject *main_dict, *sg_dict;
static SGpluginClass *parent_class = NULL;

GType
sg_plugin_function_get_type (void)
{
  static GType sg_plugin_function_type = 0;
                                                                                
  if (!sg_plugin_function_type)
    {
      static const GTypeInfo sg_plugin_function_info =
      {
        sizeof (SGpluginFunctionClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_plugin_function_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGpluginFunction),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_plugin_function_init,
        NULL,
      };
                                                                                
      sg_plugin_function_type = g_type_register_static (GTK_TYPE_SG_PLUGIN, "SGpluginFunction",
                                               &sg_plugin_function_info, 0);
    }
                                                                                
  return sg_plugin_function_type;
}

static void
sg_plugin_function_class_init (SGpluginFunctionClass *klass)
{
  GObjectClass *object_class;
  SGpluginClass *plugin_class;

  parent_class = (SGpluginClass *) g_type_class_peek_parent (klass);

  object_class = (GObjectClass *) klass;
  plugin_class = (SGpluginClass *) klass;

  object_class->finalize = sg_plugin_function_finalize;
}


SGpluginFunction *
sg_plugin_function_new ()
{
  SGpluginFunction *plugin;
  plugin = SG_PLUGIN_FUNCTION(g_object_new(sg_plugin_function_get_type(), NULL));
  return plugin;
}

static void
sg_plugin_function_init(SGpluginFunction *plugin)
{
  gint i;
  for(i = 0; i < 10; i++) {
    plugin->param[i] = NULL;
  }
  plugin->ivar = NULL;
  plugin->function = NULL;
  plugin->action = function_default_action;
}

static void
sg_plugin_function_finalize(GObject *object)
{
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(object);
  gint i;

  for(i = 0; i < 10; i++) {
    if(plugin->param[i]) g_free(plugin->param[i]);
    plugin->param[i] = NULL;
  }
  if(plugin->ivar) g_free(plugin->ivar);
  plugin->ivar = NULL;

  if(plugin->function) g_free(plugin->function);
  plugin->function = NULL;

  if(plugin->pixmap && GTK_IS_WIDGET(plugin->pixmap)) 
    gtk_widget_destroy(GTK_WIDGET(plugin->pixmap));
  plugin->pixmap = NULL;
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

gdouble
sg_plugin_function_action(SGpluginFunction *plugin, gdouble *params, gdouble ival, gboolean *error, gpointer data)
{ 
  *error = FALSE;

  if(plugin->action) return (plugin->action(plugin, params, ival, error, data));

  *error = TRUE;
  return 0.0;
} 

SGpluginFunction *
sg_plugin_function_get(const gchar *name)
{
  GList *list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_FUNCTION(list->data)){
      SGpluginFunction *c = SG_PLUGIN_FUNCTION(list->data);
      if(strcmp(SG_PLUGIN(c)->name, name) == 0) return c;
    }
    list = list->next;
  }
  return NULL;
}

static gdouble
function_default_action(SGpluginFunction *plugin, gdouble *par_val, gdouble ival, gboolean *error, gpointer data)
{
  gint i;
  gdouble dval;
  PyObject *object = NULL;

  *error = FALSE;

  object=Py_BuildValue("d",ival);
  if (!object) { *error = TRUE; return 0.0; }
  PyDict_SetItemString (sg_dict, plugin->ivar, object);
                                                                               
  for(i = 0; i < plugin->nparam; i++){
    object=Py_BuildValue("d", par_val[i]);
    if (!object) { *error = TRUE; return 0.0; }
    PyDict_SetItemString (sg_dict, plugin->param[i], object);
  }
                                                                               
  object = PyRun_String (plugin->function, Py_eval_input, main_dict, sg_dict);
  if (!object) {
    sg_python_error_report(object);
    *error = TRUE;
    return 0.0;
  }
                                                                               
  dval = PyFloat_AsDouble (object);
  return dval;
}
