/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include "sg_plugin.h"

static GList *sg_plugin_list;

static void sg_plugin_class_init                       (SGpluginClass *klass);
static void sg_plugin_init                             (SGplugin *plugin);
static void sg_plugin_finalize                         (GObject *object);
static GObjectClass *parent_class = NULL;

GType
sg_plugin_get_type (void)
{
  static GType sg_plugin_type = 0;
                                                                                
  if (!sg_plugin_type)
    {
      static const GTypeInfo sg_plugin_info =
      {
        sizeof (SGpluginClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_plugin_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGplugin),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_plugin_init,
        NULL,
      };
                                                                                
      sg_plugin_type = g_type_register_static (G_TYPE_OBJECT, "SGplugin",
                                               &sg_plugin_info, 0);
    }
                                                                                
  return sg_plugin_type;
}

static void
sg_plugin_class_init (SGpluginClass *klass)
{
  GObjectClass *object_class;
  SGpluginClass *plugin_class;

  parent_class = (GObjectClass *) g_type_class_peek_parent (klass);

  object_class = (GObjectClass *) klass;
  plugin_class = (SGpluginClass *) klass;

  object_class->finalize = sg_plugin_finalize;
}


SGplugin *
sg_plugin_new ()
{
  SGplugin *plugin;
  plugin = SG_PLUGIN(g_object_new(sg_plugin_get_type(), NULL));
  return plugin;
}

static void
sg_plugin_init(SGplugin *plugin)
{
  plugin->name = NULL;
  plugin->description = NULL;
  plugin->handle = NULL;
}

static void
sg_plugin_finalize(GObject *object)
{
  SGplugin *plugin;

  plugin = SG_PLUGIN(object);

  if(plugin->name) g_free(plugin->name);
  plugin->name = NULL;

  if(plugin->description) g_free(plugin->description);
  plugin->description = NULL;

  if(plugin->handle) g_module_close(plugin->handle);
  plugin->handle = NULL;
}

void
sg_plugin_set_name(SGplugin *plugin, const gchar *name)
{
  if(plugin->name) g_free(plugin->name);
  plugin->name = NULL;
  if(name) plugin->name = g_strdup(name);
}

void
sg_plugin_set_description(SGplugin *plugin, const gchar *description)
{
  if(plugin->description) g_free(plugin->description);
  plugin->description = NULL;
  if(description) plugin->description = g_strdup(description);
}

/***********************************************************************
 */

void
sg_plugins_init()
{
  sg_plugin_list = NULL;
}

void
sg_plugins_quit()
{
  GList *list;

  list = sg_plugin_list;
  while(list){
    SGplugin *plugin;
    plugin = SG_PLUGIN(list->data);

    if(plugin) g_object_unref(G_OBJECT(plugin));
    sg_plugin_list = g_list_remove_link(sg_plugin_list, list);
    g_list_free_1(list);
    list = sg_plugin_list;
  }

  sg_plugin_list = NULL;
}

GList *
sg_plugins()
{
  return sg_plugin_list;
}

void
sg_plugin_add(SGplugin *plugin)
{
  sg_plugin_list = g_list_append(sg_plugin_list, plugin);
}

SGplugin *
sg_plugin_get(const gchar *name)
{
  GList *list = sg_plugin_list;
  while(list){
    SGplugin *plugin = SG_PLUGIN(list->data);
    if(strcmp(plugin->name, name) == 0) return plugin;
    list = list->next;
  }
  return NULL; 
}
