/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_property_dialog.h"
#include "sg_ticks_dialog.h"
#include "sg.h"
#include <math.h>

typedef struct
{
  SGlayer *layer;
  GtkPlot *plot;

  GtkPlotAxis *axis;

  GtkPlotAxis *axis1, *axis2;

  GtkPlotOrientation orientation;

  GtkWidget *thick_spin;
  GtkWidget *length_spin;
  GtkWidget *color_combo;

  GtkPlotTicks *ticks;
} SGticksDialog;

static void sg_ticks_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_ticks_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_ticks_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_ticks_dialog_new(SGlayer *layer, GtkPlotAxis *axis)
{
  GtkWidget *widget;
  SGticksDialog *dialog;

  dialog = g_new0(SGticksDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->ok = sg_ticks_dialog_ok;
  SG_PROPERTY_DIALOG(widget)->apply = sg_ticks_dialog_apply;
  dialog->layer = layer;
  dialog->axis = axis;
  sg_ticks_dialog_init_gui(widget);

  return widget;
}

static void
sg_ticks_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_ticks_dialog_apply(d, data);
}

static void
sg_ticks_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  GtkColorCombo *combo;
  GdkColor color;
  gfloat line_width;
  gint ticks_length;
  SGticksDialog *dialog = (SGticksDialog *)d->data;
  GtkPlot *plot = dialog->plot;

  line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->thick_spin));
  ticks_length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->length_spin));
  combo = GTK_COLOR_COMBO(dialog->color_combo);
  color = gtk_color_combo_get_selection(combo);
                                                                               
  dialog->axis->ticks_length = ticks_length;
                                                                               
  dialog->axis->ticks_width = line_width / 2;

  gtk_plot_axis_set_attributes(dialog->axis, line_width, &color);
  dialog->axis->labels_attr.fg = color;

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(GTK_PLOT_CANVAS_CHILD(dialog->layer)->parent));
}

static void
init_dialog(SGticksDialog *dialog)
{
  GdkColor line_color;
  gfloat line_width;
  GtkPlotLineStyle line_style;
  GtkPlot *plot = dialog->plot;

  if(strcmp(SG_PLUGIN(dialog->layer->plugin)->name,"layer_3d") != 0)
    gtk_plot_axis_get_attributes(gtk_plot_get_axis(plot, 0), &line_width, &line_color);
  else
    gtk_plot3d_frame_get_attributes(GTK_PLOT3D(plot), &line_style, &line_width, &line_color);
                                                                               
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin), line_width);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->length_spin), plot->top->ticks_length);
                                                                               
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo), line_color);
}

static void 
sg_ticks_dialog_init_gui (GtkWidget *dialog_frame)
{
  GtkPlot *plot;
  GtkWidget *label;
  GtkWidget *table;
  GtkAdjustment *adjust;
  GtkWidget *main_table;
  GtkWidget *main_box;
  GtkWidget *frame;
  SGticksDialog *dialog;

  dialog = (SGticksDialog *)SG_PROPERTY_DIALOG(dialog_frame)->data;
  plot = dialog->plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE,5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 0);
  gtk_container_add(GTK_CONTAINER(dialog_frame), main_box);
                                                                               
  main_table = gtk_table_new (3, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 2);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 2);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);
                                                                               
  /* Poperties */
                                                                               
  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach (GTK_TABLE (main_table), frame, 0, 1, 0, 1, 0, 0, 5, 5);
                                                                               
  table=gtk_table_new(3, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);
                                                                               
  label = gtk_label_new( _("Thickness(pts) ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 0, 1, 0, 0, 5, 5);
  label = gtk_label_new( _("Ticks length ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 1, 2, 0, 0, 5, 5);
  label = gtk_label_new( _("Color ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach(GTK_TABLE(table), label, 0, 1, 2, 3, 0, 0, 5, 5);
  adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->thick_spin = gtk_spin_button_new(adjust, 0, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin), TRUE);
                                                                               
  adjust = (GtkAdjustment *)gtk_adjustment_new(0., 0., 20., 1., 1., 0.);
  dialog->length_spin = gtk_spin_button_new(adjust, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->length_spin), TRUE);
                                                                               
  dialog->color_combo = gtk_color_combo_new();
                                                                               
  gtk_table_attach(GTK_TABLE(table), dialog->thick_spin, 1, 2, 0, 1, 0, 0, 5, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->length_spin, 1, 2, 1, 2, 0, 0, 5,
5);
  gtk_table_attach(GTK_TABLE(table), dialog->color_combo, 1, 2, 2, 3, 0, 0, 5,
5);

  init_dialog(dialog);
}

GtkWidget *
sg_ticks_dialog (SGlayer *layer,
                 GtkWidget *ok_button,
                 GtkWidget *apply_button,
                 GtkWidget *cancel_button)
{
  GtkWidget *dialog;
  GtkWidget *notebook;
  GtkPlot *plot;

  notebook = gtk_notebook_new();

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;

  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_3d") == 0){
      dialog = sg_ticks_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("X") ));
      dialog = sg_ticks_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Y") ));
      dialog = sg_ticks_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Z") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_2d") == 0){
      dialog = sg_ticks_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Left") ));
      dialog = sg_ticks_dialog_new(layer, plot->right);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Right") ));
      dialog = sg_ticks_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Top") ));
      dialog = sg_ticks_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Bottom") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name,"layer_polar") == 0){
      dialog = sg_ticks_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Angles") ));
      dialog = sg_ticks_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
                                     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("R") ));
  }


  return notebook;
}

