/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

typedef struct
{
  GtkWidget *scale_max_entry;
  GtkWidget *size_max_entry;
  GtkWidget *show_scale_check;
  GtkWidget *precision_spin;
  GtkWidget *format_combo;
  GtkWidget *labels_prefix_entry;
  GtkWidget *labels_suffix_entry;
  GtkPlotData *dataset;
} SGscaleDialog;

static void sg_symbol_scale_dialog_apply            (SGpropertyDialog *d, 
					      gpointer data);
static void sg_symbol_scale_dialog_ok               (SGpropertyDialog *d, 
					      gpointer data);
static void sg_symbol_scale_dialog_init_gui         (GtkWidget *frame);

static gchar *symbol_styles[] = { N_("Empty"), N_("Filled"), N_("Opaque"), NULL };

GtkWidget *
sg_symbol_scale_dialog_new(GtkPlotData *dataset)
{
  GtkWidget *widget;
  SGscaleDialog *dialog;

  dialog = g_new0(SGscaleDialog, 1);
  dialog->dataset = dataset;

  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_symbol_scale_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_symbol_scale_dialog_ok;
  sg_symbol_scale_dialog_init_gui(widget);

  return widget;
}

static void
sg_symbol_scale_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_symbol_scale_dialog_apply(d, data);
}

static void     
init_dialog(SGscaleDialog *dialog)
{
  gchar text[100];
  GtkPlotData *data = dialog->dataset; 

  g_snprintf(text, 100, "%f", GTK_PLOT_BUBBLE(data)->scale_max);
  gtk_entry_set_text(GTK_ENTRY(dialog->scale_max_entry), text);

  g_snprintf(text, 100, "%d", GTK_PLOT_BUBBLE(data)->size_max);
  gtk_entry_set_text(GTK_ENTRY(dialog->size_max_entry), text);

  if(GTK_PLOT_BUBBLE(data)->labels_prefix){
    g_snprintf(text, 100, "%s", GTK_PLOT_BUBBLE(data)->labels_prefix);
  } else {
    g_snprintf(text, 100, "");
  }

  gtk_entry_set_text(GTK_ENTRY(dialog->labels_prefix_entry), text);

  if(GTK_PLOT_BUBBLE(data)->labels_suffix){
    g_snprintf(text, 100, "%s", GTK_PLOT_BUBBLE(data)->labels_suffix);
  } else {
    g_snprintf(text, 100, "");
  }
  gtk_entry_set_text(GTK_ENTRY(dialog->labels_suffix_entry), text);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->show_scale_check),
                                   GTK_PLOT_BUBBLE(data)->show_scale);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), GTK_PLOT_BUBBLE(data)->labels_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin),
                            GTK_PLOT_BUBBLE(data)->labels_precision);
}

static void     
sg_symbol_scale_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *label;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;
  SGscaleDialog *dialog = (SGscaleDialog *)SG_PROPERTY_DIALOG(frame)->data;
  gchar *label_formats[] = { N_("Decimal: 1.000"),
                             N_("Scientific: 1.E10"),
                             N_("Scientific: 1x10^10"),
                             NULL };

  gtk_frame_set_label(GTK_FRAME(frame), _("Scale") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new(9, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new( _("Scale max.:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);

  dialog->scale_max_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->scale_max_entry,1,2,1,2);
  sg_entry_set_numeric(GTK_ENTRY(dialog->scale_max_entry), 6);

  label = gtk_label_new( _("Size max.:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

  dialog->size_max_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->size_max_entry,1,2,3,4);
  sg_entry_set_numeric(GTK_ENTRY(dialog->size_max_entry), 0);

  label = gtk_label_new( _("Labels") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);
                                                                                
  dialog->format_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->format_combo, 1,2,4,5);
  sg_combo_set_items(GTK_COMBO(dialog->format_combo), label_formats);

  label = gtk_label_new( _("Precision") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);
                                                                                
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 8., 1., 1., 0.);
  dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->precision_spin, 1, 2, 5, 6);

  label = gtk_label_new( _("Labels prefix:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,6,7);

  dialog->labels_prefix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->labels_prefix_entry,1,2,6,7);

  label = gtk_label_new( _("Labels suffix:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,7,8);

  dialog->labels_suffix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->labels_suffix_entry,1,2,7,8);

  dialog->show_scale_check = gtk_check_item_new_with_label( _("Show scale") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->show_scale_check, 0, 2, 8, 9);

  gtk_widget_size_request(dialog->scale_max_entry, &req);
  gtk_widget_set_usize(dialog->scale_max_entry, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->size_max_entry, req.width / 2, req.height);
  gtk_widget_show_all(table);

  init_dialog(dialog);
}

static void     
sg_symbol_scale_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  GtkWidget *child;
  gdouble scale;
  gint size;
  const gchar *text;
  SGscaleDialog *dialog = (SGscaleDialog *)d->data;
  GtkPlotBubble *data = GTK_PLOT_BUBBLE(dialog->dataset);

  data->show_scale = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->show_scale_check));

  text = gtk_entry_get_text(GTK_ENTRY(dialog->scale_max_entry));
  scale = atof(text);
  GTK_PLOT_BUBBLE(data)->scale_max = scale;

  text = gtk_entry_get_text(GTK_ENTRY(dialog->size_max_entry));
  size = atoi(text);
  GTK_PLOT_BUBBLE(data)->size_max = size;

  text = gtk_entry_get_text(GTK_ENTRY(dialog->labels_prefix_entry));
  if(text && strlen(text) > 0)
    gtk_plot_bubble_set_labels_prefix(GTK_PLOT_BUBBLE(data), text);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->labels_suffix_entry));
  if(text && strlen(text) > 0)
    gtk_plot_bubble_set_labels_suffix(GTK_PLOT_BUBBLE(data), text);

  gtk_plot_bubble_set_labels_precision(GTK_PLOT_BUBBLE(data), gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin)));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data;
  gtk_plot_bubble_set_labels_style(GTK_PLOT_BUBBLE(data), gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), child));
}

