/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_property_dialog.h"
#include "sg.h"

static gchar *line_styles[] = { 
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

typedef struct
{
  GtkPlotData *dataset;
  GtkWidget *grid_check;
  GtkWidget *mesh_check;
  GtkWidget *line_color_combo;
  GtkWidget *line_width_spin;
  GtkWidget *line_style_combo;
  GtkWidget *xline_color_combo;
  GtkWidget *transparent_check;
  GtkWidget *lines_check;
  GtkWidget *height_gradient_check;
  GtkWidget *bg_color_combo;
  GtkWidget *fg_color_combo;
  gboolean is_2d;
  gboolean transparent;
  gboolean height_gradient;
} SGsurfaceDialog;

static void sg_surface_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_surface_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_surface_dialog_init_gui         (GtkWidget *w);

GtkWidget *
sg_surface_dialog_new(GtkPlotData *dataset, gboolean is_2d, gboolean transparent, gboolean height_gradient)
{
  GtkWidget *widget;
  SGsurfaceDialog *dialog;

  dialog = g_new0(SGsurfaceDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_surface_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_surface_dialog_ok;
  dialog->dataset = dataset;
  dialog->is_2d = is_2d;
  dialog->transparent = transparent;
  dialog->height_gradient = height_gradient;
  sg_surface_dialog_init_gui(widget);

  return widget;
}

static void
sg_surface_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_surface_dialog_apply(d, data);
}

static void     
init_dialog (SGsurfaceDialog *dialog)
{
  GtkPlotSurface *data;

  data = GTK_PLOT_SURFACE(dialog->dataset);


  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list), GTK_PLOT_SURFACE(data)->mesh_line.line_style);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->line_width_spin), GTK_PLOT_SURFACE(data)->mesh_line.line_width);

  if(!dialog->is_2d) {
    sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_color_combo), data->color);
    sg_color_combo_init(GTK_COLOR_COMBO(dialog->bg_color_combo), data->shadow);
    sg_color_combo_init(GTK_COLOR_COMBO(dialog->xline_color_combo), data->grid_background);
  }

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->line_color_combo), data->grid_foreground);

  if(dialog->height_gradient)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->height_gradient_check),
                                 GTK_PLOT_SURFACE(data)->use_height_gradient);
  if(dialog->transparent)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->transparent_check),
                                 GTK_PLOT_SURFACE(data)->transparent);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->grid_check), data->show_grid);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->mesh_check), data->show_mesh);
}

static void
sg_surface_dialog_init_gui (GtkWidget *widget)
{
  GtkWidget *frame;
  GtkWidget *table, *table1;
  GtkWidget *label;
  SGsurfaceDialog *dialog;
  GtkAdjustment *adj;
  GtkRequisition req;

  dialog = (SGsurfaceDialog *)SG_PROPERTY_DIALOG(widget)->data;

  table = gtk_table_new(2, 1, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_container_add(GTK_CONTAINER(widget), table);

  if(!dialog->is_2d){
    frame = gtk_frame_new( _("Colors") );
    gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
    gtk_table_attach_defaults(GTK_TABLE(table),frame,0,1,0,1);
  
    table1 = gtk_table_new(4, 2, FALSE);
    gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
    gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
    gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
    gtk_container_add (GTK_CONTAINER (frame), table1);
  
    label = gtk_label_new( _("Foreground") );
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);
  
    dialog->fg_color_combo = gtk_color_combo_new();
    gtk_widget_set_usize(dialog->fg_color_combo, 30, 18);
    gtk_table_attach_defaults(GTK_TABLE(table1), dialog->fg_color_combo,1,2,0,1);

    label = gtk_label_new( _("Background") );
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);
  
    dialog->bg_color_combo = gtk_color_combo_new();
    gtk_widget_set_usize(dialog->bg_color_combo, 30, 18);
    gtk_table_attach_defaults(GTK_TABLE(table1),dialog->bg_color_combo,1,2,1,2);
  }

  if(dialog->transparent) {
    dialog->transparent_check = gtk_check_item_new_with_label( _("Transparent") );
    gtk_table_attach_defaults(GTK_TABLE(table1),dialog->transparent_check,0,2,2,3);
  }
  if(dialog->height_gradient){
    dialog->height_gradient_check = gtk_check_item_new_with_label( _("Use height gradient") );
    gtk_table_attach_defaults(GTK_TABLE(table1),dialog->height_gradient_check,0,2,3,4);
  }

  frame = gtk_frame_new( _("Lines") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach_defaults(GTK_TABLE(table),frame,1,2,0,1);

  table1 = gtk_table_new(2, 4, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table1), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table1), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table1), 5);
  gtk_container_add (GTK_CONTAINER (frame), table1);

  label = gtk_label_new( _("Type") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,0,1);
  dialog->line_style_combo = gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->line_style_combo)->entry), FALSE);
  sg_combo_set_items(GTK_COMBO(dialog->line_style_combo), line_styles);
  gtk_table_attach_defaults(GTK_TABLE(table1),dialog->line_style_combo,1,2,0,1);
  gtk_widget_size_request(dialog->line_style_combo, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->line_style_combo, req.width, req.height);
                                                                                
  label = gtk_label_new( _("Width") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,1,2);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., .5, 1., 0.);
  dialog->line_width_spin = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->line_width_spin), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table1),dialog->line_width_spin,1,2,1,2);
  gtk_widget_set_usize(dialog->line_width_spin, req.width, req.height);

  label = gtk_label_new( _("Foreground") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,2,3);

  dialog->line_color_combo = gtk_color_combo_new();
  gtk_widget_set_usize(dialog->line_color_combo, 30, 18);
  gtk_table_attach_defaults(GTK_TABLE(table1),dialog->line_color_combo,1,2,2,3);

  if(!dialog->is_2d){
    label = gtk_label_new( _("Background") );
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table1),label,0,1,3,4);

    dialog->xline_color_combo = gtk_color_combo_new();
    gtk_widget_set_usize(dialog->xline_color_combo, 30, 18);
    gtk_table_attach_defaults(GTK_TABLE(table1),dialog->xline_color_combo,1,2,3,4);
  }

  dialog->grid_check = gtk_check_item_new_with_label( _("Show grid") );
  gtk_table_attach_defaults(GTK_TABLE(table1),dialog->grid_check,0,2,4,5);
  dialog->mesh_check = gtk_check_item_new_with_label( _("Show mesh") );
  gtk_table_attach_defaults(GTK_TABLE(table1),dialog->mesh_check,0,2,5,6);
//      dialog->transparent_check = gtk_check_item_new_with_label( _("Transparent") );
//      gtk_table_attach_defaults(GTK_TABLE(table1),dialog->transparent_check,0,2,4,5);

  init_dialog(dialog);
}

static void     
sg_surface_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGsurfaceDialog *dialog = (SGsurfaceDialog *)d->data;
  GtkPlotSurface *data;
  GdkColor color;
  gint row, col;
  GtkWidget *child = NULL;

  data = GTK_PLOT_SURFACE(dialog->dataset);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list)->selection->data;
  data->mesh_line.line_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->line_style_combo)->list), child);
  data->mesh_line.line_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->line_width_spin));

  if(!dialog->is_2d){
    row = GTK_COLOR_COMBO(dialog->fg_color_combo)->row;
    col = GTK_COLOR_COMBO(dialog->fg_color_combo)->column;
    color = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->fg_color_combo), row, col);
    data->color = color;

    row = GTK_COLOR_COMBO(dialog->bg_color_combo)->row;
    col = GTK_COLOR_COMBO(dialog->bg_color_combo)->column;
    color = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->bg_color_combo), row, col);
    data->shadow = color;

    row = GTK_COLOR_COMBO(dialog->xline_color_combo)->row;
    col = GTK_COLOR_COMBO(dialog->xline_color_combo)->column;
    color = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->xline_color_combo), row, col);
    data->grid_background = color;
  }

  row = GTK_COLOR_COMBO(dialog->line_color_combo)->row;
  col = GTK_COLOR_COMBO(dialog->line_color_combo)->column;
  color = gtk_color_combo_get_color_at(GTK_COLOR_COMBO(dialog->line_color_combo), row, col);
  data->grid_foreground = color;

  data->show_grid = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->grid_check));
  data->show_mesh = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->mesh_check));

  if(dialog->transparent)
    data->transparent = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_check));
  if(dialog->height_gradient)
    data->use_height_gradient = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->height_gradient_check));

}

