/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

typedef struct
{
  GtkWidget *arrow_combo;
  GtkWidget *arrow_origin_check;
  GtkWidget *arrow_end_check;
  GtkWidget *arrow_length_spin;
  GtkWidget *arrow_width_spin;
  GtkWidget *arrow_center_check;
  GtkWidget *arrow_relative_check;
  GtkWidget *scale_entry;
  GtkPlotData *dataset;
} SGarrowDialog;

static void sg_segment_dialog_apply            (SGpropertyDialog *d, 
					      gpointer data);
static void sg_segment_dialog_ok               (SGpropertyDialog *d, 
					      gpointer data);
static void sg_segment_dialog_init_gui         (GtkWidget *frame);

static gchar *symbol_styles[] = { N_("Empty"), N_("Filled"), N_("Opaque"), NULL };

GtkWidget *
sg_segment_dialog_new(GtkPlotData *dataset)
{
  GtkWidget *widget;
  SGarrowDialog *dialog;

  dialog = g_new0(SGarrowDialog, 1);
  dialog->dataset = dataset;

  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_segment_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_segment_dialog_ok;
  sg_segment_dialog_init_gui(widget);

  return widget;
}

static void
sg_segment_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_segment_dialog_apply(d, data);
}

static void     
init_dialog(SGarrowDialog *dialog)
{
  gchar text[100];
  GtkPlotData *data = dialog->dataset; 

  g_snprintf(text, 100, "%f", gtk_plot_data_get_a_scale(data));
  gtk_entry_set_text(GTK_ENTRY(dialog->scale_entry), text);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list),
                       GTK_PLOT_SEGMENT(data)->arrow_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->arrow_length_spin),
                            GTK_PLOT_SEGMENT(data)->arrow_length);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->arrow_width_spin),
                                GTK_PLOT_SEGMENT(data)->arrow_width);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->arrow_center_check),
                                   GTK_PLOT_SEGMENT(data)->centered);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->arrow_relative_check),
                                   GTK_PLOT_SEGMENT(data)->relative);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->arrow_origin_check),
                               (GTK_PLOT_SEGMENT(data)->arrow_mask & GTK_PLOT_ARROW_ORIGIN) != 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->arrow_end_check),
                               (GTK_PLOT_SEGMENT(data)->arrow_mask & GTK_PLOT_ARROW_END) != 0);
}

static void     
sg_segment_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *label;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;
  SGarrowDialog *dialog = (SGarrowDialog *)SG_PROPERTY_DIALOG(frame)->data;

  gtk_frame_set_label(GTK_FRAME(frame), _("Arrows") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new(5, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new( _("Style:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label , 0, 1, 0, 1);
  dialog->arrow_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->arrow_combo), symbol_styles);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_combo, 1, 2, 0, 1);

  label = gtk_label_new( _("Length:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
  dialog->arrow_length_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->arrow_length_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->arrow_length_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_length_spin, 1, 2, 1, 2);

  label = gtk_label_new( _("Width:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
  dialog->arrow_width_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->arrow_width_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->arrow_width_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_width_spin, 1, 2, 2, 3);
  label = gtk_label_new( _("Scale:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);

  dialog->scale_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->scale_entry,1,2,3,4);
  sg_entry_set_numeric(GTK_ENTRY(dialog->scale_entry), 6);

  dialog->arrow_origin_check = gtk_check_item_new_with_label( _("Origin") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_origin_check, 0, 1, 4, 5);

  dialog->arrow_origin_check = gtk_check_item_new_with_label( _("End") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_end_check, 1, 2, 4, 5);

  dialog->arrow_relative_check = gtk_check_item_new_with_label( _("Relative") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_relative_check, 0, 2, 5, 6);

  gtk_widget_size_request(dialog->scale_entry, &req);
  dialog->arrow_center_check = gtk_check_item_new_with_label( _("Center") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_center_check, 0, 2, 6, 7);

  gtk_widget_size_request(dialog->scale_entry, &req);
  gtk_widget_set_usize(dialog->scale_entry, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->arrow_combo, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->arrow_width_spin, req.width / 2, req.height);
  gtk_widget_set_usize(dialog->arrow_length_spin, req.width / 2, req.height);
  gtk_widget_show_all(table);

  init_dialog(dialog);
}

static void     
sg_segment_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  GtkWidget *child;
  gdouble scale;
  gchar *text;
  SGarrowDialog *dialog = (SGarrowDialog *)d->data;
  GtkPlotSegment *data = GTK_PLOT_SEGMENT(dialog->dataset);

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list)->selection->data;

  data->arrow_style = (GtkPlotSymbolStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list), child);

  data->arrow_length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->arrow_length_spin));
  data->arrow_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->arrow_width_spin));
  data->centered = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->arrow_center_check));
  data->relative = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->arrow_relative_check));
  data->arrow_mask = 0;
  data->arrow_mask |= gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->arrow_origin_check)) ? GTK_PLOT_ARROW_ORIGIN : 0;
  data->arrow_mask |= gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->arrow_end_check)) ? GTK_PLOT_ARROW_END : 0;

  text = gtk_entry_get_text(GTK_ENTRY(dialog->scale_entry));
  scale = atof(text);
  gtk_plot_data_set_a_scale(GTK_PLOT_DATA(data), scale);
}

