/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_planes_dialog.h"
#include "sg_dialog.h"
#include "sg_property_dialog.h"
#include "sg.h"

typedef struct 
{
  GtkWidget *xy_color_combo; 
  GtkWidget *yz_color_combo; 
  GtkWidget *zx_color_combo; 
  GtkWidget *xy_check; 
  GtkWidget *yz_check; 
  GtkWidget *zx_check; 
  GtkWidget *corner_check; 
  GtkWidget *corner_color_combo; 
  GtkWidget *thick_spin; 
  GtkWidget *style_combo; 
  SGlayer *layer;
} SGplaneDialog;

static gchar *line_styles[] = {
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

static void sg_plane_dialog_apply            (SGpropertyDialog *d, gpointer data); 
static void sg_plane_dialog_ok               (SGpropertyDialog *d, gpointer data); 
static void sg_plane_dialog_init_gui         (GtkWidget *widget);
  
GtkWidget *
sg_plane_dialog_new(SGlayer *layer)
{
  GtkWidget *widget;
  SGplaneDialog *dialog;

  dialog = g_new0(SGplaneDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_plane_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_plane_dialog_ok;
  dialog->layer = layer;
  sg_plane_dialog_init_gui(widget);

  return widget;
}

static void
sg_plane_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_plane_dialog_apply(d, data);
}

static void
sg_plane_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGplaneDialog *dialog = (SGplaneDialog *)d->data;
  SGlayer *layer = dialog->layer;
  GtkPlot3D *plot;
  SGplot *parent;
  GtkPlotCanvas *canvas;
  GtkColorCombo *combo;
  GtkWidget *child;
  GdkColor color;

  plot = GTK_PLOT3D(GTK_PLOT_CANVAS_PLOT(layer)->plot);

  combo = GTK_COLOR_COMBO(dialog->xy_color_combo);
  color = gtk_color_combo_get_selection(combo);
  plot->color_xy = color;

  combo = GTK_COLOR_COMBO(dialog->yz_color_combo);
  color = gtk_color_combo_get_selection(combo);
  plot->color_yz = color;

  combo = GTK_COLOR_COMBO(dialog->zx_color_combo);
  color = gtk_color_combo_get_selection(combo);
  plot->color_zx = color;

  plot->xy_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->xy_check));
  plot->yz_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->yz_check));
  plot->zx_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->zx_check));

  plot->corner_visible = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->corner_check));

  combo = GTK_COLOR_COMBO(dialog->corner_color_combo);
  color = gtk_color_combo_get_selection(combo);
  plot->corner.color = color;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo)->list)->selection->data;
  plot->corner.line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), child);

  combo = GTK_COLOR_COMBO(dialog->zx_color_combo);
  plot->corner.line_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->thick_spin));

  parent = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  canvas = GTK_PLOT_CANVAS(parent);
  gtk_plot_canvas_paint(canvas);
  gtk_plot_canvas_refresh(canvas);
}

static void
init_dialog(SGplaneDialog *dialog)
{
  GtkPlot3D *plot;
  SGlayer *layer = dialog->layer;

  plot = GTK_PLOT3D(GTK_PLOT_CANVAS_PLOT(layer)->plot);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->xy_color_combo), plot->color_xy);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->yz_color_combo), plot->color_yz);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->zx_color_combo), plot->color_zx);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->xy_check), plot->xy_visible);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->yz_check), plot->yz_visible);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->zx_check), plot->zx_visible);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->corner_check), plot->corner_visible);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->corner_color_combo), plot->corner.color);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), plot->corner.line_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->thick_spin), plot->corner.line_width);

}

static void
sg_plane_dialog_init_gui (GtkWidget *widget)
{
  GtkWidget *frame;
  GtkWidget *main_table;
  GtkWidget *table;
  GtkWidget *label;
  GtkAdjustment *adj;
  SGplaneDialog *dialog;

  dialog = (SGplaneDialog *)SG_PROPERTY_DIALOG(widget)->data;

  /* Create widgets */
  main_table = gtk_table_new (2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 10);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_container_add(GTK_CONTAINER(widget), main_table);

  /* Poperties */

  frame = gtk_frame_new( _("Planes") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 2, 0, 1, GTK_FILL, 0, 5, 5);

  table=gtk_table_new(3, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  dialog->xy_check = gtk_check_item_new_with_label( _("XY visible") );
  dialog->yz_check = gtk_check_item_new_with_label( _("YZ visible") );
  dialog->zx_check = gtk_check_item_new_with_label( _("ZX visible") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->xy_check, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->yz_check, 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->zx_check, 0, 1, 2, 3);

  dialog->xy_color_combo = gtk_color_combo_new();
  dialog->yz_color_combo = gtk_color_combo_new();
  dialog->zx_color_combo = gtk_color_combo_new();

  gtk_table_attach_defaults(GTK_TABLE(table), dialog->xy_color_combo, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->yz_color_combo, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->zx_color_combo, 1, 2, 2, 3);

  frame = gtk_frame_new( _("Oposite corner") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 2, 1, 2, GTK_FILL, 0, 5, 5);

  table = gtk_table_new(4, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  dialog->corner_check = gtk_check_item_new_with_label( _("visible") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->corner_check, 0, 1, 0, 1);
 
  label = gtk_label_new( _("Color ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  dialog->corner_color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->corner_color_combo, 1, 2, 1, 2);

 
  label = gtk_label_new( _("Thickness ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 1., 0.);
  dialog->thick_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->thick_spin), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->thick_spin, 1, 2, 2, 3);

  label = gtk_label_new( _("Style ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);
  dialog->style_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo), line_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo)->entry), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->style_combo, 1, 2, 3, 4);


  /* connect signals */
  init_dialog(dialog);
}
