/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_line_dialog.h"
#include "sg.h"
#include "sg_dialog.h"
#include "sg_stock.h"

typedef struct
{
  GtkWidget *fg_combo;
  GtkWidget *style_combo; 
  GtkWidget *line_width_spin; 
  GtkWidget *arrow_length_spin; 
  GtkWidget *arrow_width_spin; 
  GtkWidget *arrow_combo; 
  GtkWidget *origin_item;
  GtkWidget *end_item; 
  GtkPlotCanvas *canvas; 
  GtkPlotCanvasLine *line;
} SGlineDialog;

static gchar *line_styles[] = {
  N_("None"),
  N_("Solid"),
  N_("Dotted"),
  N_("Dashed"),
  N_("Dot-dash"),
  N_("Dot-dot-dash"),
  N_("Dot-dash-dash"),
  NULL
};

static gchar *symbol_styles[] = {
  N_("Empty"),
  N_("Filled"),
  N_("Opaque"),
  NULL
};

static void
update_line(SGpropertyDialog *pdialog, gpointer data)
{
  SGlineDialog *dialog = (SGlineDialog *)data;
  GdkColor color;
  GdkColormap *colormap;
  GtkPlotCanvasLine *line;
  GtkWidget *child;
  GtkPlotLineStyle line_style;

  line = dialog->line;
  colormap = gdk_colormap_get_system();

  color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->fg_combo));

  line->line.color = color;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->style_combo)->list)->selection->data;
  line_style = (GtkPlotLineStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), child);

  line->line.line_style = line_style;
  line->line.line_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->line_width_spin));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list)->selection->data;
  line->arrow_style = (GtkPlotSymbolStyle)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list), child);

  line->arrow_length = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->arrow_length_spin));
  line->arrow_width = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->arrow_width_spin));


  line->arrow_mask = (GtkPlotCanvasArrow)0;
  line->arrow_mask = (GtkPlotCanvasArrow)(line->arrow_mask | gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->origin_item)));
  line->arrow_mask = (GtkPlotCanvasArrow)(line->arrow_mask | (2*gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->end_item))));

  gtk_plot_canvas_paint(dialog->canvas);
  gtk_plot_canvas_refresh(dialog->canvas);
}


void
sg_line_dialog (GtkPlotCanvas *canvas, GtkPlotCanvasLine *line)
{
  SGlineDialog *dialog;
  GtkWidget *ref_dialog;
  GtkWindow *window;
  GtkWidget *frame;
  GtkWidget *table;
  GtkAdjustment *adj;

  dialog = g_new0(SGlineDialog, 1);
  dialog->canvas = canvas;
  dialog->line = line;

  /* Create widgets */
  frame = sg_property_dialog_new();
  gtk_frame_set_label(GTK_FRAME(frame), _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, TRUE);

  table=gtk_table_new(9, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);

  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Color") ), 0, 1, 0, 1);
  dialog->fg_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->fg_combo, 1, 2, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Line style") ), 0, 1, 2, 3);
  dialog->style_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->style_combo), line_styles);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->style_combo)->entry), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->style_combo, 1, 2, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Line width") ), 0, 1, 3, 4);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 1., 1., 0.);
  dialog->line_width_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->line_width_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->line_width_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->line_width_spin, 1, 2, 3, 4);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Arrow style") ), 0, 1, 4, 5);
  dialog->arrow_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->arrow_combo), symbol_styles);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_combo, 1, 2, 4, 5);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Arrow length") ), 0, 1, 5, 6);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
  dialog->arrow_length_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->arrow_length_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->arrow_length_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_length_spin, 1, 2, 5, 6);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Arrow width") ), 0, 1, 6, 7);
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 40., 1., 1., 0.);
  dialog->arrow_width_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->arrow_width_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->arrow_width_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->arrow_width_spin, 1, 2, 6, 7);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Arrow at") ), 0, 1, 7, 9);

  dialog->origin_item = gtk_check_item_new_with_label( _("Origin") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->origin_item, 1, 2, 7, 8);

  dialog->end_item = gtk_check_item_new_with_label( _("End") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->end_item, 1, 2, 8, 9);

  /* initialize */

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->style_combo)->list), 
                       line->line.line_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->line_width_spin), 
                            line->line.line_width);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->arrow_combo)->list), 
                       line->arrow_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->arrow_length_spin), 
                            line->arrow_length);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->arrow_width_spin), 
                            line->arrow_width);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->origin_item),
			      line->arrow_mask & GTK_PLOT_ARROW_ORIGIN);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->end_item),
			      line->arrow_mask & GTK_PLOT_ARROW_END);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->fg_combo), line->line.color); 

  /* connect signals */

  SG_PROPERTY_DIALOG(frame)->ok = update_line;
  SG_PROPERTY_DIALOG(frame)->apply = update_line;
  ref_dialog = sg_dialog_new( _("SciGraphica: Line "), GTK_ORIENTATION_HORIZONTAL, SG_BUTTON_OK|SG_BUTTON_APPLY|SG_BUTTON_CLOSE, GTK_BUTTONBOX_START); 
  window = GTK_WINDOW(ref_dialog);

  sg_dialog_add(ref_dialog, SG_PROPERTY_DIALOG(frame));
  gtk_window_set_policy(window, FALSE, FALSE, FALSE);
  gtk_widget_show_all(frame);
  sg_dialog_run(ref_dialog, GTK_OBJECT(canvas));
}
