/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_layer_dialog.h"
#include "sg.h"
#include "sg_property_dialog.h"
#include "sg_file_dialog.h"

typedef struct
{
  GtkWidget *left_entry; 
  GtkWidget *top_entry;
  GtkWidget *width_entry; 
  GtkWidget *height_entry;
  GtkWidget *transparent_item; 
  GtkWidget *bg_button;
  GtkWidget *units_combo;
  GtkWidget *ratio_item;
  gdouble pts_width, pts_height;
  gdouble pts_left, pts_top;
  GtkWidget *color_combo; 
  SGplot *plot;
  SGlayer *layer;
  gint units;
} SGlayerDialog;

static void sg_layer_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_layer_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_layer_dialog_init_gui         (GtkWidget *frame);

GtkWidget *
sg_layer_dialog_new(SGlayer *layer)
{
  GtkWidget *widget;
  SGlayerDialog *dialog;

  dialog = g_new0(SGlayerDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_layer_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_layer_dialog_ok;
  dialog->layer = layer;
  dialog->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  sg_layer_dialog_init_gui(widget);

  return widget;
}

static void
sg_layer_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_layer_dialog_apply(d, data);
}

static void
sg_layer_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGlayerDialog *dialog = (SGlayerDialog *)d->data;
  GtkWidget *child;
  GtkColorCombo *combo;
  GdkColor color;
  gdouble left, top, width, height;
  SGpageUnit units;
  GtkPlot *plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;
  gdouble page_width = 0., page_height = 0.;

  width = atof(gtk_entry_get_text(GTK_ENTRY(dialog->width_entry)));
  height = atof(gtk_entry_get_text(GTK_ENTRY(dialog->height_entry)));
  left = atof(gtk_entry_get_text(GTK_ENTRY(dialog->left_entry)));
  top = atof(gtk_entry_get_text(GTK_ENTRY(dialog->top_entry)));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->units_combo)->list)->selection->data;
  dialog->units = units = (SGpageUnit)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->units_combo)->list), child);

  if(dialog->plot->orientation == GTK_PLOT_PORTRAIT){ 
    page_width = dialog->plot->page_width;
    page_height = dialog->plot->page_height;
  } else {
    page_width = dialog->plot->page_height;
    page_height = dialog->plot->page_width;
  }

  switch(units){
   case SG_UNIT_MM:
     dialog->pts_width = (width * PTS_MM);
     dialog->pts_height = (height * PTS_MM);
     dialog->pts_left = (left * PTS_MM);
     dialog->pts_top = (top * PTS_MM);
     break;
   case SG_UNIT_CM:
     dialog->pts_width = (width * PTS_CM);
     dialog->pts_height = (height * PTS_CM);
     dialog->pts_left = (left * PTS_CM);
     dialog->pts_top = (top * PTS_CM);
     break;
   case SG_UNIT_IN:
     dialog->pts_width = (width * PTS_IN);
     dialog->pts_height = (height * PTS_IN);
     dialog->pts_left = (left * PTS_IN);
     dialog->pts_top = (top * PTS_IN);
     break;
   case SG_UNIT_PIXEL:
     dialog->pts_width = (width * PTS_PIXEL);
     dialog->pts_height = (height * PTS_PIXEL);
     dialog->pts_left = (left * PTS_PIXEL);
     dialog->pts_top = (top * PTS_PIXEL);
     break;
   case SG_UNIT_PT:
     dialog->pts_width = (width * PTS_PT);
     dialog->pts_height = (height * PTS_PT);
     dialog->pts_left = (left * PTS_PT);
     dialog->pts_top = (top * PTS_PT);
     break;
   case SG_UNIT_PERCENT_OF_PAGE:
   default:
     dialog->pts_width = (width * page_width);
     dialog->pts_height = (height * page_height);
     dialog->pts_left = (left * page_width);
     dialog->pts_top = (top * page_height);
     break;
  }

  left = (gdouble) dialog->pts_left / (gdouble) page_width;
  top = (gdouble) dialog->pts_top / (gdouble) page_height;
  width = (gdouble) dialog->pts_width / (gdouble) page_width;
  height = (gdouble) dialog->pts_height / (gdouble) page_height;

  gtk_plot_canvas_child_move_resize(GTK_PLOT_CANVAS(dialog->plot),
				    GTK_PLOT_CANVAS_CHILD(dialog->layer),
                                    left, top, left+width, top+height);

  combo = GTK_COLOR_COMBO(dialog->color_combo);
  color = gtk_color_combo_get_selection(combo);
  plot->background = color;

  gtk_plot_set_transparent(GTK_PLOT(plot), 
            gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->transparent_item)));

/*
  plot->show_legends = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(legends_item));
*/

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
}

static void
select_image(GtkWidget *widget, gpointer data)
{
  GtkPixmap *pixmap;
  SGlayerDialog *dialog = (SGlayerDialog *)data;
  GtkWidget *filesel;
  gboolean return_value;

  pixmap = GTK_PIXMAP(gtk_pixmap_new(gdk_pixmap_new(widget->window, 10, 10, -1), NULL));
  filesel = sg_file_dialog_new(SG_PLUGIN_FILE_OPEN, NULL, "image", G_OBJECT(pixmap));
  return_value = sg_file_dialog_run(filesel);

  if(return_value) 
     gtk_plot_set_background_pixmap(GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot, GTK_PIXMAP(pixmap)->pixmap);

  gtk_widget_unref(GTK_WIDGET(pixmap));
}

static void
new_units(GtkWidget *widget, GtkWidget *child, gpointer data)
{
  gchar label[100];
  SGpageUnit new_units;
  gdouble width, height;
  gdouble left, top;
  SGlayerDialog *dialog = (SGlayerDialog *)data;

  dialog->units = new_units = (SGpageUnit)gtk_list_child_position(GTK_LIST(widget), child);

  switch(new_units){
   case SG_UNIT_MM:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_MM;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_MM;
       left = (gdouble)dialog->pts_left / (gdouble)PTS_MM;
       top = (gdouble)dialog->pts_top / (gdouble)PTS_MM;
       break;
   case SG_UNIT_CM:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_CM;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_CM;
       left = (gdouble)dialog->pts_left / (gdouble)PTS_CM;
       top = (gdouble)dialog->pts_top / (gdouble)PTS_CM;
       break;
   case SG_UNIT_IN:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_IN;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_IN;
       left = (gdouble)dialog->pts_left / (gdouble)PTS_IN;
       top = (gdouble)dialog->pts_top / (gdouble)PTS_IN;
       break;
   case SG_UNIT_PIXEL:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_PIXEL;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_PIXEL;
       left = (gdouble)dialog->pts_left / (gdouble)PTS_PIXEL;
       top = (gdouble)dialog->pts_top / (gdouble)PTS_PIXEL;
       break;
   case SG_UNIT_PT:
       width = (gdouble)dialog->pts_width / (gdouble)PTS_PT;
       height = (gdouble)dialog->pts_height / (gdouble)PTS_PT;
       left = (gdouble)dialog->pts_left / (gdouble)PTS_PT;
       top = (gdouble)dialog->pts_top / (gdouble)PTS_PT;
       break;
   case SG_UNIT_PERCENT_OF_PAGE:
   default:
       if(dialog->plot->orientation == GTK_PLOT_PORTRAIT){
         width = (gdouble)dialog->pts_width / (gdouble)dialog->plot->page_width;
         height = (gdouble)dialog->pts_height / (gdouble)dialog->plot->page_height;
         left = (gdouble)dialog->pts_left / (gdouble)dialog->plot->page_width;
         top = (gdouble)dialog->pts_top / (gdouble)dialog->plot->page_height;
       } else {
         width = (gdouble)dialog->pts_width / (gdouble)dialog->plot->page_height;
         height = (gdouble)dialog->pts_height / (gdouble)dialog->plot->page_width;
         left = (gdouble)dialog->pts_left / (gdouble)dialog->plot->page_height;
         top = (gdouble)dialog->pts_top / (gdouble)dialog->plot->page_width;
       }
       break;
  }

  sprintf(label, "%.*f", 5, width);
  gtk_entry_set_text(GTK_ENTRY(dialog->width_entry), label);
  sprintf(label, "%.*f", 5, height);
  gtk_entry_set_text(GTK_ENTRY(dialog->height_entry), label);
  sprintf(label, "%.*f", 5, left);
  gtk_entry_set_text(GTK_ENTRY(dialog->left_entry), label);
  sprintf(label, "%.*f", 5, top);
  gtk_entry_set_text(GTK_ENTRY(dialog->top_entry), label);
}

static void
preserve_ratio(GtkWidget *widget, gpointer data)
{
  GtkWidget *child;
  gint new_width, new_height;
  gdouble width, height;
  gdouble ratio;
  gchar label[100];
  SGpageUnit units;
  SGlayerDialog *dialog = (SGlayerDialog *)data;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->units_combo)->list)->selection->data;
  dialog->units = units = (SGpageUnit)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->units_combo)->list), child);


  width = atof(gtk_entry_get_text(GTK_ENTRY(dialog->width_entry)));
  height = atof(gtk_entry_get_text(GTK_ENTRY(dialog->height_entry)));
 
  if(!gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->ratio_item))) return;

  switch(units){
   case SG_UNIT_MM:
     new_width = (width * PTS_MM);
     new_height = (height * PTS_MM);
     break;
   case SG_UNIT_CM:
     new_width = (width * PTS_CM);
     new_height = (height * PTS_CM);
     break;
   case SG_UNIT_IN:
     new_width = (width * PTS_IN);
     new_height = (height * PTS_IN);
     break;
   case SG_UNIT_PIXEL:
     new_width = (width * PTS_PIXEL);
     new_height = (height * PTS_PIXEL);
     break;
   case SG_UNIT_PT:
     new_width = (width * PTS_PT);
     new_height = (height * PTS_PT);
     break;
   case SG_UNIT_PERCENT_OF_PAGE:
   default:
     if(dialog->plot->orientation == GTK_PLOT_PORTRAIT){
       new_width = (width * dialog->plot->page_width);
       new_height = (height * dialog->plot->page_height);
     } else {
       new_width = (width * dialog->plot->page_height);
       new_height = (height * dialog->plot->page_width);
     }
     break;
  }

  if(widget == dialog->width_entry){
      ratio = (gdouble)new_width / (gdouble)dialog->pts_width;
      height *= ratio;
      dialog->pts_height *= (ratio);
  }else{
      ratio = (gdouble)new_height / (gdouble)dialog->pts_height;
      width *= ratio;
      dialog->pts_width *= (ratio);
  }
  
  sprintf(label, "%.*f", 5, width);
  gtk_entry_set_text(GTK_ENTRY(dialog->width_entry), label);
  sprintf(label, "%.*f", 5, height);
  gtk_entry_set_text(GTK_ENTRY(dialog->height_entry), label);
}

static void
init_dialog(SGlayerDialog *dialog)
{
  GtkPlot *plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;
  gchar *units[] = { N_("PS points"),
                     N_("mm"),
                     N_("cm"),
                     N_("inches"),
                     N_("pixels"),
                     N_("% of page size"),
                     NULL };

  dialog->units = dialog->plot->page_units;
  sg_combo_set_items(GTK_COMBO(dialog->units_combo), units);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->units_combo)->list), SG_UNIT_MM);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->units_combo)->entry), FALSE);

  if(dialog->plot->orientation == GTK_PLOT_PORTRAIT){
    dialog->pts_width = (plot->width * dialog->plot->page_width);
    dialog->pts_height = (plot->height * dialog->plot->page_height);
    dialog->pts_left = (plot->x * dialog->plot->page_width);
    dialog->pts_top = (plot->y * dialog->plot->page_height);
  } else {
    dialog->pts_width = (plot->width * dialog->plot->page_height);
    dialog->pts_height = (plot->height * dialog->plot->page_width);
    dialog->pts_left = (plot->x * dialog->plot->page_height);
    dialog->pts_top = (plot->y * dialog->plot->page_width);
  }

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->units_combo)->list), dialog->units);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->transparent_item),
                              gtk_plot_is_transparent(GTK_PLOT(plot))); 

/*
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(legends_item),
                              plot->show_legends); 
*/

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->color_combo), GTK_PLOT(plot)->background);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->ratio_item), FALSE);
}

static void
sg_layer_dialog_init_gui (GtkWidget *frame)
{
  GtkWidget *main_table;
  GtkWidget *table;
  GtkWidget *box;
  GtkWidget *label;
  GtkRequisition req;
  SGlayerDialog *dialog;

  dialog = (SGlayerDialog *)SG_PROPERTY_DIALOG(frame)->data;

  /* Create widgets */
  main_table = gtk_table_new(2,2,FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_container_add(GTK_CONTAINER(frame), main_table);

/*----------------------------------------------*/
  frame = gtk_frame_new( _("Position") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 0, 1, 0, 0, 5, 5);

  table=gtk_table_new(5, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  box = gtk_hbox_new(TRUE, 5);
  gtk_table_attach(GTK_TABLE(table), 
                   box, 0, 4, 0, 1, 0, 0, 5, 5);
  label = gtk_label_new( _("Units ") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_box_pack_start(GTK_BOX(box), label, FALSE, FALSE, 0);
  dialog->units_combo = gtk_combo_new();
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->units_combo)->entry), FALSE);
  gtk_box_pack_end(GTK_BOX(box), dialog->units_combo, FALSE, FALSE, 0);

  gtk_table_attach(GTK_TABLE(table), gtk_hseparator_new(), 0, 4, 1, 2, 
                   GTK_FILL, GTK_SHRINK, 5, 5); 

  gtk_table_attach(GTK_TABLE(table), 
                   gtk_label_new( _("left offset ") ), 0, 1, 2, 3, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), 
                   gtk_label_new( _("top offset ") ), 0, 1, 3, 4, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), 
                   gtk_label_new( _("width ") ), 2, 3, 2, 3, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), 
                   gtk_label_new( _("height ") ), 2, 3, 3, 4, 0, 0, 0, 5);
  dialog->left_entry = gtk_entry_new();
  dialog->top_entry = gtk_entry_new();
  dialog->width_entry = gtk_entry_new();
  dialog->height_entry = gtk_entry_new();

  gtk_widget_size_request(dialog->left_entry, &req);
  req.width /= 2;
  gtk_widget_set_usize(dialog->left_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->top_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->width_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->height_entry, req.width, req.height);

  gtk_table_attach(GTK_TABLE(table), dialog->left_entry, 1, 2, 2, 3, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->top_entry, 1, 2, 3, 4, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->width_entry, 3, 4, 2, 3, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->height_entry, 3, 4, 3, 4, 0, 0, 0, 5);

  gtk_table_attach(GTK_TABLE(table), 
      dialog->ratio_item = gtk_check_item_new_with_label( _("preserve aspect ratio") ),
      0, 4, 4, 5, GTK_FILL, 0, 0, 0);
  
 /*--------------------------------------*/
  frame = gtk_frame_new( _("Properties") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 0, 1, 0, GTK_FILL, 5, 5);

  table=gtk_table_new(3, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  dialog->transparent_item = gtk_check_item_new_with_label( _("transparent") );
  gtk_table_attach(GTK_TABLE(table), dialog->transparent_item, 0, 2, 0, 1, 0, 0, 0, 5);

  gtk_table_attach(GTK_TABLE(table), 
                   gtk_label_new( _("Background") ), 0, 1, 1, 2, 0, 0, 0, 5);
  dialog->color_combo = gtk_color_combo_new();
  gtk_table_attach(GTK_TABLE(table), dialog->color_combo, 1, 2, 1, 2, 0, 0, 0, 5);

/*
  legends_item = gtk_check_item_new_with_label("show legends");
  gtk_table_attach(GTK_TABLE(table), legends_item, 0, 2, 2, 3, 0, 0, 0, 5);
*/
  dialog->bg_button = gtk_button_new_with_label( _("Browse...") );

  gtk_table_attach(GTK_TABLE(table), gtk_label_new( _("Use image") ), 0, 1, 2, 3, 0, 0, 0, 5);
  gtk_table_attach(GTK_TABLE(table), dialog->bg_button, 1, 2, 2, 3, 0, 0, 0, 5);
/*--------------------------------------*/
  /* connect signals */

  sg_entry_set_numeric(GTK_ENTRY(dialog->left_entry), 5);
  sg_entry_set_numeric(GTK_ENTRY(dialog->top_entry), 5);
  sg_entry_set_numeric(GTK_ENTRY(dialog->width_entry), 5);
  sg_entry_set_numeric(GTK_ENTRY(dialog->height_entry), 5);

  gtk_signal_connect (GTK_OBJECT (dialog->bg_button),
                      "clicked",
                      GTK_SIGNAL_FUNC (select_image),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->units_combo)->list),
                      "select_child",
                      GTK_SIGNAL_FUNC (new_units),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (dialog->width_entry),
                      "activate",
                      GTK_SIGNAL_FUNC (preserve_ratio),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (dialog->height_entry),
                      "activate",
                      GTK_SIGNAL_FUNC (preserve_ratio),
                      dialog);

  init_dialog(dialog);
}
