/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_labels_dialog.h"
#include "sg.h"
#include "sg_layer.h"
#include "sg_dialog.h"
#include "sg_property_dialog.h"

typedef struct
{
  SGplot *plot;
  SGlayer *layer;
  GtkPlotAxis *axis;

  GtkWidget *font_combo;
  GtkWidget *format_combo;
  GtkWidget *offset_spin;
  GtkWidget *angle_spin;
  GtkWidget *precision_spin;
  GtkWidget *custom_item;
  GtkWidget *plugin_button;
  GtkWidget *plugin_list;
  GtkWidget *plugin_dialog;
  GtkWidget *edit_button;
  GtkWidget *prefix_entry;
  GtkWidget *suffix_entry;
} SGlabelsDialog;

static void sg_labels_dialog_apply            (SGpropertyDialog *d, gpointer data); 
static void sg_labels_dialog_ok               (SGpropertyDialog *d, gpointer data); 
static void sg_labels_dialog_init_gui         (GtkWidget *frame);
static void select_plugin                     (GtkWidget *widget, 
                                               gpointer data);
 
 
GtkWidget *
sg_labels_dialog_new(SGlayer *layer, GtkPlotAxis *axis)
{
  GtkWidget *widget;
  SGlabelsDialog *dialog;

  dialog = g_new0(SGlabelsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->apply = sg_labels_dialog_apply;
  SG_PROPERTY_DIALOG(widget)->ok = sg_labels_dialog_ok;
  dialog->layer = layer;
  dialog->axis = axis;
  dialog->plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(layer)->parent);
  sg_labels_dialog_init_gui(widget);

  return widget;
}

static void
sg_labels_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_labels_dialog_apply(d, data);
}

static void
sg_labels_dialog_apply(SGpropertyDialog *d, gpointer data)
{
  SGlabelsDialog *dialog = (SGlabelsDialog *)d->data;
  GtkPlot *plot;
  GtkPlotAxis *axis;
  GdkColor bg, fg;
  GtkWidget *child;
  gint the_axis = 0;
  const gchar *text;
  const gchar *font;
  gint height, angle;
  GtkPSFont *psfont;

  plot = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot;
  axis = dialog->axis;

  if(axis == plot->left) the_axis = GTK_PLOT_AXIS_LEFT;
  if(axis == plot->right) the_axis = GTK_PLOT_AXIS_RIGHT;
  if(axis == plot->top) the_axis = GTK_PLOT_AXIS_TOP;
  if(axis == plot->bottom) the_axis = GTK_PLOT_AXIS_BOTTOM;

  psfont = gtk_font_combo_get_psfont(GTK_FONT_COMBO(dialog->font_combo));
  font = psfont->psname;
  height = gtk_font_combo_get_font_height(GTK_FONT_COMBO(dialog->font_combo));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data;
  axis->label_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), child);

  axis->label_precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin));
  axis->labels_offset = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->offset_spin));
  angle = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->angle_spin));
  axis->custom_labels = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->custom_item));

  switch(the_axis){
    case GTK_PLOT_AXIS_LEFT:
     if(angle == 90 || angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 0)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     if(angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     break;
    case GTK_PLOT_AXIS_RIGHT:
     if(angle == 90 || angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 0)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     if(angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     break;
    case GTK_PLOT_AXIS_TOP:
     if(angle == 0 || angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 90)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
     if(angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     break;
    case GTK_PLOT_AXIS_BOTTOM:
     if(angle == 0 || angle == 180)
                axis->labels_attr.justification = GTK_JUSTIFY_CENTER; 
     if(angle == 90)
                axis->labels_attr.justification = GTK_JUSTIFY_RIGHT; 
     if(angle == 270)
                axis->labels_attr.justification = GTK_JUSTIFY_LEFT; 
  }

  fg = axis->labels_attr.fg;
  bg = axis->labels_attr.bg; 
  gtk_plot_axis_set_labels_attributes(axis,
                                      font,
                                      height,
                                      angle,
                                      &fg,
                                      &bg, 
                                      TRUE, 
                                      axis->labels_attr.justification);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->prefix_entry));
  if(text && strlen(text) > 0)
     gtk_plot_axis_set_labels_prefix(axis, text);
  else
     gtk_plot_axis_set_labels_prefix(axis, NULL);

  text = gtk_entry_get_text(GTK_ENTRY(dialog->suffix_entry));
  if(text && strlen(text) > 0)
     gtk_plot_axis_set_labels_suffix(axis, text);
  else
     gtk_plot_axis_set_labels_suffix(axis, NULL);

  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));
}


static void 
init_dialog(SGlabelsDialog *dialog)
{
  gchar *label_formats[] = { N_("Decimal: 1.000"), 
                             N_("Scientific: 1.E10"), 
                             N_("Scientific: 1x10^10"), 
                             NULL };
  GtkPSFont *font;

  sg_combo_set_items(GTK_COMBO(dialog->format_combo), label_formats);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), dialog->axis->label_style); 

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->format_combo)->entry), FALSE);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->offset_spin), 
                            dialog->axis->labels_offset);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin), 
                            dialog->axis->label_precision);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->angle_spin), 
                            dialog->axis->labels_attr.angle);

  if(dialog->axis->labels_prefix)
     gtk_entry_set_text(GTK_ENTRY(dialog->prefix_entry), dialog->axis->labels_prefix);

  if(dialog->axis->labels_suffix)
     gtk_entry_set_text(GTK_ENTRY(dialog->suffix_entry), dialog->axis->labels_suffix);

  font = gtk_psfont_get_by_name(dialog->axis->labels_attr.font);
  gtk_font_combo_select(GTK_FONT_COMBO(dialog->font_combo),
                        font->family,
                        FALSE,
                        FALSE,
                        dialog->axis->labels_attr.height);

  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->custom_item), dialog->axis->custom_labels);
}

static gint
add_row(GtkWidget *widget,
        gint row, gint col, gint *new_row, gint *new_col,
        gpointer data)
{
  GtkSheet *sheet = GTK_SHEET(widget);
                                                                                
  if(row == *new_row){
    gtk_sheet_add_row(sheet, 1);
    *new_row = *new_row + 1;
  }
  return TRUE;
}

static void
edit_labels(GtkWidget *widget, gpointer data)
{
  SGlabelsDialog *ldialog = (SGlabelsDialog *)data;
  GtkWidget *sw;
  GtkWidget *sheet = NULL;
  SGdialog *dialog;
  SGdialogButton rvalue;
  gint i = 0;
  gchar **array_data = NULL;
  GtkPlotArray *array = NULL;
                                                                                
  dialog = SG_DIALOG(sg_dialog_new( _( "Edit data" ), GTK_ORIENTATION_VERTICAL,
SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_END));
                                                                                
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 200, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(dialog->box),sw,TRUE,TRUE,0);

  if(ldialog->axis->tick_labels){
    gint nrows = ldialog->axis->tick_labels->size;
    gchar **_data = ldialog->axis->tick_labels->data.data_string;
    sheet = gtk_sheet_new(nrows, 1, "");
    for(i = 0; i < nrows; i++) {
      if(_data && _data[i]) 
        gtk_sheet_set_cell_text(GTK_SHEET(sheet), i, 0, _data[i]);
    } 
  }else{
    sheet = gtk_sheet_new(20, 1, "");
  }
  gtk_container_add(GTK_CONTAINER(sw), sheet);
  gtk_widget_ref(sheet);
  gtk_widget_show_all(dialog->box);
  gtk_sheet_column_button_add_label(GTK_SHEET(sheet), 0, _("Labels") );
  gtk_signal_connect(GTK_OBJECT(GTK_SHEET(sheet)), "traverse",
                     GTK_SIGNAL_FUNC(add_row), NULL);
  gtk_widget_show_all(dialog->box);
                                                                                
  rvalue = sg_dialog_run(GTK_WIDGET(dialog), GTK_OBJECT(widget));
                                                                                
  if(rvalue == SG_BUTTON_CANCEL) {
    gtk_widget_destroy(sheet);
    return;
  }
                                                                                
  array_data = g_new(gchar *, GTK_SHEET(sheet)->maxallocrow+1);
  for(i = 0; i <= GTK_SHEET(sheet)->maxallocrow; i++){
    gchar *text = gtk_sheet_cell_get_text(GTK_SHEET(sheet), i, 0);
    if(text)
      array_data[i] = g_strdup(text);
    else
      array_data[i] = NULL;
  }

  array = GTK_PLOT_ARRAY(gtk_plot_array_new(NULL, array_data, GTK_SHEET(sheet)->maxallocrow+1, GTK_TYPE_STRING, TRUE));
  gtk_plot_axis_set_tick_labels(ldialog->axis, array);
  gtk_widget_destroy(sheet);
}

static void
real_select_plugin(GtkWidget *widget, gpointer data)
{
  SGlabelsDialog *dialog = (SGlabelsDialog *)data;
  SGpluginArray *plugin;
  GList *selection = NULL;
  gint row;

  selection = GTK_CLIST(dialog->plugin_list)->selection;
  if(!selection) return;
                                                                             
  row = GPOINTER_TO_INT(selection->data);
  plugin = (SGpluginArray *)gtk_clist_get_row_data(GTK_CLIST(dialog->plugin_list), row);

  if(plugin){
    GtkPlotArray *array;
    gpointer app_data;
    SGapplication *app = NULL;
    app_data = g_object_get_data(G_OBJECT(dialog->plot), "application");
    if(app_data) app = SG_APPLICATION(app_data);
    gtk_widget_hide(dialog->plugin_dialog);
    array = sg_plugin_array_action(plugin, app, NULL);
    if(array) {
      gtk_plot_axis_set_tick_labels(dialog->axis, array);
    }
  }
}

static void
select_plugin(GtkWidget *button, gpointer data)
{
  GtkWidget *sw;
  gint nrows = 0;
  GList *list;
  SGlabelsDialog *ldialog = (SGlabelsDialog *)data;
  SGdialog *dialog;
  GtkWidget *clist;

  dialog = SG_DIALOG(sg_dialog_new(_( "Pick labels source" ), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_END));
  ldialog->plugin_dialog = GTK_WIDGET(dialog);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 200, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);  
  gtk_box_pack_start(GTK_BOX(dialog->box),sw,TRUE,TRUE,0);
  ldialog->plugin_list = clist = gtk_clist_new(1);
  gtk_clist_set_row_height(GTK_CLIST(clist), 20);
  gtk_clist_set_column_min_width(GTK_CLIST(clist), 0, 400);
  gtk_container_add(GTK_CONTAINER(sw), clist);
                                                                             
  list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_ARRAY(list->data)){
      SGpluginArray *plugin = SG_PLUGIN_ARRAY(list->data);
      gchar *text[1];

      text[0] = SG_PLUGIN(plugin)->description;
      gtk_clist_append(GTK_CLIST(clist), text);
      gtk_clist_set_row_data(GTK_CLIST(clist), nrows, plugin);
      nrows++;
    }
    list = list->next;
  }
                                                                             
  gtk_signal_connect(GTK_OBJECT(dialog->button_ok), "clicked",
                     GTK_SIGNAL_FUNC(real_select_plugin), ldialog);
  gtk_widget_show_all(dialog->box);
  sg_dialog_run(GTK_WIDGET(dialog), NULL);
}

static void 
sg_labels_dialog_init_gui (GtkWidget *dialog_frame)
{
  GtkWidget *main_box;
  GtkWidget *table, *main_table;
  GtkWidget *label;
  GtkWidget *frame;
  GtkAdjustment *adj;
  SGlabelsDialog *dialog;
  GtkWidget *box;

  dialog = (SGlabelsDialog *)SG_PROPERTY_DIALOG(dialog_frame)->data;

  /* Create widgets */
  main_box = gtk_hbox_new (FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(main_box), 5);
  gtk_container_add(GTK_CONTAINER(dialog_frame), main_box);

  main_table = gtk_table_new(2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(main_table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(main_table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(main_table), 5);
  gtk_box_pack_start (GTK_BOX (main_box), main_table, FALSE, FALSE, 0);

/*----------------------------------------------*/
  frame = gtk_frame_new( _("Format") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 0, 1, 1, 2, GTK_EXPAND|GTK_FILL, GTK_FILL, 5, 5);

  table = gtk_table_new(8, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  dialog->custom_item = gtk_check_item_new_with_label( _("Custom") );
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->custom_item, 0, 1, 1, 2);

  box = gtk_hbox_new(5, TRUE);
  dialog->plugin_button = gtk_button_new_with_label( _("Select...") ); 
  gtk_box_pack_start(GTK_BOX(box), dialog->plugin_button, TRUE, TRUE, 5);
  dialog->edit_button = gtk_button_new_with_label( _("Edit") ); 
  gtk_box_pack_start(GTK_BOX(box), dialog->edit_button, TRUE, TRUE, 5);
  gtk_table_attach_defaults(GTK_TABLE(table), box, 1, 2, 1, 2);

  label = gtk_label_new( _("Format") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 3, 4);

  dialog->format_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->format_combo, 1, 2, 3, 4);

  label = gtk_label_new( _("Precision") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 4, 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 8., 1., 1., 0.);
  dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->precision_spin, 1, 2, 4, 5);

  label = gtk_label_new( _("Prefix") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 5, 6);

  dialog->prefix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->prefix_entry, 1, 2, 5, 6);

  label = gtk_label_new( _("Suffix") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., .5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 6, 7);

  dialog->suffix_entry = gtk_entry_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->suffix_entry, 1, 2, 6, 7);
  

  dialog->font_combo = gtk_font_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->font_combo, 0, 2, 7, 8);

/*----------------------------------------------*/
  frame = gtk_frame_new( _("Position") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_table_attach(GTK_TABLE(main_table), frame, 1, 2, 1, 2, GTK_EXPAND|GTK_FILL, GTK_FILL, 5, 5);

  table = gtk_table_new(3, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(frame), table);

  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Offset") ), 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            gtk_label_new( _("Rotate") ), 0, 1, 1, 2);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 60., 1., 1., 0.);
  dialog->offset_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->offset_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->offset_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->offset_spin, 1, 2, 0, 1);
  
  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 270., 90., 90., 0.);
  dialog->angle_spin = gtk_spin_button_new(adj, 90, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->angle_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->angle_spin), 0);
  gtk_entry_set_editable(GTK_ENTRY(dialog->angle_spin), FALSE);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->angle_spin, 1, 2, 1, 2);
  
  /* connect signals */
  gtk_signal_connect(GTK_OBJECT(dialog->plugin_button), "clicked",
		     GTK_SIGNAL_FUNC(select_plugin), dialog);
  gtk_signal_connect(GTK_OBJECT(dialog->edit_button), "clicked",
		     GTK_SIGNAL_FUNC(edit_labels), dialog);


  gtk_widget_show_all(main_box);

  init_dialog(dialog);
}

GtkWidget *
sg_labels_dialog(SGlayer *layer,
		 GtkWidget *ok_button,
		 GtkWidget *apply_button,
		 GtkWidget *cancel_button)
{
  GtkWidget *dialog;
  GtkWidget *notebook;
  GtkPlot *plot;

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;
  notebook = gtk_notebook_new();

  if(strcmp(SG_PLUGIN(layer->plugin)->name, "layer_3d") == 0){
      dialog = sg_labels_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("X") ));
      dialog = sg_labels_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Y") ));
      dialog = sg_labels_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Z") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name, "layer_2d") == 0){
      dialog = sg_labels_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Left") ));
      dialog = sg_labels_dialog_new(layer, plot->right);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Right") ));
      dialog = sg_labels_dialog_new(layer, plot->top);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Top") ));
      dialog = sg_labels_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Bottom") ));
  }
  if(strcmp(SG_PLUGIN(layer->plugin)->name, "layer_polar") == 0){
      dialog = sg_labels_dialog_new(layer, plot->bottom);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("Angles") ));
      dialog = sg_labels_dialog_new(layer, plot->left);
      sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog),
				     ok_button, apply_button, cancel_button);
      gtk_notebook_append_page(GTK_NOTEBOOK(notebook), dialog,
                               gtk_label_new( _("R") ));
  }
  return notebook;
}
