/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#define GTK_ENABLE_BROKEN
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_dialog.h"
#include "sg_plugin_function.h"
#include "sg_fit_dialog.h"
#include "sg_stock.h"
#include "lm.h"


typedef struct
{
  SGplot *plot;
  GtkPlotData *dataset;
  SGlayer *layer;

  SGpluginFunction *function;
  
  GtkWidget *dialog;

  GtkWidget *clist;
  GtkWidget *fentry;
  GtkWidget *nspin;
  GtkWidget *pentry;
  GtkWidget *iventry;
  GtkWidget *dventry;

  GtkWidget *marker_check;
  GtkWidget *marker1_spin;
  GtkWidget *marker1_entry;
  GtkWidget *marker2_spin;
  GtkWidget *marker2_entry;
  gdouble from_value;
  gdouble to_value;
  gdouble *xvalues;
  gdouble *yvalues;
  gint npoints;

  gboolean apply_constraints;

  gdouble pval[10];
  GtkWidget *val_entry[10];
  GtkWidget *var_check[10];
  GtkWidget *plot_button;

  GtkWidget *cancel_button;
  GtkWidget *next_button;
  GtkWidget *back_button;
  GtkWidget *main_box;

} SGfitDialog;

static gint pick_dataset(GtkWidget *widget, gpointer data);
static gint pick_function(GtkWidget *widget, gpointer data);
static gint start_fit(GtkWidget *widget, gpointer data);
static gint plot_dataset(GtkWidget *widget, gpointer data);

static gint
create_dataset(GtkWidget *widget, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  SGpluginFunction *plugin = dialog->function;
  SGdataset *new_dataset = NULL;
  gboolean do_plot;

  do_plot = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->plot_button));
  if(do_plot){
/*
    gdouble *x = NULL, *new_x = NULL, *new_y = NULL;
    gint npoints;
    gint i;
    gboolean error;

    x = gtk_plot_data_get_x(dataset, &npoints);
    new_x = g_new0(gdouble, npoints);
    new_y = g_new0(gdouble, npoints);
    for(i = 0; i < npoints; i++){
      new_x[i] = x[i];
      new_y[i] = sg_plugin_function_action(plugin, dialog->pval, x[i], &error, NULL); 
    }

    new_dataset = sg_plugin_iterator_construct(sg_plugin_iterator_get("sg_dataset_file"), sg_plugin_style_get("lpoints_2d"));
    if(new_dataset){
      sg_dataset_set_points(new_dataset, "x", new_x, npoints);
      sg_dataset_set_points(new_dataset, "y", new_y, npoints);
      sg_dataset_set_name(new_dataset, SG_PLUGIN(plugin)->name);
      sg_list_add(dialog->plot->datasets, G_OBJECT(new_dataset), new_dataset->name);
      sg_layer_add_dataset_autosymbol(dialog->layer, new_dataset);
    }
*/

    new_dataset = sg_plugin_iterator_construct(sg_plugin_iterator_get("sg_dataset_function"), sg_plugin_style_get("lines_2d"));
    if(new_dataset){
      gchar *text = NULL;
      gchar aux_text[100];
      gint i;

      g_snprintf(aux_text, 100, "%s = %g", plugin->param[0], dialog->pval[0]);
      text = g_strdup(aux_text); 
      for(i = 1; i < plugin->nparam; i++){
        g_snprintf(aux_text, 100, "; %s = %g", plugin->param[i], dialog->pval[i]);
        text = g_strconcat(text, aux_text, NULL); 
      }
      g_snprintf(aux_text, 100, "; y = %s", plugin->function);
      text = g_strconcat(text, aux_text, NULL); 
      g_object_set(G_OBJECT(new_dataset), 
                   "SGdatasetFunction::exp", text, NULL);
      sg_dataset_set_name(new_dataset, SG_PLUGIN(plugin)->name);
      g_free(text);
      sg_list_add(dialog->plot->datasets, G_OBJECT(new_dataset), new_dataset->name);
      sg_layer_add_dataset_autosymbol(dialog->layer, new_dataset);
    }

  } 
/*
  sg_layer_refresh_datasets(dialog->layer);
*/
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(dialog->plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(dialog->plot));

  if(dialog->xvalues) g_free(dialog->xvalues);
  dialog->xvalues = NULL;
  if(dialog->yvalues) g_free(dialog->yvalues);
  dialog->yvalues = NULL;

  gtk_widget_destroy(dialog->dialog);
  return FALSE;
}


static void
create_wizard_window(SGfitDialog *dialog, const gchar *title)
{
  if(dialog->dialog) gtk_widget_destroy(dialog->dialog);
  dialog->dialog = sg_dialog_new(title, GTK_ORIENTATION_VERTICAL, SG_BUTTON_BACK|SG_BUTTON_NEXT|SG_BUTTON_CANCEL, GTK_BUTTONBOX_END);
  gtk_window_position (GTK_WINDOW (dialog->dialog), GTK_WIN_POS_CENTER);
  dialog->main_box = SG_DIALOG(dialog->dialog)->box;
  dialog->next_button = SG_DIALOG(dialog->dialog)->button_next;
  dialog->back_button = SG_DIALOG(dialog->dialog)->button_back;
  dialog->cancel_button = SG_DIALOG(dialog->dialog)->button_cancel;
}

void 
marker1_changed(GtkAdjustment *adj, SGfitDialog *dialog)
{
  gdouble *x;
  gint n;
  gchar text[100];

  if(!dialog->dataset) return;
 
  x = gtk_plot_data_get_x(dialog->dataset, &n);

  n = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->marker1_spin));
  g_snprintf(text, 100, "%g", x[n]); 
  gtk_entry_set_text(GTK_ENTRY(dialog->marker1_entry), text); 
  dialog->from_value = x[n];
}

void 
marker2_changed(GtkAdjustment *adj, SGfitDialog *dialog)
{
  gdouble *x;
  gint n;
  gchar text[100];

  if(!dialog->dataset) return;
 
  x = gtk_plot_data_get_x(dialog->dataset, &n);

  n = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->marker2_spin));
  g_snprintf(text, 100, "%g", x[n]); 
  gtk_entry_set_text(GTK_ENTRY(dialog->marker2_entry), text); 
  dialog->to_value = x[n];
}

void
select_dataset(GtkCList *clist, gint row, gint col, GdkEvent *event, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  GtkAdjustment *adj;
  gdouble *x;
  gint n;
  GList *markers;

  dialog->dataset = GTK_PLOT_DATA(gtk_clist_get_row_data(clist, row));
  x = gtk_plot_data_get_x(dialog->dataset, &n);
  adj = (GtkAdjustment *)gtk_adjustment_new(0, 0, n-1, 1, 8, 8); 
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(dialog->marker1_spin), adj);
  adj = (GtkAdjustment *)gtk_adjustment_new(0, 0, n-1, 1, 8, 8); 
  gtk_spin_button_set_adjustment(GTK_SPIN_BUTTON(dialog->marker2_spin), adj);

  gtk_signal_connect(GTK_OBJECT(GTK_SPIN_BUTTON(dialog->marker1_spin)->adjustment), "value_changed",
                     GTK_SIGNAL_FUNC(marker1_changed), dialog);
  gtk_signal_connect(GTK_OBJECT(GTK_SPIN_BUTTON(dialog->marker2_spin)->adjustment), "value_changed",
                     GTK_SIGNAL_FUNC(marker2_changed), dialog);

  n = 0;
  markers = dialog->dataset->markers;
  while(markers){
    GtkPlotMarker *marker = (GtkPlotMarker *)markers->data;
    if(n == 0){
      gtk_adjustment_set_value(GTK_SPIN_BUTTON(dialog->marker1_spin)->adjustment, marker->point);
    } else {
      gtk_adjustment_set_value(GTK_SPIN_BUTTON(dialog->marker2_spin)->adjustment, marker->point);
    }
    n++;
    markers = markers->next;
  }

}

static gint
pick_dataset (GtkWidget *widget, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  GtkWidget *table;
  GtkWidget *sw;
  GtkWidget *frame;
  GtkWidget *label;
  GList *list;

  if(dialog->xvalues) g_free(dialog->xvalues);
  dialog->xvalues = NULL;
  if(dialog->yvalues) g_free(dialog->yvalues);
  dialog->yvalues = NULL;

  /* Create widgets */

  create_wizard_window(dialog, _("Pick dataset"));

  table=gtk_table_new(1, 2, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_box_pack_start(GTK_BOX(dialog->main_box), table, TRUE, TRUE, 0);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize(sw, 120, 200);
  gtk_table_attach_defaults(GTK_TABLE(table), sw, 0, 1, 0, 1);

  dialog->clist = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), dialog->clist);

  list = GTK_PLOT_CANVAS_PLOT(dialog->layer)->plot->data_sets;
  while(list){
    GtkPlotData *dataset = GTK_PLOT_DATA(list->data);
    SGdataset *parent = SG_DATASET(dataset->link);
    gchar *text[1];
    gint nrows;
    GdkColormap *colormap;
    GdkPixmap *pixmap;
    GtkWidget *clist = dialog->clist;

    colormap = gdk_colormap_get_system();

    if(!dataset->is_function && !dataset->is_iterator){
      pixmap = SG_PLUGIN_STYLE(parent->constructor)->pixmap->pixmap;
      gdk_pixmap_ref(pixmap);
                                                                               
      nrows = GTK_CLIST(clist)->rows;
                                                                               
      text[0] = NULL;
      gtk_clist_append(GTK_CLIST(clist), text);
                                                                               
      text[0] = parent->name;
      gtk_clist_set_pixtext(GTK_CLIST(clist), nrows, 0, text[0], 5, pixmap, NULL);
      gtk_clist_set_row_data(GTK_CLIST(clist), nrows, dataset);
    }
                                                                               
    list = list->next;
  }

  frame = gtk_frame_new( _("Constraints") );
  gtk_table_attach_defaults(GTK_TABLE(table), frame, 1, 2, 0, 1);

  table=gtk_table_new(5, 3, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);

  /* From */
  label = gtk_label_new( _("From point") );
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  dialog->marker1_spin = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(0, 0, 0, 1, 0, 0), 0, 0);
  gtk_entry_set_editable(GTK_ENTRY(dialog->marker1_spin), TRUE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->marker1_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->marker1_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->marker1_spin, 1, 2, 0, 1);
  
  dialog->marker1_entry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(dialog->marker1_entry), TRUE);
  sg_entry_set_numeric(GTK_ENTRY(dialog->marker1_entry), 10);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->marker1_entry, 2, 3, 0, 1);
  
  /* To */
  label = gtk_label_new( _("To point") );
  gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);
  dialog->marker2_spin = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(0, 0, 0, 1, 0, 0), 0, 0);
  gtk_entry_set_editable(GTK_ENTRY(dialog->marker2_spin), TRUE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->marker2_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->marker2_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->marker2_spin, 1, 2, 1, 2);
  
  dialog->marker2_entry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(dialog->marker2_entry), TRUE);
  sg_entry_set_numeric(GTK_ENTRY(dialog->marker2_entry), 10);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->marker2_entry, 2, 3, 1, 2);
 
  dialog->marker_check = gtk_check_item_new_with_label(_("Use constraints"));
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->marker_check, 0, 3, 2, 3);

  /* initialize */

  /* connect signals */
  gtk_widget_set_sensitive(dialog->back_button, FALSE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);

  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked",
                     GTK_SIGNAL_FUNC(pick_function), dialog);
  gtk_signal_connect(GTK_OBJECT(dialog->clist), "select_row",
                     GTK_SIGNAL_FUNC(select_dataset), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, GTK_OBJECT(dialog->plot));

  return FALSE;
}

static void
action(double *p, double *hx, int m, int n, void *data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  SGpluginFunction *plugin = dialog->function;
  GtkPlotData *dataset = dialog->dataset;
  gint i;
  gboolean error;
  gdouble *x;

  for(i = 0; i < dialog->npoints; i++){
    hx[i] = sg_plugin_function_action(plugin, p, dialog->xvalues[i], &error, NULL); 
  }
}

static gboolean
iteration(GtkWidget *button, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  SGpluginFunction *plugin = dialog->function;
  GtkPlotData *dataset = dialog->dataset;
  gdouble *y;
  gint i, npoints;
  gdouble *work;
  gdouble info[LM_INFO_SZ];

  npoints = dialog->npoints;
  y = dialog->yvalues;

  work = g_new0(gdouble, LM_DER_WORKSZ(plugin->nparam, npoints));
 
  for(i = 0; i < plugin->nparam; i++){
    const gchar *text = gtk_entry_get_text(GTK_ENTRY(dialog->val_entry[i]));
    dialog->pval[i] = atof(text);
  }
  
/*  dlevmar_dif(action, dialog->pval, y, plugin->nparam, npoints, 1, NULL, info, work, dialog);
*/
  dlevmar_dif(action, dialog->pval, y, plugin->nparam, npoints, 1, NULL, info, NULL, dialog);
  g_free(work);

  for(i = 0; i < plugin->nparam; i++){
    gchar text[20];
    gboolean change = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->var_check[i])); 
    g_snprintf(text, 20, "%g", dialog->pval[i]);
    if(change) gtk_entry_set_text(GTK_ENTRY(dialog->val_entry[i]), text);
  }

  return FALSE;
}

static gboolean
iterations(GtkWidget *button, gpointer data)
{
  gint i;
  for(i = 0; i < 10; i++) iteration(button, data);
  return FALSE;
}

static gint
start_fit (GtkWidget *widget, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;
  GtkWidget *table;
  SGpluginFunction *plugin = SG_PLUGIN_FUNCTION(dialog->function);
  GtkWidget *label;
  GtkWidget *button;
  gint i = 0, j;

  /* Create widgets */

  if(!dialog->function) return FALSE;
  create_wizard_window(dialog, _("Start fit"));

  table=gtk_table_new(10, 4, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_box_pack_start(GTK_BOX(dialog->main_box), table, TRUE, TRUE, 0);

  label = gtk_label_new(_("Param"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 0, 1);
  label = gtk_label_new(_("Vary"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
  label = gtk_label_new(_("Value"));
  gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 4, 1, 2);
  for(j = 0; j < plugin->nparam; j++){
    i = j+2;
    gchar val[100];

    label = gtk_label_new(plugin->param[j]);
    gtk_misc_set_alignment(GTK_MISC(label), 0.5, 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, i, i+1);

    dialog->var_check[j] = gtk_check_item_new();
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->var_check[j]), TRUE);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->var_check[j], 1, 2, i, i+1);

    dialog->val_entry[j] = gtk_entry_new();
    gtk_entry_set_editable(GTK_ENTRY(dialog->val_entry[j]), TRUE);
    sg_entry_set_numeric(GTK_ENTRY(dialog->val_entry[j]), 12);
    g_snprintf(val, 100, "%g", plugin->par_value[j]);
    dialog->pval[j] = plugin->par_value[j];
    gtk_entry_set_text(GTK_ENTRY(dialog->val_entry[j]), val);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->val_entry[j], 2, 3, i, i+1);

  }
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 4, i+1, i+2);

  button = gtk_button_new_with_label(_("Iterate"));
  gtk_table_attach_defaults(GTK_TABLE(table), button, 0, 1, i+2, i+3);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(iteration), dialog);

  button = gtk_button_new_with_label(_("10 Iter"));
  gtk_table_attach_defaults(GTK_TABLE(table), button, 1, 2, i+2, i+3);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
                     GTK_SIGNAL_FUNC(iterations), dialog);

  /* initialize */

  /* connect signals */
  gtk_widget_set_sensitive(dialog->back_button, TRUE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);
  gtk_signal_connect(GTK_OBJECT(dialog->back_button), "clicked",
                     GTK_SIGNAL_FUNC(pick_function), dialog);
  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked",
                     GTK_SIGNAL_FUNC(plot_dataset), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, GTK_OBJECT(dialog->plot));

  return FALSE;
}

static gint
plot_dataset (GtkWidget *widget, gpointer data)
{
  SGfitDialog *dialog = (SGfitDialog *)data;

  /* Create widgets */

  create_wizard_window(dialog, _("Plot curve"));
  dialog->plot_button = gtk_check_item_new_with_label(_("Plot result")); 
  gtk_box_pack_start(GTK_BOX(dialog->main_box), dialog->plot_button, TRUE, TRUE, 0);
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->plot_button), TRUE);

  /* connect signals */
  gtk_widget_set_sensitive(dialog->back_button, TRUE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);
  gtk_signal_connect(GTK_OBJECT(dialog->back_button), "clicked",
                     GTK_SIGNAL_FUNC(start_fit), dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked",
                     GTK_SIGNAL_FUNC(create_dataset), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, GTK_OBJECT(dialog->plot));

  return FALSE;
}

void
select_function(GtkCList *clist, gint row, gint col, GdkEvent *event, gpointer data)
{
  SGpluginFunction *plugin;
  gchar *text = NULL;
  gint i;
  SGfitDialog *dialog = (SGfitDialog *)data;

  dialog->function = plugin = SG_PLUGIN_FUNCTION(gtk_clist_get_row_data(clist, row));

  gtk_editable_delete_text(GTK_EDITABLE(dialog->fentry), 0, -1);
  gtk_text_insert(GTK_TEXT(dialog->fentry), NULL, NULL, NULL, plugin->function, -1);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->nspin), plugin->nparam);

  text = g_strdup(plugin->param[0]); 
  for(i = 1; i < plugin->nparam; i++){
    gchar *new_text = NULL;
    new_text = g_strconcat(text, ", ", plugin->param[i], NULL);
    g_free(text);
    text = new_text;
  }

  gtk_entry_set_text(GTK_ENTRY(dialog->pentry), text);
  g_free(text);
  
  gtk_entry_set_text(GTK_ENTRY(dialog->iventry), plugin->ivar);
}

static gint
pick_function (GtkWidget *widget, gpointer data)
{
  GtkWidget *table;
  GtkWidget *label;
  GtkWidget *sw;
  SGfitDialog *dialog = (SGfitDialog *)data;
  GList *list;
  gdouble *x = NULL;
  gdouble *y = NULL;
  gint n;

  if(!dialog->dataset) return FALSE;

  if(dialog->marker_check) dialog->apply_constraints = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->marker_check));
  if(dialog->xvalues) g_free(dialog->xvalues);
  if(dialog->yvalues) g_free(dialog->yvalues);

  x = gtk_plot_data_get_x(dialog->dataset, &n);
  y = gtk_plot_data_get_y(dialog->dataset, &n);
  dialog->xvalues = g_new0(gdouble, n);
  dialog->yvalues = g_new0(gdouble, n);

  if(dialog->apply_constraints){
    gint i;
    dialog->npoints = 0;
    for(i = 0; i < n; i++){
      if(x[i] >= dialog->from_value && x[i] <= dialog->to_value){
        dialog->xvalues[dialog->npoints] = x[i];
        dialog->yvalues[dialog->npoints] = y[i];
        dialog->npoints++;
      }
    }
  } else {
    gint i;
    dialog->xvalues = g_new0(gdouble, n);
    dialog->yvalues = g_new0(gdouble, n);
    dialog->npoints = n;
    for(i = 0; i < n; i++){
      dialog->xvalues[i] = x[i];
      dialog->yvalues[i] = y[i];
    }
  }

  /* Create widgets */
  create_wizard_window(dialog, _("Select fitting function"));

  table=gtk_table_new(7, 4, FALSE);  
  gtk_container_set_border_width(GTK_CONTAINER(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 10);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);

  gtk_box_pack_start(GTK_BOX(dialog->main_box), table, TRUE, TRUE, 0);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_widget_set_usize(GTK_WIDGET(sw), 120, 100);
  gtk_table_attach_defaults(GTK_TABLE(table), sw, 0, 1, 0, 6);

  dialog->clist = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), dialog->clist);
  gtk_table_attach_defaults(GTK_TABLE(table), gtk_vseparator_new(), 1, 2, 0, 6);

  label = gtk_label_new(_("Expression:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 0, 1);
  dialog->fentry = gtk_text_new(NULL, NULL);
  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(sw), dialog->fentry);
  gtk_table_attach_defaults(GTK_TABLE(table), sw, 3, 4, 0, 1);

  label = gtk_label_new(_("Number of Param.:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 1, 2);
  dialog->nspin = gtk_spin_button_new((GtkAdjustment *)gtk_adjustment_new(0, 0, 8, 1, 8, 0), 0, 0);
  gtk_entry_set_editable(GTK_ENTRY(dialog->nspin), FALSE);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->nspin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->nspin), 0);
  gtk_widget_set_state(GTK_WIDGET(dialog->nspin), GTK_STATE_INSENSITIVE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->nspin, 3, 4, 1, 2);

  label = gtk_label_new(_("Parameters:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
  dialog->pentry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(dialog->pentry), FALSE);
  gtk_widget_set_state(GTK_WIDGET(dialog->pentry), GTK_STATE_INSENSITIVE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->pentry, 3, 4, 2, 3);

  label = gtk_label_new(_("Independent var.:"));
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 3, 4);
  dialog->iventry = gtk_entry_new();
  gtk_entry_set_editable(GTK_ENTRY(dialog->iventry), FALSE);
  gtk_widget_set_state(GTK_WIDGET(dialog->iventry), GTK_STATE_INSENSITIVE);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->iventry, 3, 4, 3, 4);

  /* initialize */

  list = sg_plugins();
  while(list){
    SGplugin *plugin = SG_PLUGIN(list->data);
    if(GTK_IS_SG_PLUGIN_FUNCTION(plugin)){
      gchar *text[1];
      gint nrows = GTK_CLIST(dialog->clist)->rows;

      text[0] = plugin->name;
      gtk_clist_append(GTK_CLIST(dialog->clist), text);
      gtk_clist_set_row_data(GTK_CLIST(dialog->clist), nrows, plugin);
    }
    list = list->next;
  }

  /* connect signals */
  gtk_widget_set_sensitive(dialog->back_button, TRUE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);
  gtk_signal_connect(GTK_OBJECT(dialog->clist), "select_row",
                     GTK_SIGNAL_FUNC(select_function), dialog);
  gtk_signal_connect(GTK_OBJECT(dialog->back_button), "clicked",
                     GTK_SIGNAL_FUNC(pick_dataset), dialog);
  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked",
                     GTK_SIGNAL_FUNC(start_fit), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, GTK_OBJECT(dialog->plot));

  return FALSE;
}

void
sg_fit_dialog(SGplot *plot)
{
  SGfitDialog *dialog;

  dialog = g_new0(SGfitDialog, 1);
  dialog->plot = plot;
  dialog->layer = dialog->plot->active_layer;
  dialog->xvalues = dialog->yvalues = NULL;
  dialog->npoints = 0;
  dialog->apply_constraints = FALSE;

  dialog->dialog = NULL;
  dialog->dataset = NULL;

  pick_dataset(NULL, dialog);
}


