/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_dataset_dialog.h"
#include "sg_dataset.h"
#include "sg_layer.h"
#include "sg_dialog.h"
#include "sg_plugin_style.h"
#include "sg_plugin_iterator.h"
#include "sg_application.h"

typedef struct
{
  SGdataset *dataset;
  SGpluginLayer *layer;
  SGpluginIterator *iterator;
  SGpluginStyle *style;
  SGapplication *app;
 
  GtkWidget *dialog;
  GtkWidget *main_box;
  GtkWidget *cancel_button;
  GtkWidget *next_button;
  GtkWidget *back_button;
  GtkWidget *iterators_list;
  GtkWidget *constructors_list;

} SGdatasetDialog;

static gint pick_dataset(GtkWidget *widget, gpointer data);
static gint pick_iterator(GtkWidget *widget, gpointer data);

static gint
cancel_clicked(GtkWidget *button, gpointer data)
{
  SGdatasetDialog *dialog = (SGdatasetDialog *)data;
  dialog->dataset = NULL;
  return TRUE;
}

static void 
create_wizard_window(SGdatasetDialog *dialog, const gchar *title)
{
  if(dialog->dialog && GTK_IS_WIDGET(dialog->dialog)) 
    gtk_widget_destroy(dialog->dialog);

  dialog->dialog = sg_dialog_new(title, GTK_ORIENTATION_VERTICAL, SG_BUTTON_BACK|SG_BUTTON_NEXT|SG_BUTTON_CANCEL, GTK_BUTTONBOX_END); 
  gtk_window_position (GTK_WINDOW (dialog->dialog), GTK_WIN_POS_CENTER);
  dialog->main_box = SG_DIALOG(dialog->dialog)->box;
  dialog->next_button = SG_DIALOG(dialog->dialog)->button_next;
  dialog->back_button = SG_DIALOG(dialog->dialog)->button_back;
  dialog->cancel_button = SG_DIALOG(dialog->dialog)->button_cancel;
}

static gint
create_dataset(GtkWidget *widget, gpointer data)
{
  GList *selection = NULL;
  GtkIconListItem *item = NULL;
  SGdatasetDialog *dialog = (SGdatasetDialog *)data;

  selection = GTK_ICON_LIST(dialog->constructors_list)->selection;
  if(!selection) return TRUE;

  item = (GtkIconListItem *)selection->data;
  dialog->style = (SGpluginStyle *)item->link;

  if(dialog->dialog && GTK_IS_WIDGET(dialog->dialog)) gtk_widget_destroy(dialog->dialog);

  dialog->dataset = sg_plugin_iterator_new_dialog(dialog->iterator, dialog->style, dialog->app);

  return FALSE;
}

static gint
pick_dataset(GtkWidget *widget, gpointer data)
{
  GList *selection = NULL;
  gint row;
  GtkWidget *sw;
  GList *list = NULL;
  SGdatasetDialog *dialog = (SGdatasetDialog *)data;

  selection = GTK_CLIST(dialog->iterators_list)->selection;
  if(!selection) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  dialog->iterator = (SGpluginIterator *)gtk_clist_get_row_data(GTK_CLIST(dialog->iterators_list), row);

  if(dialog->iterator->style){
    SGpluginStyle *c = sg_plugin_style_get(dialog->iterator->style);
    if(c) {
      dialog->style = c;
      dialog->dataset = sg_plugin_iterator_new_dialog(dialog->iterator, dialog->style, dialog->app);
      if(dialog->dialog && GTK_IS_WIDGET(dialog->dialog)) gtk_widget_destroy(dialog->dialog);
      return FALSE;
    } else {
      return TRUE;
    }
  }

  create_wizard_window(dialog, _("Pick dataset style"));

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 300, 200);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(dialog->main_box),sw,TRUE,TRUE,0);
  dialog->constructors_list = gtk_icon_list_new(20,GTK_ICON_LIST_TEXT_RIGHT);
  gtk_icon_list_set_text_space(GTK_ICON_LIST(dialog->constructors_list), 140);
  gtk_icon_list_set_editable(GTK_ICON_LIST(dialog->constructors_list), FALSE);
  GTK_ICON_LIST(dialog->constructors_list)->compare_func = NULL;
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), dialog->constructors_list);

  list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_STYLE(list->data)){
      gint nindep_req, ndep_req, nindep, ndep;
      SGpluginStyle *c = SG_PLUGIN_STYLE(list->data);
      SGpluginIterator *iter = dialog->iterator;
      gboolean enough_dep, enough_indep;
      
      nindep = sg_plugin_style_num_independent_dimensions(c);
      ndep = sg_plugin_style_num_dependent_dimensions(c);
      nindep_req = sg_plugin_style_min_independent_dimensions(c);
      ndep_req = sg_plugin_style_min_dependent_dimensions(c);

      enough_indep = iter->max_independent_dims == -1 || iter->max_independent_dims >= nindep_req; 
      enough_indep = enough_indep && (iter->min_independent_dims == 0 || iter->min_independent_dims == nindep_req); 

      enough_dep = iter->max_dependent_dims == -1 || iter->max_dependent_dims >= ndep_req; 
      enough_dep = enough_dep && (iter->min_dependent_dims == 0 || iter->min_dependent_dims == ndep_req); 

      if(enough_dep && enough_indep && strcmp(c->layer,SG_PLUGIN(dialog->layer)->name) == 0){
        GdkPixmap *pixmap; 
        GdkBitmap *mask; 
        pixmap = c->pixmap->pixmap;
        mask = c->pixmap->mask;
        gtk_icon_list_add_from_pixmap(GTK_ICON_LIST(dialog->constructors_list), 
                                      pixmap, mask, SG_PLUGIN(c)->description, c);
      }
    }
    list = list->next;
  }

  gtk_widget_set_sensitive(dialog->back_button, TRUE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);
  gtk_signal_connect(GTK_OBJECT(dialog->back_button), "clicked", 
                     GTK_SIGNAL_FUNC(pick_iterator), dialog);

  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked", 
                     GTK_SIGNAL_FUNC(create_dataset), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, NULL);

  return FALSE;
}

static gint
pick_iterator(GtkWidget *widget, gpointer data)
{
  GtkWidget *sw;
  gint nrows = 0;
  GList *list;
  SGdatasetDialog *dialog = (SGdatasetDialog *)data;
  guint miniter = 0, maxiter = 0, nlayer = 0;
  gint i;
 
  create_wizard_window(dialog, _("Create dataset from:"));

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_container_set_border_width(GTK_CONTAINER(sw), 5);
  gtk_widget_set_usize(sw, 200, 200);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(dialog->main_box),sw,TRUE,TRUE,0);
  dialog->iterators_list = gtk_clist_new(1);
  gtk_clist_set_row_height(GTK_CLIST(dialog->iterators_list), 20);
  gtk_clist_set_column_min_width(GTK_CLIST(dialog->iterators_list), 0, 400);
  gtk_container_add(GTK_CONTAINER(sw), dialog->iterators_list);

  for(i = 0; i < 4; i++)
    if(dialog->layer->dim_mask & (1 << i)) nlayer++;

  list = sg_plugins();
  while(list){
    if(GTK_IS_SG_PLUGIN_ITERATOR(list->data)){
      SGpluginIterator *iter = SG_PLUGIN_ITERATOR(list->data);
      gchar *text[1];

      miniter = iter->min_independent_dims + iter->min_dependent_dims;
      maxiter = iter->max_independent_dims + iter->max_dependent_dims;

      if(miniter >= nlayer || iter->max_independent_dims == -1 || iter->max_dependent_dims == -1){
        GdkPixmap *pixmap = NULL;
        GdkBitmap *mask = NULL;
        text[0] = SG_PLUGIN(iter)->description;

	pixmap = iter->pixmap->pixmap;
	mask = iter->pixmap->mask;
        if(pixmap) gdk_pixmap_ref(pixmap);
        if(mask) gdk_bitmap_ref(mask);
 
        gtk_clist_append(GTK_CLIST(dialog->iterators_list), text);
        gtk_clist_set_pixtext(GTK_CLIST(dialog->iterators_list), nrows, 0, text[0], 5, pixmap, mask);
        gtk_clist_set_row_data(GTK_CLIST(dialog->iterators_list), nrows, iter);
        nrows++;
      }
    }
    list = list->next;
  } 

  gtk_widget_set_sensitive(dialog->back_button, FALSE);
  gtk_widget_set_sensitive(dialog->next_button, TRUE);

  gtk_signal_connect(GTK_OBJECT(dialog->next_button), "clicked", 
                     GTK_SIGNAL_FUNC(pick_dataset), dialog);
  gtk_widget_show_all(dialog->main_box);
  sg_dialog_run(dialog->dialog, NULL);

  return FALSE;
}  
 
SGdataset *
sg_dataset_dialog(SGapplication *app, SGpluginLayer *layer)
{
  SGdatasetDialog *dialog;
  SGdataset *dataset;

  dialog = g_new0(SGdatasetDialog, 1);
  dialog->layer = layer;

  dialog->app = app;

  dialog->dialog = NULL;
  dialog->dataset = NULL;
  dialog->iterator = NULL;
  dialog->style = NULL;
  dialog->iterators_list = NULL;

  pick_iterator(NULL, dialog);

  dataset = dialog->dataset;
  g_free(dialog);

  return dataset;
}

