/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg_convert_dialog.h"
#include "sg_matrix_convert.h"
#include "sg.h"
#include "sg_stock.h"
#include "sg_dialog.h"
#include "sg_misc_dialogs.h"

typedef struct
{
  GtkWidget *column_entry[3];
  GtkWidget *left_button[3];
  GtkWidget *right_button[3];
  GtkWidget *entries_table;
  
  GtkWidget *main_table;
  GtkWidget *columns_list;
  GtkWidget *box;

  SGworksheet *worksheet;
  gboolean convert_mode;
} SGconvertDialog;

static GtkWidget *columns_dialog	(SGconvertDialog *dialog);
static void select_columns		(GtkWidget *widget, 
					 gpointer data);


static gint
get_column_from_name(SGconvertDialog *dialog, const gchar *name)
{
  SGworksheet *worksheet = dialog->worksheet;
  gint n = 0;

  for(n = 0; n <= GTK_SHEET(worksheet)->maxcol; n++){
    if(strcmp(name, GTK_SHEET(worksheet)->column[n].name) == 0) return n;
  }

  return -1;
}

static void 
ok_pressed(SGpropertyDialog *pdialog, gpointer data)
{
  SGconvertDialog *dialog = (SGconvertDialog *)data;
  SGworksheet *worksheet, *matrix;
  gint i, narg = 0;
  gint column[3];
  const gchar *column_name[3];

  for(i = 0; i < 3; i++){
    column[i] = -1;
    column_name[i] = gtk_entry_get_text(GTK_ENTRY(dialog->column_entry[i]));
    column[i] = get_column_from_name(dialog, column_name[i]);
    if(column[i] != -1) narg++;
  }

  if(narg < 3) return;

  worksheet = dialog->worksheet;

  if(dialog->convert_mode)
    matrix = sg_worksheet_to_matrix_xyz(worksheet, column[0], column[1], column[2]);
  else
    matrix = sg_worksheet_to_matrix_columns(worksheet, column[0], column[1], column[2]);

  if(!matrix)
      sg_message_dialog( _("Data does not conform with a quadrilateral mesh"), 0);
}

static void
update_columns(SGconvertDialog *dialog)
{
  gint i, nrows;
  SGworksheet *worksheet = dialog->worksheet;
  gchar *text;
  gchar name[100];

  nrows = GTK_CLIST(dialog->columns_list)->rows;
  for(i = 0; i <= nrows; i++){
    gtk_clist_remove(GTK_CLIST(dialog->columns_list), 0);
  }

  for(i = 0; i <= GTK_SHEET(worksheet)->maxcol; i++){
     if(!GTK_SHEET(worksheet)->column[i].name)
        sprintf(name,"%d",i);
     else
        sprintf(name,"%s",GTK_SHEET(worksheet)->column[i].name);
     text=g_strdup(name);
     gtk_clist_append(GTK_CLIST(dialog->columns_list), &text);
  };

}

void
sg_convert_dialog (SGworksheet *worksheet, gboolean mode)
{
  GtkWidget *ref_dialog;
  SGconvertDialog *dialog;
  GtkWindow *window;
  GtkWidget *frame;

  dialog = g_new0(SGconvertDialog, 1);
  dialog->convert_mode = mode;
  dialog->worksheet = worksheet;

  /* Create widgets */

  frame = sg_property_dialog_new();
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, TRUE);

  gtk_container_add(GTK_CONTAINER(frame), columns_dialog(dialog));

  SG_PROPERTY_DIALOG(frame)->ok = ok_pressed;
  SG_PROPERTY_DIALOG(frame)->apply = ok_pressed;
  ref_dialog = sg_dialog_new( _("SciGraphica: Convert to XYZ matrix"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_APPLY|SG_BUTTON_CLOSE, GTK_BUTTONBOX_SPREAD);
  window = GTK_WINDOW(ref_dialog);

  /* Show widgets */
  gtk_window_set_policy(window, FALSE, FALSE, FALSE);
  gtk_window_set_modal(window, TRUE);
  sg_dialog_add(ref_dialog, SG_PROPERTY_DIALOG(frame));
  gtk_widget_show_all (GTK_WIDGET(frame));
  sg_dialog_run(ref_dialog, GTK_OBJECT(worksheet));
}


static GtkWidget *
columns_dialog (SGconvertDialog *dialog)
{
  GtkWidget *frame, *vbox;
  GtkWidget *box, *sw;
  GtkWidget *action_area;

  vbox = gtk_vbox_new(FALSE, 5);

  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_START);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);
  gtk_box_pack_start (GTK_BOX (vbox), action_area, FALSE, FALSE, 0);

/*********************************************************************/

  frame = gtk_frame_new( _("Columns") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  gtk_box_pack_start (GTK_BOX (vbox), frame, FALSE, FALSE, 0);

  dialog->box = box = gtk_hbox_new(FALSE, 5);
  gtk_container_set_border_width(GTK_CONTAINER(box), 5);
  gtk_container_add (GTK_CONTAINER (frame), box);

  sw = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_set_usize(sw, 180, 160);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start (GTK_BOX (box), sw, FALSE, FALSE, 0);

  dialog->columns_list = gtk_clist_new(1);
  gtk_container_add(GTK_CONTAINER(sw), dialog->columns_list);

  select_columns(NULL, dialog);

/********************************************************************/
  return vbox;
}

static gint
set_column(GtkWidget *widget, gpointer data)
{
  SGconvertDialog *dialog = (SGconvertDialog *)data;
  GList *selection;
  const gchar *text;
  GtkWidget *name_entry = NULL;
  gint row;
  gint i;

  for(i = 0; i < 3; i++)
    if(dialog->right_button[i] == widget) name_entry = dialog->column_entry[i];

  selection = GTK_CLIST(dialog->columns_list)->selection;
  if(!selection) return TRUE;

  text = gtk_entry_get_text(GTK_ENTRY(name_entry));
  if(strlen(text) > 0) return TRUE;

  row = GPOINTER_TO_INT(selection->data);
  gtk_clist_get_text(GTK_CLIST(dialog->columns_list), row, 0, &text);

  gtk_entry_set_text(GTK_ENTRY(name_entry), text);
  gtk_clist_remove(GTK_CLIST(dialog->columns_list), row);
  return TRUE;
}

static gint
restore_column(GtkWidget *widget, gpointer data)
{
  SGconvertDialog *dialog = (SGconvertDialog *)data;
  const gchar *text;
  gint col;
  GtkWidget *name_entry = NULL;
  gint i;

  for(i = 0; i < 3; i++)
    if(dialog->left_button[i] == widget) name_entry = dialog->column_entry[i];

  text = gtk_entry_get_text(GTK_ENTRY(name_entry));
  if(strlen(text) == 0) return TRUE;

  col = get_column_from_name(dialog, text);
  gtk_clist_insert(GTK_CLIST(dialog->columns_list), col, &text);

  gtk_entry_set_text(GTK_ENTRY(name_entry), "");
  return TRUE;
}

static void
select_columns(GtkWidget *widget, gpointer data)
{
  SGconvertDialog *dialog = (SGconvertDialog *)data;
  GtkWidget *box;
  GtkWidget *column_label;
  GtkWidget *button_box;
  gint i, nrows = 0;
  gchar *label[3] = {"X:","Y:","Z:"};

  box = dialog->box;

  update_columns(dialog);

  if(dialog->entries_table && GTK_IS_WIDGET(dialog->entries_table))
     gtk_container_remove(GTK_CONTAINER(box), dialog->entries_table);

  dialog->entries_table = gtk_table_new(3, 3, FALSE);
  gtk_table_set_col_spacings(GTK_TABLE(dialog->entries_table), 5);

  for(i = 0; i < 3; i++){
      button_box = gtk_hbox_new(TRUE, 0);
      gtk_table_attach_defaults(GTK_TABLE (dialog->entries_table), button_box, 
                                0, 1, nrows, nrows + 1);

      dialog->left_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(dialog->left_button[i]), 
                        gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(dialog->left_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), dialog->left_button[i], FALSE, FALSE, 0);
 
      dialog->right_button[i] = gtk_button_new();
      gtk_container_add(GTK_CONTAINER(dialog->right_button[i]), 
                        gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_OUT));
      gtk_widget_set_usize(dialog->right_button[i], 20, 20);
      gtk_box_pack_start(GTK_BOX(button_box), dialog->right_button[i], FALSE, FALSE, 0);

      column_label = gtk_label_new(label[i]);
      gtk_misc_set_alignment(GTK_MISC(column_label), .5, .5);
      gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), column_label, 
                                1, 2, nrows, nrows + 1);    
      dialog->column_entry[i] = gtk_entry_new();
      gtk_entry_set_editable(GTK_ENTRY(dialog->column_entry[i]), FALSE);
      gtk_table_attach_defaults(GTK_TABLE(dialog->entries_table), dialog->column_entry[i], 
                                2, 3, nrows, nrows + 1);    
      gtk_entry_set_text(GTK_ENTRY(dialog->column_entry[i]), "");

      gtk_signal_connect(GTK_OBJECT(dialog->left_button[i]), "clicked",
                         (GtkSignalFunc) restore_column, dialog);
      gtk_signal_connect(GTK_OBJECT(dialog->right_button[i]), "clicked",
                         (GtkSignalFunc) set_column, dialog);

      nrows++;
  }

  gtk_box_pack_start(GTK_BOX(box), dialog->entries_table, FALSE, FALSE, 0);
  gtk_widget_show_all(dialog->entries_table);
}


