/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_column_dialog.h"
#include "sg_stock.h"

typedef struct
{
  GtkWidget *name_entry;
  GtkWidget *apply_check;
  GtkWidget *type_combo;
  GtkWidget *format_combo;
  GtkWidget *internal_combo;
  GtkWidget *precision_spin;

  gint column;
  GtkSheet *sheet;

} SGcolumnDialog;

static void
update_worksheet(SGpropertyDialog *pdialog, gpointer data)
{
  SGcolumnDialog *dialog = (SGcolumnDialog *)data;
  GtkWidget *child;
  const gchar *name;
  gint precision;
  SGcolumnformat format;
  SGcolumntype type;
  SGcolumninternal internal;
  gboolean all_columns;
  gint width;

  if(!GTK_IS_SG_MATRIX(dialog->sheet)){
    name = gtk_entry_get_text(GTK_ENTRY(dialog->name_entry));

    gtk_sheet_set_column_title(dialog->sheet, dialog->column, name);
    gtk_sheet_column_button_add_label(dialog->sheet, dialog->column, name);

/*
    width = gdk_string_width(GTK_WIDGET(dialog->sheet)->style->font, name) + 12;
    if(width > dialog->sheet->column[dialog->column].width)
      gtk_sheet_set_column_width(dialog->sheet, dialog->column, width); 
  
*/
  }

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(dialog->type_combo)->list)->selection->data;
  type = (SGcolumntype)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->type_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(dialog->format_combo)->list)->selection->data;
  format = (SGcolumnformat)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(dialog->internal_combo)->list)->selection->data;
  internal = (SGcolumninternal)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->internal_combo)->list), child);

  precision = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->precision_spin));

  if(!GTK_IS_SG_MATRIX(dialog->sheet))
    all_columns = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->apply_check));
  else
    all_columns = TRUE;

  gtk_sheet_freeze(dialog->sheet);
  if(!GTK_IS_SG_MATRIX(dialog->sheet)){
    if(all_columns || GTK_IS_SG_MATRIX(dialog->sheet)){
      gint i;
      for(i = MAX(dialog->column, 0); i <= dialog->sheet->maxcol; i++)
        sg_worksheet_column_set_format(SG_WORKSHEET(dialog->sheet), i, 
                                       type,
                                       format,
                                       internal,
                                       precision);
    }
    sg_worksheet_column_set_format(SG_WORKSHEET(dialog->sheet), dialog->column, 
                                   type,
                                   format,
                                   internal,
                                   precision);
  }else {
    gint i;
    for(i = 0; i <= dialog->sheet->maxcol; i++)
      sg_worksheet_column_set_format(SG_WORKSHEET(dialog->sheet), i, 
                                     type,
                                     format,
                                     internal,
                                     precision);
    sg_matrix_set_format(SG_MATRIX(dialog->sheet),  
                                   type,
                                   format,
                                   internal,
                                   precision);
  }

  gtk_sheet_thaw(dialog->sheet);
}


static void
type_changed(GtkWidget *widget, gpointer data)
{
  SGcolumnDialog *dialog = (SGcolumnDialog *)data;
  GtkWidget *child;
  gint type;
  gint internal;
 
  child = (GtkWidget*)GTK_LIST(GTK_COMBO(dialog->type_combo)->list)->selection->data;
  type = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->type_combo)->list), child);

  child = (GtkWidget*)GTK_LIST(GTK_COMBO(dialog->internal_combo)->list)->selection->data;
  internal = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->internal_combo)->list), child);

  if(type == SG_TYPE_NUMBER){
      gtk_widget_set_sensitive(dialog->internal_combo, TRUE);
      gtk_widget_set_sensitive(dialog->format_combo, TRUE);
      if(internal == SG_INTERNAL_INTEGER)
        gtk_widget_set_sensitive(dialog->precision_spin, FALSE);
      else
        gtk_widget_set_sensitive(dialog->precision_spin, TRUE);
  } else {
      gtk_widget_set_sensitive(dialog->precision_spin, FALSE);
      gtk_widget_set_sensitive(dialog->format_combo, FALSE);
      gtk_widget_set_sensitive(dialog->internal_combo, FALSE);
  }

}


void
sg_column_dialog (SGworksheet *worksheet, gint col)
{
  GtkWidget *ref_dialog;
  SGcolumnDialog *dialog;
  GtkWindow *window;
  GtkWidget *frame;
  GtkWidget *label;
  GtkWidget *table;
  GtkSheetColumn *column = NULL;
  SGcolumn *wcolumn = NULL;
  GtkAdjustment *adj;
  gchar text[100];
  gchar *types[] = { N_("Number"),
                     N_("Text"),
                     N_("Date"),
                     N_("Time"),
                     NULL
		   };
  gchar *formats[] = { N_("Decimal (1000)"),
                       N_("Scientific (1.E3)"),
                       NULL
                     };
  gchar *internals[] = { N_("Integer"),
                         N_("Double"),
                         NULL
                       };

  dialog = g_new0(SGcolumnDialog, 1);
  dialog->column = col;
  dialog->sheet = GTK_SHEET(worksheet);
  if(!GTK_IS_SG_MATRIX(worksheet)){
    column = &(GTK_SHEET(worksheet)->column[col]);
    wcolumn = &worksheet->column[dialog->column];
  } else {
    wcolumn = &(SG_MATRIX(worksheet)->format);
  }


  /* Create widgets */
  frame = sg_property_dialog_new();
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, TRUE);

  table = gtk_table_new(7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  if(!GTK_IS_SG_MATRIX(worksheet)){
    dialog->name_entry = gtk_entry_new();   

    label = gtk_label_new( _("Column name") ),
    gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
    gtk_table_attach_defaults(GTK_TABLE(table), 
                              label,
                              0, 1, 0, 1);
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->name_entry, 1, 2, 0, 1);


    gtk_table_attach_defaults(GTK_TABLE(table), gtk_hseparator_new(), 0, 2, 1, 2);
  }

  dialog->type_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->type_combo), types);

  label = gtk_label_new( _("Data type") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->type_combo, 1, 2, 2, 3);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->type_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->type_combo)->list), 
                       wcolumn->type); 
  if(GTK_IS_SG_MATRIX(worksheet))
    gtk_widget_set_sensitive(dialog->type_combo, FALSE);

  dialog->format_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->format_combo), formats);

  label = gtk_label_new( _("Format") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->format_combo, 1, 2, 3, 4);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->format_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->format_combo)->list), 
                       wcolumn->format); 


  label = gtk_label_new( _("Precision") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 4, 5);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 16., 1., 1., 0.);
  dialog->precision_spin = gtk_spin_button_new(adj, 0, 0);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->precision_spin), TRUE);
  gtk_spin_button_set_digits(GTK_SPIN_BUTTON(dialog->precision_spin), 0);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->precision_spin, 1, 2, 4, 5);


  dialog->internal_combo = gtk_combo_new();
  sg_combo_set_items(GTK_COMBO(dialog->internal_combo), internals);

  label = gtk_label_new( _("Representation") ),
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            label,
                            0, 1, 5, 6);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->internal_combo, 1, 2, 5, 6);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->internal_combo)->entry), FALSE);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->internal_combo)->list), 
                       wcolumn->internal); 

  if(!GTK_IS_SG_MATRIX(worksheet)){
    dialog->apply_check = gtk_check_item_new_with_label( _("Apply to all columns to the right") );
    gtk_table_attach_defaults(GTK_TABLE(table), dialog->apply_check, 0, 2, 6, 7);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dialog->apply_check), FALSE);
  }

  SG_PROPERTY_DIALOG(frame)->ok = update_worksheet;
  SG_PROPERTY_DIALOG(frame)->apply = update_worksheet;
  ref_dialog = sg_dialog_new(NULL, GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_SPREAD);
  window = GTK_WINDOW(ref_dialog); 

  if(!GTK_IS_SG_MATRIX(worksheet))
    gtk_window_set_title (window, _("SciGraphica: Column properties") );
  else
    gtk_window_set_title (window, _("SciGraphica: Matrix properties") );

  /* connect signals */
  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->type_combo)->entry), 
                      "changed",
                      GTK_SIGNAL_FUNC (type_changed),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->internal_combo)->entry), 
                      "changed",
                      GTK_SIGNAL_FUNC (type_changed),
                      dialog);

  /* Show widgets */

  if(!GTK_IS_SG_MATRIX(worksheet)){
    if(column->name)
      sprintf(text,"%s",column->name);
    else
      sprintf(text,"%d",col);

    gtk_entry_set_text(GTK_ENTRY(dialog->name_entry), text);
    gtk_widget_grab_focus(dialog->name_entry);
  }

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->precision_spin), 
                            worksheet->column[dialog->column].precision);

  type_changed(NULL, dialog);

  sg_dialog_add(ref_dialog, SG_PROPERTY_DIALOG(frame));
  gtk_window_set_policy(window, FALSE, FALSE, FALSE);
  gtk_window_set_modal(window, TRUE);
  gtk_widget_show_all (GTK_WIDGET(frame));
  sg_dialog_run(ref_dialog, GTK_OBJECT(worksheet));
}
