/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include "sg.h"
#include "sg_property_dialog.h"

typedef struct 
{
  GtkWidget *width_entry;
  GtkWidget *symbol_style_combo;
  GtkWidget *symbol_width_spin;
  GtkWidget *border_color_combo;
  GtkWidget *symbol_color_combo;
  GtkPlotData *dataset;
} SGbarsDialog;

static void sg_bars_dialog_apply            (SGpropertyDialog *d, gpointer data);
static void sg_bars_dialog_ok               (SGpropertyDialog *d, gpointer data);
static void sg_bars_dialog_init_gui         (GtkWidget *d);

static gchar *symbol_styles[] = 
{
  N_("Empty"),
  N_("Filled"),
  N_("Opaque"),
  NULL
};

GtkWidget *
sg_bars_dialog_new(GtkPlotData *dataset)
{
  GtkWidget *widget;
  SGbarsDialog *dialog;

  dialog = g_new0(SGbarsDialog, 1);
  widget = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(widget), dialog, TRUE);
  SG_PROPERTY_DIALOG(widget)->ok = sg_bars_dialog_ok;
  SG_PROPERTY_DIALOG(widget)->apply = sg_bars_dialog_apply;
  dialog->dataset = dataset;
  sg_bars_dialog_init_gui(widget);

  return widget;
}

static void
sg_bars_dialog_ok(SGpropertyDialog *d, gpointer data)
{
  sg_bars_dialog_apply(d, data);
}

static void     
init_dialog          (SGbarsDialog *dialog)
{
  GtkPlotData *data = dialog->dataset;
  gchar text[100];
  GdkColor symbol_color, border_color;
  GtkPlotSymbolStyle symbol_style;
  GtkPlotSymbolType symbol_type;
  gfloat symbol_width;
  gint symbol_size;

  gtk_plot_data_get_symbol(data,
                           &symbol_type, &symbol_style,
                           &symbol_size, &symbol_width,
                           &symbol_color, &border_color);


  snprintf(text, 100, "%f", GTK_PLOT_BAR(data)->width*2);
  gtk_entry_set_text(GTK_ENTRY(dialog->width_entry), text);

  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->symbol_style_combo)->list), symbol_style);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->symbol_width_spin), symbol_width);

  sg_color_combo_init(GTK_COLOR_COMBO(dialog->symbol_color_combo), symbol_color);
  sg_color_combo_init(GTK_COLOR_COMBO(dialog->border_color_combo), border_color);
}

static void     
sg_bars_dialog_init_gui           (GtkWidget *frame)
{
  GtkWidget *label;
  GtkWidget *table;
  GtkRequisition req;
  GtkAdjustment *adj;
  SGbarsDialog *dialog = (SGbarsDialog *)SG_PROPERTY_DIALOG(frame)->data;

  gtk_frame_set_label(GTK_FRAME(frame), _("Bars") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  table = gtk_table_new(7, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);
  gtk_table_set_row_spacings(GTK_TABLE(table), 10);
  gtk_container_add (GTK_CONTAINER (frame), table);

  label = gtk_label_new( _("Width:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,0,1);

  dialog->width_entry = gtk_entry_new();
  sg_entry_set_numeric(GTK_ENTRY(dialog->width_entry), 15);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->width_entry,1,2,0,1);
  gtk_widget_size_request(dialog->width_entry, &req);
  req.width /= 2;

  label = gtk_label_new( _("Style:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,1,2);

  dialog->symbol_style_combo = gtk_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->symbol_style_combo,1,2,1,2);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->symbol_style_combo)->entry), FALSE);

  sg_combo_set_items(GTK_COMBO(dialog->symbol_style_combo), symbol_styles);

  label = gtk_label_new( _("Thickness:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,2,3);

  adj = (GtkAdjustment *)gtk_adjustment_new(0., 0., 6., 0.5, 1., 0.);
  dialog->symbol_width_spin = gtk_spin_button_new(adj, 0.5, 1);
  gtk_spin_button_set_numeric(GTK_SPIN_BUTTON(dialog->symbol_width_spin), TRUE);
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->symbol_width_spin,1,2,2,3);

  label = gtk_label_new( _("Border:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,3,4);
  dialog->border_color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->border_color_combo,1,2,3,4);

  label = gtk_label_new( _("Color:") );
  gtk_misc_set_alignment(GTK_MISC(label), 1., 0.5);
  gtk_table_attach_defaults(GTK_TABLE(table),label,0,1,4,5);

  dialog->symbol_color_combo = gtk_color_combo_new();
  gtk_table_attach_defaults(GTK_TABLE(table),dialog->symbol_color_combo,1,2,4,5);

  gtk_widget_set_usize(dialog->width_entry, req.width, req.height);
  gtk_widget_set_usize(dialog->symbol_width_spin, req.width, req.height);
  gtk_widget_set_usize(dialog->symbol_style_combo, req.width, req.height);

  gtk_widget_show_all(frame);

  init_dialog(dialog);
}

static void     
sg_bars_dialog_apply     (SGpropertyDialog *d, gpointer _data)
{
  SGbarsDialog *dialog = (SGbarsDialog *)d->data;
  GtkPlotData *data = dialog->dataset;
  GdkColor border;
  const gchar *text;
  GtkWidget *child;
  GdkColor symbol_color;
  gint symbol_style, symbol_type = 0;
  gfloat symbol_width; 

  text = gtk_entry_get_text(GTK_ENTRY(dialog->width_entry));
  GTK_PLOT_BAR(data)->width = atof(text)/2;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->symbol_style_combo)->list)->selection->data;
  symbol_style = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->symbol_style_combo)->list), child);

  symbol_width = gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(dialog->symbol_width_spin));

  symbol_color = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->symbol_color_combo));

  border = gtk_color_combo_get_selection(GTK_COLOR_COMBO(dialog->border_color_combo));
  gtk_plot_data_set_symbol(data,
                           (GtkPlotSymbolType) symbol_type,
                           (GtkPlotSymbolStyle) symbol_style,
                           0,
                           symbol_width,
                           &symbol_color,
                           &border);

}

