/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include <sg_layer_control.h>
#include <sg_application.h>
#include <sg_property_dialog.h>
#include <sg_style_dialog.h>
#include <sg_axis_dialog.h>
#include <sg_break_dialog.h>
#include <sg_dataset_dialog.h>
#include <sg_layer_dataset_dialog.h>
#include <sg_frame_dialog.h>
#include <sg_grids_dialog.h>
#include <sg_labels_dialog.h>
#include <sg_layer_dialog.h>
#include <sg_legends_dialog.h>
#include <sg_planes_dialog.h>
#include <sg_title_dialog.h>
#include <sg_ticks_dialog.h>
#include <sg_toolbox_icons.h>
#include "pixmaps/2d_small.xpm"
#include "pixmaps/as.xpm"

#define LABEL_LEN 1000
#define ZOOM 2.0000000000000

static void     toolbox_autoscale                       (GtkButton *button,
                                                         gpointer data);
static void     toolbox_up                              (GtkButton *button,
                                                         gpointer data);
static void     toolbox_down                            (GtkButton *button,
                                                         gpointer data);
static void     toolbox_left                            (GtkButton *button,
                                                         gpointer data);
static void     toolbox_right                           (GtkButton *button,
                                                         gpointer data);
static void     toolbox_zoomin                          (GtkButton *button,
                                                         gpointer data);
static void     toolbox_zoomout                         (GtkButton *button,
                                                         gpointer data);

void
layer_2d_init(SGplugin *_plugin)
{
  SGpluginLayer *plugin = SG_PLUGIN_LAYER(_plugin);
  GdkColormap *colormap;
  GdkPixmap *pixmap = NULL;
  GdkBitmap *mask = NULL;

  colormap = gdk_colormap_get_system();

  plugin->dim_mask = GTK_PLOT_DATA_X|GTK_PLOT_DATA_Y;

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL, _2d_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}
 
GtkPlot *
layer_2d_construct()
{
  GtkPlot *plot;

  plot = GTK_PLOT(gtk_plot_new(NULL));

  return plot;
}

static void
new_dataset_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_layer_dataset_dialog_new(SG_LAYER(p_layer));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                 dialog->ok_button,
                                 dialog->apply_button,
                                 dialog->cancel_button);
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(dialog->main_frame),
                    GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Add/remove datasets") );
}

static void
new_axis_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_axis_dialog(SG_LAYER(p_layer),
                                  dialog->ok_button,
                                  dialog->apply_button,
                                  dialog->cancel_button);
                                                                                 
  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->dialog);
  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit axes properties") );
}
                                                                                 
static void
new_break_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_break_dialog(SG_LAYER(p_layer),
                                   dialog->ok_button,
                                   dialog->apply_button,
                                   dialog->cancel_button);

  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->dialog);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit break properties") );
}

static void
new_title_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_title_dialog(SG_LAYER(p_layer),
                                   dialog->ok_button,
                                   dialog->apply_button,
                                   dialog->cancel_button);

  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->dialog);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit titles properties") );
}
                                                                                 
static void
new_grids_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_grids_dialog(SG_LAYER(p_layer),
                                   dialog->ok_button,
                                   dialog->apply_button,
                                   dialog->cancel_button);
                                                                                 
  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->main_frame);
                                                                                 
  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit grids properties") );
}
                                                                                 
static void
new_legends_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_legends_dialog_new(SG_LAYER(p_layer));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                 dialog->ok_button,
                                 dialog->apply_button,
                                 dialog->cancel_button);

  gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(dialog->main_frame),
                    GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit legends properties") );
}

static void
new_labels_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_labels_dialog(SG_LAYER(p_layer),
                                    dialog->ok_button,
                                    dialog->apply_button,
                                    dialog->cancel_button);

  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit labels properties") );
}

static void
new_ticks_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_ticks_dialog(SG_LAYER(p_layer),
                                    dialog->ok_button,
                                    dialog->apply_button,
                                    dialog->cancel_button);

  gtk_container_add(GTK_CONTAINER(dialog->main_frame), dialog->dialog);
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit tick properties") );
}

static void
new_frame_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_frame_dialog_new(SG_LAYER(p_layer));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                 dialog->ok_button,
                                 dialog->apply_button,
                                 dialog->cancel_button);

  gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(dialog->main_frame),
                    GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit frame properties") );
}
                                                                                 
static void
new_style_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  if(!GTK_PLOT_CANVAS_PLOT(p_layer)->plot->data_sets){
    dialog->dialog = gtk_label_new( _("This layer contains no datasets") );
  } else {
    dialog->dialog = sg_style_dialog_new(SG_LAYER(p_layer));
    sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                   dialog->ok_button,
                                   dialog->apply_button,
                                   dialog->cancel_button);
    gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  }

  gtk_container_add(GTK_CONTAINER(dialog->main_frame), GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit dataset properties") );
}

static void
new_layer_dialog(SGlayerControl *dialog, gpointer p_layer)
{
  dialog->dialog = sg_layer_dialog_new(SG_LAYER(p_layer));
  sg_property_dialog_set_buttons(SG_PROPERTY_DIALOG(dialog->dialog),
                                 dialog->ok_button,
                                 dialog->apply_button,
                                 dialog->cancel_button);

  gtk_frame_set_shadow_type (GTK_FRAME (dialog->dialog), GTK_SHADOW_OUT);
  gtk_container_add(GTK_CONTAINER(dialog->main_frame),
                    GTK_WIDGET(dialog->dialog));
  gtk_widget_show_all(dialog->main_frame);

  gtk_label_set_text(GTK_LABEL(dialog->label), _("Edit layer properties") );
}


void 	
layer_2d_dialog(SGpluginLayer *plugin, GtkWidget *_dialog, GtkCTreeNode *root_item, GtkObject *olayer, const gchar *title)
{
  GtkCTreeNode *item, *subitem, *subsubitem;
  SGlayerControl *dialog = SG_LAYER_CONTROL(_dialog);
  GtkCTree *tree = GTK_CTREE(dialog->tree);
  SGlayer *layer = SG_LAYER(olayer);
  SGnodeData *node;
  gchar path[1000];
  gchar *subtitle[0];
  gchar *text[20] = { N_("Legends"),
                      N_("Axes"),
                      N_("Titles"),
                      N_("Ticks labels"),
                      N_("Frame"),
                      N_("Grids"),
                      N_("Datasets"),
                      N_("Properties"),
                      N_("Planes"),
                      N_("Breaks"),
                      N_("Tick properties"),
                      NULL };

  subtitle[0] = g_strdup(title);

  item = gtk_ctree_insert_node(GTK_CTREE(tree), root_item, NULL,
                               &subtitle[0],
                               4,
                               plugin->pixmap->pixmap, 
			       plugin->pixmap->mask,
                               plugin->pixmap->pixmap, 
			       plugin->pixmap->mask,
                               FALSE, FALSE);

  gtk_ctree_node_set_row_data(GTK_CTREE(tree), item,
                              node = sg_node_data_new(2, layer,
                              new_layer_dialog, title, item));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:frame") , title);
  subitem = gtk_ctree_insert_node(GTK_CTREE(tree), item, NULL,
                                  &text[4],
                                  4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subitem,
                              node = sg_node_data_new(2, layer,
                              new_frame_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:grids") , title);
  subitem = gtk_ctree_insert_node(GTK_CTREE(tree), item, NULL,
                                  &text[5],
                                  4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subitem,
                              node = sg_node_data_new(2, layer,
                              new_grids_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);
                                                                               
  g_snprintf(path, LABEL_LEN, _("%s:legends") , title);
  subitem = gtk_ctree_insert_node(GTK_CTREE(tree), item, NULL,
                                  &text[0],
                                  4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subitem,
                              node = sg_node_data_new(2, layer,
                              new_legends_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);
                                                                               
  g_snprintf(path, LABEL_LEN, _("%s:axis") , title);
  subitem = gtk_ctree_insert_node(GTK_CTREE(tree), item, NULL,
                                  &text[1],
                                  4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subitem,
                              node = sg_node_data_new(2, layer,
                              new_axis_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);
                                                                               
                                                                               
  g_snprintf(path, LABEL_LEN, _("%s:axis:title") , title);
  subsubitem = gtk_ctree_insert_node(GTK_CTREE(tree), subitem, NULL,
                                     &text[2],
                                     4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subsubitem,
                              node = sg_node_data_new(2, layer,
                              new_title_dialog, path, subsubitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:axis:labels") , title);
  subsubitem = gtk_ctree_insert_node(GTK_CTREE(tree), subitem, NULL,
                                     &text[3],
                                     4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subsubitem,
                              node = sg_node_data_new(2, layer,
                              new_labels_dialog, path, subsubitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:axis:ticks") , title);
  subsubitem = gtk_ctree_insert_node(GTK_CTREE(tree), subitem, NULL,
                                     &text[10],
                                     4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subsubitem,
                              node = sg_node_data_new(2, layer,
                              new_ticks_dialog, path, subsubitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:axis:breaks") , title);
  subsubitem = gtk_ctree_insert_node(GTK_CTREE(tree), subitem, NULL,
                                     &text[9],
                                     4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subsubitem,
                              node = sg_node_data_new(2, layer,
                              new_break_dialog, path, subsubitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:dataset") , title);
  subitem = gtk_ctree_insert_node(GTK_CTREE(tree), item, NULL,
                                  &text[6],
                                  4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subitem,
                              node = sg_node_data_new(2, layer,
                              new_dataset_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);

  g_snprintf(path, LABEL_LEN, _("%s:dataset:style") , title);
  subsubitem = gtk_ctree_insert_node(GTK_CTREE(tree), subitem, NULL,
                                     &text[7],
                                     4, NULL, NULL, NULL, NULL, FALSE, FALSE);
  gtk_ctree_node_set_row_data(GTK_CTREE(tree), subsubitem,
                              node = sg_node_data_new(2, layer,
                              new_style_dialog, path, subitem));
  dialog->nodes = g_list_append(dialog->nodes, node);
                                                                               
  g_free(subtitle[0]);
} 

GtkWidget *
layer_2d_toolbox(SGpluginLayer *plugin, GtkObject *layer)
{
  GtkWidget *table;
  GtkWidget *button;
  GdkColormap *colormap;
  GdkPixmap *tool;
  GdkBitmap *mask;
  GtkWidget *pixmap;
  GtkTooltips *tooltips;

  table = gtk_table_new(2, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 2);
  colormap = gdk_colormap_get_system();

  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    0, 1, 0, 1, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Autoscale"), _("Autoscale") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);

  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             as_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_autoscale), layer);

  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    0, 1, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Zoom out"), _("Zoom out") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             zoom_plus_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_zoomout), layer);
                                                                                
  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    1, 2, 1, 2, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Zoom in"), _("Zoom in") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             zoom_minus_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_zoomin), layer);
                                                                                
  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    0, 1, 2, 3, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Up"), _("Up") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             arrow_up_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_up), layer);

  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    1, 2, 2, 3, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Down"), _("Down") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             arrow_down_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_down), layer);
                                                                                
  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    0, 1, 3, 4, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Left"), _("Left") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             arrow_left_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_pixmap_unref(tool);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_left), layer);
                                                                                
  button = gtk_button_new();
  gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
  gtk_table_attach (GTK_TABLE(table),
                    button,
                    1, 2, 3, 4, GTK_SHRINK, GTK_SHRINK, 0, 0);
  tooltips = gtk_tooltips_new();
  gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips),
                       button, _("Right"), _("Right") );
  gtk_tooltips_enable(GTK_TOOLTIPS(tooltips));
  gtk_tooltips_set_delay(GTK_TOOLTIPS(tooltips), 0);
  tool=gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, &mask, NULL,
                                             arrow_right_xpm);
  pixmap = gtk_pixmap_new(tool, mask);
  gtk_container_add(GTK_CONTAINER(button),pixmap);
  gtk_widget_show(pixmap);
  gdk_bitmap_unref(mask);
  gtk_signal_connect(GTK_OBJECT(button),
                     "clicked",
                     GTK_SIGNAL_FUNC(toolbox_right), layer);

  gtk_widget_show_all(table);
  return table;
}

static void
toolbox_left(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  step = real_plot->bottom->ticks.step;
                                                                                
  if(real_plot->bottom->ticks.scale == GTK_PLOT_SCALE_LOG10)
    gtk_plot_set_xrange(real_plot,
                        real_plot->xmin / pow(real_plot->xmax/real_plot->xmin,0.2),
                        real_plot->xmax / pow(real_plot->xmax/real_plot->xmin,0.2));
                                                                                
  else
    gtk_plot_set_xrange(real_plot,
                        real_plot->xmin - step,
                        real_plot->xmax - step);
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
                                                                                
}
                                                                                
static void
toolbox_right(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  step = real_plot->bottom->ticks.step;
                                                                                
  if(real_plot->bottom->ticks.scale == GTK_PLOT_SCALE_LOG10)
    gtk_plot_set_xrange(real_plot,
                        real_plot->xmin * pow(real_plot->xmax/real_plot->xmin,0.2),
                        real_plot->xmax * pow(real_plot->xmax/real_plot->xmin,0.2));
                                                                                
  else
    gtk_plot_set_xrange(real_plot,
                        real_plot->xmin + step,
                        real_plot->xmax + step);
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}
                                                                                
static void
toolbox_up(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  step = real_plot->left->ticks.step;
                                                                                
  if(real_plot->left->ticks.scale == GTK_PLOT_SCALE_LOG10)
    gtk_plot_set_yrange(real_plot,
                        real_plot->ymin * pow(real_plot->ymax/real_plot->ymin,0.2),
                        real_plot->ymax * pow(real_plot->ymax/real_plot->ymin,0.2));
                                                                                
  else
    gtk_plot_set_yrange(real_plot,
                        real_plot->ymin + step,
                        real_plot->ymax + step);
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}
                                                                                
static void
toolbox_down(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
  gdouble step;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  step = real_plot->left->ticks.step;
                                                                                
  if(real_plot->left->ticks.scale == GTK_PLOT_SCALE_LOG10)
    gtk_plot_set_yrange(real_plot,
                        real_plot->ymin / pow(real_plot->ymax/real_plot->ymin,0.2),
                        real_plot->ymax / pow(real_plot->ymax/real_plot->ymin,0.2));
                                                                                
  else
    gtk_plot_set_yrange(real_plot,
                        real_plot->ymin - step,
                        real_plot->ymax - step);
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}
                                                                                
static void
toolbox_autoscale(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  gtk_plot_autoscale(GTK_PLOT(real_plot));
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}
                                                                                
static void
toolbox_zoomin(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  if (real_plot->left->ticks.scale == GTK_PLOT_SCALE_LOG10) {
      gtk_plot_set_yrange(real_plot,
                      (pow(real_plot->ymin,(1+ZOOM)/2)*pow(real_plot->ymax,(1-ZOOM)/2)),
                      (pow(real_plot->ymin,(1-ZOOM)/2)*pow(real_plot->ymax,(1+ZOOM)/2)));
  } else {
      gtk_plot_set_yrange(real_plot,
                      ((real_plot->ymin)*(1+ZOOM)/2)+((real_plot->ymax)*(1-ZOOM)/2),
                      ((real_plot->ymin)*(1-ZOOM)/2)+((real_plot->ymax)*(1+ZOOM)/2));
      gtk_plot_set_ticks(real_plot, GTK_PLOT_AXIS_Y,
                   real_plot->left->ticks.step * ZOOM,
                   real_plot->left->ticks.nminor);
  }
  if (real_plot->bottom->ticks.scale == GTK_PLOT_SCALE_LOG10) {
      gtk_plot_set_xrange(real_plot,
                      (pow(real_plot->xmin,(1+ZOOM)/2)*pow(real_plot->xmax,(1-ZOOM)/2)),
                      (pow(real_plot->xmin,(1-ZOOM)/2)*pow(real_plot->xmax,(1+ZOOM)/2)));
  } else {
      gtk_plot_set_xrange(real_plot,
                     ((real_plot->xmin)*(1+ZOOM)/2)+((real_plot->xmax)*(1-ZOOM)/2),
                     ((real_plot->xmin)*(1-ZOOM)/2)+((real_plot->xmax)*(1+ZOOM)/2));
      gtk_plot_set_ticks(real_plot, GTK_PLOT_AXIS_X,
                   real_plot->bottom->ticks.step * ZOOM,
                   real_plot->bottom->ticks.nminor);
  }
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}
                                                                                
static void
toolbox_zoomout(GtkButton *button, gpointer data)
{
  SGplot *plot;
  GtkPlot *real_plot;
                                                                                
  real_plot = GTK_PLOT_CANVAS_PLOT(data)->plot;
  plot = SG_PLOT(GTK_PLOT_CANVAS_CHILD(data)->parent);
                                                                                
  if (real_plot->left->ticks.scale == GTK_PLOT_SCALE_LOG10) {
      gtk_plot_set_yrange(real_plot,
                      (pow(real_plot->ymin,(1+1/ZOOM)/2)*pow(real_plot->ymax,(1-1/ZOOM)/2)),
                      (pow(real_plot->ymin,(1-1/ZOOM)/2)*pow(real_plot->ymax,(1+1/ZOOM)/2)));
  } else {
      gtk_plot_set_yrange(real_plot,
                      ((real_plot->ymin)*(1+1/ZOOM)/2)+((real_plot->ymax)*(1-1/ZOOM)/2),
                      ((real_plot->ymin)*(1-1/ZOOM)/2)+((real_plot->ymax)*(1+1/ZOOM)/2));
  }
  if (real_plot->bottom->ticks.scale == GTK_PLOT_SCALE_LOG10) {
      gtk_plot_set_xrange(real_plot,
                      (pow(real_plot->xmin,(1+1/ZOOM)/2)*pow(real_plot->xmax,(1-1/ZOOM)/2)),
                      (pow(real_plot->xmin,(1-1/ZOOM)/2)*pow(real_plot->xmax,(1+1/ZOOM)/2)));
  } else {
      gtk_plot_set_xrange(real_plot,
                      ((real_plot->xmin)*(1+1/ZOOM)/2)+((real_plot->xmax)*(1-1/ZOOM)/2),
                      ((real_plot->xmin)*(1-1/ZOOM)/2)+((real_plot->xmax)*(1+1/ZOOM)/2));
  }
                                                                                
  gtk_plot_set_ticks(real_plot, GTK_PLOT_AXIS_X,
                     real_plot->bottom->ticks.step / ZOOM,
                     real_plot->bottom->ticks.nminor);
                                                                                
  gtk_plot_set_ticks(real_plot, GTK_PLOT_AXIS_Y,
                     real_plot->left->ticks.step / ZOOM,
                     real_plot->left->ticks.nminor);
                                                                                
  gtk_plot_canvas_paint(GTK_PLOT_CANVAS(plot));
  gtk_plot_canvas_refresh(GTK_PLOT_CANVAS(plot));
}

