/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include "sg_dataset_matrix.h"

#define P_(string) string

#define NAME_LEN 100

static void sg_dataset_matrix_class_init	(SGdatasetMatrixClass *klass);
static void sg_dataset_matrix_init		(SGdatasetMatrix *dataset);
static void sg_dataset_matrix_finalize		(GObject *object);
static void sg_dataset_matrix_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_matrix_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_matrix_refresh		(SGdataset *dataset);
static void update_arrays                       (SGdataset *dataset);
static gboolean sg_dataset_matrix_connected	(SGdataset *dataset, 
						 gpointer data);

enum {
  ARG_0,
  ARG_MATRIX,
};

static SGdatasetClass *parent_class = NULL;

GType
sg_dataset_matrix_get_type (void)
{
  static GType dataset_type = 0;
                                                                                
  if (!dataset_type)
    {
      static const GTypeInfo dataset_info =
      {
        sizeof (SGdatasetMatrixClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_dataset_matrix_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGdatasetMatrix),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_dataset_matrix_init,
        NULL,
      };
                                                                                
      dataset_type = g_type_register_static (GTK_TYPE_SG_DATASET, "SGdatasetMatrix", &dataset_info, 0);
    }
                                                                                
  return dataset_type;
}
                          
static void
sg_dataset_matrix_class_init (SGdatasetMatrixClass *klass)
{
  GObjectClass *object_class;
  SGdatasetClass *dataset_class;

  parent_class = (SGdatasetClass *) g_type_class_peek_parent(klass);
  object_class = (GObjectClass *) klass;
  dataset_class = (SGdatasetClass *) klass;

  object_class->finalize = sg_dataset_matrix_finalize;
  object_class->set_property = sg_dataset_matrix_set_property;
  object_class->get_property = sg_dataset_matrix_get_property;
                                                                                
  g_object_class_install_property (object_class,
                           ARG_MATRIX,
  g_param_spec_pointer ("matrix",
                           P_("Matrix"),
                           P_("Matrix"),
                           G_PARAM_READABLE|G_PARAM_WRITABLE));

  dataset_class->refresh = sg_dataset_matrix_refresh;
  dataset_class->connected = sg_dataset_matrix_connected;
}

SGdataset *
sg_dataset_matrix_new ()
{
  SGdataset *dataset;

  dataset = SG_DATASET(g_object_new(sg_dataset_matrix_get_type(),NULL));

  return dataset;
}

static void
sg_dataset_matrix_get_property (GObject      *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetMatrix *dataset;
                                                                                
  dataset = SG_DATASET_MATRIX (object);
                                                                                
  switch(prop_id){
    case ARG_MATRIX:
      g_value_set_pointer(value,dataset->matrix);
      break;
  }
}
                                                                                
static void
sg_dataset_matrix_set_property (GObject      *object,
                         guint            prop_id,
                         const GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetMatrix *dataset;
                                                                                
  dataset = SG_DATASET_MATRIX (object);
                                                                                
  switch(prop_id){
    case ARG_MATRIX:
      dataset->matrix = SG_MATRIX(g_value_get_pointer(value));
      break;
  }
}

static void
sg_dataset_matrix_init(SGdatasetMatrix *dataset)
{
  dataset->matrix = NULL;
  sg_dataset_set_description(SG_DATASET(dataset), _("Using matrix data"));
}

void
sg_dataset_matrix_set  	  (SGdatasetMatrix *dataset,
			   SGmatrix *matrix)
{
  dataset->matrix = matrix;
  sg_dataset_refresh(SG_DATASET(dataset));
}

static void
sg_dataset_matrix_refresh(SGdataset *dataset)
{
  SGworksheet *worksheet;
  GtkSheet *sheet;
  gint numrows = 0, numcols = 0;
  SGdatasetMatrix *real_data;
  GList *list;

  real_data = SG_DATASET_MATRIX(dataset);

  if(!real_data->matrix) return;

  update_arrays(dataset);

  worksheet = SG_WORKSHEET(real_data->matrix);
  sheet = GTK_SHEET(worksheet);

  numrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet));
  numcols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet));

  list = dataset->children;
  while(list){
    GtkPlotData *child = GTK_PLOT_DATA(list->data);

    child->num_points = numrows * numcols;
    child->num_points = numrows * numcols;

    if(GTK_IS_PLOT_SURFACE(child)){
      GTK_PLOT_SURFACE(child)->nx = numcols;
      GTK_PLOT_SURFACE(child)->ny = numrows;
      GTK_PLOT_SURFACE(child)->recalc_dt = TRUE;
    }

    gtk_plot_data_update(child); 
    list = list->next;
  }
}

static void
sg_dataset_matrix_finalize(GObject *object)
{
  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
update_arrays(SGdataset *dataset)
{
  SGdatasetMatrix *real_data;
  SGmatrix *matrix;
  SGworksheet *worksheet;
  gint row = 0, col = 0;
  gint nrows = 0, ncols = 0;
  GList *list = NULL;

  real_data = SG_DATASET_MATRIX(dataset);
  matrix = real_data->matrix;
  worksheet = SG_WORKSHEET(matrix);

  nrows = gtk_sheet_get_rows_count(GTK_SHEET(worksheet));
  ncols = gtk_sheet_get_columns_count(GTK_SHEET(worksheet));

  list = dataset->arrays->arrays;
  while(list){
    GtkPlotArray *array = GTK_PLOT_ARRAY(list->data);
    if(strcmp(array->name, "x") == 0){
      gtk_plot_array_free(array);
      if(matrix->nx > 0 && matrix->x_values){
        gdouble *new_array = g_new0(gdouble, matrix->nx*matrix->ny);
        gint i = 0;
        for(row = 0; row < matrix->ny; row++){
          for(col = 0; col < matrix->nx; col++){
            new_array[i++] = matrix->x_values[col];
          }
        }
        array->data.data_double = new_array;
        array->size = matrix->nx*matrix->ny;
        array->own_data = TRUE;
      } else {
        gdouble *new_array = g_new0(gdouble, nrows*ncols);
        gdouble xstep = (matrix->xmax - matrix->xmin) / ncols;
        gint i = 0;
        for(row = 0; row < nrows; row++){
          for(col = 0; col < ncols; col++){
            new_array[i++] = col * xstep;
          }
        }
        array->data.data_double = new_array;
        array->size = ncols*nrows;
        array->own_data = TRUE;
      }
    } else if(strcmp(array->name, "y") == 0){
      gtk_plot_array_free(array);
      if(matrix->nx > 0 && matrix->x_values){
        gdouble *new_array = g_new0(gdouble, matrix->nx*matrix->ny);
        gint i = 0;
        for(row = 0; row < matrix->ny; row++){
          for(col = 0; col < matrix->nx; col++){
            new_array[i++] = matrix->y_values[row];
          }
        }
        array->data.data_double = new_array;
        array->size = matrix->nx*matrix->ny;
        array->own_data = TRUE;
      } else {
        gdouble *new_array = g_new0(gdouble, nrows*ncols);
        gdouble ystep = (matrix->ymax - matrix->ymin) / nrows;
        gint i = 0;
        for(row = 0; row < nrows; row++){
          for(col = 0; col < ncols; col++){
            new_array[i++] = row * ystep;
          }
        }
        array->data.data_double = new_array;
        array->size = ncols*nrows;
        array->own_data = TRUE;
      }
    } else if(array->required){
      gdouble *new_array = g_new0(gdouble, nrows*ncols);
      gint i = 0;
      gboolean error;
      gtk_plot_array_free(array);
      for(row = 0; row < nrows; row++){
        for(col = 0; col < ncols; col++){
          gdouble val = sg_worksheet_cell_get_double(worksheet, row, col, &error);
          if(error) val = 0;
          new_array[i++] = val;
        }
      }
      array->data.data_double = new_array;
      array->size = nrows*ncols;
      array->own_data = TRUE;
    }
    list = list->next;
  }
}

static gboolean
sg_dataset_matrix_connected(SGdataset *dataset, gpointer object)
{
  SGdatasetMatrix *data = SG_DATASET_MATRIX(dataset);
  
  if(data->matrix && data->matrix == object) return TRUE;

  return FALSE;
}

