/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <math.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <sg.h>
#include <sg_python_expr.h>
#include "sg_dataset_function.h"
#include "../pixmaps/g_function.xpm"

#define P_(string) string

static void sg_dataset_function_class_init	(SGdatasetFunctionClass *klass);
static void sg_dataset_function_init		(SGdatasetFunction *dataset);
static void sg_dataset_function_finalize		(GObject *object);
static void sg_dataset_function_set_property    (GObject *object,
                                                 guint prop_id,
                                                 const GValue *value,
                                                 GParamSpec *pspec);
static void sg_dataset_function_get_property    (GObject *object,
                                                 guint prop_id,
                                                 GValue *value,
                                                 GParamSpec *pspec);
                                                                                

static void sg_dataset_function_set_iterator	(SGdataset *dataset);
static GtkPlotData* sg_dataset_function_new_child	(SGdataset *dataset);

enum {
  ARG_0,
  ARG_EXP,
};

static gdouble function                                 (GtkPlot *plot,
                                                         GtkPlotData *data,
                                                         gdouble x,
                                                         gboolean *error);


static SGdatasetClass *parent_class = NULL;

GType
sg_dataset_function_get_type (void)
{
  static GType dataset_type = 0;
                                                                                
  if (!dataset_type)
    {
      static const GTypeInfo dataset_info =
      {
        sizeof (SGdatasetFunctionClass),
        NULL,           /* base_init */
        NULL,           /* base_finalize */
        (GClassInitFunc) sg_dataset_function_class_init,
        NULL,           /* class_finalize */
        NULL,           /* class_data */
        sizeof (SGdatasetFunction),
        0,              /* n_preallocs */
        (GInstanceInitFunc) sg_dataset_function_init,
        NULL,
      };
                                                                                
      dataset_type = g_type_register_static (GTK_TYPE_SG_DATASET, "SGdatasetFunction", &dataset_info, 0);
    }
                                                                                
  return dataset_type;
}

static void
sg_dataset_function_class_init (SGdatasetFunctionClass *klass)
{
  GObjectClass *object_class;
  SGdatasetClass *dataset_class;

  parent_class = (SGdatasetClass *) g_type_class_peek_parent(klass);
  object_class = (GObjectClass *) klass;
  dataset_class = (SGdatasetClass *) klass;

  object_class->finalize = sg_dataset_function_finalize;
  object_class->set_property = sg_dataset_function_set_property;
  object_class->get_property = sg_dataset_function_get_property;

  g_object_class_install_property (object_class,
                           ARG_EXP,
  g_param_spec_string ("exp",
                           P_("Expression"),
                           P_("Expression"),
                           NULL,
                           G_PARAM_READABLE|G_PARAM_WRITABLE));

  dataset_class->set_iterator = sg_dataset_function_set_iterator;
  dataset_class->new_child = sg_dataset_function_new_child;
}

SGdataset *
sg_dataset_function_new ()
{
  SGdataset *dataset;

  dataset = SG_DATASET(g_object_new(sg_dataset_function_get_type(),NULL));

  return dataset;
}

static void
sg_dataset_function_get_property (GObject      *object,
                         guint            prop_id,
                         GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetFunction *dataset;
                                                                                
  dataset = SG_DATASET_FUNCTION (object);
                                                                                
  switch(prop_id){
    case ARG_EXP:
      g_value_set_string(value,dataset->exp);
      break;
  }
}

static void
sg_dataset_function_set_property (GObject      *object,
                         guint            prop_id,
                         const GValue          *value,
                         GParamSpec      *pspec)
{
  SGdatasetFunction *dataset;
                                                                                
  dataset = SG_DATASET_FUNCTION (object);
                                                                                
  switch(prop_id){
    case ARG_EXP:
      if(dataset->exp) g_free(dataset->exp); 
      dataset->exp = g_strdup(g_value_get_string(value));
      break;
  }
}

static void
sg_dataset_function_init(SGdatasetFunction *dataset)
{
  dataset->exp = NULL;
  sg_dataset_set_description(SG_DATASET(dataset), _("Function f(x)"));
}

static GtkPlotData * 
sg_dataset_function_new_child	(SGdataset *dataset)
{
  GdkPixmap *pixmap = NULL;
  GdkColormap *colormap = gdk_colormap_get_system();
  GtkPlotData *new_data;

  new_data = SG_DATASET_CLASS(parent_class)->new_child(dataset);

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, colormap, NULL, NULL, g_function_xpm);

/*
  gtk_widget_finalize(GTK_WIDGET(dataset->constructor->pixmap));
  dataset->constructor->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, NULL));
  gdk_pixmap_unref(pixmap);
*/

  new_data->line.line_style = GTK_PLOT_LINE_SOLID;
  new_data->symbol.symbol_type = GTK_PLOT_SYMBOL_NONE;
  new_data->is_function = TRUE;
  new_data->function = function;
  return new_data;
}


static void
sg_dataset_function_set_iterator(SGdataset *dataset)
{
  GList *list;

  list = dataset->children;
  while(list){
    GtkPlotData *real_data = GTK_PLOT_DATA(list->data);
    real_data->is_iterator = FALSE;
    real_data->iterator = NULL;
    real_data->is_function = TRUE;
    real_data->function = function;
    list = list->next;
  }
}

void
sg_dataset_function_set   (SGdatasetFunction *dataset,
			   const gchar *exp)
{
  if(dataset->exp) g_free(dataset->exp);
  dataset->exp = NULL;

  dataset->exp = g_strdup(exp);
}

static void
sg_dataset_function_finalize(GObject *object)
{
  if(SG_DATASET_FUNCTION(object)->exp) g_free(SG_DATASET_FUNCTION(object)->exp);
  SG_DATASET_FUNCTION(object)->exp = NULL;

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static gdouble
function (GtkPlot *plot, GtkPlotData *data, gdouble x, gboolean *error)
{
  SGdatasetFunction *dataset;
  gdouble val;

  dataset = SG_DATASET_FUNCTION(gtk_plot_data_get_link(data));
  if (sg_eval_func(dataset->exp, x, &val)) *error=FALSE;
  else *error = TRUE;

  return val;
}

