/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include "sg_worksheet_file.h"
#include <scigraphica/sg.h>
#include "pixmaps/sg_small.xpm"

static void write_header_tex			(SGworksheetfile *file); 
static void write_footer_tex			(SGworksheetfile *file); 
static void write_col_title_tex			(SGworksheetfile *file, 
                                                 gint col);
static void write_cell_tex			(SGworksheetfile *file,  
                                                 gint row, gint col);
static void new_row_tex				(SGworksheetfile *file,
                                                 gint row); 
static void new_column_tex			(SGworksheetfile *file, 
                                                 gint col); 

static gboolean tex_export_default 		(SGpluginFile *plugin,
                              			 const gchar *filename,
						 FILE *opened,
                              			 GObject **object, 
						 gpointer data);

void
SGworksheet_tex_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".tex");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

void
SGmatrix_tex_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".tex");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
SGworksheet_tex_export 	     (SGpluginFile *plugin,
                              const gchar *filename,
			      FILE *opened,
                              GObject **object, gpointer data)
{  
  return tex_export_default(plugin, filename, opened, object, data);
}  

gboolean
SGmatrix_tex_export 	     (SGpluginFile *plugin,
                              const gchar *filename,
			      FILE *opened,
                              GObject **object, gpointer data)
{  
  return tex_export_default(plugin, filename, opened, object, data);
}  

static gboolean
tex_export_default 	     (SGpluginFile *plugin,
                              const gchar *filename,
			      FILE *opened,
                              GObject **object, gpointer data)
{  
  SGworksheetfile *file;
  SGworksheet *worksheet;
  GtkSheet *sheet;

  worksheet = SG_WORKSHEET(*object);
  sheet = GTK_SHEET(worksheet);
  file = sg_worksheet_file_new(worksheet, filename);

  if(opened)
    file->stream = opened;
  else
    file->stream = sg_file_open(file->filename, "w");

  if (!file->stream){
       g_warning( _("ERROR: Cannot write to file: %s"), filename);
       g_free(file);
       return FALSE;
  }

  file->write_header = write_header_tex; 
  file->write_footer = write_footer_tex; 
  file->write_col_title = write_col_title_tex; 
  file->write_cell = write_cell_tex; 
  file->new_row = new_row_tex; 
  file->new_column = new_column_tex; 

  file->titlecolor = g_strdup("\"BBBBBB\"");
  file->cellcolor = g_strdup("\"FFFFFF\"");

  sg_worksheet_file_export(file, NULL);

  if(!opened) sg_file_close(file->stream);

  g_free(file->filename);
  g_free(file->titlecolor);
  g_free(file->cellcolor);
  g_free(file);

  return TRUE;
}

static void
write_header_tex(SGworksheetfile *file)
{
  gint col;

/*  
  sg_file_printf(file->stream,"\\documentstyle{article}");
  sg_file_printf(file->stream,"\\begin{document}");
*/
  sg_file_printf(file->stream,"\\begin{tabular}{");

  for(col = file->range.col0; col <= file->range.coli; col++)
    sg_file_printf(file->stream,"|c");

  sg_file_printf(file->stream,"|}\n\n");

  sg_file_printf(file->stream, "\\hline\n"); 
}

static void
write_footer_tex(SGworksheetfile *file)
{
  sg_file_printf(file->stream, "\\\\\n"); 
  sg_file_printf(file->stream, "\\hline\n"); 

  sg_file_printf(file->stream, "\n\\end{tabular}");
/*
  sg_file_printf(file->stream,"\\end{document}");
*/
}

static void
new_row_tex(SGworksheetfile *file, gint row)
{

  sg_file_printf(file->stream, "\\\\\n"); 
  sg_file_printf(file->stream, "\\hline\n"); 

}

static void
new_column_tex(SGworksheetfile *file, gint col)
{

  sg_file_printf(file->stream, " & "); 

}

static void
write_col_title_tex(SGworksheetfile *file, gint col) 
{
  gchar *title;
 
  title = GTK_SHEET(file->worksheet)->column[col].name;

  sg_file_printf(file->stream, "%s",title);
}

static void
write_cell_tex(SGworksheetfile *file, gint row, gint col)
{
  gchar *text;
  
  text = sg_worksheet_cell_get_text(file->worksheet, row, col);

  if(text && strlen(text) > 0)
           sg_file_printf(file->stream, "%s",text);
}

