/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_worksheet_file.h"
#include "pixmaps/sg_small.xpm"

static void write_header_html			(SGworksheetfile *file); 
static void write_footer_html			(SGworksheetfile *file); 
static void write_col_title_html		(SGworksheetfile *file, 
						 gint col); 
static void write_row_title_html		(SGworksheetfile *file, 
						 gint row); 
static void write_cell_html			(SGworksheetfile *file,  
						 gint row, gint col); 
static void new_row_html			(SGworksheetfile *file, 
                                                 gint row); 
static void justify_html			(SGworksheetfile *file, 
						 GtkJustification just);

static gboolean html_export_default 		(SGpluginFile *plugin,
                             			 const gchar *filename,
						 FILE *opened,
                              			 GObject **object,
						 gpointer data);

void
SGworksheet_html_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".htm*");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

void
SGmatrix_html_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".htm*");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
SGworksheet_html_export (SGpluginFile *plugin,
                         const gchar *filename,
		         FILE *opened,
                         GObject **object,
		         gpointer data)
{  
  return html_export_default(plugin, filename, opened, object, data);
}

gboolean
SGmatrix_html_export (SGpluginFile *plugin,
                      const gchar *filename,
		      FILE *opened,
                      GObject **object,
		      gpointer data)
{  
  return html_export_default(plugin, filename, opened, object, data);
}

static gboolean
html_export_default (SGpluginFile *plugin,
                     const gchar *filename,
		     FILE *opened,
                     GObject **object,
		     gpointer data)
{  
  SGworksheetfile *file;
  GtkSheet *sheet;
  SGworksheet *worksheet;

  worksheet = SG_WORKSHEET(*object);
  sheet = GTK_SHEET(worksheet);
  file = sg_worksheet_file_new(worksheet, filename);

  if(opened)
    file->stream = opened;
  else
    file->stream = sg_file_open(file->filename, "w");

  if (!file->stream){
       g_warning( _("ERROR: Cannot write to file: %s"), filename);
       g_free(file);
       return FALSE;
  }

  file->write_header = write_header_html; 
  file->write_footer = write_footer_html; 
  file->write_col_title = write_col_title_html; 
  file->write_row_title = write_row_title_html; 
  file->write_cell = write_cell_html; 
  file->new_row = new_row_html; 

  file->titlecolor = g_strdup("\"BBBBBB\"");
  file->cellcolor = g_strdup("\"FFFFFF\"");

  sg_worksheet_file_export(file, NULL);

  if(!opened) sg_file_close(file->stream);

  g_free(file->filename);
  g_free(file->titlecolor);
  g_free(file->cellcolor);
  g_free(file);

  return TRUE;
}

static void
write_header_html(SGworksheetfile *file)
{
  sg_file_printf(file->stream,"<BODY>\n");
  sg_file_printf(file->stream,"<HTML>\n");

  sg_file_printf(file->stream, "<FONT=\"lucida,helvetica\">\n"); 
/*
  sg_file_printf(file->stream, "<TABLE WIDTH=80%% BORDER=1 CELLSPACING=0 CELLPADDING=2> \n");
*/
  sg_file_printf(file->stream, "<TABLE BORDER=1 CELLSPACING=0 CELLPADDING=2> \n");

  sg_file_printf(file->stream, "<TR>\n"); 
  sg_file_printf(file->stream,"<TD BGCOLOR=%s> <BR> </TD>\n", file->titlecolor);
}

static void
write_footer_html(SGworksheetfile *file)
{
  sg_file_printf(file->stream, "</TR>\n\n");
  sg_file_printf(file->stream, "</TABLE>\n\n");
  sg_file_printf(file->stream,"</BODY>\n");
  sg_file_printf(file->stream,"</HTML>\n");
}

static void
new_row_html(SGworksheetfile *file, gint row)
{

  sg_file_printf(file->stream, "</TR>\n"); 
  sg_file_printf(file->stream, "<TR>\n"); 

}

static void
write_col_title_html(SGworksheetfile *file, gint col)
{
  gchar *title;

  title = GTK_SHEET(file->worksheet)->column[col].name;

  sg_file_printf(file->stream, "<TD BGCOLOR=%s ", file->titlecolor);
  justify_html(file, GTK_JUSTIFY_CENTER);

  if(title && strlen(title) > 0)
    sg_file_printf(file->stream, ">\n<B> %s </B> </TD>\n", title);
  else
    sg_file_printf(file->stream, ">\n <BR> </TD>\n");

}

static void
write_row_title_html(SGworksheetfile *file, gint row)
{
  gchar *title;

  title = GTK_SHEET(file->worksheet)->row[row].name;

  sg_file_printf(file->stream, "<TD BGCOLOR=%s ", file->titlecolor);
  justify_html(file, GTK_JUSTIFY_CENTER);

  if(title && strlen(title) > 0)
    sg_file_printf(file->stream, ">\n<B> %s </B> </TD>\n", title);
  else
    sg_file_printf(file->stream, ">\n<B> %d </B> </TD>\n", row);
/*
    sg_file_printf(file->stream, ">\n <BR> %d </TD>\n", row);
*/
}

static void
write_cell_html(SGworksheetfile *file, gint row, gint col)
{
  gchar *text;
  GtkSheetCellAttr attr;

  text = sg_worksheet_cell_get_text(file->worksheet, row, col);
  gtk_sheet_get_attributes(GTK_SHEET(file->worksheet),
                           row, col, &attr);

  sg_file_printf(file->stream, "<TD BGCOLOR=%s ", file->cellcolor);
  justify_html(file, attr.justification);

  if(text && strlen(text) > 0) {
    gchar *s;
    sg_file_printf(file->stream, ">\n %s </TD>\n", s = xml_get_string(text));
  } else {
    sg_file_printf(file->stream, ">\n <BR> </TD>\n");
  }
}

static void
justify_html(SGworksheetfile *file, GtkJustification just)
{
  switch (just) {
    case GTK_JUSTIFY_LEFT:
      sg_file_printf(file->stream, "ALIGN=LEFT");
      break;
    case GTK_JUSTIFY_RIGHT:
      sg_file_printf(file->stream, "ALIGN=RIGHT");
      break;
    case GTK_JUSTIFY_CENTER:
      sg_file_printf(file->stream, "ALIGN=CENTER");
      break;
    case GTK_JUSTIFY_FILL:
    default:
      break;
  }
}
