/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_types.h"

                                                                             
enum
{
  ARG_UNKNOWN,
  ARG_COLOR,
  ARG_LINE,
  ARG_VECTOR,
  ARG_PIXMAP,
  ARG_BITMAP,
  ARG_TEXT,
  ARG_ARRAY,
  ARG_LAST,
};

static void put_text(FILE *stream, const gchar *name, guint type, GtkPlotText *text, gint level);
static void put_line(FILE *stream, const gchar *name, guint type, GtkPlotLine *line, gint level);
static void put_pixmap(FILE *stream, const gchar *name, guint type, GdkPixmap *pixmap, gint level);
static void put_array(FILE *stream, const gchar *name, guint type, GtkPlotArray *array, gint level);
static void put_pixmap(FILE *stream, const gchar *name, guint type, GdkPixmap *pixmap, gint level);
static void put_bitmap(FILE *stream, const gchar *name, guint type, GdkBitmap *pixmap, gint level);
static void color_to_hex(gint pixel, gchar string[5]);

static guint
get_pointer_type(const gchar *name)
{
  guint i, arg;
  gchar *pattern[] = {"", "color", "line", "vector", "pixmap", "bitmap", "text", "array"};
                                                       
  for(arg = 1; arg < ARG_LAST; arg++){                         
    for(i = 0; i < strlen(name) - strlen(pattern[arg]) + 1; i++) {
      guint j;
      for(j = 0; j < strlen(pattern[arg]); j++){
        if(name[i+j] != pattern[arg][j]) break;
      }
      if(j == strlen(pattern[arg])) return arg;
    }
  }
  return ARG_UNKNOWN;
}

gboolean
sg_object_file_export_xml   (FILE *stream, GObject *object, gint level)
{  
  GType type;
  GObject *_object = NULL;

  if(!object) return FALSE;
  if(!G_IS_OBJECT(object)) return FALSE;
  type = G_OBJECT_TYPE(object);
                                                                                
  sg_file_printf(stream, "%*s<sgp:Object Type=\"%s\">\n", level*2, " ", gtk_type_name(type));

  level++;
  
/*                                                                              
  while(type != GTK_TYPE_OBJECT && type != G_TYPE_OBJECT && type != GTK_TYPE_CONTAINER && type != GTK_TYPE_FIXED && type != GTK_TYPE_MISC) {
*/
  {
    GParamSpec **args = NULL;
    guint32* flags = NULL;
    guint n_args;
    guint i;
    guint arg_uint;
    gint arg_int;
    gdouble arg_double;
    gboolean arg_bool;
    gchar *arg_string;
    gpointer arg_pointer;
    guint arg_enum;
    GObject *arg_object;
    gchar arg_name[200];

    args = g_object_class_list_properties(G_OBJECT_GET_CLASS(G_OBJECT(object)), &n_args);

    for(i = 0; i < n_args; i++){
      gchar *c;
      g_snprintf(arg_name,200,"%s::%s",g_type_name(args[i]->owner_type),args[i]->name);
      for(c = arg_name; c && *c != '\0'; c++) if(*c == '-') *c ='_';

      if(args[i]->owner_type == GTK_TYPE_WIDGET && strcmp(arg_name,"GtkWidget::visible") != 0) continue;
      if(args[i]->owner_type == GTK_TYPE_OBJECT || args[i]->owner_type == GTK_TYPE_CONTAINER || args[i]->owner_type == GTK_TYPE_MISC) continue;

/*
      printf("Argument %u is called %s and has type %s\n", i, arg_name, gtk_type_name(args[i]->value_type));
*/

      switch(args[i]->value_type){
        case G_TYPE_UINT:
          g_object_get(G_OBJECT(object), arg_name, &arg_uint, NULL);
          sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%u\"/>\n", level*2, " ", arg_name,_GTK_TYPE_UINT,arg_uint);
          break;
        case G_TYPE_ENUM:
          g_object_get(G_OBJECT(object), arg_name, &arg_enum, NULL);
          sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%u\"/>\n", level*2, " ", arg_name,_GTK_TYPE_ENUM,arg_enum);
          break;
        case G_TYPE_INT:
          g_object_get(G_OBJECT(object), arg_name, &arg_int, NULL);
          sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%d\"/>\n", level*2, " ", arg_name,_GTK_TYPE_INT,arg_int);
          break;
        case G_TYPE_DOUBLE:
          g_object_get(G_OBJECT(object), arg_name, &arg_double, NULL);
          sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%e\"/>\n", level*2, " ", arg_name,_GTK_TYPE_DOUBLE,arg_double);
          break;
        case G_TYPE_BOOLEAN:
          g_object_get(G_OBJECT(object), arg_name, &arg_bool, NULL);
          sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%u\"/>\n", level*2, " " , arg_name,_GTK_TYPE_BOOL,arg_bool);
          break;
        case G_TYPE_STRING:
          g_object_get(G_OBJECT(object), arg_name, &arg_string, NULL);
          if(arg_string) {
            gchar *s;
            sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%s\"/>\n", level*2, " " , arg_name,_GTK_TYPE_STRING, s = xml_get_string(arg_string));
          }
          break;
        case G_TYPE_OBJECT:
          g_object_get(G_OBJECT(object), arg_name, &arg_object, NULL);
          _object = arg_object;
          if(_object && G_IS_OBJECT(_object)){
            sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%s\">\n",  level*2, " " , arg_name, _GTK_TYPE_OBJECT, g_type_name(G_OBJECT_TYPE(arg_object)));
            sg_object_file_export_xml(stream, _object, level+1);
            sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " "); 
          }
          break;
        case G_TYPE_POINTER:
          g_object_get(G_OBJECT(object), arg_name, &arg_pointer, NULL);
          if(get_pointer_type(arg_name) == ARG_COLOR){
            GdkColor *color = (GdkColor *)arg_pointer;
            if(color) sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GdkColor\" R=\"%d\" G=\"%d\" B=\"%d\"/>\n", level*2, " ", arg_name,_GTK_TYPE_POINTER,color->red,color->green,color->blue);
          } 
          if(get_pointer_type(arg_name) == ARG_VECTOR){
            GtkPlotVector *v = (GtkPlotVector *)arg_pointer;
            if(v) sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GtkPlotVector\" X=\"%f\" Y=\"%f\" Z=\"%f\"/>\n", level*2, " ", arg_name,_GTK_TYPE_POINTER,v->x,v->y,v->z);
          }
          if(get_pointer_type(arg_name) == ARG_TEXT){
            GtkPlotText *text = (GtkPlotText *)arg_pointer;
            put_text(stream, arg_name, _GTK_TYPE_POINTER, text, level);
          }
          if(get_pointer_type(arg_name) == ARG_PIXMAP){
            GdkPixmap *pixmap = (GdkPixmap *)arg_pointer;
            put_pixmap(stream, arg_name, _GTK_TYPE_POINTER, pixmap, level);
          }
          if(get_pointer_type(arg_name) == ARG_LINE){
            GtkPlotLine *line = (GtkPlotLine *)arg_pointer;
            put_line(stream, arg_name, _GTK_TYPE_POINTER, line, level);
          }
          if(get_pointer_type(arg_name) == ARG_BITMAP){
            GdkBitmap *bitmap = (GdkBitmap *)arg_pointer;
            put_bitmap(stream, arg_name, _GTK_TYPE_POINTER, bitmap, level);
          }
          if(get_pointer_type(arg_name) == ARG_ARRAY){
            if(GTK_IS_PLOT_ARRAY(object))
              put_array(stream, arg_name, _GTK_TYPE_POINTER, GTK_PLOT_ARRAY(object), level);
          }
          break;
        default:
          if(G_TYPE_IS_ENUM(args[i]->value_type)){
            g_object_get(G_OBJECT(object), arg_name, &arg_enum, NULL);
            sg_file_printf(stream,"%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%u\"/>\n", level*2, " ", arg_name,_GTK_TYPE_ENUM,arg_enum);
          } else {
            g_object_get(G_OBJECT(object), arg_name, &arg_object, NULL);
            if(arg_object && G_IS_OBJECT(arg_object)){
              _object = G_OBJECT(arg_object);
              sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"%s\">\n",  level*2, " " , arg_name, _GTK_TYPE_OBJECT, g_type_name(G_OBJECT_TYPE(_object)));
              sg_object_file_export_xml(stream, _object, level+1);
              sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " "); 
            }
          }
          break;
                                                                                
      }
    }
                                                                                
    if(args) g_free(args);
    args = NULL;
    if(flags) g_free(flags);
    flags = NULL;
                                                                                
    type = gtk_type_parent(type);
  }

  level--;

  sg_file_printf(stream, "%*s</sgp:Object>\n", level*2, " ");
                                                                                
  return TRUE;
}

static void
put_text(FILE *stream, const gchar *name, guint type, GtkPlotText *text, gint level)
{
  sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GtkPlotText\" X=\"%f\" Y=\"%f\" Angle=\"%d\" Justification=\"%d\" Transparent=\"%d\">\n", level*2, " ", name, type, text->x, text->y, text->angle, text->justification, text->transparent);
  sg_file_printf(stream, "%*s  <sgp:Border Style=\"%d\" Space=\"%d\" Width=\"%d\" Shadow=\"%d\"/>\n", level*2, " ", text->border, text->border_space, text->border_width, text->shadow_width);
  sg_file_printf(stream, "%*s  <sgp:Foreground R=\"%d\" G=\"%d\" B=\"%d\"/>\n",level*2, " ",  text->fg.red, text->fg.green, text->fg.blue);
  sg_file_printf(stream, "%*s  <sgp:Background R=\"%d\" G=\"%d\" B=\"%d\"/>\n",level*2, " ",  text->bg.red, text->bg.green, text->bg.blue);
  sg_file_printf(stream, "%*s  <sgp:Font Name=\"%s\" Height=\"%d\"/>\n",level*2, " ",  text->font, text->height);
                                                                                
  if(text->text && strlen(text->text) > 0) {
    gchar *s;
    sg_file_printf(stream, "%*s  <sgp:String>%s</sgp:String>\n",level*2, " ",  s = xml_get_string(text->text));
  }
                                                                                
  sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
}

static void
put_line(FILE *stream, const gchar *name, guint type, GtkPlotLine *line, gint level)
{
  sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GtkPlotLine\" Style=\"%d\" Width=\"%f\" R=\"%d\" G=\"%d\" B=\"%d\"/>\n", level*2, " ", name, type, line->line_style, line->line_width, line->color.red, line->color.green, line->color.blue);
}

static void
put_pixmap(FILE *stream, const gchar *name, guint type, GdkPixmap *pixmap, gint level)
{
  GdkPixbuf *pixbuf;
  gint width, height;
  gint x, y;
  gint i;
  gint ncolors = 0;
  GdkColor *colors = NULL;
  guchar *pixels = NULL;
  gint rowstride;
  gint pos;
  
  if(!pixmap) return;

  gdk_window_get_size(pixmap, &width, &height);

  if(width == 0 && height == 0) return;

  pixbuf = gdk_pixbuf_get_from_drawable(NULL, pixmap, gdk_drawable_get_colormap(pixmap), 0, 0, 0, 0, width, height);

  if(!pixbuf) return;

  pixels = gdk_pixbuf_get_pixels(pixbuf);
  rowstride = gdk_pixbuf_get_rowstride(pixbuf);
  pos = 0;
  
  for(y = 0; y < height; y++){
    for(x = 0; x < width; x++){
      GdkColor color;
      gboolean stored;
      guchar *p;

      pos = y * rowstride + x * 4;
      p = &pixels[pos];
      color.red = p[0];
      color.green = p[1];
      color.blue = p[2];

      stored = FALSE;
      for(i = 0; i < ncolors; i++)
        if(gdk_color_equal(&colors[i], &color)) stored = TRUE;
                                                                                
      if(!stored){
         colors = (GdkColor *)g_realloc(colors, (ncolors+1)*sizeof(GdkColor));
         colors[ncolors] = color;
         ncolors++;
      }
    }
  }
                                                                                
  sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GdkPixmap\" Width=\"%d\" Height=\"%d\" Colors=\"%d\">\n", level*2, " ", name, type, width, height, ncolors);

  sg_file_printf(stream, "%*s  <sgp:Pixmap>\n", level*2, " ");
                                                                                
  for(i = 0; i < ncolors; i++){
    gchar r[5], g[5], b[5];
    color_to_hex(colors[i].red, r);
    color_to_hex(colors[i].green, g);
    color_to_hex(colors[i].blue, b);
                                                                                
/*
    printf("%s %s %s %d %d %d\n",r,g,b,colors[i].red,colors[i].green,colors[i].blue);
*/
                                                                                
    sg_file_printf(stream,"%*s    <sgp:XPMColor>%s%s%s</sgp:XPMColor>\n", level*2, " ", r, g, b);
  }
                                                                                
  sg_file_printf(stream, "%*s    <sgp:Pixels>\n", level*2, " ");
  for(y = 0; y < height; y++){
    for(x = 0; x < width; x++){
      GdkColor color;
      gboolean stored;
      guchar *p;
      gchar string[5];

      pos = y * rowstride + x * 4;
      p = &pixels[pos];
      color.red = p[0];
      color.green = p[1];
      color.blue = p[2];

      for(i = 0; i < ncolors; i++)
         if(gdk_color_equal(&colors[i],&color)) break;

      color_to_hex(i, string);

      sg_file_printf(stream,"%s",string);
      if(fmod(x+1, 13) == 0) sg_file_printf(stream,"\n");
    }
    sg_file_printf(stream,"\n");
  }
                                                                                
  sg_file_printf(stream, "%*s    </sgp:Pixels>\n", level*2, " ");
  sg_file_printf(stream, "%*s  </sgp:Pixmap>\n", level*2, " ");
  sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
                                                                                
  gdk_pixbuf_unref(pixbuf);
}

static void
put_bitmap(FILE *stream, const gchar *name, guint type, GdkBitmap *bitmap, gint level)
{
  GdkImage *image = NULL;
  gint x, y, width, height;

  if(!bitmap) return;
                                                                                
  gdk_window_get_size(bitmap, &width, &height);

  if(width == 0 && height == 0) return;

  image = gdk_image_get(bitmap,
                        0, 0,
                        width, height);

  if(!image) return;

  sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"GdkBitmap\" Width=\"%d\" Height=\"%d\">\n", level*2, " ", name, type, width, height);
  sg_file_printf(stream, "%*s  <sgp:Bitmap>\n", level*2, " ");
  sg_file_printf(stream, "%*s    <sgp:Pixels>\n", level*2, " ");
  for(y = 0; y < height; y++){
    for(x = 0; x < width; x++){
      gulong pixel;
                                                                                
      pixel = gdk_image_get_pixel(image, x, y);
                                                                               
      if(pixel == 0) 
        sg_file_printf(stream,"0");
      else
        sg_file_printf(stream,"1");
      if(fmod(x+1, 80) == 0) sg_file_printf(stream,"\n");
    }
    sg_file_printf(stream,"\n");
  }
                                                                                
  sg_file_printf(stream, "%*s    </sgp:Pixels>\n", level*2, " ");
  sg_file_printf(stream, "%*s  </sgp:Bitmap>\n", level*2, " ");
  sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");

  gdk_image_destroy(image);
}

static void
put_array(FILE *stream, const gchar *name, guint type, GtkPlotArray *array, gint level)
{
  if(array->type == GTK_TYPE_STRING){
    gchar **data = gtk_plot_array_get_string(array);
    gint size = gtk_plot_array_get_size(array);
                                                                                
    if(data && size > 0){
      gint n = 0;

      sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"Array\">\n", level*2, " ", name, type);

      for(n = 0; n < size; n++){
        if(data[n])
          sg_file_printf(stream, "%*s  <sgp:Point Index=\"%d\" Value=\"%s\"/>\n", level*2, " ", n, data[n]);
        else
          sg_file_printf(stream, "%*s  <sgp:Point Index=\"%d\" Value=\"\"/>\n", level*2, " ", n);
      }

      sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
    }

  } else if(array->type == GTK_TYPE_DOUBLE){
    gdouble *data = gtk_plot_array_get_double(array);
    gint size = gtk_plot_array_get_size(array);

    if(data && size > 0){
      gint n = 0;

      sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"Array\">\n", level*2, " ", name, type);

      for(n = 0; n < size; n++){
        sg_file_printf(stream, "%*s  <sgp:Point Index=\"%d\" Value=\"%e\"/>\n", level*2, " ", n, data[n]);
      }

      sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
    }
  } else if(array->type == GTK_TYPE_FLOAT){
    gfloat *data = gtk_plot_array_get_float(array);
    gint size = gtk_plot_array_get_size(array);

    if(data && size > 0){
      gint n = 0;

      sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"Array\">\n", level*2, " ", name, type);

      for(n = 0; n < size; n++){
        sg_file_printf(stream, "%*s  <sgp:Point Index=\"%d\" Value=\"%e\"/>\n", level*2, " ", n, data[n]);
      }

      sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
    }
  } else if(array->type == GTK_TYPE_INT){
    gint *data = gtk_plot_array_get_int(array);
    gint size = gtk_plot_array_get_size(array);

    if(data && size > 0){
      gint n = 0;

      sg_file_printf(stream, "%*s<sgp:Arg Name=\"%s\" Type=\"%u\" Value=\"Array\">\n", level*2, " ", name, type);

      for(n = 0; n < size; n++){
        sg_file_printf(stream, "%*s  <sgp:Point Index=\"%d\" Value=\"%d\"/>\n", level*2, " ", n, data[n]);
      }

      sg_file_printf(stream, "%*s</sgp:Arg>\n", level*2, " ");
    }
  }
}

static void
color_to_hex(gint pixel, gchar string[5])
{
  gint i, n;
  gint powers[4] = {1, 16, 256, 4096};
                                                                                
  for(i=3; i>=0; i--){
     n = pixel / powers[i];
     pixel -= n * powers[i];
     if(n < 10)
       string[3-i] = '0' + n;
     else
       string[3-i] = 'A' + n - 10;
  }
  string[4] = '\0';
}
