/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "pixmaps/sg_small.xpm"
#include <libxml/xmlreader.h>

typedef struct
{
  gint row, col;
  gchar *last_node;
  SGworksheet *worksheet;
} parser_state;

void
SGmatrix_xml_open_init(SGplugin *_plugin)
{
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  GtkSheet *sheet = GTK_SHEET(state->worksheet);
  xmlChar *name = xmlTextReaderName(reader);
  SGmatrix *matrix = SG_MATRIX(state->worksheet);

  /* Start Element */

  if(xmlTextReaderNodeType(reader) == 1){
    if(strcmp(name,"sgw:Range") == 0){ 
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Xmin") == 0) matrix->xmin = atof(value);
        if(strcmp(child, "Xmax") == 0) matrix->xmax = atof(value);
        if(strcmp(child, "Ymin") == 0) matrix->ymin = atof(value);
        if(strcmp(child, "Ymax") == 0) matrix->ymax = atof(value);

        xmlFree(child);
        xmlFree(value);
      }
    }
    if(strcmp(name,"sgw:Xvalues") == 0){ 
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "NX") == 0) matrix->nx = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
      if(matrix->nx > 0) matrix->x_values = g_new0(gdouble, matrix->nx);
    }
    if(strcmp(name,"sgw:Yvalues") == 0){ 
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "NY") == 0) matrix->ny = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
      if(matrix->ny > 0) matrix->y_values = g_new0(gdouble, matrix->ny);
    }
    if(strcmp(name,"sgw:X") == 0){ 
      gint index = 0;
      gdouble val = 0.;
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "index") == 0) index = atoi(value);
        if(strcmp(child, "value") == 0) val = atof(value);

        xmlFree(child);
        xmlFree(value);
      }
      if(matrix->nx > 0 && matrix->x_values){
	 matrix->x_values[index] = val;
      }
    }
    if(strcmp(name,"sgw:Y") == 0){ 
      gint index = 0;
      gdouble val = 0.;
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "index") == 0) index = atoi(value);
        if(strcmp(child, "value") == 0) val = atof(value);

        xmlFree(child);
        xmlFree(value);
      }
      if(matrix->ny > 0 && matrix->y_values){
	 matrix->y_values[index] = val;
      }
    }
/* FIXME
    if(strcmp(name,"sgw:Geometry") == 0){ 
      gint width = 500, height = 350;
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Width") == 0) width = atoi(value);
        if(strcmp(child, "Height") == 0) height = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
      state->worksheet->width = width;
      state->worksheet->height = height;
      if(state->worksheet->is_mapped)
         gtk_window_set_default_size(GTK_WINDOW(state->worksheet), width, height);
    }
*/
    if(strcmp(name,"sgw:MatrixFormat") == 0){ 
      SGcolumntype type = SG_TYPE_NUMBER;
      SGcolumnformat format = SG_FORMAT_DECIMAL;
      SGcolumninternal internal = SG_INTERNAL_DOUBLE;
      gint precision = 3;
      gchar *exp = NULL;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Exp") == 0) exp = g_strdup(value);
        if(strcmp(child, "Internal") == 0) internal = (SGcolumninternal)atoi(value);
        if(strcmp(child, "Format") == 0) format = (SGcolumnformat)atoi(value);
        if(strcmp(child, "Precision") == 0 ||
           strcmp(child, "Presicion") == 0) precision = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
      sg_matrix_set_exp(matrix, exp);
      matrix->format.type = type;
      matrix->format.format = format;
      matrix->format.internal = internal;
      matrix->format.precision = precision;
    }
    if(strcmp(name,"sgw:Column") == 0){ 
      gint ncol = 0;
      gint width = 80;
      gchar *title = NULL;
      SGcolumntype type = SG_TYPE_NONE;
      SGcolumnformat format = SG_FORMAT_DECIMAL;
      SGcolumninternal internal = SG_INTERNAL_DOUBLE;
      gint precision = 3;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Width") == 0) width = atoi(value);
        if(strcmp(child, "No") == 0) ncol = atoi(value);
        if(strcmp(child, "Title") == 0) title = g_strdup(value);
        if(strcmp(child, "Type") == 0) type = (SGcolumntype)atoi(value);
        if(strcmp(child, "Internal") == 0) internal = (SGcolumninternal)atoi(value);
        if(strcmp(child, "Format") == 0) format = (SGcolumnformat)atoi(value);
        if(strcmp(child, "Precision") == 0 ||
           strcmp(child, "Presicion") == 0) precision = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
      state->col = ncol;
      gtk_sheet_set_column_width(sheet, ncol, width);
      gtk_sheet_set_column_title(sheet, ncol, title);
      gtk_sheet_column_button_add_label(sheet, ncol, title);
      g_free(title);

      if(type == SG_TYPE_NONE){ /* before 0.7 */
        type = (SGcolumntype)format;
        format = SG_FORMAT_DECIMAL;
      }

      sg_worksheet_column_set_format(state->worksheet, ncol, type, format, internal, precision);
    }

    if(strcmp(name,"sgw:Cell") == 0){ 
      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Row") == 0) state->row = atoi(value);
        if(strcmp(child, "Col") == 0) state->col = atoi(value);

        xmlFree(child);
        xmlFree(value);
      }
    }

    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  /* Text node */

  if(state->last_node && xmlTextReaderNodeType(reader) == 3){
    xmlChar *value = xmlTextReaderValue(reader);
    if(!value) return;

    if(strcmp(state->last_node, "sgw:Name") == 0){
      sg_worksheet_rename(state->worksheet, value);
    }
    if(strcmp(state->last_node, "sgw:Begin") == 0){
      sg_worksheet_set_begin(state->worksheet, atoi(value));
    }
    if(strcmp(state->last_node, "sgw:End") == 0){
      sg_worksheet_set_end(state->worksheet, atoi(value));
    }
    if(strcmp(state->last_node, "sgw:Content") == 0){
      sg_worksheet_cell_set(state->worksheet, state->row, state->col, value, TRUE, TRUE);
    }
    if(strcmp(state->last_node, "sgw:Formula") == 0){
      sg_worksheet_cell_set(state->worksheet, state->row, state->col, value, TRUE, TRUE);
    }
    if(strcmp(state->last_node, "sgw:ColFormula") == 0){
      sg_worksheet_column_set_exp(state->worksheet, state->col, value);
    }
    if(strcmp(state->last_node, "sgw:MaxRow") == 0)
      sg_worksheet_add_rows(state->worksheet, 
		            atoi(value)-GTK_SHEET(state->worksheet)->maxrow);

    if(strcmp(state->last_node, "sgw:MaxCol") == 0)
      sg_worksheet_add_columns(state->worksheet,
		               atoi(value)-GTK_SHEET(state->worksheet)->maxcol);

    xmlFree(value);
  }

  /* End Element */

  if(xmlTextReaderNodeType(reader) == 15){
    if(state->last_node) g_free(state->last_node);
    state->last_node = NULL;
  }

  xmlFree(name);
}

gboolean
SGmatrix_xml_open   (SGpluginFile *plugin, 
		     const gchar *filename, 
		     FILE *stream,
                     GObject **object, gpointer data)
{  
  SGmatrix *matrix = NULL;
  xmlTextReaderPtr reader;
  gint ret_val;
  parser_state *state;

  state = g_new0(parser_state, 1);
  state->last_node = NULL;

  if(data)
    reader = (xmlTextReaderPtr)data;
  else
    reader = xmlNewTextReaderFilename(filename);

  if(!reader) return FALSE;

  matrix = sg_matrix_new("",20,5);
  state->worksheet = SG_WORKSHEET(matrix);
  *object = G_OBJECT(matrix);

  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgw:Matrix") == 0) {
      sg_worksheet_update_exp_all(SG_WORKSHEET(*object));
      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      g_free(state);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(!data) xmlFreeTextReader(reader);
  if(state->last_node) g_free(state->last_node);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


