/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "sg_object_file_xml.h"
#include "sg_layer_file_xml.h"
#include "pixmaps/sg_small.xpm"

static void write_header		(FILE *file, SGlayer *layer);
static void write_footer		(FILE *file);
static void save_layer			(FILE *file, SGlayer *layer);
static void save_dataset		(FILE *file, GtkPlotData *dataset);

void
layer_2d_xml_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgp");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
layer_2d_xml_export     (SGpluginFile *plugin, 
                        const gchar *filename, 
                        FILE *opened,
                        GObject **object, gpointer data)
{
  return(sg_layer_export_xml(plugin, filename, opened, object, data));
}

void
layer_3d_xml_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgp");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
layer_3d_xml_export     (SGpluginFile *plugin, 
                        const gchar *filename, 
                        FILE *opened,
                        GObject **object, gpointer data)
{
  return(sg_layer_export_xml(plugin, filename, opened, object, data));
}

void
layer_polar_xml_export_init(SGpluginFile *plugin)
{
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_EXPORT;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sgp");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

gboolean
layer_polar_xml_export     (SGpluginFile *plugin, 
                        const gchar *filename, 
                        FILE *opened,
                        GObject **object, gpointer data)
{
  return(sg_layer_export_xml(plugin, filename, opened, object, data));
}

gboolean
sg_layer_export_xml     (SGpluginFile *plugin, 
                         const gchar *filename, 
                         FILE *opened,
                         GObject **object, gpointer data)
{
  FILE *file;
  SGlayer *layer;

  layer = SG_LAYER(*object);

  if(opened)
    file = opened;
  else
    file = sg_file_open(filename, "wb");
   
  if (!file){
       g_warning( _("ERROR: Cannot write to file: %s"), filename);
       return FALSE;
  } 

  if(!opened)
    sg_file_printf(file,"<?xml version=\"1.0\"?>\n");

  write_header(file, SG_LAYER(*object));
  save_layer(file, SG_LAYER(*object));
  write_footer(file);

  if(!opened) sg_file_close(file);
  
  return TRUE;
}

static void
write_header(FILE *file, SGlayer *layer)
{
  sg_file_printf(file,"  <sgp:Layer Plugin=\"%s\" xmlns:sgp=\"http://scigraphica.sourceforge.net\">\n", SG_PLUGIN(layer->plugin)->name);
  sg_file_printf(file, "    <sgp:Doc Version=\"3.0\"/>\n");
  sg_file_printf(file, "    <sgp:Summary>\n");
  sg_file_printf(file, "      <sgp:Item>\n");
  sg_file_printf(file, "        <sgp:name>application</sgp:name>\n");
  sg_file_printf(file, "        <sgp:val-string>scigraphica</sgp:val-string>\n");
  sg_file_printf(file, "      </sgp:Item>\n");
  sg_file_printf(file, "      <sgp:Item>\n");
  sg_file_printf(file, "        <sgp:name>author</sgp:name>\n");
  sg_file_printf(file, "        <sgp:val-string>%s</sgp:val-string>\n", g_get_real_name());
  sg_file_printf(file, "      </sgp:Item>\n");
  sg_file_printf(file, "    </sgp:Summary>\n");
}

static void
save_layer(FILE *file, SGlayer *layer)
{
  GtkPlot *plot;
  GList *datasets;

  plot = GTK_PLOT_CANVAS_PLOT(layer)->plot;

  sg_object_file_export_xml(file, G_OBJECT(layer), 2);
  sg_object_file_export_xml(file, G_OBJECT(plot), 2);

  datasets = plot->data_sets;
  while(datasets){
    save_dataset(file, GTK_PLOT_DATA(datasets->data));
    datasets = datasets->next;
  } 
}

static void
write_footer(FILE *file)
{
  sg_file_printf(file,"  </sgp:Layer>\n");
}

static void
save_dataset(FILE *file, GtkPlotData *dataset)
{
  SGpluginFile *plugin;
  GObject *o = G_OBJECT(dataset);

  plugin = sg_plugin_file_get("xml", "sg_dataset", SG_PLUGIN_FILE_EXPORT); 

  if(plugin) plugin->action(plugin, NULL, file, &o, NULL);

}

