/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <Python.h>
#include <scigraphica/sg.h>
#include <scigraphica/sg_python.h>
#include <scigraphica/sg_python_expr.h>
#include "sg_import_dialog.h"
#include "import_data.h"

#define GTK_TYPE_SG_IMPORT_DIALOG (sg_import_dialog_get_type ())
#define SG_IMPORT_DIALOG(obj) (GTK_CHECK_CAST ((obj), GTK_TYPE_SG_IMPORT_DIALOG, SGimportDialog))

typedef struct
{
  GtkWindow parent;

  SGdelimiter delimiter;

  gchar *file;
  SGworksheet *worksheet;
  gboolean return_val;
  
  gint begin_line;
  gint end_line;
  gint blocknum;
  gboolean read_all_lines;
  gint read_method;
  gboolean use_custom_delimiter;
  gint skip_lines;
  
  gchar *comment_string;
  gchar *block_start;
  gchar *custom_delimiter;
  
  gboolean rename_worksheets;
  gboolean precision_mod;
  gboolean rename_columns;
  
  GtkWidget *delimiter_item;
  GtkWidget *delimiter_combo;
  GtkWidget *method_combo;
  GtkWidget *delimiter_entry;
  GtkWidget *comment_entry;
  
  GtkWidget *block_entry;
  GtkWidget *skip_spin;
  GtkWidget *skip_spin2;
  GtkWidget *skip_spin3;
  
  GtkWidget *rename_columns_item;
  GtkWidget *rename_worksheets_item;
  
  GtkWidget *lines_frame;
  GtkWidget *block_frame;
  
  GtkWidget *endline_item;
  
  GtkWidget *cancel_button;
  GtkWidget *ok_button;

  GtkPlotArrayList *data;

} SGimportDialog;

typedef struct
{
  GtkWindowClass parent_class;
} SGimportDialogClass;

static void sg_import_dialog_class_init   	(SGimportDialogClass *klass);
static void sg_import_dialog_init         	(SGimportDialog *dialog);
static void sg_import_dialog_destroy      	(GtkObject *o);
static void sg_import_dialog_construct		(SGimportDialog *dialog);
static gboolean sg_import_dialog_import_block	(SGimportDialog *dialog);

static GtkWindowClass *parent_class = NULL;

GtkType
sg_import_dialog_get_type(void)
{
  static GtkType sg_import_dialog_type = 0;

  if (!sg_import_dialog_type)
                        {
    GtkTypeInfo sg_import_dialog_info =
    {
      "SGimportDialog",
      sizeof (SGimportDialog),
      sizeof (SGimportDialogClass),
      (GtkClassInitFunc) sg_import_dialog_class_init,
      (GtkObjectInitFunc) sg_import_dialog_init,
      /* reserved_1 */ NULL,
      /* reserved_2 */ NULL,
      (GtkClassInitFunc) NULL,
    };

    sg_import_dialog_type = gtk_type_unique (gtk_window_get_type(), &sg_import_dialog_info);
  }


  return sg_import_dialog_type;
}

static void
sg_import_dialog_class_init (SGimportDialogClass *klass)
{
  GtkObjectClass *object_class;

  object_class = (GtkObjectClass*) klass;

  parent_class = (GtkWindowClass *)gtk_type_class (gtk_window_get_type ());

  object_class->destroy = sg_import_dialog_destroy;
};

static void
sg_import_dialog_init(SGimportDialog *dialog)
{
  dialog->delimiter = SG_DELIMITER_UNKNOWN;
  dialog->begin_line = 1;
  dialog->end_line = 1;
  dialog->blocknum = 1;
  dialog->read_all_lines = TRUE;
  dialog->read_method = 0;
  dialog->use_custom_delimiter = FALSE;
  dialog->skip_lines = 0;
  dialog->file = NULL;
  
  dialog->comment_string = g_strdup("#");
  dialog->block_start = g_strdup("#");
  dialog->custom_delimiter = g_strdup("&");
  
  dialog->rename_worksheets = TRUE;
  dialog->precision_mod = TRUE;
  dialog->rename_columns = FALSE;

  dialog->data = NULL;
}

static void
sg_import_dialog_destroy(GtkObject *object)
{
  SGimportDialog *dialog = SG_IMPORT_DIALOG(object);

  if(dialog->comment_string) g_free(dialog->comment_string);
  dialog->comment_string = NULL;
  if(dialog->block_start) g_free(dialog->block_start);
  dialog->block_start = NULL;
  if(dialog->custom_delimiter) g_free(dialog->custom_delimiter);
  dialog->custom_delimiter = NULL;
  if(dialog->file) g_free(dialog->file);
  dialog->file = NULL;
  if(dialog->data) g_object_unref(G_OBJECT(dialog->data));
  dialog->data = NULL;

  if (GTK_OBJECT_CLASS (parent_class)->destroy)
    (*GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

GtkWidget *
sg_import_dialog_new(const gchar *filename, SGworksheet *worksheet)
{
  SGimportDialog *dialog;

  dialog = SG_IMPORT_DIALOG(gtk_widget_new(sg_import_dialog_get_type(), NULL));

  dialog->file = g_strdup(filename);
  dialog->worksheet = worksheet;
  GTK_WINDOW(dialog)->type = GTK_WINDOW_TOPLEVEL;
  gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, FALSE);
  gtk_window_set_title(GTK_WINDOW(dialog), _("Scigraphica: Import text file") );

  sg_import_dialog_construct(dialog);

  return GTK_WIDGET(dialog);
}

static void
update_options(GtkWidget *widget, gpointer data)
{ 
  GtkWidget *child;
  SGimportDialog *dialog = SG_IMPORT_DIALOG(widget);

  dialog->use_custom_delimiter = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->delimiter_item));
  if (dialog->use_custom_delimiter)
  {   
      gtk_widget_set_sensitive(dialog->delimiter_entry,TRUE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,FALSE);
  }
  else
  {   
      gtk_widget_set_sensitive(dialog->delimiter_entry,FALSE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,TRUE);
  }

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list)->selection->data;
  dialog->delimiter = (SGdelimiter)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list), child);

  if(dialog->custom_delimiter) g_free(dialog->custom_delimiter);
  dialog->custom_delimiter = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->delimiter_entry)));
  if(dialog->comment_string) g_free(dialog->comment_string);
  dialog->comment_string = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->comment_entry)));
  if(dialog->block_start) g_free(dialog->block_start);
  dialog->block_start = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->block_entry)));

  dialog->begin_line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin));
  dialog->end_line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin2));
  dialog->blocknum = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin3));

  dialog->rename_columns = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->rename_columns_item));

  dialog->rename_worksheets = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->rename_worksheets_item));

/*
  dialog->precision_mod= gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(precision_item));
*/
  dialog->precision_mod = FALSE;

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->method_combo)->list)->selection->data;
  dialog->read_method = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->method_combo)->list), child);

  dialog->read_all_lines = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->endline_item));

  if (dialog->read_all_lines)
     gtk_widget_set_sensitive(dialog->skip_spin2,FALSE);
  else
     gtk_widget_set_sensitive(dialog->skip_spin2,TRUE);

  switch (dialog->read_method)
  { 
    case 0:
      gtk_widget_set_sensitive(dialog->lines_frame,TRUE);
      gtk_widget_set_sensitive(dialog->block_frame,FALSE);
      break;
    case 1:
      gtk_widget_set_sensitive(dialog->lines_frame,FALSE);
      gtk_widget_set_sensitive(dialog->block_frame,TRUE);
      break;
  }

}

static void
mw_destroy(GtkWidget *widget, gpointer data)
{
  gtk_main_quit();
}

static void
import_file(GtkWidget *widget, gpointer data)
{ 
  SGimportDialog *dialog = SG_IMPORT_DIALOG(widget);

  update_options(GTK_WIDGET(dialog), NULL);
  if (dialog->file){
    sg_import_dialog_import_block(dialog);

    if(dialog->rename_worksheets) {
      gpointer odata = gtk_object_get_data(GTK_OBJECT(dialog->worksheet), "application");
      gchar *name = &dialog->file[strlen(dialog->file)-1];
      for(; name && *name != G_DIR_SEPARATOR; name--) {}
      if(!name) 
        name = dialog->file;
      else
        name++;
     
      sg_worksheet_rename(dialog->worksheet, name);
      if(odata){
        sg_application_rename_worksheet(SG_APPLICATION(odata), dialog->worksheet);
      }
    }
  }

  gtk_widget_destroy(widget);
}

static void
cancel_clicked(GtkWidget *widget, gpointer data)
{ 
  gtk_widget_destroy(widget);
}


static void
init_dialog(SGimportDialog *dialog)
{
  gchar *delimiters[] = { _("unknown"),
                          _("comma(,)"),
                          _("tab"),
                          _("space"),
                          NULL };
  gchar *methods[] = { _("lines"),
                       _("blocks"),
                       NULL };

  sg_combo_set_items(GTK_COMBO(dialog->delimiter_combo), delimiters);
  sg_combo_set_items(GTK_COMBO(dialog->method_combo), methods);
    
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list), dialog->delimiter);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->method_combo)->list), dialog->read_method);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin), dialog->skip_lines);

  gtk_entry_set_text(GTK_ENTRY(dialog->delimiter_entry), dialog->custom_delimiter);
  gtk_entry_set_text(GTK_ENTRY(dialog->block_entry), dialog->block_start);

  if(dialog->use_custom_delimiter)
  {
      gtk_widget_set_sensitive(dialog->delimiter_entry,TRUE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,FALSE);
  }
  else
  {
      gtk_widget_set_sensitive(dialog->delimiter_entry,FALSE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,TRUE);
  }


  gtk_entry_set_text(GTK_ENTRY(dialog->comment_entry), dialog->comment_string);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->rename_columns_item), 
                              dialog->rename_columns);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->rename_worksheets_item),
                              dialog->rename_worksheets);
/*
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(precision_item),
                              dialog->precision_mod);
*/
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->endline_item),
                              dialog->read_all_lines);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->delimiter_item),
                              dialog->use_custom_delimiter);

  switch (dialog->read_method)
  { case 0:
      gtk_widget_set_sensitive(dialog->lines_frame,TRUE);
      gtk_widget_set_sensitive(dialog->block_frame,FALSE);
      break;
    case 1:
      gtk_widget_set_sensitive(dialog->lines_frame,FALSE);
      gtk_widget_set_sensitive(dialog->block_frame,TRUE);
      break;
  }

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin),(gfloat)dialog->begin_line);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin2),(gfloat)dialog->end_line);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin3),(gfloat)dialog->blocknum);

}


static void
sg_import_dialog_construct(SGimportDialog *dialog)
{
  GtkWidget *window = GTK_WIDGET(dialog);
  GtkWidget *frame;
  GtkWidget *main_box;
  GtkWidget *action_area;
  GtkWidget *table;
  GtkWidget *comment_label;
  GtkWidget *delimiter_label;
  GtkWidget *method_label;
  GtkWidget *block_label;
  GtkAdjustment *adj,*adj2,*adj3;
  GtkRequisition req;
  gchar name[80];

  dialog->return_val = FALSE;

  /* Create widgets */
  main_box= gtk_table_new(2, 3, FALSE);

  gtk_container_add (GTK_CONTAINER (window), main_box);

  g_snprintf(name,80, _("File structure") );
  frame = gtk_frame_new(name);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), frame, 0, 2, 0, 1,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(3, 4, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  dialog->delimiter_item = gtk_check_item_new_with_label( _("Use custom:") );
  dialog->delimiter_combo = gtk_combo_new();
  dialog->delimiter_entry = gtk_entry_new();
  dialog->comment_entry = gtk_entry_new();
  comment_label=gtk_label_new( _("Comment (matches any character in string):") );
  delimiter_label=gtk_label_new( _("Delimiter:") );
  method_label=gtk_label_new( _("Read file as:") );
  dialog->method_combo = gtk_combo_new();

  gtk_widget_size_request(dialog->delimiter_entry, &req);
  gtk_widget_set_usize(dialog->delimiter_entry, 20, req.height);
  gtk_widget_set_usize(dialog->comment_entry, 20, req.height);
  gtk_widget_set_usize(dialog->delimiter_combo, req.width/3, req.height);
  gtk_widget_set_usize(dialog->method_combo, req.width/3, req.height);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->delimiter_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->method_combo)->entry), FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table), delimiter_label, 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_combo, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_item, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_entry, 2, 3, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table), comment_label, 0, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->comment_entry, 2, 3, 3, 4);

  gtk_table_attach_defaults(GTK_TABLE(table), method_label, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->method_combo, 1, 2, 0, 1);

 /* Import as lines */
  dialog->lines_frame = gtk_frame_new( _("Read lines") );
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->lines_frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), dialog->lines_frame, 0, 1, 1, 2,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(1, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(dialog->lines_frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->begin_line, 1., 1e6, 1., 10., 10.);
  dialog->skip_spin = gtk_spin_button_new(adj, 0, 0);   
  gtk_widget_set_usize(dialog->skip_spin, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Start line") ),
                            0, 1, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin,
                            1, 2, 0, 1);
  dialog->endline_item = gtk_check_item_new_with_label( _("Read all lines") );
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->endline_item,
                            0, 2, 1, 2);


  /* End line */
  adj2 = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->end_line, 1.,1e6, 1., 10., 10.);
  dialog->skip_spin2 = gtk_spin_button_new(adj2, 0, 0);
  gtk_widget_set_usize(dialog->skip_spin2, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin2), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("End line") ),
                            0, 1, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin2,
                            1, 2, 2, 3);


 /* Import as blocks */
  dialog->block_frame = gtk_frame_new( _("Read blocks") );
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->block_frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), dialog->block_frame, 1, 2, 1, 2,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(1, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(dialog->block_frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj3 = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->blocknum, 1., 1e6, 1., 10., 10.);
  dialog->skip_spin3 = gtk_spin_button_new(adj3, 0, 0);
  gtk_widget_set_usize(dialog->skip_spin3, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin3), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Block number") ),
                            0, 1, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin3,
                            1, 2, 0, 1);

  block_label=gtk_label_new( _("Start of block contains:") );
  dialog->block_entry = gtk_entry_new();
  gtk_widget_set_usize(dialog->block_entry, req.width/3, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            block_label,
                            0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->block_entry,
                            1, 2, 1, 2);



/* Worksheet options */
  frame = gtk_frame_new( _("Worsheet options") );
  gtk_frame_set_shadow_type (GTK_FRAME (frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), frame, 0, 2, 2, 3,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(2, 1, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  dialog->rename_columns_item = gtk_check_item_new_with_label(
                        _("Rename columns if 1st. line contains labels") );
  dialog->rename_worksheets_item = gtk_check_item_new_with_label(
                        _("Rename worksheet to data file name") );
/*
  precision_item = gtk_check_item_new_with_label(
                        "Modify decimal number precision on import");
*/

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->rename_columns_item,
                            0, 1, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->rename_worksheets_item,
                            0, 1, 1, 2);

/*
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            precision_item,
                            0, 1, 2, 3);
*/


  /* Action Area */
  action_area = gtk_hbutton_box_new ();
  gtk_button_box_set_layout(GTK_BUTTON_BOX(action_area), GTK_BUTTONBOX_SPREAD);
  gtk_button_box_set_spacing(GTK_BUTTON_BOX(action_area), 5);

  gtk_table_attach(GTK_TABLE(main_box), action_area, 0, 2, 3, 4,
                  (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		  (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);
  gtk_widget_show (action_area);

  dialog->ok_button = sg_pixmap_button (sg_stock_pixmap_widget_new(SG_STOCK_PIXMAP_OPEN), _("Import") );
  dialog->cancel_button = sg_stock_button ("Button_Cancel");

  GTK_WIDGET_SET_FLAGS (dialog->ok_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), dialog->ok_button, TRUE, TRUE, 0);
  gtk_widget_grab_default (dialog->ok_button);
  gtk_widget_show (dialog->ok_button);

  GTK_WIDGET_SET_FLAGS (dialog->cancel_button, GTK_CAN_DEFAULT);
  gtk_box_pack_start (GTK_BOX (action_area), dialog->cancel_button, TRUE, TRUE, 0);
  gtk_widget_show (dialog->cancel_button);
}

static gboolean 
sg_import_dialog_run(SGimportDialog *dialog)
{
  GtkWidget *window = GTK_WIDGET(dialog);

  /* Show widgets */

  gtk_widget_show_all (window);

  init_dialog(dialog);
  update_options(GTK_WIDGET(dialog), NULL);

  /* connect signals */
  gtk_signal_connect_object (GTK_OBJECT (dialog->cancel_button), "clicked",
                             GTK_SIGNAL_FUNC (cancel_clicked),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (dialog->delimiter_item), "toggled",
                             GTK_SIGNAL_FUNC (update_options),
                             GTK_OBJECT (window));

  gtk_signal_connect_object (GTK_OBJECT (dialog->endline_item), "toggled",
                             GTK_SIGNAL_FUNC (update_options),
                             GTK_OBJECT (window));


  gtk_signal_connect_object (GTK_OBJECT (GTK_COMBO(dialog->method_combo)->entry),
                             "changed",GTK_SIGNAL_FUNC (update_options),
                             GTK_OBJECT (window));


  gtk_signal_connect_object (GTK_OBJECT (dialog->ok_button), "clicked",
                             GTK_SIGNAL_FUNC (import_file), GTK_OBJECT(dialog));

  gtk_signal_connect (GTK_OBJECT (dialog), "destroy",
                      GTK_SIGNAL_FUNC (mw_destroy), NULL);

  gtk_main();
  return TRUE;
}


static gboolean
sg_import_dialog_import_block(SGimportDialog *dialog)
{
  GtkSheet *sheet;
  GArray *array = NULL;
  GPtrArray *titles = NULL, *sarray = NULL;
  PyArrayObject *parray = NULL;
  PyObject *list = NULL, *tlist = NULL;
  gint row, col, x, y, temp_line = 0;
  guint i;
  gchar delim[20]="";
  SGworksheet *worksheet = dialog->worksheet;
  gchar *file = dialog->file;

  sheet = GTK_SHEET(worksheet);
  row = 0;
  col = 0;

  if (dialog->use_custom_delimiter)
      strncat(delim, dialog->custom_delimiter, 20);
  else
      switch (dialog->delimiter)
      {
          case SG_DELIMITER_COMMA:
             strncat(delim,",",20);
             break;
          default:
             break;
      }

  strncat(delim, " \t\n", 20);
  gtk_sheet_get_active_cell(sheet, &row, &col);

  if (dialog->precision_mod)
  {  
     if (dialog->read_method==0)
     {
         if (!dialog->read_all_lines) temp_line=dialog->end_line;
         array = read_table(file, dialog->comment_string, delim, NULL, 0, dialog->begin_line, temp_line, &x, &y, &titles);
     }
     else
     {
         array = read_table(file, dialog->comment_string, delim, dialog->block_start, dialog->blocknum, 0, 0, &x, &y, &titles);
     }

     if (array)
     {
         parray = (PyArrayObject *)python_read_build_array(array, x, y);
         Py_INCREF(parray);
     }
   
     if (!array || !parray)
     {
       g_warning( _("ERROR: Could not import data from: %s"), file);
       return FALSE;
     }
   
     python_insert_object(worksheet, row, col, (PyObject *)parray,
                          GTK_ORIENTATION_VERTICAL, TRUE, FALSE);
     g_array_free(array, TRUE);
     Py_XDECREF(parray);
  } /* !dialog->precision_mod*/
  else
  { 
     if (dialog->read_method == 0)
     {
         if (!dialog->read_all_lines) temp_line = dialog->end_line;
         sarray = read_table_string(file, dialog->comment_string, delim, NULL, 0, dialog->begin_line, temp_line, &x, &y, &titles);
     }
     else
     {
         sarray = read_table_string(file, dialog->comment_string, delim, dialog->block_start, dialog->blocknum, 0, 0, &x, &y, &titles);
     }
     if (sarray)
     {
         list = python_read_build_list(sarray, x, y);
         Py_INCREF(list);
     }
   
     if (!sarray || !list)
     {
       g_warning( _("ERROR: Could not import data from: %s"), file);
       return FALSE;
     }
   
     python_insert_object(worksheet, 
                          (dialog->rename_columns ? row : (titles && titles->len > 0 ? row + 1 : row)), 
                          col, list,
                          GTK_ORIENTATION_VERTICAL, TRUE, TRUE);
     g_ptr_array_free_strings(sarray, TRUE, TRUE);

     if(titles && titles->len > 0 && dialog->rename_columns)
        for (i = 0; i < titles->len; i++)
         { 
             gtk_sheet_column_button_add_label(sheet, col+i, (gchar*)titles->pdata[i]);
             gtk_sheet_set_column_title(sheet, col+i, (gchar*)titles->pdata[i]);
         }

     if (!dialog->rename_columns && titles && titles->len > 0)
     {
         tlist = python_read_build_list(titles, x, 1);
         Py_INCREF(tlist);
         python_insert_object(worksheet, row, col, tlist,
                              GTK_ORIENTATION_VERTICAL, TRUE, TRUE);
         Py_XDECREF(tlist);
     }
  }

  if (titles) g_ptr_array_free_strings(titles, TRUE, TRUE);
  Py_XDECREF(list);

  return TRUE;
}

gboolean
sg_import_dialog (const gchar *path, SGworksheet *worksheet)
{
  SGimportDialog *dialog;
  GtkWidget *widget;

  widget = sg_import_dialog_new(path, worksheet);
  dialog = SG_IMPORT_DIALOG(widget);
  return(sg_import_dialog_run(dialog));
}

