/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdlib.h>
#include <gtk/gtk.h>
#include <gtkextra/gtkextra.h>
#include <Python.h>
#include <scigraphica/sg.h>
#include <scigraphica/sg_python.h>
#include <scigraphica/sg_python_expr.h>
#include "sg_import_dialog.h"
#include "import_data.h"

typedef struct
{
  SGdelimiter delimiter;

  gchar *file;
  SGdataset *dataset;
  gboolean return_val;
  
  gint begin_line;
  gint end_line;
  gint blocknum;
  gboolean read_all_lines;
  gint read_method;
  gboolean use_custom_delimiter;
  gint skip_lines;
  
  gchar *comment_string;
  gchar *block_start;
  gchar *custom_delimiter;
  
  GtkWidget *delimiter_item;
  GtkWidget *delimiter_combo;
  GtkWidget *method_combo;
  GtkWidget *delimiter_entry;
  GtkWidget *comment_entry;
  
  GtkWidget *block_entry;
  GtkWidget *skip_spin;
  GtkWidget *skip_spin2;
  GtkWidget *skip_spin3;
  
  GtkWidget *lines_frame;
  GtkWidget *block_frame;
  
  GtkWidget *endline_item;
  
  GtkPlotArrayList *data;

} SGdatasetImportDialog;

static void sg_dataset_import_dialog_init         	(SGdatasetImportDialog *dialog);
static void sg_dataset_import_dialog_destroy      	(SGdatasetImportDialog *dialog);
static void sg_dataset_import_dialog_apply      	(SGpropertyDialog *dialog, gpointer data);
static void sg_dataset_import_dialog_cancel      	(SGpropertyDialog *dialog, gpointer data);
static void sg_dataset_import_dialog_init_gui		(GtkWidget *frame, SGdatasetImportDialog *dialog);
static GtkPlotArrayList* sg_dataset_import_dialog_import_block	(SGdatasetImportDialog *dialog);

static void
sg_dataset_import_dialog_init(SGdatasetImportDialog *dialog)
{
  dialog->delimiter = SG_DELIMITER_UNKNOWN;
  dialog->begin_line = 1;
  dialog->end_line = 1;
  dialog->blocknum = 1;
  dialog->read_all_lines = TRUE;
  dialog->read_method = 0;
  dialog->use_custom_delimiter = FALSE;
  dialog->skip_lines = 0;
  dialog->file = NULL;
  
  dialog->comment_string = g_strdup("#");
  dialog->block_start = g_strdup("#");
  dialog->custom_delimiter = g_strdup("&");
  
  dialog->data = NULL;
}

static void
sg_dataset_import_dialog_destroy(SGdatasetImportDialog *dialog)
{
  if(dialog->comment_string) g_free(dialog->comment_string);
  dialog->comment_string = NULL;
  if(dialog->block_start) g_free(dialog->block_start);
  dialog->block_start = NULL;
  if(dialog->custom_delimiter) g_free(dialog->custom_delimiter);
  dialog->custom_delimiter = NULL;
  if(dialog->file) g_free(dialog->file);
  dialog->file = NULL;
  if(dialog->data) g_object_unref(G_OBJECT(dialog->data));
  dialog->data = NULL;
  g_free(dialog);
}

static void
update_options(GtkWidget *widget, gpointer data)
{ 
  GtkWidget *child;
  SGdatasetImportDialog *dialog = (SGdatasetImportDialog *)data;

  dialog->use_custom_delimiter = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->delimiter_item));
  if (dialog->use_custom_delimiter)
  {   
      gtk_widget_set_sensitive(dialog->delimiter_entry,TRUE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,FALSE);
  }
  else
  {   
      gtk_widget_set_sensitive(dialog->delimiter_entry,FALSE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,TRUE);
  }

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list)->selection->data;
  dialog->delimiter = (SGdelimiter)gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list), child);

  if(dialog->custom_delimiter) g_free(dialog->custom_delimiter);
  dialog->custom_delimiter = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->delimiter_entry)));
  if(dialog->comment_string) g_free(dialog->comment_string);
  dialog->comment_string = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->comment_entry)));
  if(dialog->block_start) g_free(dialog->block_start);
  dialog->block_start = g_strdup(gtk_entry_get_text(GTK_ENTRY(dialog->block_entry)));

  dialog->begin_line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin));
  dialog->end_line = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin2));
  dialog->blocknum = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(dialog->skip_spin3));

  child = (GtkWidget *)GTK_LIST(GTK_COMBO(dialog->method_combo)->list)->selection->data;
  dialog->read_method = gtk_list_child_position(GTK_LIST(GTK_COMBO(dialog->method_combo)->list), child);

  dialog->read_all_lines = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(dialog->endline_item));

  if (dialog->read_all_lines)
     gtk_widget_set_sensitive(dialog->skip_spin2,FALSE);
  else
     gtk_widget_set_sensitive(dialog->skip_spin2,TRUE);

  switch (dialog->read_method)
  { 
    case 0:
      gtk_widget_set_sensitive(dialog->lines_frame,TRUE);
      gtk_widget_set_sensitive(dialog->block_frame,FALSE);
      break;
    case 1:
      gtk_widget_set_sensitive(dialog->lines_frame,FALSE);
      gtk_widget_set_sensitive(dialog->block_frame,TRUE);
      break;
  }

}

static void
sg_dataset_import_dialog_apply(SGpropertyDialog *d, gpointer data)
{ 
  SGdatasetImportDialog *dialog = (SGdatasetImportDialog *)data;

  update_options(NULL, data);
  if (dialog->file){
    GtkPlotArrayList *arrays;
    arrays = sg_dataset_import_dialog_import_block(dialog);
    dialog->return_val = sg_edit_file_dialog(dialog->dataset, dialog->dataset->constructor->layer, arrays);
    if(dialog->return_val){   
      gchar *c = &dialog->file[strlen(dialog->file)];
      sg_dataset_set_description(dialog->dataset, dialog->file);
     
      for(; c && *c != G_DIR_SEPARATOR; c--) {}
      if(!c) 
        c = dialog->file;
      else 
        c++;
      sg_dataset_set_name(dialog->dataset, c);
    }
  }
}

static void
sg_dataset_import_dialog_cancel(SGpropertyDialog *d, gpointer data)
{ 
  SGdatasetImportDialog *dialog = (SGdatasetImportDialog *)data;
  dialog->return_val = FALSE;
}


static void
init_dialog(SGdatasetImportDialog *dialog)
{
  gchar *delimiters[] = { _("unknown"),
                          _("comma(,)"),
                          _("tab"),
                          _("space"),
                          NULL };
  gchar *methods[] = { _("lines"),
                       _("blocks"),
                       NULL };

  sg_combo_set_items(GTK_COMBO(dialog->delimiter_combo), delimiters);
  sg_combo_set_items(GTK_COMBO(dialog->method_combo), methods);
    
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->delimiter_combo)->list), dialog->delimiter);
  gtk_list_select_item(GTK_LIST(GTK_COMBO(dialog->method_combo)->list), dialog->read_method);

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin), dialog->skip_lines);

  gtk_entry_set_text(GTK_ENTRY(dialog->delimiter_entry), dialog->custom_delimiter);
  gtk_entry_set_text(GTK_ENTRY(dialog->block_entry), dialog->block_start);

  if(dialog->use_custom_delimiter)
  {
      gtk_widget_set_sensitive(dialog->delimiter_entry,TRUE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,FALSE);
  }
  else
  {
      gtk_widget_set_sensitive(dialog->delimiter_entry,FALSE);
      gtk_widget_set_sensitive(dialog->delimiter_combo,TRUE);
  }


  gtk_entry_set_text(GTK_ENTRY(dialog->comment_entry), dialog->comment_string);

  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->endline_item),
                              dialog->read_all_lines);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(dialog->delimiter_item),
                              dialog->use_custom_delimiter);

  switch (dialog->read_method)
  { case 0:
      gtk_widget_set_sensitive(dialog->lines_frame,TRUE);
      gtk_widget_set_sensitive(dialog->block_frame,FALSE);
      break;
    case 1:
      gtk_widget_set_sensitive(dialog->lines_frame,FALSE);
      gtk_widget_set_sensitive(dialog->block_frame,TRUE);
      break;
  }

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin),(gfloat)dialog->begin_line);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin2),(gfloat)dialog->end_line);
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(dialog->skip_spin3),(gfloat)dialog->blocknum);

}


static void
sg_dataset_import_dialog_init_gui(GtkWidget *frame, SGdatasetImportDialog *dialog)
{
  GtkWidget *file_frame;
  GtkWidget *main_box;
  GtkWidget *table;
  GtkWidget *comment_label;
  GtkWidget *delimiter_label;
  GtkWidget *method_label;
  GtkWidget *block_label;
  GtkAdjustment *adj,*adj2,*adj3;
  GtkRequisition req;
  gchar name[80];

  dialog->return_val = FALSE;

  /* Create widgets */
  main_box= gtk_table_new(2, 3, FALSE);
  gtk_container_add(GTK_CONTAINER(frame), main_box);
  gtk_frame_set_shadow_type (GTK_FRAME (frame), 0);

  g_snprintf(name,80, _("File structure") );
  file_frame = gtk_frame_new(name);
  gtk_frame_set_shadow_type (GTK_FRAME (file_frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), file_frame, 0, 2, 0, 1,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(3, 4, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(file_frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  dialog->delimiter_item = gtk_check_item_new_with_label( _("Use custom:") );
  dialog->delimiter_combo = gtk_combo_new();
  dialog->delimiter_entry = gtk_entry_new();
  dialog->comment_entry = gtk_entry_new();
  comment_label=gtk_label_new( _("Comment (matches any character in string):") );
  delimiter_label=gtk_label_new( _("Delimiter:") );
  method_label=gtk_label_new( _("Read file as:") );
  dialog->method_combo = gtk_combo_new();

  gtk_widget_size_request(dialog->delimiter_entry, &req);
  gtk_widget_set_usize(dialog->delimiter_entry, 20, req.height);
  gtk_widget_set_usize(dialog->comment_entry, 20, req.height);
  gtk_widget_set_usize(dialog->delimiter_combo, req.width/3, req.height);
  gtk_widget_set_usize(dialog->method_combo, req.width/3, req.height);

  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->delimiter_combo)->entry), FALSE);
  gtk_entry_set_editable(GTK_ENTRY(GTK_COMBO(dialog->method_combo)->entry), FALSE);

  gtk_table_attach_defaults(GTK_TABLE(table), delimiter_label, 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_combo, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_item, 1, 2, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->delimiter_entry, 2, 3, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table), comment_label, 0, 2, 3, 4);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->comment_entry, 2, 3, 3, 4);

  gtk_table_attach_defaults(GTK_TABLE(table), method_label, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), dialog->method_combo, 1, 2, 0, 1);

 /* Import as lines */
  dialog->lines_frame = gtk_frame_new( _("Read lines") );
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->lines_frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), dialog->lines_frame, 0, 1, 1, 2,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(1, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(dialog->lines_frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->begin_line, 1., 1e6, 1., 10., 10.);
  dialog->skip_spin = gtk_spin_button_new(adj, 0, 0);   
  gtk_widget_set_usize(dialog->skip_spin, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Start line") ),
                            0, 1, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin,
                            1, 2, 0, 1);
  dialog->endline_item = gtk_check_item_new_with_label( _("Read all lines") );
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->endline_item,
                            0, 2, 1, 2);


  /* End line */
  adj2 = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->end_line, 1.,1e6, 1., 10., 10.);
  dialog->skip_spin2 = gtk_spin_button_new(adj2, 0, 0);
  gtk_widget_set_usize(dialog->skip_spin2, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin2), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("End line") ),
                            0, 1, 2, 3);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin2,
                            1, 2, 2, 3);


 /* Import as blocks */
  dialog->block_frame = gtk_frame_new( _("Read blocks") );
  gtk_frame_set_shadow_type (GTK_FRAME (dialog->block_frame), GTK_SHADOW_ETCHED_IN);

  gtk_table_attach(GTK_TABLE(main_box), dialog->block_frame, 1, 2, 1, 2,
                   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND),10,5);

  table = gtk_table_new(1, 2, FALSE);
  gtk_container_set_border_width(GTK_CONTAINER(table), 10);
  gtk_container_add(GTK_CONTAINER(dialog->block_frame), table);
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_table_set_col_spacings(GTK_TABLE(table), 5);

  adj3 = (GtkAdjustment *)gtk_adjustment_new((gfloat) dialog->blocknum, 1., 1e6, 1., 10., 10.);
  dialog->skip_spin3 = gtk_spin_button_new(adj3, 0, 0);
  gtk_widget_set_usize(dialog->skip_spin3, 40, req.height);

  gtk_entry_set_editable(GTK_ENTRY(dialog->skip_spin3), TRUE);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            gtk_label_new( _("Block number") ),
                            0, 1, 0, 1);

  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->skip_spin3,
                            1, 2, 0, 1);

  block_label=gtk_label_new( _("Start of block contains:") );
  dialog->block_entry = gtk_entry_new();
  gtk_widget_set_usize(dialog->block_entry, req.width/3, req.height);
  gtk_table_attach_defaults(GTK_TABLE(table),
                            block_label,
                            0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), 
                            dialog->block_entry,
                            1, 2, 1, 2);

  init_dialog(dialog);
  update_options(NULL, dialog);

  /* connect signals */
  gtk_signal_connect (GTK_OBJECT (dialog->delimiter_item), "toggled",
                      GTK_SIGNAL_FUNC (update_options),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (dialog->endline_item), "toggled",
                      GTK_SIGNAL_FUNC (update_options),
                      dialog);

  gtk_signal_connect (GTK_OBJECT (GTK_COMBO(dialog->method_combo)->entry),
                      "changed",GTK_SIGNAL_FUNC (update_options),
                      dialog);
}


static GtkPlotArrayList*
sg_dataset_import_dialog_import_block(SGdatasetImportDialog *dialog)
{
  GPtrArray *titles = NULL, *sarray = NULL;
  gint x, y, temp_line = 0;
  gint row, col;
  gchar delim[20]="";
  gchar *file = dialog->file;
  GtkPlotArrayList *arrays = NULL;
  gchar **svalues = NULL;
  
  arrays = GTK_PLOT_ARRAY_LIST(gtk_plot_array_list_new());

  if (dialog->use_custom_delimiter)
      strncat(delim, dialog->custom_delimiter, 20);
  else
      switch (dialog->delimiter)
      {
          case SG_DELIMITER_COMMA:
             strncat(delim,",",20);
             break;
          default:
             break;
      }

  strncat(delim, " \t\n", 20);

  if (dialog->read_method==0)
  {
    if (!dialog->read_all_lines) temp_line = dialog->end_line;
    sarray = read_table_string(file, dialog->comment_string, delim, NULL, 0, dialog->begin_line, temp_line, &x, &y, &titles);
  }
  else
  {
    sarray = read_table_string(file, dialog->comment_string, delim, dialog->block_start, dialog->blocknum, 0, 0, &x, &y, &titles);
  }

  if (!sarray)
  {
    g_warning( _("ERROR: Could not import data from: %s"), file);
    g_object_unref(G_OBJECT(arrays));
    return NULL;
  }

  svalues = (gchar **) sarray->pdata;
  for(col = 0; col < x; col++){
    gchar **values;
    gchar title[200];
    GtkPlotArray *dim;
    values = g_malloc(y*sizeof(gchar *));
    for(row = 0; row < y; row++) {
      values[row] = g_strdup(svalues[col + x * row]);
    }
    g_snprintf(title, 200, "Column %i", col);
    if(titles && titles->len > 0) 
      g_snprintf(title, 200, (gchar *)titles->pdata[col]);
    dim = GTK_PLOT_ARRAY(gtk_plot_array_new(title, values, y, GTK_TYPE_STRING, TRUE)); 
    gtk_plot_array_list_add(arrays, dim);
    g_object_unref(G_OBJECT(dim));
  };

  g_ptr_array_free_strings(sarray, TRUE, TRUE);
  if (titles) g_ptr_array_free_strings(titles, TRUE, TRUE);
  return arrays;
}

gboolean 
sg_dataset_import_dialog(const gchar *filename, SGdataset *dataset)
{
  SGdatasetImportDialog *dialog;
  GtkWidget *frame;
  GtkWidget *window;
  gboolean return_value = FALSE;
                                                                                
  dialog = g_new0(SGdatasetImportDialog, 1);
  dialog->dataset = dataset;
  dialog->return_val = FALSE;
                                                                                
  /* Create widgets */
                                                                                
  frame = sg_property_dialog_new();
  sg_property_dialog_set_data(SG_PROPERTY_DIALOG(frame), dialog, FALSE);
  SG_PROPERTY_DIALOG(frame)->ok = sg_dataset_import_dialog_apply;
  SG_PROPERTY_DIALOG(frame)->apply = sg_dataset_import_dialog_apply;
  SG_PROPERTY_DIALOG(frame)->cancel = sg_dataset_import_dialog_cancel;

  sg_dataset_import_dialog_init(dialog);
  sg_dataset_import_dialog_init_gui(frame, dialog);
  dialog->file = g_strdup(filename);

/********************************************************************/
  window = sg_dialog_new( _("SciGraphica: Import Text File"), GTK_ORIENTATION_VERTICAL, SG_BUTTON_OK|SG_BUTTON_CANCEL, GTK_BUTTONBOX_SPREAD);
  gtk_window_set_policy(GTK_WINDOW(window), FALSE, FALSE, FALSE);
  gtk_window_position (GTK_WINDOW(window), GTK_WIN_POS_CENTER);
  gtk_widget_show_all(frame);
                                                                                
  sg_dialog_add(window, SG_PROPERTY_DIALOG(frame));
  gtk_widget_show_all(frame);
  sg_dialog_run(window, NULL);
                                                                                
  return_value = dialog->return_val;
  sg_dataset_import_dialog_destroy(dialog);
  return return_value;
}

