/*  SciGraphica - Scientific graphics and data manipulation
 *  Copyright (C) 2001 Adrian E. Feiguin <feiguin@ifir.edu.ar>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <gtkextra/gtkextra.h>
#include <scigraphica/sg.h>
#include "pixmaps/sg_small.xpm"
#include <libxml/xmlreader.h>

typedef struct
{
  gchar *last_node;
  FILE *stream;
  SGdataset *dataset;
} parser_state;

void
sg_dataset_xml_open_init(SGplugin *_plugin)
{
  SGpluginFile *plugin = SG_PLUGIN_FILE(_plugin);
  GdkPixmap *pixmap;
  GdkBitmap *mask;

  plugin->mode = SG_PLUGIN_FILE_OPEN;
  plugin->nfilters = 1;
  sprintf(plugin->filter[0],".sg");

  pixmap = gdk_pixmap_colormap_create_from_xpm_d(NULL, gdk_colormap_get_system(), &mask, NULL, sg_small_xpm);
  plugin->pixmap = GTK_PIXMAP(gtk_pixmap_new(pixmap, mask));
  gdk_pixmap_unref(pixmap);
  gdk_bitmap_unref(mask);
}

static void
process_node(xmlTextReaderPtr reader, parser_state *state)
{
  xmlChar *name = xmlTextReaderName(reader);
  GObject *object;

  /* Start Element */
  if(xmlTextReaderNodeType(reader) == 1){

    if(strcmp(name,"sgp:Iterator") == 0){ 
      gchar *iterator_name = NULL;
      gchar *constructor = NULL;
      SGpluginIterator *iterator = NULL;
      SGpluginStyle *style = NULL;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Name") == 0) iterator_name = g_strdup(value);
        if(strcmp(child, "Constructor") == 0) constructor = g_strdup(value);

        xmlFree(child);
        xmlFree(value);
      }

      if(iterator_name && constructor){
	iterator = sg_plugin_iterator_get(iterator_name);
	style = sg_plugin_style_get(constructor);
        if(iterator && style){
          SGpluginFile *iterator_plugin = NULL;
          SGpluginFile *style_plugin = NULL;
	  GObject *child = NULL;

 	  state->dataset = SG_DATASET(sg_plugin_iterator_construct(iterator, style));
	  object = G_OBJECT(state->dataset);

          iterator_plugin = sg_plugin_file_get("xml",iterator_name,SG_PLUGIN_FILE_OPEN);

          if(!iterator_plugin)
            g_warning("Unknown method to read %s\n",iterator_name);
          else{
            iterator_plugin->action(iterator_plugin, NULL, 
                                    state->stream, &object, reader);
          }

/*
          style_plugin = sg_plugin_file_get("xml",constructor,SG_PLUGIN_FILE_OPEN);
          if(!style_plugin)
            g_warning("Unknown method to read %s\n",constructor);
          else{
	    child = G_OBJECT(sg_dataset_new_child(state->dataset));
	    sg_dataset_add_child(state->dataset, GTK_PLOT_DATA(child));
            style_plugin->action(style_plugin, NULL, 
                                 state->stream, &child, reader);
          }
*/

        } else {
          g_warning("Plugin %s with constructor %s not found\n", iterator_name, constructor);
        }
      }
      
      if(iterator_name) g_free(iterator_name);
      if(constructor) g_free(constructor);
      iterator_name = constructor = NULL;
    }

    if(strcmp(name,"sgp:Style") == 0){ 
      gchar *constructor = NULL;
      SGpluginFile *style_plugin = NULL;
      GObject *child = NULL;

      while(xmlTextReaderMoveToNextAttribute(reader)){
        xmlChar *child = xmlTextReaderName(reader);
        xmlChar *value = xmlTextReaderValue(reader);

        if(strcmp(child, "Name") == 0) constructor = g_strdup(value);

        xmlFree(child);
        xmlFree(value);
      }

      if(constructor){
        style_plugin = sg_plugin_file_get("xml",constructor,SG_PLUGIN_FILE_OPEN);
        if(!style_plugin){
          g_warning("Unknown method to read %s\n",constructor);
        }else{
          child = G_OBJECT(sg_dataset_new_child(state->dataset));
	  sg_dataset_add_child(state->dataset, GTK_PLOT_DATA(child));
          sg_plugin_file_action(style_plugin, NULL, 
                               state->stream, &child, reader);
        }

        g_free(constructor);
        constructor = NULL;
      }
    }

    if(state->last_node) g_free(state->last_node);
    state->last_node = g_strdup(name);
  }

  /* End Element */

  if(xmlTextReaderNodeType(reader) == 15){
    if(state->last_node) g_free(state->last_node);
    state->last_node = NULL;
  }

  xmlFree(name);
}

gboolean 
sg_dataset_xml_open   (SGpluginFile *plugin, 
		       const gchar *filename, 
		       FILE *stream,
                       GObject **object, gpointer data)
{  
  xmlTextReaderPtr reader;
  gint ret_val;
  parser_state *state;

  state = g_new0(parser_state, 1);
  state->last_node = NULL;
  state->stream = stream;

  if(data)
    reader = (xmlTextReaderPtr)data;
  else
    reader = xmlNewTextReaderFilename(filename);

  if(!reader) return FALSE;

  ret_val = xmlTextReaderRead(reader);
  while(ret_val == 1){
    xmlChar *name = xmlTextReaderName(reader);

    process_node(reader, state);

    if(xmlTextReaderNodeType(reader) == 15 && strcmp(name, "sgp:Dataset") == 0){
      if(name) xmlFree(name);
      if(state->last_node) g_free(state->last_node);
      *object = G_OBJECT(state->dataset);
      return TRUE;
    }

    xmlFree(name);
    ret_val = xmlTextReaderRead(reader);
  }
  if(!data) xmlFreeTextReader(reader);
  if(state->last_node) g_free(state->last_node);
  g_free(state);

  if(ret_val != 0) return FALSE;

  return TRUE;
}


