/*
 * ksi_gc.c
 * gc function
 *
 * Copyright (C) 1997-2010, ivan demakov
 *
 * The software is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 *
 * The software is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with the software; see the file COPYING.LESSER.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 *
 *
 * Author:        Ivan Demakov <ksion@users.sourceforge.net>
 * Creation date: Wed Nov 26 23:14:46 1997
 * Last Update:   Fri Jan 22 20:07:20 2010
 *
 */


#include "ksi_gc.h"
#include "ksi_int.h"

#if WIN32
#  define GC_THREADS 1
#  include <gc/gc.h>
#else
#  include <gc/gc.h>
#endif

#include <gmp.h>


void*
ksi_malloc (size_t sz)
{
    return GC_malloc(sz);
}

void*
ksi_malloc_data (size_t sz)
{
    return GC_malloc_atomic_ignore_off_page(sz);
}

void*
ksi_malloc_eternal (size_t sz)
{
    return GC_malloc_uncollectable(sz);
}

void*
ksi_realloc (void* ptr, size_t sz)
{
    return GC_realloc(ptr, sz);
}

void
ksi_free (void* ptr)
{
    GC_free(ptr);
}

void*
ksi_base_ptr(void* ptr)
{
    return GC_base(ptr);
}

void
ksi_register_finalizer (void* obj, ksi_finalizer_function proc, void* data)
{
    GC_finalization_proc old_proc;
    void* old_data;

    GC_register_finalizer_ignore_self (obj, proc, data, &old_proc, &old_data);
}

int
ksi_expand_heap (unsigned sz)
{
    return GC_expand_hp(sz);
}

void
ksi_set_max_heap (unsigned sz)
{
    GC_set_max_heap_size(sz);
}

unsigned
ksi_get_heap_size (void)
{
    return GC_get_heap_size();
}

unsigned
ksi_get_heap_free (void)
{
    return GC_get_free_bytes();
}

unsigned
ksi_gcollections (void)
{
    return GC_gc_no;
}

int
ksi_gcollect (int full)
{
    if (full) {
        while (GC_collect_a_little());
        GC_gcollect();
        return 0;
    }

    return GC_collect_a_little();
}

void
ksi_enable_gc (void)
{
//  GC_dont_gc = 0;
    GC_enable();
}

void
ksi_disable_gc (void)
{
//  GC_dont_gc = 1;
    GC_disable();
}

void
ksi_set_gc_warn(ksi_gc_warn_function warn)
{
    GC_set_warn_proc(warn);
}

static void
ksi_gc_warn (char* msg, GC_word arg)
{
    if (ksi_int_data) {
        ksi_warn (msg, arg);
    } else {
        static const char *wrn = "GC warning";
#ifdef WIN32
        MessageBox (NULL, msg, wrn, MB_TASKMODAL | MB_ICONEXCLAMATION | MB_OK);
#else
        write (2, wrn, strlen (wrn));
        write (2, msg, strlen (msg));
        write (2, "\n", 1);
#endif
    }
}

static void*
alloc_gmp(size_t sz)
{
    return GC_malloc(sz);
}

static void*
realloc_gmp(void *ptr, size_t old_sz, size_t new_sz)
{
    return GC_realloc(ptr, new_sz);
}

static void
free_gmp(void *ptr, size_t sz)
{
    GC_free(ptr);
}


void
ksi_init_gc (void)
{
    static int inited = 0;
    if (inited)
        return;
    inited++;

#if defined(WIN32) && defined(GC_DLL)
    GC_use_DllMain();
#endif

    GC_INIT();
    GC_set_warn_proc (ksi_gc_warn);

#if GC_INCREMENTAL
    GC_enable_incremental ();
#endif

    mp_set_memory_functions(alloc_gmp, realloc_gmp, free_gmp);
//    GC_expand_hp(1024*1024);
}


/* End of code */
