
#line 1 "./scanner.rl"
/*  Copyright (C) 2011 CZ.NIC, z.s.p.o. <knot-dns@labs.nic.cz>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <config.h>
#include <stdint.h>			// uint32_t
#include <stdlib.h>			// calloc
#include <stdio.h>			// sprintf
#include <libgen.h>			// dirname
#include <stdbool.h>			// bool
#include <math.h>			// pow
#include <string.h>			// strdup
#include <sys/types.h>			// (OpenBSD)
#include <sys/socket.h>			// AF_INET (BSD)
#include <netinet/in.h>			// in_addr (BSD)
#include <arpa/inet.h>			// inet_pton

#include "zscanner/scanner.h"
#include "zscanner/error.h"		// error codes
#include "zscanner/file_loader.h"	// file_loader
#include "zscanner/scanner_functions.h"	// Base64
#include "zscanner/descriptor.h"	// KNOT_RRTYPE_A

/*! \brief Shorthand for setting warning data. */
#define WARN(code) { s->error_code = code; }
/*! \brief Shorthand for setting error data. */
#define ERR(code)   { s->error_code = code; s->stop = true; }

/*!
 * \brief Empty function which is called if no callback function is specified.
 */
static inline void noop(const scanner_t *s)
{
	(void)s;
}

/*!
 * \brief Writes record type number to r_data.
 *
 * \param type		Type number.
 * \param rdata_tail	Position where to write type number to.
 */
static inline void type_num(const uint16_t type, uint8_t **rdata_tail)
{
	*((uint16_t *)*rdata_tail) = htons(type);
	*rdata_tail += 2;
}

/*!
 * \brief Sets bit to bitmap window.
 *
 * \param type		Type number.
 * \param s		Scanner context.
 */
static inline void window_add_bit(const uint16_t type, scanner_t *s) {
	uint8_t win      = type / 256;
	uint8_t bit_pos  = type % 256;
	uint8_t byte_pos = bit_pos / 8;

	((s->windows[win]).bitmap)[byte_pos] |= 128 >> (bit_pos % 8);

	if ((s->windows[win]).length < byte_pos + 1) {
		(s->windows[win]).length = byte_pos + 1;
	}

	if (s->last_window < win) {
		s->last_window = win;
	}
}

// Include scanner file (in Ragel).

#line 88 "scanner.c"
static const short _zone_scanner_actions[] = {
	0, 1, 0, 1, 1, 1, 2, 1, 
	3, 1, 4, 1, 6, 1, 9, 1, 
	10, 1, 11, 1, 13, 1, 14, 1, 
	16, 1, 19, 1, 20, 1, 22, 1, 
	23, 1, 25, 1, 26, 1, 27, 1, 
	29, 1, 30, 1, 39, 1, 40, 1, 
	41, 1, 43, 1, 45, 1, 46, 1, 
	47, 1, 48, 1, 50, 1, 52, 1, 
	53, 1, 55, 1, 57, 1, 59, 1, 
	60, 1, 64, 1, 65, 1, 68, 1, 
	69, 1, 71, 1, 72, 1, 75, 1, 
	77, 1, 78, 1, 79, 1, 80, 1, 
	81, 1, 82, 1, 83, 1, 84, 1, 
	86, 1, 88, 1, 90, 1, 94, 1, 
	95, 1, 99, 1, 100, 1, 104, 1, 
	105, 1, 106, 1, 107, 1, 108, 1, 
	109, 1, 110, 1, 111, 1, 112, 1, 
	113, 1, 114, 1, 115, 1, 116, 1, 
	117, 1, 119, 1, 120, 1, 121, 1, 
	122, 1, 147, 1, 148, 1, 149, 1, 
	150, 1, 151, 1, 152, 1, 153, 1, 
	154, 1, 155, 1, 156, 1, 157, 1, 
	158, 1, 159, 1, 160, 1, 161, 1, 
	162, 1, 163, 1, 164, 1, 165, 1, 
	166, 1, 167, 1, 168, 1, 169, 1, 
	170, 1, 171, 1, 172, 1, 173, 1, 
	174, 1, 175, 1, 176, 1, 177, 1, 
	178, 1, 179, 1, 180, 1, 181, 1, 
	182, 1, 183, 1, 184, 1, 185, 1, 
	186, 1, 187, 1, 188, 1, 189, 1, 
	190, 1, 191, 1, 192, 1, 193, 1, 
	194, 1, 195, 1, 196, 1, 197, 1, 
	198, 1, 199, 1, 200, 1, 201, 1, 
	202, 1, 203, 1, 204, 1, 205, 1, 
	206, 1, 207, 1, 208, 1, 209, 1, 
	210, 1, 211, 1, 212, 1, 213, 1, 
	214, 1, 215, 1, 216, 1, 217, 1, 
	218, 1, 219, 1, 220, 1, 221, 1, 
	222, 1, 223, 1, 224, 1, 226, 1, 
	228, 1, 229, 1, 230, 1, 231, 1, 
	238, 1, 239, 1, 244, 1, 246, 1, 
	251, 1, 254, 1, 255, 1, 256, 1, 
	257, 1, 259, 1, 260, 1, 261, 1, 
	263, 2, 0, 43, 2, 1, 0, 2, 
	1, 26, 2, 1, 260, 2, 1, 302, 
	2, 2, 26, 2, 3, 26, 2, 4, 
	64, 2, 4, 68, 2, 5, 6, 2, 
	9, 10, 2, 12, 13, 2, 14, 10, 
	2, 14, 11, 2, 15, 20, 2, 15, 
	68, 2, 15, 75, 2, 16, 0, 2, 
	16, 2, 2, 16, 3, 2, 16, 74, 
	2, 18, 0, 2, 19, 9, 2, 22, 
	62, 2, 22, 99, 2, 22, 110, 2, 
	23, 0, 2, 23, 1, 2, 23, 2, 
	2, 23, 3, 2, 23, 227, 2, 24, 
	21, 2, 25, 1, 2, 25, 2, 2, 
	25, 3, 2, 25, 84, 2, 26, 1, 
	2, 26, 2, 2, 26, 3, 2, 27, 
	4, 2, 27, 259, 2, 27, 263, 2, 
	28, 21, 2, 29, 0, 2, 29, 1, 
	2, 29, 2, 2, 29, 3, 2, 29, 
	227, 2, 30, 35, 2, 31, 30, 2, 
	32, 64, 2, 32, 244, 2, 32, 259, 
	2, 32, 263, 2, 36, 232, 2, 36, 
	233, 2, 36, 234, 2, 36, 235, 2, 
	36, 236, 2, 36, 237, 2, 39, 0, 
	2, 39, 1, 2, 39, 2, 2, 39, 
	3, 2, 40, 0, 2, 40, 1, 2, 
	40, 2, 2, 40, 3, 2, 41, 0, 
	2, 41, 1, 2, 41, 2, 2, 41, 
	3, 2, 42, 258, 2, 43, 1, 2, 
	43, 2, 2, 43, 3, 2, 50, 2, 
	2, 50, 3, 2, 50, 41, 2, 50, 
	84, 2, 51, 52, 2, 53, 1, 2, 
	53, 2, 2, 53, 3, 2, 54, 259, 
	2, 55, 0, 2, 56, 57, 2, 58, 
	59, 2, 60, 0, 2, 60, 55, 2, 
	66, 19, 2, 70, 62, 2, 71, 2, 
	2, 71, 3, 2, 72, 4, 2, 73, 
	19, 2, 75, 4, 2, 79, 1, 2, 
	79, 26, 2, 81, 82, 2, 83, 1, 
	2, 83, 2, 2, 83, 3, 2, 84, 
	1, 2, 84, 2, 2, 84, 3, 2, 
	85, 86, 2, 87, 259, 2, 88, 89, 
	2, 90, 91, 2, 92, 93, 2, 92, 
	94, 2, 92, 95, 2, 96, 97, 2, 
	98, 259, 2, 101, 259, 2, 102, 0, 
	2, 118, 259, 2, 120, 0, 2, 121, 
	0, 2, 122, 0, 2, 123, 0, 2, 
	124, 0, 2, 125, 0, 2, 126, 0, 
	2, 127, 0, 2, 128, 0, 2, 129, 
	0, 2, 130, 0, 2, 131, 0, 2, 
	132, 0, 2, 133, 0, 2, 134, 0, 
	2, 135, 0, 2, 136, 0, 2, 137, 
	0, 2, 138, 0, 2, 139, 0, 2, 
	140, 0, 2, 141, 0, 2, 142, 0, 
	2, 143, 0, 2, 144, 259, 2, 145, 
	259, 2, 146, 259, 2, 147, 1, 2, 
	147, 2, 2, 147, 3, 2, 148, 1, 
	2, 148, 2, 2, 148, 3, 2, 149, 
	1, 2, 149, 2, 2, 149, 3, 2, 
	150, 1, 2, 150, 2, 2, 150, 3, 
	2, 151, 1, 2, 151, 2, 2, 151, 
	3, 2, 152, 1, 2, 152, 2, 2, 
	152, 3, 2, 153, 1, 2, 153, 2, 
	2, 153, 3, 2, 154, 1, 2, 154, 
	2, 2, 154, 3, 2, 155, 1, 2, 
	155, 2, 2, 155, 3, 2, 156, 1, 
	2, 156, 2, 2, 156, 3, 2, 157, 
	1, 2, 157, 2, 2, 157, 3, 2, 
	158, 1, 2, 158, 2, 2, 158, 3, 
	2, 159, 1, 2, 159, 2, 2, 159, 
	3, 2, 160, 1, 2, 160, 2, 2, 
	160, 3, 2, 161, 1, 2, 161, 2, 
	2, 161, 3, 2, 162, 1, 2, 162, 
	2, 2, 162, 3, 2, 163, 1, 2, 
	163, 2, 2, 163, 3, 2, 164, 1, 
	2, 164, 2, 2, 164, 3, 2, 165, 
	1, 2, 165, 2, 2, 165, 3, 2, 
	166, 1, 2, 166, 2, 2, 166, 3, 
	2, 167, 1, 2, 167, 2, 2, 167, 
	3, 2, 168, 1, 2, 168, 2, 2, 
	168, 3, 2, 169, 1, 2, 169, 2, 
	2, 169, 3, 2, 170, 1, 2, 170, 
	2, 2, 170, 3, 2, 171, 1, 2, 
	171, 2, 2, 171, 3, 2, 172, 1, 
	2, 172, 2, 2, 172, 3, 2, 173, 
	1, 2, 173, 2, 2, 173, 3, 2, 
	174, 1, 2, 174, 2, 2, 174, 3, 
	2, 175, 1, 2, 175, 2, 2, 175, 
	3, 2, 176, 1, 2, 176, 2, 2, 
	176, 3, 2, 177, 1, 2, 177, 2, 
	2, 177, 3, 2, 178, 1, 2, 178, 
	2, 2, 178, 3, 2, 179, 1, 2, 
	179, 2, 2, 179, 3, 2, 180, 1, 
	2, 180, 2, 2, 180, 3, 2, 181, 
	1, 2, 181, 2, 2, 181, 3, 2, 
	182, 1, 2, 182, 2, 2, 182, 3, 
	2, 183, 1, 2, 183, 2, 2, 183, 
	3, 2, 184, 1, 2, 184, 2, 2, 
	184, 3, 2, 185, 1, 2, 185, 2, 
	2, 185, 3, 2, 186, 1, 2, 186, 
	2, 2, 186, 3, 2, 187, 1, 2, 
	187, 2, 2, 187, 3, 2, 188, 1, 
	2, 188, 2, 2, 188, 3, 2, 189, 
	1, 2, 189, 2, 2, 189, 3, 2, 
	190, 1, 2, 190, 2, 2, 190, 3, 
	2, 191, 1, 2, 191, 2, 2, 191, 
	3, 2, 192, 1, 2, 192, 2, 2, 
	192, 3, 2, 193, 1, 2, 193, 2, 
	2, 193, 3, 2, 194, 1, 2, 194, 
	2, 2, 194, 3, 2, 195, 1, 2, 
	195, 2, 2, 195, 3, 2, 196, 1, 
	2, 196, 2, 2, 196, 3, 2, 197, 
	1, 2, 197, 2, 2, 197, 3, 2, 
	198, 1, 2, 198, 2, 2, 198, 3, 
	2, 199, 1, 2, 199, 2, 2, 199, 
	3, 2, 200, 1, 2, 200, 2, 2, 
	200, 3, 2, 201, 1, 2, 201, 2, 
	2, 201, 3, 2, 202, 1, 2, 202, 
	2, 2, 202, 3, 2, 203, 1, 2, 
	203, 2, 2, 203, 3, 2, 204, 1, 
	2, 204, 2, 2, 204, 3, 2, 205, 
	1, 2, 205, 2, 2, 205, 3, 2, 
	206, 1, 2, 206, 2, 2, 206, 3, 
	2, 207, 1, 2, 207, 2, 2, 207, 
	3, 2, 208, 1, 2, 208, 2, 2, 
	208, 3, 2, 209, 1, 2, 209, 2, 
	2, 209, 3, 2, 210, 1, 2, 210, 
	2, 2, 210, 3, 2, 211, 1, 2, 
	211, 2, 2, 211, 3, 2, 212, 1, 
	2, 212, 2, 2, 212, 3, 2, 213, 
	1, 2, 213, 2, 2, 213, 3, 2, 
	214, 1, 2, 214, 2, 2, 214, 3, 
	2, 215, 1, 2, 215, 2, 2, 215, 
	3, 2, 216, 1, 2, 216, 2, 2, 
	216, 3, 2, 217, 1, 2, 217, 2, 
	2, 217, 3, 2, 218, 1, 2, 218, 
	2, 2, 218, 3, 2, 219, 1, 2, 
	219, 2, 2, 219, 3, 2, 220, 1, 
	2, 220, 2, 2, 220, 3, 2, 221, 
	1, 2, 221, 2, 2, 221, 3, 2, 
	222, 1, 2, 222, 2, 2, 222, 3, 
	2, 223, 1, 2, 223, 2, 2, 223, 
	3, 2, 224, 1, 2, 224, 2, 2, 
	224, 3, 2, 225, 0, 2, 228, 1, 
	2, 228, 2, 2, 228, 3, 2, 229, 
	1, 2, 229, 2, 2, 229, 3, 2, 
	230, 1, 2, 230, 2, 2, 230, 3, 
	2, 231, 1, 2, 231, 2, 2, 231, 
	3, 2, 238, 1, 2, 238, 2, 2, 
	238, 3, 2, 239, 1, 2, 239, 2, 
	2, 239, 3, 2, 242, 0, 2, 243, 
	259, 2, 245, 99, 2, 250, 99, 2, 
	259, 4, 2, 260, 1, 2, 260, 26, 
	2, 262, 260, 2, 263, 4, 2, 263, 
	259, 2, 264, 258, 2, 265, 258, 2, 
	266, 258, 2, 267, 258, 2, 268, 258, 
	2, 269, 258, 2, 270, 258, 2, 271, 
	258, 2, 272, 258, 2, 273, 258, 2, 
	274, 258, 2, 275, 258, 2, 276, 258, 
	2, 277, 258, 2, 278, 258, 2, 279, 
	258, 2, 280, 258, 2, 281, 258, 2, 
	282, 258, 2, 283, 258, 2, 284, 258, 
	2, 285, 258, 2, 286, 258, 2, 287, 
	258, 2, 288, 258, 2, 289, 258, 2, 
	290, 258, 2, 291, 258, 2, 292, 258, 
	2, 293, 258, 2, 294, 258, 2, 295, 
	258, 2, 296, 258, 2, 297, 258, 2, 
	298, 258, 2, 299, 258, 2, 300, 258, 
	2, 301, 258, 2, 302, 1, 3, 0, 
	43, 1, 3, 0, 43, 2, 3, 0, 
	43, 3, 3, 1, 76, 0, 3, 1, 
	102, 0, 3, 1, 225, 0, 3, 1, 
	242, 0, 3, 1, 260, 26, 3, 1, 
	260, 302, 3, 1, 302, 26, 3, 7, 
	1, 8, 3, 11, 17, 0, 3, 16, 
	74, 2, 3, 16, 74, 3, 3, 19, 
	9, 10, 3, 23, 1, 0, 3, 25, 
	1, 84, 3, 25, 1, 302, 3, 25, 
	2, 84, 3, 25, 3, 84, 3, 25, 
	262, 260, 3, 25, 264, 258, 3, 27, 
	80, 4, 3, 27, 259, 4, 3, 27, 
	263, 4, 3, 27, 263, 259, 3, 29, 
	1, 0, 3, 31, 30, 260, 3, 32, 
	144, 259, 3, 32, 146, 259, 3, 32, 
	263, 4, 3, 32, 263, 259, 3, 34, 
	36, 232, 3, 34, 36, 233, 3, 34, 
	36, 234, 3, 34, 36, 235, 3, 34, 
	36, 236, 3, 34, 36, 237, 3, 36, 
	232, 1, 3, 36, 232, 2, 3, 36, 
	232, 3, 3, 36, 233, 1, 3, 36, 
	233, 2, 3, 36, 233, 3, 3, 36, 
	234, 1, 3, 36, 234, 2, 3, 36, 
	234, 3, 3, 36, 235, 1, 3, 36, 
	235, 2, 3, 36, 235, 3, 3, 36, 
	236, 1, 3, 36, 236, 2, 3, 36, 
	236, 3, 3, 36, 237, 1, 3, 36, 
	237, 2, 3, 36, 237, 3, 3, 42, 
	258, 1, 3, 42, 258, 2, 3, 42, 
	258, 3, 3, 42, 258, 260, 3, 44, 
	32, 64, 3, 44, 32, 259, 3, 44, 
	32, 263, 3, 49, 31, 30, 3, 50, 
	41, 0, 3, 50, 41, 1, 3, 50, 
	41, 2, 3, 50, 41, 3, 3, 50, 
	84, 1, 3, 50, 84, 2, 3, 50, 
	84, 3, 3, 60, 55, 0, 3, 61, 
	56, 57, 3, 63, 0, 1, 3, 66, 
	19, 9, 3, 67, 0, 1, 3, 73, 
	19, 9, 3, 76, 0, 1, 3, 79, 
	24, 21, 3, 79, 26, 1, 3, 79, 
	26, 2, 3, 79, 26, 3, 3, 81, 
	24, 21, 3, 81, 82, 260, 3, 84, 
	1, 302, 3, 87, 98, 259, 3, 87, 
	144, 259, 3, 88, 89, 0, 3, 88, 
	89, 1, 3, 88, 89, 2, 3, 88, 
	89, 3, 3, 90, 91, 0, 3, 90, 
	91, 1, 3, 90, 91, 2, 3, 90, 
	91, 3, 3, 96, 97, 0, 3, 96, 
	97, 1, 3, 96, 97, 2, 3, 96, 
	97, 3, 3, 101, 103, 244, 3, 101, 
	253, 259, 3, 144, 145, 259, 3, 185, 
	225, 0, 3, 186, 225, 0, 3, 187, 
	225, 0, 3, 188, 225, 0, 3, 189, 
	225, 0, 3, 190, 225, 0, 3, 191, 
	225, 0, 3, 192, 225, 0, 3, 193, 
	225, 0, 3, 194, 225, 0, 3, 195, 
	225, 0, 3, 196, 225, 0, 3, 197, 
	225, 0, 3, 198, 225, 0, 3, 199, 
	225, 0, 3, 200, 225, 0, 3, 201, 
	225, 0, 3, 202, 225, 0, 3, 203, 
	225, 0, 3, 204, 225, 0, 3, 205, 
	225, 0, 3, 206, 225, 0, 3, 207, 
	225, 0, 3, 208, 225, 0, 3, 209, 
	225, 0, 3, 210, 225, 0, 3, 211, 
	225, 0, 3, 212, 225, 0, 3, 213, 
	225, 0, 3, 214, 225, 0, 3, 215, 
	225, 0, 3, 216, 225, 0, 3, 217, 
	225, 0, 3, 218, 225, 0, 3, 219, 
	225, 0, 3, 220, 225, 0, 3, 221, 
	225, 0, 3, 222, 225, 0, 3, 223, 
	225, 0, 3, 224, 225, 0, 3, 241, 
	31, 30, 3, 246, 247, 0, 3, 246, 
	248, 0, 3, 249, 101, 259, 3, 251, 
	252, 0, 3, 253, 101, 259, 3, 259, 
	4, 80, 3, 260, 24, 21, 3, 260, 
	302, 1, 3, 263, 259, 4, 3, 264, 
	258, 1, 3, 264, 258, 2, 3, 264, 
	258, 3, 3, 264, 258, 260, 3, 265, 
	258, 1, 3, 265, 258, 2, 3, 265, 
	258, 3, 3, 265, 258, 260, 3, 266, 
	258, 1, 3, 266, 258, 2, 3, 266, 
	258, 3, 3, 266, 258, 260, 3, 267, 
	258, 1, 3, 267, 258, 2, 3, 267, 
	258, 3, 3, 267, 258, 260, 3, 268, 
	258, 1, 3, 268, 258, 2, 3, 268, 
	258, 3, 3, 268, 258, 260, 3, 269, 
	258, 1, 3, 269, 258, 2, 3, 269, 
	258, 3, 3, 269, 258, 260, 3, 270, 
	258, 1, 3, 270, 258, 2, 3, 270, 
	258, 3, 3, 270, 258, 260, 3, 271, 
	258, 1, 3, 271, 258, 2, 3, 271, 
	258, 3, 3, 271, 258, 260, 3, 272, 
	258, 1, 3, 272, 258, 2, 3, 272, 
	258, 3, 3, 272, 258, 260, 3, 273, 
	258, 1, 3, 273, 258, 2, 3, 273, 
	258, 3, 3, 273, 258, 260, 3, 274, 
	258, 1, 3, 274, 258, 2, 3, 274, 
	258, 3, 3, 274, 258, 260, 3, 275, 
	258, 1, 3, 275, 258, 2, 3, 275, 
	258, 3, 3, 275, 258, 260, 3, 276, 
	258, 1, 3, 276, 258, 2, 3, 276, 
	258, 3, 3, 276, 258, 260, 3, 277, 
	258, 1, 3, 277, 258, 2, 3, 277, 
	258, 3, 3, 277, 258, 260, 3, 278, 
	258, 1, 3, 278, 258, 2, 3, 278, 
	258, 3, 3, 278, 258, 260, 3, 279, 
	258, 1, 3, 279, 258, 2, 3, 279, 
	258, 3, 3, 279, 258, 260, 3, 280, 
	258, 1, 3, 280, 258, 2, 3, 280, 
	258, 3, 3, 280, 258, 260, 3, 281, 
	258, 1, 3, 281, 258, 2, 3, 281, 
	258, 3, 3, 281, 258, 260, 3, 282, 
	258, 1, 3, 282, 258, 2, 3, 282, 
	258, 3, 3, 282, 258, 260, 3, 283, 
	258, 1, 3, 283, 258, 2, 3, 283, 
	258, 3, 3, 283, 258, 260, 3, 284, 
	258, 1, 3, 284, 258, 2, 3, 284, 
	258, 3, 3, 284, 258, 260, 3, 285, 
	258, 1, 3, 285, 258, 2, 3, 285, 
	258, 3, 3, 285, 258, 260, 3, 286, 
	258, 1, 3, 286, 258, 2, 3, 286, 
	258, 3, 3, 286, 258, 260, 3, 287, 
	258, 1, 3, 287, 258, 2, 3, 287, 
	258, 3, 3, 287, 258, 260, 3, 288, 
	258, 1, 3, 288, 258, 2, 3, 288, 
	258, 3, 3, 288, 258, 260, 3, 289, 
	258, 1, 3, 289, 258, 2, 3, 289, 
	258, 3, 3, 289, 258, 260, 3, 290, 
	258, 1, 3, 290, 258, 2, 3, 290, 
	258, 3, 3, 290, 258, 260, 3, 291, 
	258, 1, 3, 291, 258, 2, 3, 291, 
	258, 3, 3, 291, 258, 260, 3, 292, 
	258, 1, 3, 292, 258, 2, 3, 292, 
	258, 3, 3, 292, 258, 260, 3, 293, 
	258, 1, 3, 293, 258, 2, 3, 293, 
	258, 3, 3, 293, 258, 260, 3, 294, 
	258, 1, 3, 294, 258, 2, 3, 294, 
	258, 3, 3, 294, 258, 260, 3, 295, 
	258, 1, 3, 295, 258, 2, 3, 295, 
	258, 3, 3, 295, 258, 260, 3, 296, 
	258, 1, 3, 296, 258, 2, 3, 296, 
	258, 3, 3, 296, 258, 260, 3, 297, 
	258, 1, 3, 297, 258, 2, 3, 297, 
	258, 3, 3, 297, 258, 260, 3, 298, 
	258, 1, 3, 298, 258, 2, 3, 298, 
	258, 3, 3, 298, 258, 260, 3, 299, 
	258, 1, 3, 299, 258, 2, 3, 299, 
	258, 3, 3, 299, 258, 260, 3, 300, 
	258, 1, 3, 300, 258, 2, 3, 300, 
	258, 3, 3, 300, 258, 260, 3, 301, 
	258, 1, 3, 301, 258, 2, 3, 301, 
	258, 3, 3, 301, 258, 260, 3, 302, 
	1, 26, 4, 5, 7, 1, 8, 4, 
	14, 11, 17, 0, 4, 16, 67, 0, 
	1, 4, 25, 1, 84, 302, 4, 25, 
	1, 262, 260, 4, 25, 1, 264, 258, 
	4, 25, 2, 262, 260, 4, 25, 2, 
	264, 258, 4, 25, 3, 262, 260, 4, 
	25, 3, 264, 258, 4, 25, 264, 258, 
	260, 4, 27, 32, 263, 4, 4, 27, 
	44, 32, 263, 4, 27, 263, 259, 4, 
	4, 31, 30, 24, 21, 4, 32, 263, 
	259, 4, 4, 34, 31, 30, 35, 4, 
	34, 36, 232, 1, 4, 34, 36, 232, 
	2, 4, 34, 36, 232, 3, 4, 34, 
	36, 233, 1, 4, 34, 36, 233, 2, 
	4, 34, 36, 233, 3, 4, 34, 36, 
	234, 1, 4, 34, 36, 234, 2, 4, 
	34, 36, 234, 3, 4, 34, 36, 235, 
	1, 4, 34, 36, 235, 2, 4, 34, 
	36, 235, 3, 4, 34, 36, 236, 1, 
	4, 34, 36, 236, 2, 4, 34, 36, 
	236, 3, 4, 34, 36, 237, 1, 4, 
	34, 36, 237, 2, 4, 34, 36, 237, 
	3, 4, 36, 234, 242, 0, 4, 36, 
	235, 242, 0, 4, 36, 236, 242, 0, 
	4, 36, 237, 242, 0, 4, 37, 33, 
	31, 30, 4, 38, 33, 31, 30, 4, 
	42, 258, 1, 260, 4, 44, 32, 4, 
	64, 4, 50, 49, 31, 30, 4, 50, 
	63, 0, 1, 4, 66, 19, 9, 10, 
	4, 71, 76, 0, 1, 4, 73, 19, 
	9, 10, 4, 81, 82, 24, 21, 4, 
	87, 144, 145, 259, 4, 88, 89, 1, 
	0, 4, 90, 91, 1, 0, 4, 96, 
	97, 1, 0, 4, 185, 1, 225, 0, 
	4, 186, 1, 225, 0, 4, 187, 1, 
	225, 0, 4, 188, 1, 225, 0, 4, 
	189, 1, 225, 0, 4, 190, 1, 225, 
	0, 4, 191, 1, 225, 0, 4, 192, 
	1, 225, 0, 4, 193, 1, 225, 0, 
	4, 194, 1, 225, 0, 4, 195, 1, 
	225, 0, 4, 196, 1, 225, 0, 4, 
	197, 1, 225, 0, 4, 198, 1, 225, 
	0, 4, 199, 1, 225, 0, 4, 200, 
	1, 225, 0, 4, 201, 1, 225, 0, 
	4, 202, 1, 225, 0, 4, 203, 1, 
	225, 0, 4, 204, 1, 225, 0, 4, 
	205, 1, 225, 0, 4, 206, 1, 225, 
	0, 4, 207, 1, 225, 0, 4, 208, 
	1, 225, 0, 4, 209, 1, 225, 0, 
	4, 210, 1, 225, 0, 4, 211, 1, 
	225, 0, 4, 212, 1, 225, 0, 4, 
	213, 1, 225, 0, 4, 214, 1, 225, 
	0, 4, 215, 1, 225, 0, 4, 216, 
	1, 225, 0, 4, 217, 1, 225, 0, 
	4, 218, 1, 225, 0, 4, 219, 1, 
	225, 0, 4, 220, 1, 225, 0, 4, 
	221, 1, 225, 0, 4, 222, 1, 225, 
	0, 4, 223, 1, 225, 0, 4, 224, 
	1, 225, 0, 4, 264, 258, 1, 260, 
	4, 264, 258, 1, 302, 4, 265, 258, 
	1, 260, 4, 266, 258, 1, 260, 4, 
	267, 258, 1, 260, 4, 268, 258, 1, 
	260, 4, 269, 258, 1, 260, 4, 270, 
	258, 1, 260, 4, 271, 258, 1, 260, 
	4, 272, 258, 1, 260, 4, 273, 258, 
	1, 260, 4, 274, 258, 1, 260, 4, 
	275, 258, 1, 260, 4, 276, 258, 1, 
	260, 4, 277, 258, 1, 260, 4, 278, 
	258, 1, 260, 4, 279, 258, 1, 260, 
	4, 280, 258, 1, 260, 4, 281, 258, 
	1, 260, 4, 282, 258, 1, 260, 4, 
	283, 258, 1, 260, 4, 284, 258, 1, 
	260, 4, 285, 258, 1, 260, 4, 286, 
	258, 1, 260, 4, 287, 258, 1, 260, 
	4, 288, 258, 1, 260, 4, 289, 258, 
	1, 260, 4, 290, 258, 1, 260, 4, 
	291, 258, 1, 260, 4, 292, 258, 1, 
	260, 4, 293, 258, 1, 260, 4, 294, 
	258, 1, 260, 4, 295, 258, 1, 260, 
	4, 296, 258, 1, 260, 4, 297, 258, 
	1, 260, 4, 298, 258, 1, 260, 4, 
	299, 258, 1, 260, 4, 300, 258, 1, 
	260, 4, 301, 258, 1, 260, 5, 16, 
	74, 76, 0, 1, 5, 25, 1, 264, 
	258, 260, 5, 25, 1, 264, 258, 302, 
	5, 27, 32, 263, 259, 4, 5, 31, 
	30, 260, 24, 21, 5, 34, 36, 234, 
	242, 0, 5, 34, 36, 235, 242, 0, 
	5, 34, 36, 236, 242, 0, 5, 34, 
	36, 237, 242, 0, 5, 36, 234, 1, 
	242, 0, 5, 36, 235, 1, 242, 0, 
	5, 36, 236, 1, 242, 0, 5, 36, 
	237, 1, 242, 0, 5, 44, 32, 263, 
	259, 4, 5, 81, 82, 260, 24, 21, 
	5, 240, 37, 33, 31, 30, 5, 264, 
	258, 1, 260, 302, 5, 264, 258, 260, 
	302, 1, 6, 25, 1, 264, 258, 260, 
	302, 6, 27, 44, 32, 263, 259, 4, 
	6, 34, 36, 234, 1, 242, 0, 6, 
	34, 36, 235, 1, 242, 0, 6, 34, 
	36, 236, 1, 242, 0, 6, 34, 36, 
	237, 1, 242, 0
};

static const short _zone_scanner_cond_offsets[] = {
	0, 0, 2, 4, 6, 8, 10, 12, 
	14, 14, 14, 17, 19, 22, 24, 26, 
	28, 28, 28, 28, 30, 35, 35, 35, 
	35, 37, 37, 37, 37, 37, 39, 39, 
	39, 39, 41, 41, 41, 41, 43, 45, 
	45, 45, 45, 45, 47, 47, 49, 49, 
	49, 49, 49, 51, 51, 53, 55, 55, 
	55, 55, 55, 55, 55, 55, 57, 57, 
	57, 59, 61, 61, 61, 63, 63, 65, 
	65, 67, 69, 69, 69, 69, 69, 71, 
	73, 73, 73, 73, 73, 75, 75, 77, 
	79, 79, 81, 83, 83, 83, 83, 83, 
	85, 85, 85, 87, 87, 89, 89, 89, 
	89, 91, 93, 93, 93, 95, 95, 97, 
	97, 99, 99, 99, 99, 101, 101, 101, 
	101, 103, 103, 105, 105, 105, 105, 107, 
	110, 112, 112, 114, 116, 118, 120, 120, 
	122, 125, 125, 125, 125, 125, 125, 125, 
	125, 125, 125, 127, 129, 132, 134, 136, 
	139, 141, 141, 143, 146, 148, 150, 152, 
	154, 156, 159, 161, 163, 165, 167, 169, 
	171, 173, 175, 177, 179, 181, 183, 185, 
	187, 189, 191, 194, 196, 196, 196, 196, 
	196, 196, 196, 196, 196, 196, 198, 200, 
	202, 205, 207, 209, 211, 213, 215, 217, 
	219, 221, 223, 225, 227, 229, 231, 233, 
	235, 237, 239, 244, 249, 254, 254, 254, 
	256, 256, 256, 256, 258, 258, 260, 263, 
	265, 267, 272, 277, 280, 285, 287, 289, 
	291, 293, 295, 297, 299, 301, 303, 305, 
	307, 309, 311, 314, 317, 319, 322, 322, 
	322, 322, 322, 322, 322, 322, 322, 322, 
	322, 322, 322, 322, 322, 322, 322, 322, 
	322, 323, 323, 323, 323, 323, 324, 326, 
	328, 330, 332, 332, 334, 334, 336, 339, 
	341, 343, 343, 345, 347, 347, 347, 347, 
	347, 347, 347, 349, 352, 354, 356, 358, 
	360, 360, 362, 364, 364, 364, 364, 364, 
	364, 364, 366, 369, 371, 374, 377, 377, 
	377, 377, 377, 379, 382, 382, 384, 386, 
	388, 388, 388, 390, 393, 393, 393, 393, 
	395, 395, 395, 395, 397, 397, 397, 397, 
	397, 399, 399, 399, 399, 401, 401, 401, 
	401, 403, 405, 405, 405, 405, 405, 407, 
	407, 409, 409, 409, 409, 409, 411, 411, 
	411, 411, 411, 411, 411, 411, 413, 413, 
	413, 415, 417, 417, 417, 419, 419, 421, 
	421, 423, 425, 425, 425, 425, 425, 427, 
	429, 429, 429, 429, 429, 431, 431, 433, 
	435, 435, 437, 439, 439, 439, 439, 439, 
	441, 441, 441, 443, 443, 445, 445, 445, 
	445, 447, 449, 449, 449, 451, 451, 453, 
	453, 455, 455, 455, 455, 457, 457, 457, 
	457, 459, 459, 461, 461, 461, 461, 463, 
	463, 463, 463, 465, 465, 467, 469, 471, 
	473, 475, 475, 477, 480, 483, 486, 488, 
	490, 492, 494, 494, 496, 499, 502, 504, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 507, 507, 
	507, 507, 507, 507, 507, 507, 509, 511, 
	513, 515, 517, 519, 521, 523, 525, 527, 
	529, 531, 531, 531, 531, 531, 534, 536, 
	536, 538, 541, 543, 543, 545, 548, 550, 
	550, 552, 555, 558, 561, 561, 563, 565, 
	565, 568, 568, 570, 572, 572, 575, 575, 
	577, 579, 579, 582, 582, 584, 586, 589, 
	589, 589, 589, 591, 593, 595, 597, 599, 
	601, 603, 605, 607, 609, 611, 613, 615, 
	617, 619, 621, 621, 623, 625, 627, 629, 
	631, 633, 635, 637, 640, 642, 644, 647, 
	649, 651, 653, 656, 658, 660, 662, 665, 
	667, 669, 671, 674, 676, 679, 681, 683, 
	686, 689, 692, 694, 697, 699, 701, 704, 
	707, 707, 709, 711, 713, 715, 717, 719, 
	719, 722, 725, 728, 728, 730, 732, 734, 
	736, 738, 740, 742, 744, 746, 748, 748, 
	751, 754, 757, 760, 763, 763, 765, 767, 
	769, 771, 773, 775, 777, 780, 783, 786, 
	788, 788, 788, 788, 788, 788, 790, 793, 
	793, 793, 793, 793, 795, 797, 799, 801, 
	803, 805, 805, 807, 810, 813, 816, 819, 
	819, 821, 823, 825, 827, 827, 829, 832, 
	835, 838, 838, 840, 842, 844, 846, 848, 
	850, 856, 867, 869, 872, 878, 881, 892, 
	895, 898, 901, 903, 905, 907, 909, 915, 
	918, 921, 923, 925, 927, 929, 935, 938, 
	941, 943, 945, 947, 949, 955, 958, 961, 
	964, 964, 966, 968, 970, 972, 974, 976, 
	978, 980, 982, 984, 986, 988, 990, 992, 
	994, 996, 998, 1001, 1004, 1007, 1010, 1013, 
	1016, 1019, 1022, 1022, 1022, 1024, 1024, 1024, 
	1024, 1026, 1026, 1028, 1028, 1028, 1028, 1030, 
	1030, 1030, 1030, 1032, 1032, 1032, 1032, 1032, 
	1034, 1034, 1034, 1034, 1036, 1036, 1036, 1036, 
	1038, 1040, 1040, 1040, 1040, 1040, 1042, 1042, 
	1044, 1044, 1044, 1044, 1044, 1046, 1046, 1046, 
	1046, 1046, 1046, 1046, 1046, 1048, 1048, 1048, 
	1050, 1052, 1052, 1052, 1054, 1054, 1056, 1056, 
	1058, 1060, 1060, 1060, 1060, 1060, 1062, 1064, 
	1064, 1064, 1064, 1064, 1066, 1066, 1068, 1070, 
	1070, 1072, 1074, 1074, 1074, 1074, 1074, 1076, 
	1076, 1076, 1078, 1078, 1080, 1080, 1080, 1080, 
	1082, 1084, 1084, 1084, 1086, 1086, 1088, 1088, 
	1090, 1090, 1090, 1090, 1092, 1092, 1092, 1092, 
	1094, 1094, 1096, 1096, 1096, 1096, 1098, 1098, 
	1098, 1098, 1098, 1100, 1102, 1104, 1106, 1108, 
	1110, 1112, 1115, 1118, 1121, 1121, 1123, 1123, 
	1125, 1127, 1129, 1131, 1133, 1135, 1137, 1139, 
	1139, 1139, 1139, 1139, 1139, 1139, 1139, 1139, 
	1139, 1139, 1139, 1139, 1139, 1139, 1142, 1142, 
	1144, 1147, 1150, 1153, 1153, 1155, 1157, 1159, 
	1161, 1163, 1165, 1165, 1165, 1165, 1168, 1171, 
	1174, 1174, 1176, 1178, 1180, 1182, 1184, 1186, 
	1186, 1188, 1191, 1194, 1197, 1200, 1200, 1202, 
	1204, 1204, 1207, 1207, 1209, 1211, 1211, 1211, 
	1211, 1211, 1211, 1211, 1211, 1211, 1211, 1214, 
	1214, 1214, 1214, 1214, 1214, 1214, 1214, 1214, 
	1214, 1214, 1214, 1214, 1216, 1218, 1220, 1222, 
	1224, 1226, 1228, 1230, 1232, 1234, 1234, 1234, 
	1234, 1234, 1234, 1234, 1234, 1236, 1236, 1238, 
	1241, 1241, 1243, 1246, 1246, 1248, 1251, 1253, 
	1253, 1255, 1258, 1261, 1261, 1261, 1261, 1261, 
	1261, 1261, 1261, 1261, 1261, 1263, 1266, 1266, 
	1266, 1268, 1271, 1273, 1276, 1278, 1281, 1283, 
	1286, 1286, 1286, 1286, 1286, 1288, 1291, 1291, 
	1293, 1296, 1296, 1298, 1301, 1301, 1307, 1310, 
	1321, 1324, 1335, 1338, 1338, 1338, 1338, 1338, 
	1338, 1338, 1338, 1340, 1343, 1343, 1343, 1343
};

static const char _zone_scanner_cond_lengths[] = {
	0, 2, 2, 2, 2, 2, 2, 2, 
	0, 0, 3, 2, 3, 2, 2, 2, 
	0, 0, 0, 2, 5, 0, 0, 0, 
	2, 0, 0, 0, 0, 2, 0, 0, 
	0, 2, 0, 0, 0, 2, 2, 0, 
	0, 0, 0, 2, 0, 2, 0, 0, 
	0, 0, 2, 0, 2, 2, 0, 0, 
	0, 0, 0, 0, 0, 2, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 2, 0, 0, 0, 0, 2, 2, 
	0, 0, 0, 0, 2, 0, 2, 2, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	0, 0, 2, 0, 2, 0, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	2, 0, 2, 0, 0, 0, 2, 3, 
	2, 0, 2, 2, 2, 2, 0, 2, 
	3, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 2, 3, 2, 2, 3, 
	2, 0, 2, 3, 2, 2, 2, 2, 
	2, 3, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 3, 2, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 2, 2, 2, 
	3, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 5, 5, 5, 0, 0, 2, 
	0, 0, 0, 2, 0, 2, 3, 2, 
	2, 5, 5, 3, 5, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 3, 3, 2, 3, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	1, 0, 0, 0, 0, 1, 2, 2, 
	2, 2, 0, 2, 0, 2, 3, 2, 
	2, 0, 2, 2, 0, 0, 0, 0, 
	0, 0, 2, 3, 2, 2, 2, 2, 
	0, 2, 2, 0, 0, 0, 0, 0, 
	0, 2, 3, 2, 3, 3, 0, 0, 
	0, 0, 2, 3, 0, 2, 2, 2, 
	0, 0, 2, 3, 0, 0, 0, 2, 
	0, 0, 0, 2, 0, 0, 0, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	2, 2, 0, 0, 0, 0, 2, 0, 
	2, 0, 0, 0, 0, 2, 0, 0, 
	0, 0, 0, 0, 0, 2, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 2, 0, 0, 0, 0, 2, 2, 
	0, 0, 0, 0, 2, 0, 2, 2, 
	0, 2, 2, 0, 0, 0, 0, 2, 
	0, 0, 2, 0, 2, 0, 0, 0, 
	2, 2, 0, 0, 2, 0, 2, 0, 
	2, 0, 0, 0, 2, 0, 0, 0, 
	2, 0, 2, 0, 0, 0, 2, 0, 
	0, 0, 2, 0, 2, 2, 2, 2, 
	2, 0, 2, 3, 3, 3, 2, 2, 
	2, 2, 0, 2, 3, 3, 2, 3, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 0, 0, 0, 0, 3, 2, 0, 
	2, 3, 2, 0, 2, 3, 2, 0, 
	2, 3, 3, 3, 0, 2, 2, 0, 
	3, 0, 2, 2, 0, 3, 0, 2, 
	2, 0, 3, 0, 2, 2, 3, 0, 
	0, 0, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 0, 2, 2, 2, 2, 2, 
	2, 2, 2, 3, 2, 2, 3, 2, 
	2, 2, 3, 2, 2, 2, 3, 2, 
	2, 2, 3, 2, 3, 2, 2, 3, 
	3, 3, 2, 3, 2, 2, 3, 3, 
	0, 2, 2, 2, 2, 2, 2, 0, 
	3, 3, 3, 0, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 0, 3, 
	3, 3, 3, 3, 0, 2, 2, 2, 
	2, 2, 2, 2, 3, 3, 3, 2, 
	0, 0, 0, 0, 0, 2, 3, 0, 
	0, 0, 0, 2, 2, 2, 2, 2, 
	2, 0, 2, 3, 3, 3, 3, 0, 
	2, 2, 2, 2, 0, 2, 3, 3, 
	3, 0, 2, 2, 2, 2, 2, 2, 
	6, 11, 2, 3, 6, 3, 11, 3, 
	3, 3, 2, 2, 2, 2, 6, 3, 
	3, 2, 2, 2, 2, 6, 3, 3, 
	2, 2, 2, 2, 6, 3, 3, 3, 
	0, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 3, 3, 3, 3, 3, 3, 
	3, 3, 0, 0, 2, 0, 0, 0, 
	2, 0, 2, 0, 0, 0, 2, 0, 
	0, 0, 2, 0, 0, 0, 0, 2, 
	0, 0, 0, 2, 0, 0, 0, 2, 
	2, 0, 0, 0, 0, 2, 0, 2, 
	0, 0, 0, 0, 2, 0, 0, 0, 
	0, 0, 0, 0, 2, 0, 0, 2, 
	2, 0, 0, 2, 0, 2, 0, 2, 
	2, 0, 0, 0, 0, 2, 2, 0, 
	0, 0, 0, 2, 0, 2, 2, 0, 
	2, 2, 0, 0, 0, 0, 2, 0, 
	0, 2, 0, 2, 0, 0, 0, 2, 
	2, 0, 0, 2, 0, 2, 0, 2, 
	0, 0, 0, 2, 0, 0, 0, 2, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	0, 0, 2, 2, 2, 2, 2, 2, 
	2, 3, 3, 3, 0, 2, 0, 2, 
	2, 2, 2, 2, 2, 2, 2, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 3, 0, 2, 
	3, 3, 3, 0, 2, 2, 2, 2, 
	2, 2, 0, 0, 0, 3, 3, 3, 
	0, 2, 2, 2, 2, 2, 2, 0, 
	2, 3, 3, 3, 3, 0, 2, 2, 
	0, 3, 0, 2, 2, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 3, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 0, 0, 0, 
	0, 0, 0, 0, 2, 0, 2, 3, 
	0, 2, 3, 0, 2, 3, 2, 0, 
	2, 3, 3, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 2, 3, 0, 0, 
	2, 3, 2, 3, 2, 3, 2, 3, 
	0, 0, 0, 0, 2, 3, 0, 2, 
	3, 0, 2, 3, 0, 6, 3, 11, 
	3, 11, 3, 0, 0, 0, 0, 0, 
	0, 0, 2, 3, 0, 0, 0, 0
};

static const short _zone_scanner_cond_keys[] = {
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 58, 59, 59, 60, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 58, 59, 59, 60, 127, 
	-128, 9, 10, 10, 11, 58, 59, 59, 
	60, 127, -128, 9, 10, 10, 11, 58, 
	59, 59, 60, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 58, 59, 59, 60, 127, 
	-128, 9, 10, 10, 11, 58, 59, 59, 
	60, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 58, 59, 59, 
	60, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 10, 10, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 9, 9, 10, 10, 
	32, 32, 40, 40, 41, 41, 59, 59, 
	9, 9, 10, 10, 32, 32, 40, 40, 
	41, 41, 43, 43, 47, 47, 48, 57, 
	59, 59, 65, 90, 97, 122, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	9, 9, 10, 10, 32, 32, 40, 40, 
	41, 41, 59, 59, -128, 9, 10, 10, 
	11, 127, 9, 9, 10, 10, 32, 32, 
	40, 40, 41, 41, 43, 43, 47, 47, 
	48, 57, 59, 59, 65, 90, 97, 122, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 9, 9, 10, 10, 32, 32, 
	40, 40, 41, 41, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 9, 9, 10, 10, 32, 32, 
	40, 40, 41, 41, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 9, 9, 10, 10, 32, 32, 
	40, 40, 41, 41, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, 10, 10, 59, 59, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, 10, 10, 59, 59, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 10, 10, 59, 59, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 10, 10, 
	59, 59, -128, 9, 10, 10, 11, 127, 
	10, 10, 59, 59, -128, 9, 10, 10, 
	11, 127, 9, 9, 10, 10, 32, 32, 
	40, 40, 41, 41, 59, 59, -128, 9, 
	10, 10, 11, 127, 9, 9, 10, 10, 
	32, 32, 40, 40, 41, 41, 43, 43, 
	47, 47, 48, 57, 59, 59, 65, 90, 
	97, 122, -128, 9, 10, 10, 11, 127, 
	9, 9, 10, 10, 32, 32, 40, 40, 
	41, 41, 43, 43, 47, 47, 48, 57, 
	59, 59, 65, 90, 97, 122, -128, 9, 
	10, 10, 11, 127, 10, 10, 59, 59, 
	-128, 9, 10, 10, 11, 127, 0
};

static const char _zone_scanner_cond_spaces[] = {
	0, 0, 0, 0, 0, 0, 5, 5, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 5, 5, 0, 0, 0, 5, 5, 
	0, 0, 0, 0, 5, 5, 0, 0, 
	0, 0, 0, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 0, 0, 0, 0, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 5, 5, 5, 5, 5, 5, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 5, 
	5, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 5, 5, 
	5, 5, 5, 5, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 2, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 5, 5, 0, 0, 0, 5, 5, 
	5, 5, 5, 5, 5, 5, 0, 0, 
	0, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 0, 0, 0, 0, 0, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 5, 5, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 5, 5, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 5, 
	5, 5, 5, 5, 5, 5, 5, 5, 
	5, 5, 5, 5, 5, 0, 0, 0, 
	5, 5, 5, 5, 0, 0, 0, 5, 
	5, 5, 5, 5, 5, 0, 0, 0, 
	5, 5, 5, 5, 5, 5, 0, 0, 
	0, 5, 5, 5, 5, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 1, 
	1, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 5, 5, 5, 5, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 8, 11, 8, 8, 8, 11, 
	3, 6, 3, 3, 3, 3, 3, 3, 
	6, 3, 3, 1, 1, 6, 6, 6, 
	4, 9, 4, 4, 4, 9, 7, 7, 
	7, 8, 11, 8, 8, 8, 3, 3, 
	3, 11, 3, 3, 10, 10, 10, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 8, 11, 8, 
	8, 8, 11, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 8, 11, 8, 8, 8, 11, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 8, 11, 8, 
	8, 8, 11, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 1, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 5, 5, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	5, 5, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 5, 5, 5, 5, 
	5, 5, 0, 0, 5, 5, 0, 0, 
	0, 5, 5, 0, 0, 0, 5, 5, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 0, 0, 0, 5, 5, 0, 
	0, 0, 5, 5, 0, 0, 0, 5, 
	5, 0, 0, 0, 5, 5, 0, 0, 
	0, 5, 5, 0, 0, 0, 5, 5, 
	0, 0, 0, 5, 5, 0, 0, 0, 
	5, 5, 0, 0, 0, 4, 9, 4, 
	4, 4, 9, 7, 7, 7, 3, 6, 
	3, 3, 3, 3, 3, 3, 6, 3, 
	3, 6, 6, 6, 8, 11, 8, 8, 
	8, 3, 3, 3, 11, 3, 3, 10, 
	10, 10, 5, 5, 0, 0, 0, 0
};

static const short _zone_scanner_key_offsets[] = {
	0, 0, 38, 56, 90, 106, 123, 131, 
	139, 140, 141, 144, 163, 169, 219, 239, 
	275, 279, 281, 283, 293, 307, 309, 311, 
	313, 323, 329, 331, 333, 335, 345, 349, 
	351, 353, 363, 365, 367, 369, 379, 389, 
	391, 393, 395, 396, 406, 407, 417, 419, 
	421, 423, 425, 435, 439, 445, 479, 481, 
	483, 485, 487, 489, 491, 493, 503, 507, 
	509, 519, 529, 535, 536, 546, 547, 557, 
	559, 569, 579, 583, 585, 587, 589, 599, 
	609, 615, 617, 619, 621, 631, 633, 643, 
	655, 657, 668, 680, 682, 684, 686, 688, 
	698, 700, 702, 712, 718, 728, 730, 732, 
	734, 744, 754, 762, 764, 774, 776, 786, 
	788, 798, 800, 802, 804, 814, 820, 822, 
	824, 834, 836, 846, 848, 850, 852, 864, 
	867, 905, 909, 915, 951, 969, 977, 989, 
	997, 1000, 1001, 1007, 1009, 1011, 1013, 1015, 
	1017, 1019, 1025, 1031, 1067, 1070, 1088, 1124, 
	1127, 1135, 1147, 1155, 1158, 1174, 1222, 1240, 
	1259, 1297, 1303, 1317, 1331, 1381, 1393, 1407, 
	1417, 1427, 1439, 1451, 1465, 1477, 1491, 1501, 
	1515, 1531, 1545, 1551, 1565, 1567, 1569, 1571, 
	1573, 1575, 1581, 1583, 1585, 1591, 1599, 1619, 
	1657, 1663, 1681, 1731, 1743, 1757, 1767, 1777, 
	1789, 1801, 1815, 1827, 1841, 1851, 1865, 1881, 
	1895, 1910, 1948, 1962, 1976, 1990, 1992, 1994, 
	2004, 2006, 2008, 2010, 2020, 2022, 2032, 2038, 
	2052, 2066, 2082, 2096, 2099, 2113, 2123, 2135, 
	2143, 2151, 2161, 2171, 2183, 2193, 2205, 2213, 
	2225, 2239, 2251, 2257, 2263, 2273, 2279, 2280, 
	2281, 2292, 2299, 2315, 2330, 2332, 2334, 2336, 
	2351, 2357, 2363, 2369, 2381, 2383, 2385, 2387, 
	2399, 2405, 2411, 2413, 2415, 2417, 2423, 2429, 
	2437, 2457, 2465, 2466, 2476, 2488, 2498, 2501, 
	2507, 2523, 2524, 2542, 2550, 2551, 2560, 2562, 
	2564, 2566, 2575, 2583, 2586, 2592, 2604, 2612, 
	2630, 2631, 2649, 2657, 2658, 2667, 2669, 2671, 
	2673, 2682, 2690, 2696, 2712, 2715, 2718, 2725, 
	2732, 2740, 2748, 2765, 2768, 2769, 2779, 2817, 
	2833, 2835, 2837, 2847, 2850, 2854, 2856, 2858, 
	2868, 2870, 2872, 2874, 2884, 2890, 2892, 2894, 
	2896, 2906, 2910, 2912, 2914, 2924, 2926, 2928, 
	2930, 2940, 2950, 2952, 2954, 2956, 2957, 2967, 
	2968, 2978, 2980, 2982, 2984, 2986, 2996, 2998, 
	3000, 3002, 3004, 3006, 3008, 3010, 3020, 3024, 
	3026, 3036, 3046, 3052, 3053, 3063, 3064, 3074, 
	3076, 3086, 3096, 3100, 3102, 3104, 3106, 3116, 
	3126, 3132, 3134, 3136, 3138, 3148, 3150, 3160, 
	3172, 3174, 3185, 3197, 3199, 3201, 3203, 3205, 
	3215, 3217, 3219, 3229, 3235, 3245, 3247, 3249, 
	3251, 3261, 3271, 3279, 3281, 3291, 3293, 3303, 
	3305, 3315, 3317, 3319, 3321, 3331, 3337, 3339, 
	3341, 3351, 3353, 3363, 3365, 3367, 3369, 3381, 
	3383, 3385, 3387, 3397, 3399, 3409, 3415, 3423, 
	3431, 3443, 3449, 3465, 3468, 3471, 3474, 3480, 
	3489, 3499, 3511, 3517, 3533, 3536, 3539, 3547, 
	3550, 3562, 3570, 3574, 3580, 3582, 3589, 3591, 
	3593, 3595, 3597, 3598, 3599, 3601, 3603, 3605, 
	3606, 3612, 3614, 3618, 3619, 3621, 3623, 3625, 
	3627, 3633, 3635, 3637, 3639, 3641, 3642, 3643, 
	3645, 3647, 3649, 3650, 3651, 3652, 3658, 3659, 
	3660, 3662, 3664, 3666, 3667, 3668, 3669, 3675, 
	3677, 3679, 3681, 3683, 3685, 3687, 3689, 3695, 
	3697, 3699, 3701, 3703, 3705, 3707, 3711, 3713, 
	3715, 3721, 3723, 3725, 3731, 3733, 3735, 3739, 
	3741, 3742, 3748, 3750, 3752, 3755, 3762, 3764, 
	3766, 3768, 3770, 3771, 3772, 3774, 3776, 3778, 
	3779, 3785, 3786, 3787, 3793, 3794, 3795, 3801, 
	3815, 3823, 3825, 3827, 3829, 3831, 3833, 3839, 
	3845, 3847, 3849, 3851, 3853, 3855, 3861, 3865, 
	3867, 3873, 3875, 3877, 3883, 3885, 3887, 3889, 
	3895, 3897, 3899, 3905, 3909, 3911, 3917, 3919, 
	3921, 3927, 3929, 3931, 3933, 3939, 3941, 3943, 
	3949, 3952, 3961, 3970, 3976, 3985, 3991, 4006, 
	4012, 4020, 4028, 4036, 4054, 4062, 4080, 4088, 
	4106, 4114, 4132, 4140, 4152, 4160, 4163, 4171, 
	4183, 4191, 4194, 4202, 4214, 4222, 4225, 4233, 
	4245, 4253, 4256, 4259, 4262, 4268, 4274, 4286, 
	4292, 4295, 4304, 4310, 4325, 4331, 4334, 4336, 
	4344, 4359, 4365, 4368, 4374, 4384, 4400, 4403, 
	4410, 4423, 4425, 4433, 4443, 4451, 4461, 4470, 
	4478, 4484, 4492, 4500, 4510, 4518, 4528, 4537, 
	4545, 4551, 4560, 4562, 4576, 4588, 4602, 4614, 
	4628, 4640, 4654, 4664, 4667, 4680, 4693, 4696, 
	4709, 4722, 4732, 4735, 4748, 4761, 4771, 4774, 
	4787, 4800, 4810, 4813, 4819, 4822, 4830, 4838, 
	4841, 4844, 4847, 4853, 4856, 4864, 4872, 4875, 
	4878, 4880, 4888, 4896, 4904, 4912, 4920, 4935, 
	4941, 4944, 4947, 4950, 4952, 4960, 4968, 4976, 
	4988, 4994, 5006, 5012, 5024, 5030, 5045, 5051, 
	5054, 5057, 5060, 5063, 5066, 5072, 5078, 5086, 
	5094, 5106, 5112, 5125, 5127, 5130, 5133, 5136, 
	5149, 5151, 5152, 5155, 5158, 5160, 5172, 5175, 
	5176, 5183, 5190, 5192, 5200, 5212, 5218, 5226, 
	5234, 5246, 5252, 5268, 5271, 5274, 5277, 5280, 
	5282, 5290, 5298, 5306, 5318, 5324, 5340, 5343, 
	5346, 5349, 5351, 5359, 5369, 5375, 5383, 5391, 
	5398, 5432, 5445, 5447, 5450, 5464, 5467, 5508, 
	5517, 5520, 5523, 5529, 5537, 5545, 5554, 5591, 
	5594, 5597, 5603, 5611, 5619, 5632, 5673, 5676, 
	5679, 5685, 5693, 5701, 5716, 5750, 5753, 5756, 
	5759, 5787, 5799, 5811, 5817, 5825, 5833, 5841, 
	5849, 5857, 5865, 5873, 5881, 5889, 5897, 5912, 
	5918, 5931, 5933, 5936, 5939, 5942, 5945, 5948, 
	5951, 5954, 5957, 5959, 5961, 5967, 5969, 5971, 
	5973, 5979, 5981, 5987, 5991, 5993, 5995, 6001, 
	6003, 6005, 6007, 6013, 6019, 6021, 6023, 6025, 
	6031, 6035, 6037, 6039, 6045, 6047, 6049, 6051, 
	6057, 6063, 6065, 6067, 6069, 6070, 6076, 6077, 
	6083, 6085, 6087, 6089, 6091, 6097, 6099, 6101, 
	6103, 6105, 6107, 6109, 6111, 6117, 6121, 6123, 
	6129, 6135, 6141, 6142, 6148, 6149, 6155, 6157, 
	6163, 6169, 6173, 6175, 6177, 6179, 6185, 6191, 
	6197, 6199, 6201, 6203, 6209, 6211, 6217, 6225, 
	6227, 6234, 6242, 6244, 6246, 6248, 6250, 6256, 
	6258, 6260, 6266, 6272, 6278, 6280, 6282, 6284, 
	6290, 6296, 6304, 6306, 6312, 6314, 6320, 6322, 
	6328, 6330, 6332, 6334, 6340, 6346, 6348, 6350, 
	6356, 6358, 6364, 6366, 6368, 6370, 6378, 6387, 
	6393, 6399, 6401, 6409, 6417, 6425, 6437, 6443, 
	6456, 6458, 6461, 6464, 6467, 6474, 6476, 6478, 
	6486, 6494, 6502, 6510, 6518, 6531, 6537, 6549, 
	6555, 6562, 6568, 6575, 6582, 6588, 6595, 6607, 
	6613, 6614, 6615, 6616, 6617, 6618, 6621, 6627, 
	6639, 6642, 6645, 6648, 6650, 6658, 6666, 6674, 
	6682, 6690, 6703, 6709, 6715, 6727, 6730, 6733, 
	6736, 6738, 6746, 6754, 6762, 6770, 6778, 6790, 
	6796, 6812, 6815, 6818, 6821, 6824, 6826, 6834, 
	6843, 6852, 6855, 6857, 6865, 6877, 6883, 6889, 
	6895, 6896, 6902, 6908, 6914, 6920, 6927, 6930, 
	6936, 6942, 6943, 6949, 6955, 6962, 6968, 6974, 
	6975, 6981, 6987, 6994, 7012, 7045, 7091, 7139, 
	7157, 7205, 7223, 7256, 7319, 7382, 7382, 7382, 
	7382, 7394, 7394, 7394, 7394, 7410, 7410, 7427, 
	7430, 7430, 7468, 7471, 7471, 7487, 7490, 7502, 
	7502, 7518, 7521, 7524, 7524, 7524, 7524, 7524, 
	7524, 7524, 7524, 7524, 7524, 7540, 7543, 7543, 
	7543, 7553, 7556, 7568, 7571, 7583, 7586, 7598, 
	7601, 7601, 7601, 7601, 7601, 7614, 7617, 7617, 
	7633, 7636, 7636, 7652, 7655, 7655, 7669, 7672, 
	7685, 7688, 7729, 7738, 7738, 7738, 7738, 7738, 
	7738, 7738, 7738, 7754, 7757, 7757, 7757, 7757
};

static const short _zone_scanner_trans_keys[] = {
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	778, 827, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	1034, 1083, 9, 32, 40, 41, 65, 70, 
	80, 97, 102, 112, 2058, 2107, 2314, 2363, 
	2570, 2619, 9, 32, 40, 41, 92, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 778, 
	827, 1034, 1083, 10, 35, 1034, 896, 1151, 
	9, 32, 40, 41, 92, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 778, 1034, 640, 895, 896, 
	1151, 9, 32, 40, 41, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1802, 1851, 2058, 2107, 2314, 
	2363, 2570, 2619, -128, 8, 11, 47, 48, 
	57, 60, 127, 9, 32, 40, 41, 68, 
	72, 77, 83, 87, 100, 104, 109, 115, 
	119, 778, 827, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 778, 
	827, 1034, 1083, 69, 78, 101, 110, 82, 
	114, 84, 116, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 777, 778, 800, 
	808, 809, 827, 1033, 1034, 1056, 1064, 1065, 
	1083, 896, 1151, 65, 97, 77, 109, 69, 
	101, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 72, 78, 83, 104, 110, 
	115, 67, 99, 73, 105, 68, 100, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 83, 97, 115, 77, 109, 69, 
	101, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 75, 107, 69, 101, 89, 
	121, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 85, 117, 73, 
	105, 52, 54, 56, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 52, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 73, 105, 78, 110, 70, 102, 79, 
	111, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 78, 80, 110, 112, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 65, 67, 68, 69, 72, 73, 75, 
	76, 77, 78, 80, 82, 83, 84, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1034, 1083, 80, 
	112, 83, 115, 69, 101, 67, 99, 75, 
	107, 69, 101, 89, 121, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 69, 
	88, 101, 120, 89, 121, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 51, 54, 79, 80, 111, 112, 50, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 52, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 67, 99, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 73, 88, 105, 120, 78, 
	110, 70, 102, 79, 111, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 73, 83, 97, 105, 115, 80, 
	112, 84, 116, 82, 114, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 68, 
	100, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 69, 
	101, 2058, 2107, 2314, 2363, 2570, 2619, 67, 
	99, 9, 32, 40, 41, 51, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	80, 112, 2058, 2107, 2314, 2363, 2570, 2619, 
	65, 97, 82, 114, 65, 97, 77, 109, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 84, 116, 82, 114, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	80, 82, 84, 112, 114, 116, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	83, 115, 73, 105, 71, 103, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 79, 80, 82, 83, 111, 112, 
	114, 115, 65, 97, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 70, 102, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 86, 118, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 72, 104, 
	70, 102, 80, 112, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 76, 88, 
	89, 108, 120, 121, 83, 115, 65, 97, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 84, 116, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 80, 112, 
	69, 101, 48, 57, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 78, 80, 110, 112, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 1034, 896, 1151, 
	10, 73, 79, 84, 105, 111, 116, 78, 
	110, 67, 99, 76, 108, 85, 117, 68, 
	100, 69, 101, 32, 59, 9, 10, 40, 
	41, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 1034, 
	1083, 48, 57, 1034, 896, 1151, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 65, 67, 68, 69, 
	72, 73, 75, 76, 77, 78, 80, 82, 
	83, 84, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	1034, 1083, 48, 57, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 68, 
	72, 77, 83, 87, 100, 104, 109, 115, 
	119, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 1034, 896, 1151, 9, 32, 
	40, 41, 65, 70, 80, 97, 102, 112, 
	2058, 2107, 2314, 2363, 2570, 2619, 9, 32, 
	40, 41, 58, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 97, 99, 100, 101, 104, 105, 
	107, 108, 109, 110, 112, 114, 115, 116, 
	2058, 2107, 2314, 2363, 2570, 2619, -128, 8, 
	11, 47, 48, 57, 60, 127, 9, 32, 
	40, 41, 65, 70, 80, 97, 102, 112, 
	1802, 1851, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 92, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 778, 827, 1034, 1083, -128, 
	8, 11, 58, 60, 127, 9, 32, 40, 
	41, 778, 827, 1034, 1083, -128, 8, 11, 
	58, 60, 127, 9, 32, 40, 41, 58, 
	65, 67, 68, 69, 72, 73, 75, 76, 
	77, 78, 80, 82, 83, 84, 92, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	47, 48, 57, 60, 127, 9, 32, 40, 
	41, 69, 78, 101, 110, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 72, 78, 83, 
	104, 110, 115, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 85, 117, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 73, 105, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 78, 
	80, 110, 112, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 69, 88, 101, 120, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 51, 
	54, 79, 80, 111, 112, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 73, 88, 105, 
	120, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 65, 73, 83, 97, 105, 115, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 84, 
	116, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 80, 82, 84, 112, 114, 116, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 79, 
	80, 82, 83, 111, 112, 114, 115, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 76, 
	88, 89, 108, 120, 121, 778, 827, 1034, 
	1083, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 73, 79, 84, 105, 111, 
	116, 778, 827, 1034, 1083, 82, 114, 73, 
	105, 71, 103, 73, 105, 78, 110, 32, 
	59, 9, 10, 40, 41, 84, 116, 76, 
	108, 32, 59, 9, 10, 40, 41, 9, 
	32, 40, 41, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 778, 827, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 778, 827, 1034, 1083, 48, 
	57, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 65, 70, 80, 97, 102, 
	112, 1802, 1851, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 1802, 1851, 2058, 2107, 2314, 
	2363, 2570, 2619, -128, 8, 11, 47, 48, 
	57, 60, 127, 9, 32, 40, 41, 69, 
	78, 101, 110, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 72, 78, 83, 104, 110, 
	115, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 85, 117, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 73, 105, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 78, 80, 110, 
	112, 778, 827, 1034, 1083, 9, 32, 40, 
	41, 69, 88, 101, 120, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 51, 54, 79, 
	80, 111, 112, 778, 827, 1034, 1083, 9, 
	32, 40, 41, 73, 88, 105, 120, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 65, 
	73, 83, 97, 105, 115, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 84, 116, 778, 
	827, 1034, 1083, 9, 32, 40, 41, 80, 
	82, 84, 112, 114, 116, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 79, 80, 82, 
	83, 111, 112, 114, 115, 778, 827, 1034, 
	1083, 9, 32, 40, 41, 76, 88, 89, 
	108, 120, 121, 778, 827, 1034, 1083, 9, 
	32, 35, 40, 41, 778, 827, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 9, 32, 
	40, 41, 65, 67, 68, 69, 72, 73, 
	75, 76, 77, 78, 80, 82, 83, 84, 
	97, 99, 100, 101, 104, 105, 107, 108, 
	109, 110, 112, 114, 115, 116, 778, 827, 
	1034, 1083, 48, 57, 777, 778, 800, 808, 
	809, 827, 1033, 1034, 1056, 1064, 1065, 1083, 
	896, 1151, 777, 778, 800, 808, 809, 827, 
	1033, 1034, 1056, 1064, 1065, 1083, 896, 1151, 
	777, 778, 800, 808, 809, 1033, 1034, 1056, 
	1064, 1065, 640, 895, 896, 1151, 65, 97, 
	65, 97, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 83, 115, 68, 100, 
	66, 98, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 76, 108, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	778, 1034, 640, 895, 896, 1151, 9, 32, 
	40, 41, 778, 827, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	778, 827, 1034, 1083, -128, 8, 11, 58, 
	60, 127, 777, 778, 800, 808, 809, 827, 
	1033, 1034, 1056, 1064, 1065, 1083, 640, 895, 
	896, 1151, 777, 778, 800, 808, 809, 1033, 
	1034, 1056, 1064, 1065, 640, 895, 896, 1151, 
	1034, 896, 1151, 777, 778, 800, 808, 809, 
	827, 1033, 1034, 1056, 1064, 1065, 1083, 896, 
	1151, 9, 32, 40, 41, 69, 78, 101, 
	110, 1034, 1083, 9, 32, 40, 41, 72, 
	78, 83, 104, 110, 115, 1034, 1083, 9, 
	32, 40, 41, 85, 117, 1034, 1083, 9, 
	32, 40, 41, 73, 105, 1034, 1083, 9, 
	32, 40, 41, 78, 80, 110, 112, 1034, 
	1083, 9, 32, 40, 41, 69, 88, 101, 
	120, 1034, 1083, 9, 32, 40, 41, 51, 
	54, 79, 80, 111, 112, 1034, 1083, 9, 
	32, 40, 41, 73, 88, 105, 120, 1034, 
	1083, 9, 32, 40, 41, 65, 73, 83, 
	97, 105, 115, 1034, 1083, 9, 32, 40, 
	41, 84, 116, 1034, 1083, 9, 32, 40, 
	41, 80, 82, 84, 112, 114, 116, 1034, 
	1083, 9, 32, 40, 41, 79, 80, 82, 
	83, 111, 112, 114, 115, 1034, 1083, 9, 
	32, 40, 41, 76, 88, 89, 108, 120, 
	121, 1034, 1083, 778, 1034, 640, 895, 896, 
	1151, 778, 1034, 640, 895, 896, 1151, 9, 
	32, 40, 41, 78, 80, 110, 112, 1034, 
	1083, 778, 1034, 640, 895, 896, 1151, 10, 
	10, 42, 46, 64, 92, 95, 45, 57, 
	65, 90, 97, 122, 32, 46, 59, 9, 
	10, 40, 41, 32, 42, 45, 59, 92, 
	95, 9, 10, 40, 41, 47, 57, 65, 
	90, 97, 122, 32, 46, 59, 92, 95, 
	9, 10, 40, 41, 45, 57, 65, 90, 
	97, 122, 48, 57, 48, 57, 48, 57, 
	32, 46, 59, 92, 95, 9, 10, 40, 
	41, 45, 57, 65, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 32, 59, 9, 
	10, 40, 41, 34, 92, 33, 58, 60, 
	126, 32, 33, 59, 92, 9, 10, 35, 
	39, 40, 41, 42, 126, 48, 57, 48, 
	57, 48, 57, 32, 33, 59, 92, 9, 
	10, 35, 39, 40, 41, 42, 126, 9, 
	34, 92, 522, 32, 126, 32, 59, 9, 
	10, 40, 41, 48, 57, 48, 57, 48, 
	57, 9, 34, 92, 522, 32, 126, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 778, 827, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 10, 9, 32, 40, 41, 778, 827, 
	1034, 1083, 48, 57, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 48, 57, 
	9, 32, 40, 41, 778, 827, 1034, 1083, 
	48, 57, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 42, 
	46, 92, 95, 1034, 1083, 45, 57, 65, 
	90, 97, 122, 46, 9, 32, 40, 41, 
	42, 45, 92, 95, 778, 827, 1034, 1083, 
	47, 57, 65, 90, 97, 122, 9, 32, 
	40, 41, 778, 827, 1034, 1083, 10, 46, 
	92, 95, 45, 57, 65, 90, 97, 122, 
	48, 57, 48, 57, 48, 57, 46, 92, 
	95, 45, 57, 65, 90, 97, 122, 9, 
	32, 40, 41, 778, 827, 1034, 1083, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	778, 827, 1034, 1083, 9, 32, 40, 41, 
	42, 46, 92, 95, 778, 827, 1034, 1083, 
	45, 57, 65, 90, 97, 122, 46, 9, 
	32, 40, 41, 42, 45, 92, 95, 778, 
	827, 1034, 1083, 47, 57, 65, 90, 97, 
	122, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 10, 46, 92, 95, 45, 57, 65, 
	90, 97, 122, 48, 57, 48, 57, 48, 
	57, 46, 92, 95, 45, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 778, 827, 
	1034, 1083, 778, 1034, 640, 895, 896, 1151, 
	9, 32, 40, 41, 42, 46, 92, 95, 
	1034, 1083, 45, 57, 65, 90, 97, 122, 
	1034, 896, 1151, 1034, 896, 1151, 43, 47, 
	57, 65, 90, 97, 122, 43, 47, 57, 
	65, 90, 97, 122, 43, 61, 47, 57, 
	65, 90, 97, 122, 43, 61, 47, 57, 
	65, 90, 97, 122, 9, 32, 40, 41, 
	43, 2058, 2107, 2314, 2363, 2570, 2619, 47, 
	57, 65, 90, 97, 122, 1034, 896, 1151, 
	61, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 9, 32, 40, 41, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 65, 70, 80, 
	97, 102, 112, 2058, 2107, 2314, 2363, 2570, 
	2619, 65, 97, 65, 97, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 1034, 
	896, 1151, 69, 78, 101, 110, 82, 114, 
	84, 116, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 65, 97, 77, 109, 
	69, 101, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 72, 78, 83, 104, 
	110, 115, 67, 99, 73, 105, 68, 100, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 65, 83, 97, 115, 77, 109, 
	69, 101, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 75, 107, 69, 101, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 85, 117, 
	73, 105, 52, 54, 56, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 52, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 73, 105, 78, 110, 70, 102, 
	79, 111, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 80, 112, 83, 115, 
	69, 101, 67, 99, 75, 107, 69, 101, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 69, 88, 101, 120, 
	89, 121, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 51, 54, 
	79, 80, 111, 112, 50, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 52, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 67, 99, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	73, 88, 105, 120, 78, 110, 70, 102, 
	79, 111, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 65, 73, 
	83, 97, 105, 115, 80, 112, 84, 116, 
	82, 114, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 68, 100, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 69, 101, 2058, 2107, 
	2314, 2363, 2570, 2619, 67, 99, 9, 32, 
	40, 41, 51, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 80, 112, 2058, 
	2107, 2314, 2363, 2570, 2619, 65, 97, 82, 
	114, 65, 97, 77, 109, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 84, 
	116, 82, 114, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 80, 82, 84, 
	112, 114, 116, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 83, 115, 73, 
	105, 71, 103, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 79, 
	80, 82, 83, 111, 112, 114, 115, 65, 
	97, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 70, 102, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 86, 
	118, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 72, 104, 70, 102, 80, 
	112, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 76, 88, 89, 108, 120, 
	121, 83, 115, 65, 97, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 84, 
	116, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 80, 112, 69, 101, 48, 
	57, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 83, 115, 68, 
	100, 66, 98, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 76, 108, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 65, 
	70, 97, 102, 48, 57, 65, 70, 97, 
	102, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 65, 70, 97, 
	102, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 48, 1034, 1083, 49, 
	57, 9, 32, 40, 41, 778, 827, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 48, 
	57, 65, 70, 97, 102, 9, 32, 40, 
	41, 2058, 2107, 2314, 2363, 2570, 2619, 48, 
	57, 65, 70, 97, 102, 1034, 896, 1151, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 1034, 896, 1151, 68, 69, 
	73, 80, 82, 100, 101, 105, 112, 114, 
	48, 57, 32, 59, 9, 10, 40, 41, 
	48, 57, 72, 83, 104, 115, 32, 59, 
	9, 10, 40, 41, 65, 97, 32, 45, 
	59, 9, 10, 40, 41, 78, 110, 83, 
	115, 69, 101, 67, 99, 51, 45, 83, 
	115, 72, 104, 65, 97, 49, 32, 59, 
	9, 10, 40, 41, 67, 99, 67, 68, 
	99, 100, 45, 71, 103, 79, 111, 83, 
	115, 84, 116, 32, 59, 9, 10, 40, 
	41, 83, 115, 65, 97, 80, 112, 50, 
	51, 53, 54, 83, 115, 72, 104, 65, 
	97, 50, 53, 54, 32, 59, 9, 10, 
	40, 41, 56, 52, 83, 115, 72, 104, 
	65, 97, 51, 56, 52, 32, 59, 9, 
	10, 40, 41, 78, 110, 68, 100, 73, 
	105, 82, 114, 69, 101, 67, 99, 84, 
	116, 32, 59, 9, 10, 40, 41, 82, 
	114, 73, 105, 86, 118, 65, 97, 84, 
	116, 69, 101, 68, 79, 100, 111, 78, 
	110, 83, 115, 32, 59, 9, 10, 40, 
	41, 73, 105, 68, 100, 32, 59, 9, 
	10, 40, 41, 83, 115, 65, 97, 77, 
	83, 109, 115, 68, 100, 53, 32, 59, 
	9, 10, 40, 41, 72, 104, 65, 97, 
	49, 50, 53, 32, 45, 59, 9, 10, 
	40, 41, 78, 110, 83, 115, 69, 101, 
	67, 99, 51, 45, 83, 115, 72, 104, 
	65, 97, 49, 32, 59, 9, 10, 40, 
	41, 53, 54, 32, 59, 9, 10, 40, 
	41, 49, 50, 32, 59, 9, 10, 40, 
	41, 65, 73, 79, 80, 83, 85, 97, 
	105, 111, 112, 115, 117, 48, 57, 32, 
	59, 9, 10, 40, 41, 48, 57, 67, 
	99, 80, 112, 75, 107, 73, 105, 88, 
	120, 32, 59, 9, 10, 40, 41, 65, 
	80, 83, 97, 112, 115, 67, 99, 80, 
	112, 75, 107, 73, 105, 88, 120, 32, 
	59, 9, 10, 40, 41, 71, 75, 103, 
	107, 80, 112, 32, 59, 9, 10, 40, 
	41, 73, 105, 88, 120, 32, 59, 9, 
	10, 40, 41, 80, 112, 75, 107, 73, 
	105, 32, 59, 9, 10, 40, 41, 73, 
	105, 68, 100, 32, 59, 9, 10, 40, 
	41, 71, 75, 103, 107, 80, 112, 32, 
	59, 9, 10, 40, 41, 73, 105, 88, 
	120, 32, 59, 9, 10, 40, 41, 80, 
	112, 75, 107, 73, 105, 32, 59, 9, 
	10, 40, 41, 82, 114, 73, 105, 32, 
	59, 9, 10, 40, 41, 46, 48, 57, 
	32, 46, 59, 9, 10, 40, 41, 48, 
	57, 42, 92, 95, 45, 57, 64, 90, 
	97, 122, 32, 59, 9, 10, 40, 41, 
	42, 92, 95, 45, 57, 64, 90, 97, 
	122, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 42, 92, 95, 1034, 1083, 
	45, 57, 64, 90, 97, 122, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	68, 72, 77, 83, 87, 100, 104, 109, 
	115, 119, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 32, 59, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 9, 10, 
	40, 41, 48, 57, 32, 59, 9, 10, 
	40, 41, 48, 57, 68, 72, 77, 83, 
	87, 100, 104, 109, 115, 119, 48, 57, 
	32, 59, 9, 10, 40, 41, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 68, 72, 77, 83, 87, 
	100, 104, 109, 115, 119, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 68, 72, 77, 83, 87, 100, 
	104, 109, 115, 119, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 1034, 896, 
	1151, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 68, 72, 77, 83, 87, 100, 104, 
	109, 115, 119, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 32, 59, 
	9, 10, 40, 41, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 32, 59, 
	9, 10, 40, 41, 1034, 896, 1151, 42, 
	92, 95, 45, 57, 64, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 42, 92, 95, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 32, 59, 9, 
	10, 40, 41, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 42, 92, 95, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 1034, 896, 1151, 
	32, 59, 9, 10, 40, 41, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, -128, 8, 11, 58, 60, 127, 
	1034, 896, 1151, 46, 48, 58, 65, 70, 
	97, 102, 32, 46, 59, 9, 10, 40, 
	41, 48, 58, 65, 70, 97, 102, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 78, 83, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 78, 
	83, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 46, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 78, 83, 1034, 1083, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	1034, 1083, 48, 57, 9, 32, 40, 41, 
	69, 87, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 69, 87, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 46, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 69, 87, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 45, 1034, 1083, 48, 57, 
	48, 57, 9, 32, 40, 41, 46, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	46, 109, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 9, 32, 
	40, 41, 46, 109, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 46, 109, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	1034, 896, 1151, 9, 32, 40, 41, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 109, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 1034, 896, 1151, 
	9, 32, 40, 41, 109, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 9, 32, 40, 
	41, 109, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 1034, 896, 1151, 9, 
	32, 40, 41, 109, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 9, 32, 40, 41, 
	109, 2058, 2107, 2314, 2363, 2570, 2619, 48, 
	57, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 1034, 896, 1151, 9, 32, 
	40, 41, 109, 2058, 2107, 2314, 2363, 2570, 
	2619, 48, 57, 9, 32, 40, 41, 109, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 1034, 896, 1151, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 1034, 896, 1151, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	1034, 896, 1151, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 42, 92, 95, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 32, 
	59, 9, 10, 40, 41, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, -128, 8, 
	11, 58, 60, 127, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	-128, 8, 11, 58, 60, 127, 9, 32, 
	40, 41, 1034, 1083, 9, 32, 40, 41, 
	1034, 1083, -128, 8, 11, 58, 60, 127, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 42, 92, 95, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 32, 59, 9, 
	10, 40, 41, 1034, 896, 1151, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 48, 57, 65, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 1034, 1083, 
	9, 32, 40, 41, 43, 1034, 1083, 47, 
	57, 65, 90, 97, 122, 1546, 1595, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	9, 32, 33, 40, 41, 49, 50, 2058, 
	2107, 2314, 2363, 2570, 2619, 49, 50, 58, 
	46, 48, 57, 47, 46, 57, 48, 57, 
	9, 32, 40, 41, 2058, 2107, 2314, 2363, 
	2570, 2619, 48, 57, 1034, 896, 1151, 58, 
	46, 48, 58, 65, 70, 97, 102, 47, 
	46, 58, 65, 70, 97, 102, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	9, 32, 40, 41, 1034, 1083, 48, 57, 
	65, 90, 97, 122, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 48, 49, 50, 51, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 9, 
	32, 40, 41, 46, 1034, 1083, 6153, 6176, 
	6184, 6185, 6409, 6432, 6440, 6441, 6665, 6688, 
	6696, 6697, 9482, 9531, 9738, 9787, 10250, 10299, 
	10506, 10555, 10762, 10811, 11274, 11323, 11530, 11579, 
	11786, 11835, 12298, 12347, 12554, 12603, 12810, 12859, 
	3081, 3104, 3112, 3113, 3115, 4106, 4155, 3119, 
	3129, 3137, 3162, 3169, 3194, 1546, 1595, 4106, 
	3968, 4223, 4617, 4640, 4648, 4649, 13578, 13627, 
	13834, 13883, 14346, 14395, 14602, 14651, 14858, 14907, 
	5642, 5504, 5759, 3115, 6153, 6176, 6184, 6185, 
	6409, 6432, 6440, 6441, 6665, 6688, 6696, 6697, 
	9482, 9531, 9738, 9787, 10250, 10299, 10506, 10555, 
	10762, 10811, 11274, 11323, 11530, 11579, 11786, 11835, 
	12298, 12347, 12554, 12603, 12810, 12859, 3119, 3129, 
	3137, 3162, 3169, 3194, 7690, 8202, 8714, 7552, 
	7807, 8064, 8319, 8576, 8831, 1034, 896, 1151, 
	1034, 896, 1151, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 46, 1034, 1083, 
	48, 57, 46, 6153, 6176, 6184, 6185, 6409, 
	6432, 6440, 6441, 6665, 6688, 6696, 6697, 9482, 
	9531, 9738, 9787, 10250, 10299, 10506, 10555, 10762, 
	10811, 11274, 11323, 11530, 11579, 11786, 11835, 12298, 
	12347, 12554, 12603, 12810, 12859, 48, 57, 1034, 
	896, 1151, 1034, 896, 1151, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 9, 32, 40, 41, 46, 
	1034, 1083, 48, 58, 65, 70, 97, 102, 
	46, 6153, 6176, 6184, 6185, 6409, 6432, 6440, 
	6441, 6665, 6688, 6696, 6697, 9482, 9531, 9738, 
	9787, 10250, 10299, 10506, 10555, 10762, 10811, 11274, 
	11323, 11530, 11579, 11786, 11835, 12298, 12347, 12554, 
	12603, 12810, 12859, 48, 58, 65, 70, 97, 
	102, 1034, 896, 1151, 1034, 896, 1151, 9, 
	32, 40, 41, 1034, 1083, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 1034, 1083, 48, 57, 9, 32, 40, 
	41, 42, 92, 95, 1034, 1083, 45, 57, 
	64, 90, 97, 122, 6153, 6176, 6184, 6185, 
	6409, 6432, 6440, 6441, 6665, 6688, 6696, 6697, 
	9482, 9531, 9738, 9787, 10250, 10299, 10506, 10555, 
	10762, 10811, 11274, 11323, 11530, 11579, 11786, 11835, 
	12298, 12347, 12554, 12603, 12810, 12859, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 97, 99, 100, 
	101, 104, 105, 107, 108, 109, 110, 112, 
	114, 115, 116, 9, 32, 40, 41, 65, 
	70, 80, 97, 102, 112, 1034, 1083, 9, 
	32, 40, 41, 1034, 1083, 48, 57, 65, 
	90, 97, 122, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 42, 92, 95, 
	1034, 1083, 45, 57, 64, 90, 97, 122, 
	9, 32, 40, 41, 1034, 1083, 9, 32, 
	40, 41, 43, 1034, 1083, 47, 57, 65, 
	90, 97, 122, 1546, 1595, 1034, 896, 1151, 
	1034, 896, 1151, 1034, 896, 1151, 1034, 896, 
	1151, 1034, 896, 1151, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 65, 97, 65, 
	97, 9, 32, 40, 41, 1034, 1083, 83, 
	115, 68, 100, 66, 98, 9, 32, 40, 
	41, 1034, 1083, 76, 108, 9, 32, 40, 
	41, 1034, 1083, 69, 78, 101, 110, 82, 
	114, 84, 116, 9, 32, 40, 41, 1034, 
	1083, 65, 97, 77, 109, 69, 101, 9, 
	32, 40, 41, 1034, 1083, 72, 78, 83, 
	104, 110, 115, 67, 99, 73, 105, 68, 
	100, 9, 32, 40, 41, 1034, 1083, 65, 
	83, 97, 115, 77, 109, 69, 101, 9, 
	32, 40, 41, 1034, 1083, 75, 107, 69, 
	101, 89, 121, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 85, 
	117, 73, 105, 52, 54, 56, 9, 32, 
	40, 41, 1034, 1083, 52, 9, 32, 40, 
	41, 1034, 1083, 73, 105, 78, 110, 70, 
	102, 79, 111, 9, 32, 40, 41, 1034, 
	1083, 80, 112, 83, 115, 69, 101, 67, 
	99, 75, 107, 69, 101, 89, 121, 9, 
	32, 40, 41, 1034, 1083, 69, 88, 101, 
	120, 89, 121, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 51, 
	54, 79, 80, 111, 112, 50, 9, 32, 
	40, 41, 1034, 1083, 52, 9, 32, 40, 
	41, 1034, 1083, 67, 99, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 1034, 
	1083, 73, 88, 105, 120, 78, 110, 70, 
	102, 79, 111, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 65, 
	73, 83, 97, 105, 115, 80, 112, 84, 
	116, 82, 114, 9, 32, 40, 41, 1034, 
	1083, 68, 100, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 69, 101, 1034, 
	1083, 67, 99, 9, 32, 40, 41, 51, 
	1034, 1083, 9, 32, 40, 41, 80, 112, 
	1034, 1083, 65, 97, 82, 114, 65, 97, 
	77, 109, 9, 32, 40, 41, 1034, 1083, 
	84, 116, 82, 114, 9, 32, 40, 41, 
	1034, 1083, 80, 82, 84, 112, 114, 116, 
	9, 32, 40, 41, 1034, 1083, 83, 115, 
	73, 105, 71, 103, 9, 32, 40, 41, 
	1034, 1083, 9, 32, 40, 41, 1034, 1083, 
	79, 80, 82, 83, 111, 112, 114, 115, 
	65, 97, 9, 32, 40, 41, 1034, 1083, 
	70, 102, 9, 32, 40, 41, 1034, 1083, 
	86, 118, 9, 32, 40, 41, 1034, 1083, 
	72, 104, 70, 102, 80, 112, 9, 32, 
	40, 41, 1034, 1083, 76, 88, 89, 108, 
	120, 121, 83, 115, 65, 97, 9, 32, 
	40, 41, 1034, 1083, 84, 116, 9, 32, 
	40, 41, 1034, 1083, 80, 112, 69, 101, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 42, 92, 95, 45, 57, 64, 
	90, 97, 122, 32, 59, 9, 10, 40, 
	41, 32, 59, 9, 10, 40, 41, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 90, 97, 122, 9, 32, 40, 
	41, 1034, 1083, 9, 32, 40, 41, 43, 
	1034, 1083, 47, 57, 65, 90, 97, 122, 
	1546, 1595, 1034, 896, 1151, 1034, 896, 1151, 
	1034, 896, 1151, 43, 47, 57, 65, 90, 
	97, 122, 1546, 1595, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 1034, 1083, 48, 57, 9, 32, 
	40, 41, 45, 1034, 1083, 48, 57, 65, 
	70, 97, 102, 9, 32, 40, 41, 1034, 
	1083, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 86, 97, 118, 48, 57, 65, 
	86, 97, 118, 61, 48, 57, 65, 86, 
	97, 118, 48, 57, 65, 86, 97, 118, 
	61, 48, 57, 65, 86, 97, 118, 61, 
	48, 57, 65, 86, 97, 118, 48, 57, 
	65, 86, 97, 118, 61, 48, 57, 65, 
	86, 97, 118, 32, 59, 9, 10, 40, 
	41, 48, 57, 65, 86, 97, 118, 32, 
	59, 9, 10, 40, 41, 61, 61, 61, 
	61, 61, 1034, 896, 1151, 48, 57, 65, 
	70, 97, 102, 9, 32, 40, 41, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 45, 1034, 
	1083, 48, 57, 65, 70, 97, 102, 32, 
	59, 9, 10, 40, 41, 48, 57, 65, 
	70, 97, 102, 32, 59, 9, 10, 40, 
	41, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 1034, 
	896, 1151, 1034, 896, 1151, 1034, 896, 1151, 
	48, 57, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 9, 32, 40, 41, 46, 1034, 
	1083, 48, 57, 32, 46, 59, 9, 10, 
	40, 41, 48, 57, 1034, 896, 1151, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 9, 32, 40, 41, 1034, 1083, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 48, 57, 65, 70, 97, 
	102, 48, 57, 65, 70, 97, 102, 58, 
	48, 57, 65, 70, 97, 102, 48, 57, 
	65, 70, 97, 102, 48, 57, 65, 70, 
	97, 102, 48, 57, 65, 70, 97, 102, 
	32, 58, 59, 9, 10, 40, 41, 1034, 
	896, 1151, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 45, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 32, 45, 59, 9, 10, 
	40, 41, 48, 57, 65, 70, 97, 102, 
	48, 57, 65, 70, 97, 102, 45, 48, 
	57, 65, 70, 97, 102, 48, 57, 65, 
	70, 97, 102, 32, 45, 59, 9, 10, 
	40, 41, 9, 32, 36, 40, 41, 42, 
	92, 95, 778, 827, 1034, 1083, 45, 57, 
	64, 90, 97, 122, 9, 32, 36, 40, 
	41, 42, 58, 92, 95, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	44, 45, 57, 60, 63, 64, 90, 91, 
	96, 97, 122, 123, 127, 9, 32, 36, 
	40, 41, 42, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 95, 97, 99, 100, 101, 104, 
	105, 107, 108, 109, 110, 112, 114, 115, 
	116, 778, 827, 1034, 1083, 45, 57, 64, 
	90, 98, 122, 9, 32, 36, 40, 41, 
	42, 65, 67, 68, 69, 72, 73, 75, 
	76, 77, 78, 80, 82, 83, 84, 92, 
	95, 97, 99, 100, 101, 104, 105, 107, 
	108, 109, 110, 112, 114, 115, 116, 778, 
	827, 1034, 1083, 45, 47, 48, 57, 64, 
	90, 98, 122, 9, 32, 36, 40, 41, 
	42, 92, 95, 778, 827, 1034, 1083, 45, 
	57, 64, 90, 97, 122, 9, 32, 36, 
	40, 41, 42, 65, 67, 68, 69, 72, 
	73, 75, 76, 77, 78, 80, 82, 83, 
	84, 92, 95, 97, 99, 100, 101, 104, 
	105, 107, 108, 109, 110, 112, 114, 115, 
	116, 778, 827, 1034, 1083, 45, 47, 48, 
	57, 64, 90, 98, 122, 9, 32, 36, 
	40, 41, 42, 92, 95, 778, 827, 1034, 
	1083, 45, 57, 64, 90, 97, 122, 9, 
	32, 36, 40, 41, 42, 58, 92, 95, 
	1802, 1851, 2058, 2107, 2314, 2363, 2570, 2619, 
	-128, 8, 11, 44, 45, 57, 60, 63, 
	64, 90, 91, 96, 97, 122, 123, 127, 
	9, 32, 36, 40, 41, 42, 58, 65, 
	67, 68, 69, 72, 73, 75, 76, 77, 
	78, 80, 82, 83, 84, 92, 95, 97, 
	99, 100, 101, 104, 105, 107, 108, 109, 
	110, 112, 114, 115, 116, 1802, 1851, 2058, 
	2107, 2314, 2363, 2570, 2619, -128, 8, 11, 
	44, 45, 47, 48, 57, 60, 63, 64, 
	90, 91, 96, 98, 122, 123, 127, 9, 
	32, 36, 40, 41, 42, 58, 65, 67, 
	68, 69, 72, 73, 75, 76, 77, 78, 
	80, 82, 83, 84, 92, 95, 97, 99, 
	100, 101, 104, 105, 107, 108, 109, 110, 
	112, 114, 115, 116, 1802, 1851, 2058, 2107, 
	2314, 2363, 2570, 2619, -128, 8, 11, 44, 
	45, 47, 48, 57, 60, 63, 64, 90, 
	91, 96, 98, 122, 123, 127, 32, 33, 
	59, 92, 9, 10, 35, 39, 40, 41, 
	42, 126, 9, 32, 40, 41, 42, 46, 
	92, 95, 1034, 1083, 45, 57, 65, 90, 
	97, 122, 9, 32, 40, 41, 43, 2058, 
	2107, 2314, 2363, 2570, 2619, 47, 57, 65, 
	90, 97, 122, 1034, 896, 1151, 9, 32, 
	40, 41, 65, 67, 68, 69, 72, 73, 
	75, 76, 77, 78, 80, 82, 83, 84, 
	97, 99, 100, 101, 104, 105, 107, 108, 
	109, 110, 112, 114, 115, 116, 2058, 2107, 
	2314, 2363, 2570, 2619, 1034, 896, 1151, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 48, 57, 65, 70, 97, 102, 1034, 
	896, 1151, 9, 32, 40, 41, 1034, 1083, 
	48, 57, 65, 70, 97, 102, 9, 32, 
	40, 41, 2058, 2107, 2314, 2363, 2570, 2619, 
	48, 57, 65, 70, 97, 102, 1034, 896, 
	1151, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, -128, 8, 
	11, 58, 60, 127, 1034, 896, 1151, 9, 
	32, 40, 41, 2058, 2107, 2314, 2363, 2570, 
	2619, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	1034, 896, 1151, 9, 32, 40, 41, 2058, 
	2107, 2314, 2363, 2570, 2619, 48, 57, 1034, 
	896, 1151, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 1034, 896, 
	1151, 9, 32, 33, 40, 41, 49, 50, 
	2058, 2107, 2314, 2363, 2570, 2619, 1034, 896, 
	1151, 9, 32, 40, 41, 2058, 2107, 2314, 
	2363, 2570, 2619, 48, 57, 65, 70, 97, 
	102, 1034, 896, 1151, 9, 32, 40, 41, 
	2058, 2107, 2314, 2363, 2570, 2619, 48, 57, 
	65, 70, 97, 102, 1034, 896, 1151, 4617, 
	4640, 4648, 4649, 13578, 13627, 13834, 13883, 14346, 
	14395, 14602, 14651, 14858, 14907, 5642, 5504, 5759, 
	3081, 3104, 3112, 3113, 3115, 4106, 4155, 3119, 
	3129, 3137, 3162, 3169, 3194, 4106, 3968, 4223, 
	3115, 6153, 6176, 6184, 6185, 6409, 6432, 6440, 
	6441, 6665, 6688, 6696, 6697, 9482, 9531, 9738, 
	9787, 10250, 10299, 10506, 10555, 10762, 10811, 11274, 
	11323, 11530, 11579, 11786, 11835, 12298, 12347, 12554, 
	12603, 12810, 12859, 3119, 3129, 3137, 3162, 3169, 
	3194, 7690, 8202, 8714, 7552, 7807, 8064, 8319, 
	8576, 8831, 9, 32, 40, 41, 2058, 2107, 
	2314, 2363, 2570, 2619, 48, 57, 65, 70, 
	97, 102, 1034, 896, 1151, 0
};

static const char _zone_scanner_single_lengths[] = {
	0, 36, 16, 34, 16, 11, 8, 8, 
	1, 1, 1, 13, 2, 42, 18, 36, 
	4, 2, 2, 10, 12, 2, 2, 2, 
	10, 6, 2, 2, 2, 10, 4, 2, 
	2, 10, 2, 2, 2, 10, 10, 2, 
	2, 2, 1, 10, 1, 10, 2, 2, 
	2, 2, 10, 4, 6, 34, 2, 2, 
	2, 2, 2, 2, 2, 10, 4, 2, 
	10, 10, 6, 1, 10, 1, 10, 2, 
	10, 10, 4, 2, 2, 2, 10, 10, 
	6, 2, 2, 2, 10, 2, 10, 12, 
	2, 11, 12, 2, 2, 2, 2, 10, 
	2, 2, 10, 6, 10, 2, 2, 2, 
	10, 10, 8, 2, 10, 2, 10, 2, 
	10, 2, 2, 2, 10, 6, 2, 2, 
	10, 2, 10, 2, 2, 0, 10, 1, 
	36, 4, 6, 34, 16, 6, 10, 6, 
	1, 1, 6, 2, 2, 2, 2, 2, 
	2, 2, 6, 34, 1, 16, 34, 1, 
	6, 10, 6, 1, 16, 40, 18, 13, 
	36, 2, 8, 8, 42, 12, 14, 10, 
	10, 12, 12, 14, 12, 14, 10, 14, 
	16, 14, 2, 14, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 8, 18, 36, 
	2, 18, 42, 12, 14, 10, 10, 12, 
	12, 14, 12, 14, 10, 14, 16, 14, 
	9, 36, 12, 12, 10, 2, 2, 10, 
	2, 2, 2, 10, 2, 10, 2, 8, 
	8, 12, 10, 1, 12, 10, 12, 8, 
	8, 10, 10, 12, 10, 12, 8, 12, 
	14, 12, 2, 2, 10, 2, 1, 1, 
	5, 3, 6, 5, 0, 0, 0, 5, 
	2, 2, 2, 4, 0, 0, 0, 4, 
	4, 2, 0, 0, 0, 4, 6, 6, 
	18, 8, 1, 8, 10, 8, 1, 6, 
	10, 1, 12, 8, 1, 3, 0, 0, 
	0, 3, 8, 1, 6, 6, 8, 12, 
	1, 12, 8, 1, 3, 0, 0, 0, 
	3, 8, 2, 10, 1, 1, 1, 1, 
	2, 2, 11, 1, 1, 10, 38, 16, 
	2, 2, 10, 1, 4, 2, 2, 10, 
	2, 2, 2, 10, 6, 2, 2, 2, 
	10, 4, 2, 2, 10, 2, 2, 2, 
	10, 10, 2, 2, 2, 1, 10, 1, 
	10, 2, 2, 2, 2, 10, 2, 2, 
	2, 2, 2, 2, 2, 10, 4, 2, 
	10, 10, 6, 1, 10, 1, 10, 2, 
	10, 10, 4, 2, 2, 2, 10, 10, 
	6, 2, 2, 2, 10, 2, 10, 12, 
	2, 11, 12, 2, 2, 2, 2, 10, 
	2, 2, 10, 6, 10, 2, 2, 2, 
	10, 10, 8, 2, 10, 2, 10, 2, 
	10, 2, 2, 2, 10, 6, 2, 2, 
	10, 2, 10, 2, 2, 0, 10, 2, 
	2, 2, 10, 2, 10, 6, 6, 6, 
	6, 0, 10, 1, 1, 1, 6, 7, 
	8, 6, 0, 10, 1, 1, 6, 1, 
	10, 2, 4, 2, 2, 3, 2, 2, 
	2, 2, 1, 1, 2, 2, 2, 1, 
	2, 2, 4, 1, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 1, 1, 2, 
	2, 2, 1, 1, 1, 2, 1, 1, 
	2, 2, 2, 1, 1, 1, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 4, 2, 2, 
	2, 2, 2, 2, 2, 2, 4, 2, 
	1, 2, 2, 2, 3, 3, 2, 2, 
	2, 2, 1, 1, 2, 2, 2, 1, 
	2, 1, 1, 2, 1, 1, 2, 12, 
	2, 2, 2, 2, 2, 2, 2, 6, 
	2, 2, 2, 2, 2, 2, 4, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	2, 2, 2, 4, 2, 2, 2, 2, 
	2, 2, 2, 2, 2, 2, 2, 2, 
	1, 3, 3, 2, 3, 6, 9, 6, 
	6, 6, 6, 16, 6, 16, 6, 16, 
	6, 12, 2, 10, 2, 1, 6, 10, 
	6, 1, 6, 10, 6, 1, 6, 10, 
	6, 1, 1, 1, 2, 6, 6, 2, 
	1, 3, 6, 9, 2, 1, 0, 6, 
	9, 2, 1, 2, 10, 10, 1, 1, 
	3, 0, 6, 8, 6, 8, 7, 8, 
	6, 6, 6, 8, 6, 8, 7, 8, 
	6, 7, 0, 12, 10, 12, 10, 12, 
	10, 12, 10, 1, 11, 11, 1, 11, 
	11, 10, 1, 11, 11, 10, 1, 11, 
	11, 10, 1, 6, 1, 6, 6, 1, 
	1, 1, 6, 1, 6, 6, 1, 1, 
	0, 6, 6, 6, 6, 6, 9, 2, 
	1, 1, 1, 0, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 9, 2, 1, 
	1, 1, 1, 1, 0, 6, 6, 6, 
	6, 6, 7, 2, 1, 1, 1, 13, 
	2, 1, 1, 1, 0, 10, 1, 1, 
	1, 1, 0, 6, 6, 6, 6, 6, 
	6, 0, 10, 1, 1, 1, 1, 0, 
	6, 6, 6, 6, 0, 10, 1, 1, 
	1, 0, 6, 10, 6, 6, 6, 7, 
	34, 7, 2, 1, 14, 1, 35, 3, 
	1, 1, 6, 6, 6, 7, 35, 1, 
	1, 6, 6, 6, 7, 35, 1, 1, 
	6, 6, 6, 9, 34, 1, 1, 1, 
	28, 12, 6, 6, 6, 6, 6, 6, 
	6, 6, 6, 6, 6, 6, 9, 6, 
	7, 2, 1, 1, 1, 1, 1, 1, 
	1, 1, 2, 2, 6, 2, 2, 2, 
	6, 2, 6, 4, 2, 2, 6, 2, 
	2, 2, 6, 6, 2, 2, 2, 6, 
	4, 2, 2, 6, 2, 2, 2, 6, 
	6, 2, 2, 2, 1, 6, 1, 6, 
	2, 2, 2, 2, 6, 2, 2, 2, 
	2, 2, 2, 2, 6, 4, 2, 6, 
	6, 6, 1, 6, 1, 6, 2, 6, 
	6, 4, 2, 2, 2, 6, 6, 6, 
	2, 2, 2, 6, 2, 6, 8, 2, 
	7, 8, 2, 2, 2, 2, 6, 2, 
	2, 6, 6, 6, 2, 2, 2, 6, 
	6, 8, 2, 6, 2, 6, 2, 6, 
	2, 2, 2, 6, 6, 2, 2, 6, 
	2, 6, 2, 2, 0, 6, 3, 2, 
	2, 0, 6, 6, 6, 6, 6, 7, 
	2, 1, 1, 1, 1, 2, 0, 6, 
	6, 6, 6, 6, 7, 6, 6, 0, 
	1, 0, 1, 1, 0, 1, 2, 2, 
	1, 1, 1, 1, 1, 1, 0, 6, 
	1, 1, 1, 0, 6, 6, 6, 6, 
	6, 7, 2, 0, 2, 1, 1, 1, 
	0, 6, 6, 6, 6, 6, 6, 0, 
	10, 1, 1, 1, 1, 0, 6, 7, 
	3, 1, 0, 6, 6, 0, 0, 0, 
	1, 0, 0, 0, 0, 3, 1, 0, 
	0, 1, 0, 0, 3, 0, 0, 1, 
	0, 0, 3, 12, 17, 40, 40, 12, 
	40, 12, 17, 45, 45, 0, 0, 0, 
	4, 0, 0, 0, 10, 0, 11, 1, 
	0, 38, 1, 0, 10, 1, 6, 0, 
	10, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 10, 1, 0, 0, 
	10, 1, 10, 1, 10, 1, 10, 1, 
	0, 0, 0, 0, 13, 1, 0, 10, 
	1, 0, 10, 1, 0, 14, 1, 7, 
	1, 35, 3, 0, 0, 0, 0, 0, 
	0, 0, 10, 1, 0, 0, 0, 0
};

static const char _zone_scanner_range_lengths[] = {
	0, 1, 1, 0, 0, 3, 0, 0, 
	0, 0, 1, 3, 2, 4, 1, 0, 
	0, 0, 0, 0, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 1, 1, 
	1, 0, 0, 1, 1, 1, 1, 1, 
	1, 0, 0, 0, 0, 0, 0, 0, 
	0, 2, 0, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 0, 4, 0, 3, 
	1, 2, 3, 3, 4, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 0, 0, 0, 0, 0, 
	0, 2, 0, 0, 2, 0, 1, 1, 
	2, 0, 4, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	3, 1, 1, 1, 2, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 2, 3, 
	3, 2, 2, 1, 1, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 2, 2, 0, 2, 0, 0, 
	3, 2, 5, 5, 1, 1, 1, 5, 
	2, 2, 2, 4, 1, 1, 1, 4, 
	1, 2, 1, 1, 1, 1, 0, 1, 
	1, 0, 0, 1, 1, 1, 1, 0, 
	3, 0, 3, 0, 0, 3, 1, 1, 
	1, 3, 0, 1, 0, 3, 0, 3, 
	0, 3, 0, 0, 3, 1, 1, 1, 
	3, 0, 2, 3, 1, 1, 3, 3, 
	3, 3, 3, 1, 0, 0, 0, 0, 
	0, 0, 0, 1, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 1, 1, 0, 
	0, 0, 0, 0, 0, 0, 1, 1, 
	3, 3, 3, 1, 1, 1, 0, 1, 
	1, 3, 3, 3, 1, 1, 1, 1, 
	1, 3, 0, 2, 0, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	2, 0, 0, 0, 0, 0, 0, 0, 
	2, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	2, 0, 0, 2, 0, 0, 0, 0, 
	0, 2, 0, 0, 0, 2, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	2, 0, 0, 2, 0, 0, 2, 1, 
	3, 0, 0, 0, 0, 0, 2, 0, 
	0, 0, 0, 0, 0, 2, 0, 0, 
	2, 0, 0, 2, 0, 0, 0, 2, 
	0, 0, 2, 0, 0, 2, 0, 0, 
	2, 0, 0, 0, 2, 0, 0, 2, 
	1, 3, 3, 2, 3, 0, 3, 0, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 3, 3, 1, 3, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 2, 0, 3, 2, 
	1, 3, 0, 3, 2, 1, 1, 1, 
	3, 2, 1, 2, 0, 3, 1, 3, 
	5, 1, 1, 1, 1, 1, 1, 0, 
	0, 1, 1, 1, 1, 1, 1, 0, 
	0, 1, 1, 1, 1, 1, 1, 1, 
	1, 1, 0, 1, 1, 1, 1, 1, 
	1, 0, 1, 1, 1, 0, 1, 1, 
	1, 0, 1, 0, 1, 1, 1, 1, 
	1, 1, 0, 1, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 3, 2, 
	1, 1, 1, 1, 1, 1, 1, 3, 
	0, 3, 0, 3, 0, 3, 2, 1, 
	1, 1, 1, 1, 3, 0, 1, 1, 
	3, 0, 3, 0, 1, 1, 1, 0, 
	0, 0, 1, 1, 1, 1, 1, 0, 
	3, 3, 1, 1, 3, 0, 1, 1, 
	3, 3, 3, 1, 1, 1, 1, 1, 
	1, 1, 1, 3, 3, 3, 1, 1, 
	1, 1, 1, 0, 0, 1, 1, 0, 
	0, 3, 0, 1, 0, 1, 3, 3, 
	1, 1, 0, 1, 1, 1, 1, 1, 
	1, 0, 1, 1, 3, 3, 1, 1, 
	0, 1, 1, 3, 0, 1, 1, 1, 
	0, 0, 3, 0, 1, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 3, 0, 
	3, 0, 1, 1, 1, 1, 1, 1, 
	1, 1, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 1, 1, 3, 2, 
	2, 1, 1, 1, 1, 3, 0, 3, 
	0, 1, 1, 1, 3, 0, 1, 1, 
	1, 1, 1, 1, 3, 0, 3, 3, 
	3, 3, 3, 3, 3, 3, 5, 2, 
	0, 0, 0, 0, 0, 1, 3, 3, 
	1, 1, 1, 1, 1, 1, 1, 1, 
	1, 3, 2, 3, 5, 1, 1, 1, 
	1, 1, 1, 1, 1, 1, 3, 3, 
	3, 1, 1, 1, 1, 1, 1, 1, 
	3, 1, 1, 1, 3, 3, 3, 3, 
	0, 3, 3, 3, 3, 2, 1, 3, 
	3, 0, 3, 3, 2, 3, 3, 0, 
	3, 3, 2, 3, 8, 3, 4, 3, 
	4, 3, 8, 9, 9, 0, 0, 0, 
	4, 0, 0, 0, 3, 0, 3, 1, 
	0, 0, 1, 0, 3, 1, 3, 0, 
	3, 1, 1, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 3, 1, 0, 0, 
	0, 1, 1, 1, 1, 1, 1, 1, 
	0, 0, 0, 0, 0, 1, 0, 3, 
	1, 0, 3, 1, 0, 0, 1, 3, 
	1, 3, 3, 0, 0, 0, 0, 0, 
	0, 0, 3, 1, 0, 0, 0, 0
};

static const short _zone_scanner_index_offsets[] = {
	0, 0, 38, 56, 91, 108, 123, 132, 
	141, 143, 145, 148, 165, 170, 217, 237, 
	274, 279, 282, 285, 296, 310, 313, 316, 
	319, 330, 337, 340, 343, 346, 357, 362, 
	365, 368, 379, 382, 385, 388, 399, 410, 
	413, 416, 419, 421, 432, 434, 445, 448, 
	451, 454, 457, 468, 473, 480, 515, 518, 
	521, 524, 527, 530, 533, 536, 547, 552, 
	555, 566, 577, 584, 586, 597, 599, 610, 
	613, 624, 635, 640, 643, 646, 649, 660, 
	671, 678, 681, 684, 687, 698, 701, 712, 
	725, 728, 740, 753, 756, 759, 762, 765, 
	776, 779, 782, 793, 800, 811, 814, 817, 
	820, 831, 842, 851, 854, 865, 868, 879, 
	882, 893, 896, 899, 902, 913, 920, 923, 
	926, 937, 940, 951, 954, 957, 959, 971, 
	974, 1012, 1017, 1024, 1060, 1078, 1086, 1098, 
	1106, 1109, 1111, 1118, 1121, 1124, 1127, 1130, 
	1133, 1136, 1141, 1148, 1184, 1187, 1205, 1241, 
	1244, 1252, 1264, 1272, 1275, 1292, 1337, 1356, 
	1373, 1411, 1416, 1428, 1440, 1487, 1500, 1515, 
	1526, 1537, 1550, 1563, 1578, 1591, 1606, 1617, 
	1632, 1649, 1664, 1669, 1684, 1687, 1690, 1693, 
	1696, 1699, 1704, 1707, 1710, 1715, 1724, 1744, 
	1782, 1787, 1806, 1853, 1866, 1881, 1892, 1903, 
	1916, 1929, 1944, 1957, 1972, 1983, 1998, 2015, 
	2030, 2043, 2081, 2095, 2109, 2122, 2125, 2128, 
	2139, 2142, 2145, 2148, 2159, 2162, 2173, 2178, 
	2190, 2202, 2217, 2230, 2233, 2247, 2258, 2271, 
	2280, 2289, 2300, 2311, 2324, 2335, 2348, 2357, 
	2370, 2385, 2398, 2403, 2408, 2419, 2424, 2426, 
	2428, 2437, 2443, 2455, 2466, 2468, 2470, 2472, 
	2483, 2488, 2493, 2498, 2507, 2509, 2511, 2513, 
	2522, 2528, 2533, 2535, 2537, 2539, 2545, 2552, 
	2560, 2580, 2589, 2591, 2601, 2613, 2623, 2626, 
	2633, 2647, 2649, 2665, 2674, 2676, 2683, 2685, 
	2687, 2689, 2696, 2705, 2708, 2715, 2725, 2734, 
	2750, 2752, 2768, 2777, 2779, 2786, 2788, 2790, 
	2792, 2799, 2808, 2813, 2827, 2830, 2833, 2838, 
	2843, 2849, 2855, 2870, 2873, 2875, 2886, 2925, 
	2942, 2945, 2948, 2959, 2962, 2967, 2970, 2973, 
	2984, 2987, 2990, 2993, 3004, 3011, 3014, 3017, 
	3020, 3031, 3036, 3039, 3042, 3053, 3056, 3059, 
	3062, 3073, 3084, 3087, 3090, 3093, 3095, 3106, 
	3108, 3119, 3122, 3125, 3128, 3131, 3142, 3145, 
	3148, 3151, 3154, 3157, 3160, 3163, 3174, 3179, 
	3182, 3193, 3204, 3211, 3213, 3224, 3226, 3237, 
	3240, 3251, 3262, 3267, 3270, 3273, 3276, 3287, 
	3298, 3305, 3308, 3311, 3314, 3325, 3328, 3339, 
	3352, 3355, 3367, 3380, 3383, 3386, 3389, 3392, 
	3403, 3406, 3409, 3420, 3427, 3438, 3441, 3444, 
	3447, 3458, 3469, 3478, 3481, 3492, 3495, 3506, 
	3509, 3520, 3523, 3526, 3529, 3540, 3547, 3550, 
	3553, 3564, 3567, 3578, 3581, 3584, 3586, 3598, 
	3601, 3604, 3607, 3618, 3621, 3632, 3639, 3647, 
	3655, 3665, 3669, 3683, 3686, 3689, 3692, 3699, 
	3708, 3718, 3728, 3732, 3746, 3749, 3752, 3760, 
	3763, 3775, 3781, 3786, 3791, 3794, 3800, 3803, 
	3806, 3809, 3812, 3814, 3816, 3819, 3822, 3825, 
	3827, 3832, 3835, 3840, 3842, 3845, 3848, 3851, 
	3854, 3859, 3862, 3865, 3868, 3871, 3873, 3875, 
	3878, 3881, 3884, 3886, 3888, 3890, 3895, 3897, 
	3899, 3902, 3905, 3908, 3910, 3912, 3914, 3919, 
	3922, 3925, 3928, 3931, 3934, 3937, 3940, 3945, 
	3948, 3951, 3954, 3957, 3960, 3963, 3968, 3971, 
	3974, 3979, 3982, 3985, 3990, 3993, 3996, 4001, 
	4004, 4006, 4011, 4014, 4017, 4021, 4027, 4030, 
	4033, 4036, 4039, 4041, 4043, 4046, 4049, 4052, 
	4054, 4059, 4061, 4063, 4068, 4070, 4072, 4077, 
	4091, 4097, 4100, 4103, 4106, 4109, 4112, 4117, 
	4124, 4127, 4130, 4133, 4136, 4139, 4144, 4149, 
	4152, 4157, 4160, 4163, 4168, 4171, 4174, 4177, 
	4182, 4185, 4188, 4193, 4198, 4201, 4206, 4209, 
	4212, 4217, 4220, 4223, 4226, 4231, 4234, 4237, 
	4242, 4245, 4252, 4259, 4264, 4271, 4278, 4291, 
	4298, 4306, 4314, 4322, 4340, 4348, 4366, 4374, 
	4392, 4400, 4416, 4422, 4434, 4440, 4443, 4451, 
	4463, 4471, 4474, 4482, 4494, 4502, 4505, 4513, 
	4525, 4533, 4536, 4539, 4542, 4547, 4554, 4564, 
	4569, 4572, 4579, 4586, 4599, 4604, 4607, 4609, 
	4617, 4630, 4635, 4638, 4643, 4654, 4668, 4671, 
	4676, 4685, 4687, 4695, 4705, 4713, 4723, 4732, 
	4741, 4748, 4756, 4764, 4774, 4782, 4792, 4801, 
	4810, 4817, 4826, 4828, 4842, 4854, 4868, 4880, 
	4894, 4906, 4920, 4931, 4934, 4947, 4960, 4963, 
	4976, 4989, 5000, 5003, 5016, 5029, 5040, 5043, 
	5056, 5069, 5080, 5083, 5090, 5093, 5101, 5109, 
	5112, 5115, 5118, 5125, 5128, 5136, 5144, 5147, 
	5150, 5152, 5160, 5168, 5176, 5184, 5192, 5205, 
	5210, 5213, 5216, 5219, 5221, 5229, 5237, 5245, 
	5255, 5262, 5272, 5279, 5289, 5296, 5309, 5314, 
	5317, 5320, 5323, 5326, 5329, 5333, 5340, 5348, 
	5356, 5366, 5373, 5384, 5387, 5390, 5393, 5396, 
	5410, 5413, 5415, 5418, 5421, 5423, 5435, 5438, 
	5440, 5445, 5450, 5452, 5460, 5470, 5477, 5485, 
	5493, 5503, 5507, 5521, 5524, 5527, 5530, 5533, 
	5535, 5543, 5551, 5559, 5569, 5573, 5587, 5590, 
	5593, 5596, 5598, 5606, 5617, 5624, 5632, 5640, 
	5648, 5683, 5694, 5697, 5700, 5715, 5718, 5757, 
	5764, 5767, 5770, 5777, 5785, 5793, 5802, 5839, 
	5842, 5845, 5852, 5860, 5868, 5879, 5918, 5921, 
	5924, 5931, 5939, 5947, 5960, 5995, 5998, 6001, 
	6004, 6033, 6046, 6056, 6063, 6071, 6079, 6087, 
	6095, 6103, 6111, 6119, 6127, 6135, 6143, 6156, 
	6163, 6174, 6177, 6180, 6183, 6186, 6189, 6192, 
	6195, 6198, 6201, 6204, 6207, 6214, 6217, 6220, 
	6223, 6230, 6233, 6240, 6245, 6248, 6251, 6258, 
	6261, 6264, 6267, 6274, 6281, 6284, 6287, 6290, 
	6297, 6302, 6305, 6308, 6315, 6318, 6321, 6324, 
	6331, 6338, 6341, 6344, 6347, 6349, 6356, 6358, 
	6365, 6368, 6371, 6374, 6377, 6384, 6387, 6390, 
	6393, 6396, 6399, 6402, 6405, 6412, 6417, 6420, 
	6427, 6434, 6441, 6443, 6450, 6452, 6459, 6462, 
	6469, 6476, 6481, 6484, 6487, 6490, 6497, 6504, 
	6511, 6514, 6517, 6520, 6527, 6530, 6537, 6546, 
	6549, 6557, 6566, 6569, 6572, 6575, 6578, 6585, 
	6588, 6591, 6598, 6605, 6612, 6615, 6618, 6621, 
	6628, 6635, 6644, 6647, 6654, 6657, 6664, 6667, 
	6674, 6677, 6680, 6683, 6690, 6697, 6700, 6703, 
	6710, 6713, 6720, 6723, 6726, 6728, 6736, 6743, 
	6748, 6753, 6755, 6763, 6771, 6779, 6789, 6796, 
	6807, 6810, 6813, 6816, 6819, 6824, 6827, 6829, 
	6837, 6845, 6853, 6861, 6869, 6880, 6887, 6897, 
	6901, 6906, 6910, 6915, 6920, 6924, 6929, 6937, 
	6942, 6944, 6946, 6948, 6950, 6952, 6955, 6959, 
	6969, 6972, 6975, 6978, 6980, 6988, 6996, 7004, 
	7012, 7020, 7031, 7036, 7040, 7048, 7051, 7054, 
	7057, 7059, 7067, 7075, 7083, 7091, 7099, 7109, 
	7113, 7127, 7130, 7133, 7136, 7139, 7141, 7149, 
	7158, 7165, 7168, 7170, 7178, 7188, 7192, 7196, 
	7200, 7202, 7206, 7210, 7214, 7218, 7224, 7227, 
	7231, 7235, 7237, 7241, 7245, 7251, 7255, 7259, 
	7261, 7265, 7269, 7275, 7291, 7317, 7361, 7406, 
	7422, 7467, 7483, 7509, 7564, 7619, 7620, 7621, 
	7622, 7631, 7632, 7633, 7634, 7648, 7649, 7664, 
	7667, 7668, 7707, 7710, 7711, 7725, 7728, 7738, 
	7739, 7753, 7756, 7759, 7760, 7761, 7762, 7763, 
	7764, 7765, 7766, 7767, 7768, 7782, 7785, 7786, 
	7787, 7798, 7801, 7813, 7816, 7828, 7831, 7843, 
	7846, 7847, 7848, 7849, 7850, 7864, 7867, 7868, 
	7882, 7885, 7886, 7900, 7903, 7904, 7919, 7922, 
	7933, 7936, 7975, 7982, 7983, 7984, 7985, 7986, 
	7987, 7988, 7989, 8003, 8006, 8007, 8008, 8009
};

static const short _zone_scanner_indicies[] = {
	1, 1, 2, 3, 5, 6, 7, 8, 
	9, 10, 11, 12, 13, 14, 15, 16, 
	17, 18, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	19, 20, 21, 22, 4, 0, 24, 24, 
	25, 26, 28, 29, 30, 31, 32, 28, 
	29, 30, 31, 32, 33, 34, 27, 23, 
	36, 36, 37, 38, 39, 40, 41, 42, 
	43, 44, 45, 46, 47, 48, 49, 50, 
	51, 52, 39, 40, 41, 42, 43, 44, 
	45, 46, 47, 48, 49, 50, 51, 52, 
	53, 54, 35, 56, 56, 57, 58, 59, 
	60, 61, 59, 60, 61, 62, 63, 64, 
	64, 65, 66, 55, 68, 68, 70, 71, 
	72, 73, 74, 67, 67, 75, 76, 67, 
	67, 67, 69, 78, 78, 79, 80, 81, 
	82, 81, 82, 77, 78, 78, 79, 80, 
	81, 82, 81, 82, 83, 81, 82, 85, 
	84, 73, 74, 69, 86, 86, 87, 88, 
	72, 81, 82, 81, 89, 90, 91, 81, 
	92, 67, 67, 67, 77, 81, 93, 82, 
	89, 77, 94, 94, 96, 97, 67, 99, 
	100, 101, 102, 103, 104, 105, 106, 107, 
	108, 109, 110, 111, 112, 72, 99, 100, 
	101, 102, 103, 104, 105, 106, 107, 108, 
	109, 110, 111, 112, 19, 20, 113, 114, 
	115, 116, 117, 118, 67, 67, 98, 67, 
	95, 120, 120, 121, 122, 28, 29, 30, 
	31, 32, 28, 29, 30, 31, 32, 81, 
	82, 123, 124, 27, 119, 126, 126, 127, 
	128, 39, 40, 41, 42, 43, 44, 45, 
	46, 47, 48, 49, 50, 51, 52, 39, 
	40, 41, 42, 43, 44, 45, 46, 47, 
	48, 49, 50, 51, 52, 81, 82, 129, 
	130, 125, 131, 132, 131, 132, 35, 133, 
	133, 35, 134, 134, 35, 135, 135, 136, 
	137, 138, 139, 140, 140, 141, 142, 55, 
	78, 81, 78, 79, 80, 82, 143, 81, 
	143, 144, 145, 89, 74, 77, 146, 146, 
	35, 147, 147, 35, 148, 148, 35, 149, 
	149, 150, 151, 152, 153, 154, 154, 155, 
	156, 55, 157, 158, 159, 157, 158, 159, 
	35, 160, 160, 35, 161, 161, 35, 162, 
	162, 35, 163, 163, 164, 165, 166, 167, 
	168, 168, 169, 170, 55, 171, 172, 171, 
	172, 35, 173, 173, 35, 174, 174, 35, 
	175, 175, 176, 177, 178, 179, 180, 180, 
	181, 182, 55, 183, 183, 35, 184, 184, 
	35, 185, 185, 35, 186, 186, 187, 188, 
	189, 190, 191, 191, 192, 193, 55, 194, 
	194, 195, 196, 197, 198, 199, 199, 200, 
	201, 55, 202, 202, 35, 203, 203, 35, 
	204, 205, 35, 206, 35, 207, 207, 208, 
	209, 210, 211, 212, 212, 213, 214, 55, 
	215, 35, 216, 216, 217, 218, 219, 220, 
	221, 221, 222, 223, 55, 224, 224, 35, 
	225, 225, 35, 226, 226, 35, 227, 227, 
	35, 228, 228, 229, 230, 231, 232, 233, 
	233, 234, 235, 55, 236, 237, 236, 237, 
	35, 238, 238, 239, 240, 241, 242, 35, 
	243, 243, 244, 245, 246, 247, 248, 249, 
	250, 251, 252, 253, 254, 255, 256, 257, 
	258, 259, 246, 247, 248, 249, 250, 251, 
	252, 253, 254, 255, 256, 257, 258, 259, 
	260, 261, 35, 237, 237, 35, 262, 262, 
	35, 263, 263, 35, 264, 264, 35, 265, 
	265, 35, 266, 266, 35, 267, 267, 35, 
	268, 268, 269, 270, 271, 272, 273, 273, 
	274, 275, 55, 276, 277, 276, 277, 35, 
	278, 278, 35, 279, 279, 280, 281, 282, 
	283, 284, 284, 285, 286, 55, 287, 287, 
	288, 289, 290, 291, 292, 292, 293, 294, 
	55, 295, 296, 297, 298, 297, 298, 35, 
	299, 35, 300, 300, 301, 302, 303, 304, 
	305, 305, 306, 307, 55, 308, 35, 309, 
	309, 310, 311, 312, 313, 314, 314, 315, 
	316, 55, 317, 317, 35, 318, 318, 319, 
	320, 321, 322, 323, 323, 324, 325, 55, 
	326, 326, 327, 328, 329, 330, 331, 331, 
	332, 333, 55, 334, 335, 334, 335, 35, 
	336, 336, 35, 337, 337, 35, 338, 338, 
	35, 339, 339, 340, 341, 342, 343, 344, 
	344, 345, 346, 55, 347, 347, 348, 349, 
	350, 351, 352, 352, 353, 354, 55, 355, 
	356, 357, 355, 356, 357, 35, 358, 358, 
	35, 359, 359, 35, 360, 360, 35, 361, 
	361, 362, 363, 364, 365, 366, 366, 367, 
	368, 55, 369, 369, 35, 370, 370, 371, 
	372, 373, 374, 375, 375, 376, 377, 55, 
	378, 378, 379, 380, 381, 381, 382, 383, 
	384, 384, 385, 386, 55, 387, 387, 35, 
	388, 388, 389, 390, 391, 392, 393, 394, 
	394, 395, 396, 55, 397, 397, 398, 399, 
	400, 400, 401, 402, 403, 403, 404, 405, 
	55, 406, 406, 35, 407, 407, 35, 408, 
	408, 35, 409, 409, 35, 410, 410, 411, 
	412, 413, 414, 415, 415, 416, 417, 55, 
	418, 418, 35, 419, 419, 35, 420, 420, 
	421, 422, 423, 424, 425, 425, 426, 427, 
	55, 428, 429, 430, 428, 429, 430, 35, 
	431, 431, 432, 433, 434, 435, 436, 436, 
	437, 438, 55, 439, 439, 35, 440, 440, 
	35, 441, 441, 35, 442, 442, 443, 444, 
	445, 446, 447, 447, 448, 449, 55, 450, 
	450, 451, 452, 453, 454, 455, 455, 456, 
	457, 55, 458, 459, 460, 461, 458, 459, 
	460, 461, 35, 462, 462, 35, 463, 463, 
	464, 465, 466, 467, 468, 468, 469, 470, 
	55, 471, 471, 35, 472, 472, 473, 474, 
	475, 476, 477, 477, 478, 479, 55, 480, 
	480, 35, 481, 481, 482, 483, 484, 485, 
	486, 486, 487, 488, 55, 489, 489, 35, 
	490, 490, 35, 491, 491, 35, 492, 492, 
	493, 494, 495, 496, 497, 497, 498, 499, 
	55, 500, 501, 502, 500, 501, 502, 35, 
	503, 503, 35, 504, 504, 35, 505, 505, 
	506, 507, 508, 509, 510, 510, 511, 512, 
	55, 513, 513, 35, 514, 514, 515, 516, 
	517, 518, 519, 519, 520, 521, 55, 522, 
	522, 35, 523, 523, 35, 525, 524, 527, 
	527, 528, 529, 531, 532, 533, 533, 534, 
	535, 530, 526, 260, 261, 35, 536, 536, 
	537, 538, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	5, 6, 7, 8, 9, 10, 11, 12, 
	13, 14, 15, 16, 17, 18, 19, 20, 
	539, 540, 4, 0, 541, 237, 541, 237, 
	35, 542, 542, 543, 544, 545, 546, 35, 
	547, 547, 548, 549, 551, 552, 553, 554, 
	555, 556, 557, 558, 559, 560, 561, 562, 
	563, 564, 551, 552, 553, 554, 555, 556, 
	557, 558, 559, 560, 561, 562, 563, 564, 
	565, 566, 550, 524, 567, 567, 568, 569, 
	571, 572, 573, 574, 575, 571, 572, 573, 
	574, 575, 576, 577, 570, 23, 567, 567, 
	568, 569, 576, 577, 578, 23, 580, 581, 
	582, 583, 584, 580, 581, 582, 583, 584, 
	579, 23, 585, 585, 586, 587, 589, 590, 
	588, 23, 565, 566, 35, 19, 20, 592, 
	593, 594, 592, 593, 594, 591, 595, 595, 
	591, 596, 596, 591, 597, 597, 591, 598, 
	598, 591, 599, 599, 591, 600, 600, 591, 
	601, 601, 601, 601, 591, 603, 603, 604, 
	605, 606, 607, 602, 608, 608, 609, 610, 
	5, 6, 7, 8, 9, 10, 11, 12, 
	13, 14, 15, 16, 17, 18, 5, 6, 
	7, 8, 9, 10, 11, 12, 13, 14, 
	15, 16, 17, 18, 611, 612, 4, 524, 
	611, 612, 613, 615, 615, 616, 617, 28, 
	29, 30, 31, 32, 28, 29, 30, 31, 
	32, 618, 619, 27, 614, 620, 620, 621, 
	622, 5, 6, 7, 8, 9, 10, 11, 
	12, 13, 14, 15, 16, 17, 18, 5, 
	6, 7, 8, 9, 10, 11, 12, 13, 
	14, 15, 16, 17, 18, 623, 624, 4, 
	524, 623, 624, 35, 24, 24, 25, 26, 
	33, 34, 625, 23, 627, 628, 629, 630, 
	631, 627, 628, 629, 630, 631, 626, 23, 
	632, 632, 633, 634, 636, 637, 635, 23, 
	53, 54, 35, 639, 639, 640, 641, 59, 
	60, 61, 59, 60, 61, 642, 643, 64, 
	64, 644, 645, 638, 646, 646, 647, 648, 
	67, 99, 100, 101, 102, 103, 104, 105, 
	106, 107, 108, 109, 110, 111, 112, 72, 
	99, 100, 101, 102, 103, 104, 105, 106, 
	107, 108, 109, 110, 111, 112, 649, 650, 
	67, 67, 651, 652, 67, 67, 98, 67, 
	526, 654, 654, 655, 656, 59, 60, 61, 
	59, 60, 61, 81, 82, 657, 658, 659, 
	660, 661, 662, 653, 663, 663, 664, 665, 
	72, 81, 82, 93, 89, 90, 91, 666, 
	92, 67, 67, 67, 77, 667, 667, 668, 
	669, 5, 6, 7, 8, 9, 10, 11, 
	12, 13, 14, 15, 16, 17, 18, 5, 
	6, 7, 8, 9, 10, 11, 12, 13, 
	14, 15, 16, 17, 18, 81, 82, 670, 
	671, 4, 0, 81, 670, 82, 671, 83, 
	672, 672, 673, 674, 81, 82, 81, 82, 
	82, 82, 82, 77, 672, 672, 673, 674, 
	81, 82, 81, 82, 82, 82, 82, 83, 
	675, 675, 676, 677, 67, 99, 100, 101, 
	102, 103, 104, 105, 106, 107, 108, 109, 
	110, 111, 112, 72, 99, 100, 101, 102, 
	103, 104, 105, 106, 107, 108, 109, 110, 
	111, 112, 81, 82, 670, 678, 90, 91, 
	670, 679, 67, 67, 98, 67, 95, 78, 
	78, 79, 80, 131, 132, 131, 132, 81, 
	82, 81, 82, 653, 78, 78, 79, 80, 
	157, 158, 159, 157, 158, 159, 81, 82, 
	81, 82, 653, 78, 78, 79, 80, 202, 
	202, 81, 82, 81, 82, 653, 78, 78, 
	79, 80, 224, 224, 81, 82, 81, 82, 
	653, 78, 78, 79, 80, 541, 237, 541, 
	237, 81, 82, 81, 82, 653, 78, 78, 
	79, 80, 276, 277, 276, 277, 81, 82, 
	81, 82, 653, 78, 78, 79, 80, 295, 
	296, 297, 298, 297, 298, 81, 82, 81, 
	82, 653, 78, 78, 79, 80, 334, 335, 
	334, 335, 81, 82, 81, 82, 653, 78, 
	78, 79, 80, 355, 356, 357, 355, 356, 
	357, 81, 82, 81, 82, 653, 78, 78, 
	79, 80, 418, 418, 81, 82, 81, 82, 
	653, 78, 78, 79, 80, 428, 429, 430, 
	428, 429, 430, 81, 82, 81, 82, 653, 
	78, 78, 79, 80, 458, 459, 460, 461, 
	458, 459, 460, 461, 81, 82, 81, 82, 
	653, 78, 78, 79, 80, 500, 501, 502, 
	500, 501, 502, 81, 82, 81, 82, 653, 
	81, 680, 82, 678, 77, 78, 78, 79, 
	80, 592, 593, 594, 592, 593, 594, 81, 
	82, 81, 82, 681, 682, 682, 591, 683, 
	683, 591, 684, 684, 591, 685, 685, 591, 
	686, 686, 591, 687, 687, 687, 687, 591, 
	688, 688, 591, 689, 689, 591, 690, 690, 
	690, 690, 591, 692, 692, 693, 694, 81, 
	82, 695, 696, 691, 698, 698, 699, 700, 
	28, 29, 30, 31, 32, 28, 29, 30, 
	31, 32, 81, 82, 701, 702, 27, 697, 
	703, 703, 704, 705, 5, 6, 7, 8, 
	9, 10, 11, 12, 13, 14, 15, 16, 
	17, 18, 5, 6, 7, 8, 9, 10, 
	11, 12, 13, 14, 15, 16, 17, 18, 
	81, 82, 706, 707, 4, 0, 81, 706, 
	82, 707, 125, 709, 709, 710, 711, 59, 
	60, 61, 59, 60, 61, 81, 82, 712, 
	713, 659, 660, 714, 715, 708, 716, 716, 
	717, 718, 67, 99, 100, 101, 102, 103, 
	104, 105, 106, 107, 108, 109, 110, 111, 
	112, 72, 99, 100, 101, 102, 103, 104, 
	105, 106, 107, 108, 109, 110, 111, 112, 
	81, 82, 680, 678, 90, 91, 719, 679, 
	67, 67, 98, 67, 95, 692, 692, 693, 
	694, 131, 132, 131, 132, 81, 82, 695, 
	696, 708, 692, 692, 693, 694, 157, 158, 
	159, 157, 158, 159, 81, 82, 695, 696, 
	708, 692, 692, 693, 694, 202, 202, 81, 
	82, 695, 696, 708, 692, 692, 693, 694, 
	224, 224, 81, 82, 695, 696, 708, 692, 
	692, 693, 694, 541, 237, 541, 237, 81, 
	82, 695, 696, 708, 692, 692, 693, 694, 
	276, 277, 276, 277, 81, 82, 695, 696, 
	708, 692, 692, 693, 694, 295, 296, 297, 
	298, 297, 298, 81, 82, 695, 696, 708, 
	692, 692, 693, 694, 334, 335, 334, 335, 
	81, 82, 695, 696, 708, 692, 692, 693, 
	694, 355, 356, 357, 355, 356, 357, 81, 
	82, 695, 696, 708, 692, 692, 693, 694, 
	418, 418, 81, 82, 695, 696, 708, 692, 
	692, 693, 694, 428, 429, 430, 428, 429, 
	430, 81, 82, 695, 696, 708, 692, 692, 
	693, 694, 458, 459, 460, 461, 458, 459, 
	460, 461, 81, 82, 695, 696, 708, 692, 
	692, 693, 694, 500, 501, 502, 500, 501, 
	502, 81, 82, 695, 696, 708, 720, 720, 
	85, 722, 723, 84, 84, 724, 725, 84, 
	84, 84, 721, 667, 667, 668, 669, 5, 
	6, 7, 8, 9, 10, 11, 12, 13, 
	14, 15, 16, 17, 18, 5, 6, 7, 
	8, 9, 10, 11, 12, 13, 14, 15, 
	16, 17, 18, 81, 82, 670, 671, 4, 
	95, 78, 81, 78, 79, 80, 82, 726, 
	670, 726, 727, 728, 671, 612, 77, 78, 
	81, 78, 79, 80, 82, 726, 670, 726, 
	727, 728, 671, 612, 83, 672, 81, 672, 
	673, 674, 729, 670, 729, 730, 731, 82, 
	678, 77, 732, 732, 35, 733, 733, 35, 
	734, 734, 735, 736, 737, 738, 739, 739, 
	740, 741, 55, 742, 742, 35, 743, 743, 
	35, 744, 744, 35, 745, 745, 746, 747, 
	748, 749, 750, 750, 751, 752, 55, 753, 
	753, 35, 754, 754, 755, 756, 757, 758, 
	759, 759, 760, 761, 55, 19, 113, 20, 
	114, 77, 762, 762, 763, 764, 81, 82, 
	81, 82, 20, 20, 20, 77, 762, 762, 
	763, 764, 81, 82, 81, 82, 20, 20, 
	20, 83, 762, 81, 762, 763, 764, 82, 
	765, 670, 765, 766, 767, 678, 20, 114, 
	77, 672, 81, 672, 673, 674, 768, 81, 
	768, 769, 770, 82, 89, 77, 649, 650, 
	69, 78, 81, 78, 79, 80, 82, 771, 
	670, 771, 772, 773, 678, 650, 77, 603, 
	603, 604, 605, 131, 132, 131, 132, 606, 
	607, 774, 603, 603, 604, 605, 157, 158, 
	159, 157, 158, 159, 606, 607, 774, 603, 
	603, 604, 605, 202, 202, 606, 607, 774, 
	603, 603, 604, 605, 224, 224, 606, 607, 
	774, 603, 603, 604, 605, 541, 237, 541, 
	237, 606, 607, 774, 603, 603, 604, 605, 
	276, 277, 276, 277, 606, 607, 774, 603, 
	603, 604, 605, 295, 296, 297, 298, 297, 
	298, 606, 607, 774, 603, 603, 604, 605, 
	334, 335, 334, 335, 606, 607, 774, 603, 
	603, 604, 605, 355, 356, 357, 355, 356, 
	357, 606, 607, 774, 603, 603, 604, 605, 
	418, 418, 606, 607, 774, 603, 603, 604, 
	605, 428, 429, 430, 428, 429, 430, 606, 
	607, 774, 603, 603, 604, 605, 458, 459, 
	460, 461, 458, 459, 460, 461, 606, 607, 
	774, 603, 603, 604, 605, 500, 501, 502, 
	500, 501, 502, 606, 607, 774, 19, 21, 
	20, 22, 83, 19, 539, 20, 540, 125, 
	603, 603, 604, 605, 236, 237, 236, 237, 
	606, 607, 774, 81, 129, 82, 130, 125, 
	776, 775, 778, 777, 780, 782, 783, 784, 
	781, 781, 781, 781, 779, 785, 786, 785, 
	785, 785, 779, 787, 788, 789, 787, 790, 
	789, 787, 787, 789, 789, 789, 779, 785, 
	786, 785, 792, 791, 785, 785, 791, 791, 
	791, 779, 793, 791, 795, 794, 796, 794, 
	797, 799, 797, 800, 798, 797, 797, 798, 
	798, 798, 794, 787, 787, 787, 787, 779, 
	801, 801, 801, 801, 779, 804, 805, 803, 
	803, 802, 806, 803, 806, 805, 806, 803, 
	807, 803, 802, 808, 803, 810, 809, 811, 
	809, 812, 813, 812, 815, 812, 813, 814, 
	813, 809, 816, 817, 818, 816, 816, 802, 
	806, 806, 806, 806, 819, 820, 816, 821, 
	809, 822, 809, 823, 824, 825, 823, 823, 
	809, 827, 827, 828, 829, 830, 831, 826, 
	827, 827, 828, 829, 830, 831, 833, 832, 
	835, 835, 836, 837, 839, 840, 841, 842, 
	843, 839, 840, 841, 842, 843, 844, 845, 
	844, 845, 838, 834, 835, 835, 836, 837, 
	844, 845, 844, 845, 846, 844, 845, 835, 
	835, 836, 837, 844, 845, 844, 845, 847, 
	834, 850, 851, 852, 853, 854, 850, 851, 
	852, 853, 854, 849, 848, 855, 855, 856, 
	857, 859, 860, 859, 860, 858, 834, 830, 
	831, 826, 862, 862, 863, 864, 865, 866, 
	861, 862, 862, 863, 864, 867, 869, 870, 
	868, 865, 866, 868, 868, 868, 861, 871, 
	861, 873, 873, 874, 875, 876, 877, 878, 
	877, 879, 880, 879, 880, 877, 877, 877, 
	872, 881, 881, 882, 883, 884, 885, 884, 
	885, 872, 884, 885, 871, 887, 886, 886, 
	886, 886, 861, 888, 886, 890, 889, 891, 
	889, 893, 894, 892, 892, 892, 892, 889, 
	873, 873, 874, 875, 879, 880, 879, 880, 
	872, 865, 866, 861, 895, 895, 896, 897, 
	898, 899, 613, 895, 895, 896, 897, 898, 
	899, 900, 900, 900, 901, 903, 903, 904, 
	905, 906, 907, 906, 908, 902, 910, 910, 
	911, 912, 913, 915, 916, 914, 917, 918, 
	917, 919, 914, 914, 914, 909, 921, 920, 
	922, 922, 923, 924, 925, 926, 927, 926, 
	928, 929, 928, 929, 926, 926, 926, 909, 
	930, 930, 931, 932, 917, 918, 917, 918, 
	83, 917, 918, 921, 934, 933, 933, 933, 
	933, 920, 935, 933, 937, 936, 938, 936, 
	940, 941, 939, 939, 939, 939, 936, 922, 
	922, 923, 924, 928, 929, 928, 929, 909, 
	917, 942, 918, 919, 83, 943, 943, 944, 
	945, 913, 915, 916, 914, 946, 947, 914, 
	914, 914, 920, 946, 947, 613, 898, 899, 
	613, 949, 949, 949, 949, 948, 950, 950, 
	950, 950, 948, 951, 952, 951, 951, 951, 
	948, 953, 954, 953, 953, 953, 948, 954, 
	954, 955, 956, 949, 957, 958, 959, 959, 
	960, 961, 949, 949, 949, 948, 957, 958, 
	948, 954, 948, 963, 963, 964, 965, 966, 
	967, 968, 968, 969, 970, 962, 971, 971, 
	972, 973, 974, 975, 976, 977, 978, 979, 
	980, 981, 982, 983, 984, 985, 986, 987, 
	974, 975, 976, 977, 978, 979, 980, 981, 
	982, 983, 984, 985, 986, 987, 988, 989, 
	990, 990, 991, 992, 962, 993, 993, 994, 
	995, 996, 997, 998, 996, 997, 998, 999, 
	1000, 1001, 1001, 1002, 1003, 962, 1004, 1004, 
	962, 1005, 1005, 962, 1006, 1006, 1007, 1008, 
	1009, 1010, 1011, 1011, 1012, 1013, 962, 988, 
	989, 962, 1014, 1015, 1014, 1015, 962, 1016, 
	1016, 962, 1017, 1017, 962, 1018, 1018, 1019, 
	1020, 1021, 1022, 1023, 1023, 1024, 1025, 962, 
	1026, 1026, 962, 1027, 1027, 962, 1028, 1028, 
	962, 1029, 1029, 1030, 1031, 1032, 1033, 1034, 
	1034, 1035, 1036, 962, 1037, 1038, 1039, 1037, 
	1038, 1039, 962, 1040, 1040, 962, 1041, 1041, 
	962, 1042, 1042, 962, 1043, 1043, 1044, 1045, 
	1046, 1047, 1048, 1048, 1049, 1050, 962, 1051, 
	1052, 1051, 1052, 962, 1053, 1053, 962, 1054, 
	1054, 962, 1055, 1055, 1056, 1057, 1058, 1059, 
	1060, 1060, 1061, 1062, 962, 1063, 1063, 962, 
	1064, 1064, 962, 1065, 1065, 962, 1066, 1066, 
	1067, 1068, 1069, 1070, 1071, 1071, 1072, 1073, 
	962, 1074, 1074, 1075, 1076, 1077, 1078, 1079, 
	1079, 1080, 1081, 962, 1082, 1082, 962, 1083, 
	1083, 962, 1084, 1085, 962, 1086, 962, 1087, 
	1087, 1088, 1089, 1090, 1091, 1092, 1092, 1093, 
	1094, 962, 1095, 962, 1096, 1096, 1097, 1098, 
	1099, 1100, 1101, 1101, 1102, 1103, 962, 1104, 
	1104, 962, 1105, 1105, 962, 1106, 1106, 962, 
	1107, 1107, 962, 1108, 1108, 1109, 1110, 1111, 
	1112, 1113, 1113, 1114, 1115, 962, 1116, 1116, 
	962, 1117, 1117, 962, 1118, 1118, 962, 1119, 
	1119, 962, 1120, 1120, 962, 1121, 1121, 962, 
	1122, 1122, 962, 1123, 1123, 1124, 1125, 1126, 
	1127, 1128, 1128, 1129, 1130, 962, 1131, 1132, 
	1131, 1132, 962, 1133, 1133, 962, 1134, 1134, 
	1135, 1136, 1137, 1138, 1139, 1139, 1140, 1141, 
	962, 1142, 1142, 1143, 1144, 1145, 1146, 1147, 
	1147, 1148, 1149, 962, 1150, 1151, 1152, 1153, 
	1152, 1153, 962, 1154, 962, 1155, 1155, 1156, 
	1157, 1158, 1159, 1160, 1160, 1161, 1162, 962, 
	1163, 962, 1164, 1164, 1165, 1166, 1167, 1168, 
	1169, 1169, 1170, 1171, 962, 1172, 1172, 962, 
	1173, 1173, 1174, 1175, 1176, 1177, 1178, 1178, 
	1179, 1180, 962, 1181, 1181, 1182, 1183, 1184, 
	1185, 1186, 1186, 1187, 1188, 962, 1189, 1190, 
	1189, 1190, 962, 1191, 1191, 962, 1192, 1192, 
	962, 1193, 1193, 962, 1194, 1194, 1195, 1196, 
	1197, 1198, 1199, 1199, 1200, 1201, 962, 1202, 
	1202, 1203, 1204, 1205, 1206, 1207, 1207, 1208, 
	1209, 962, 1210, 1211, 1212, 1210, 1211, 1212, 
	962, 1213, 1213, 962, 1214, 1214, 962, 1215, 
	1215, 962, 1216, 1216, 1217, 1218, 1219, 1220, 
	1221, 1221, 1222, 1223, 962, 1224, 1224, 962, 
	1225, 1225, 1226, 1227, 1228, 1229, 1230, 1230, 
	1231, 1232, 962, 1233, 1233, 1234, 1235, 1236, 
	1236, 1237, 1238, 1239, 1239, 1240, 1241, 962, 
	1242, 1242, 962, 1243, 1243, 1244, 1245, 1246, 
	1247, 1248, 1249, 1249, 1250, 1251, 962, 1252, 
	1252, 1253, 1254, 1255, 1255, 1256, 1257, 1258, 
	1258, 1259, 1260, 962, 1261, 1261, 962, 1262, 
	1262, 962, 1263, 1263, 962, 1264, 1264, 962, 
	1265, 1265, 1266, 1267, 1268, 1269, 1270, 1270, 
	1271, 1272, 962, 1273, 1273, 962, 1274, 1274, 
	962, 1275, 1275, 1276, 1277, 1278, 1279, 1280, 
	1280, 1281, 1282, 962, 1283, 1284, 1285, 1283, 
	1284, 1285, 962, 1286, 1286, 1287, 1288, 1289, 
	1290, 1291, 1291, 1292, 1293, 962, 1294, 1294, 
	962, 1295, 1295, 962, 1296, 1296, 962, 1297, 
	1297, 1298, 1299, 1300, 1301, 1302, 1302, 1303, 
	1304, 962, 1305, 1305, 1306, 1307, 1308, 1309, 
	1310, 1310, 1311, 1312, 962, 1313, 1314, 1315, 
	1316, 1313, 1314, 1315, 1316, 962, 1317, 1317, 
	962, 1318, 1318, 1319, 1320, 1321, 1322, 1323, 
	1323, 1324, 1325, 962, 1326, 1326, 962, 1327, 
	1327, 1328, 1329, 1330, 1331, 1332, 1332, 1333, 
	1334, 962, 1335, 1335, 962, 1336, 1336, 1337, 
	1338, 1339, 1340, 1341, 1341, 1342, 1343, 962, 
	1344, 1344, 962, 1345, 1345, 962, 1346, 1346, 
	962, 1347, 1347, 1348, 1349, 1350, 1351, 1352, 
	1352, 1353, 1354, 962, 1355, 1356, 1357, 1355, 
	1356, 1357, 962, 1358, 1358, 962, 1359, 1359, 
	962, 1360, 1360, 1361, 1362, 1363, 1364, 1365, 
	1365, 1366, 1367, 962, 1368, 1368, 962, 1369, 
	1369, 1370, 1371, 1372, 1373, 1374, 1374, 1375, 
	1376, 962, 1377, 1377, 962, 1378, 1378, 962, 
	1379, 962, 1380, 1380, 1381, 1382, 1384, 1385, 
	1386, 1386, 1387, 1388, 1383, 962, 1389, 1389, 
	962, 1390, 1390, 962, 1391, 1391, 962, 1392, 
	1392, 1393, 1394, 1395, 1396, 1397, 1397, 1398, 
	1399, 962, 1400, 1400, 962, 1401, 1401, 1402, 
	1403, 1404, 1405, 1406, 1406, 1407, 1408, 962, 
	1410, 1410, 1411, 1412, 1413, 1414, 1409, 1410, 
	1410, 1411, 1412, 1413, 1414, 1416, 1415, 1417, 
	1417, 1418, 1419, 1421, 1422, 1420, 1415, 1424, 
	1424, 1425, 1426, 1428, 1429, 1427, 1427, 1427, 
	1423, 1430, 1430, 1430, 1423, 1431, 1431, 1432, 
	1433, 1434, 1435, 1436, 1436, 1437, 1438, 1427, 
	1427, 1427, 1423, 1434, 1435, 1423, 1428, 1429, 
	1409, 1413, 1414, 1409, 1439, 1439, 1440, 1441, 
	1442, 1443, 1409, 1439, 1439, 1440, 1441, 1444, 
	1442, 1443, 1445, 1415, 1446, 1446, 1447, 1448, 
	1450, 1450, 1451, 1452, 1449, 1415, 1453, 1453, 
	1454, 1455, 1457, 1458, 1456, 1456, 1456, 1423, 
	1459, 1459, 1459, 1423, 1460, 1460, 1461, 1462, 
	1463, 1464, 1465, 1465, 1466, 1467, 1456, 1456, 
	1456, 1423, 1463, 1464, 1423, 1457, 1458, 1409, 
	1468, 1468, 1469, 1470, 1471, 1472, 1449, 1415, 
	1442, 1443, 1409, 1475, 1476, 1477, 1478, 1479, 
	1475, 1476, 1477, 1478, 1479, 1474, 1473, 1480, 
	1480, 1480, 1480, 1481, 1473, 1482, 1483, 1482, 
	1483, 1473, 1484, 1484, 1484, 1484, 1473, 1485, 
	1485, 1473, 1486, 1487, 1486, 1486, 1486, 1473, 
	1488, 1488, 1473, 1489, 1489, 1473, 1490, 1490, 
	1473, 1491, 1491, 1473, 1492, 1473, 1493, 1473, 
	1494, 1494, 1473, 1495, 1495, 1473, 1496, 1496, 
	1473, 1497, 1473, 1498, 1498, 1498, 1498, 1473, 
	1499, 1499, 1473, 1500, 1501, 1500, 1501, 1473, 
	1502, 1473, 1503, 1503, 1473, 1504, 1504, 1473, 
	1505, 1505, 1473, 1506, 1506, 1473, 1507, 1507, 
	1507, 1507, 1473, 1508, 1508, 1473, 1509, 1509, 
	1473, 1510, 1510, 1473, 1511, 1512, 1473, 1513, 
	1473, 1514, 1473, 1515, 1515, 1473, 1516, 1516, 
	1473, 1517, 1517, 1473, 1518, 1473, 1519, 1473, 
	1520, 1473, 1521, 1521, 1521, 1521, 1473, 1522, 
	1473, 1523, 1473, 1524, 1524, 1473, 1525, 1525, 
	1473, 1526, 1526, 1473, 1527, 1473, 1528, 1473, 
	1529, 1473, 1530, 1530, 1530, 1530, 1473, 1531, 
	1531, 1473, 1532, 1532, 1473, 1533, 1533, 1473, 
	1534, 1534, 1473, 1535, 1535, 1473, 1536, 1536, 
	1473, 1537, 1537, 1473, 1538, 1538, 1538, 1538, 
	1473, 1539, 1539, 1473, 1540, 1540, 1473, 1541, 
	1541, 1473, 1542, 1542, 1473, 1543, 1543, 1473, 
	1544, 1544, 1473, 1545, 1546, 1545, 1546, 1473, 
	1547, 1547, 1473, 1548, 1548, 1473, 1549, 1549, 
	1549, 1549, 1473, 1550, 1550, 1473, 1551, 1551, 
	1473, 1552, 1552, 1552, 1552, 1473, 1553, 1553, 
	1473, 1554, 1554, 1473, 1555, 1556, 1555, 1556, 
	1473, 1557, 1557, 1473, 1558, 1473, 1559, 1559, 
	1559, 1559, 1473, 1560, 1560, 1473, 1561, 1561, 
	1473, 1562, 1563, 1564, 1473, 1565, 1566, 1565, 
	1565, 1565, 1473, 1567, 1567, 1473, 1568, 1568, 
	1473, 1569, 1569, 1473, 1570, 1570, 1473, 1571, 
	1473, 1572, 1473, 1573, 1573, 1473, 1574, 1574, 
	1473, 1575, 1575, 1473, 1576, 1473, 1577, 1577, 
	1577, 1577, 1473, 1578, 1473, 1579, 1473, 1580, 
	1580, 1580, 1580, 1473, 1581, 1473, 1582, 1473, 
	1583, 1583, 1583, 1583, 1473, 1586, 1587, 1588, 
	1589, 1590, 1591, 1586, 1587, 1588, 1589, 1590, 
	1591, 1585, 1584, 1592, 1592, 1592, 1592, 1593, 
	1584, 1594, 1594, 1584, 1595, 1595, 1584, 1596, 
	1596, 1584, 1597, 1597, 1584, 1598, 1598, 1584, 
	1599, 1599, 1599, 1599, 1584, 1600, 1601, 1602, 
	1600, 1601, 1602, 1584, 1603, 1603, 1584, 1604, 
	1604, 1584, 1605, 1605, 1584, 1606, 1606, 1584, 
	1607, 1607, 1584, 1608, 1608, 1608, 1608, 1584, 
	1609, 1610, 1609, 1610, 1584, 1611, 1611, 1584, 
	1612, 1612, 1612, 1612, 1584, 1613, 1613, 1584, 
	1614, 1614, 1584, 1615, 1615, 1615, 1615, 1584, 
	1616, 1616, 1584, 1617, 1617, 1584, 1618, 1618, 
	1584, 1619, 1619, 1619, 1619, 1584, 1620, 1620, 
	1584, 1621, 1621, 1584, 1622, 1622, 1622, 1622, 
	1584, 1623, 1624, 1623, 1624, 1584, 1625, 1625, 
	1584, 1626, 1626, 1626, 1626, 1584, 1627, 1627, 
	1584, 1628, 1628, 1584, 1629, 1629, 1629, 1629, 
	1584, 1630, 1630, 1584, 1631, 1631, 1584, 1632, 
	1632, 1584, 1633, 1633, 1633, 1633, 1584, 1634, 
	1634, 1584, 1635, 1635, 1584, 1636, 1636, 1636, 
	1636, 1584, 1638, 1638, 1637, 1639, 1640, 1639, 
	1639, 1639, 1640, 1637, 1641, 1641, 1641, 1641, 
	1641, 1641, 69, 1642, 1642, 1642, 1642, 69, 
	1643, 1643, 1643, 1643, 1643, 1643, 69, 1644, 
	1644, 1645, 1646, 1647, 1648, 69, 1649, 1649, 
	1650, 1651, 1652, 1652, 1652, 1653, 1654, 1652, 
	1652, 1652, 69, 1655, 1655, 1656, 1657, 1658, 
	1659, 69, 1661, 1661, 1662, 1663, 1665, 1666, 
	1664, 1660, 1667, 1667, 1668, 1669, 1671, 1672, 
	1670, 1660, 1673, 1673, 1674, 1675, 1677, 1678, 
	1676, 1660, 1680, 1680, 1681, 1682, 1684, 1685, 
	1686, 1687, 1688, 1684, 1685, 1686, 1687, 1688, 
	1689, 1690, 1683, 1679, 1691, 1691, 1692, 1693, 
	1695, 1696, 1694, 1660, 1697, 1697, 1698, 1699, 
	1701, 1702, 1703, 1704, 1705, 1701, 1702, 1703, 
	1704, 1705, 1706, 1707, 1700, 1679, 1708, 1708, 
	1709, 1710, 1712, 1713, 1711, 1660, 1714, 1714, 
	1715, 1716, 1718, 1719, 1720, 1721, 1722, 1718, 
	1719, 1720, 1721, 1722, 1723, 1724, 1717, 1679, 
	1725, 1725, 1726, 1727, 1729, 1730, 1728, 1660, 
	1731, 1731, 1733, 1734, 1735, 1736, 1737, 1733, 
	1734, 1735, 1736, 1737, 1731, 1731, 1732, 1679, 
	1731, 1731, 1731, 1731, 1738, 1679, 1740, 1741, 
	1742, 1743, 1744, 1740, 1741, 1742, 1743, 1744, 
	1739, 1679, 1745, 1745, 1745, 1745, 1746, 1679, 
	1729, 1730, 69, 1714, 1714, 1715, 1716, 1723, 
	1724, 1747, 1679, 1749, 1750, 1751, 1752, 1753, 
	1749, 1750, 1751, 1752, 1753, 1748, 1679, 1754, 
	1754, 1755, 1756, 1758, 1759, 1757, 1679, 1712, 
	1713, 69, 1697, 1697, 1698, 1699, 1706, 1707, 
	1760, 1679, 1762, 1763, 1764, 1765, 1766, 1762, 
	1763, 1764, 1765, 1766, 1761, 1679, 1767, 1767, 
	1768, 1769, 1771, 1772, 1770, 1679, 1695, 1696, 
	69, 1680, 1680, 1681, 1682, 1689, 1690, 1773, 
	1679, 1775, 1776, 1777, 1778, 1779, 1775, 1776, 
	1777, 1778, 1779, 1774, 1679, 1780, 1780, 1781, 
	1782, 1784, 1785, 1783, 1679, 1677, 1678, 69, 
	1665, 1666, 69, 1653, 1654, 69, 69, 69, 
	69, 69, 1786, 1787, 1787, 1788, 1789, 1790, 
	1791, 69, 1793, 1793, 1794, 1795, 1796, 1797, 
	1792, 1792, 1792, 69, 1798, 1798, 1798, 1798, 
	69, 1796, 1797, 69, 1799, 1799, 1799, 1799, 
	1799, 1799, 69, 1800, 1800, 1801, 1802, 1803, 
	1804, 69, 1805, 1805, 1806, 1807, 1808, 1808, 
	1808, 1809, 1810, 1808, 1808, 1808, 69, 1811, 
	1811, 1811, 1811, 69, 1809, 1810, 69, 1812, 
	1660, 1813, 1813, 1814, 1815, 1817, 1818, 1816, 
	1660, 1819, 1819, 1820, 1821, 1822, 1822, 1822, 
	1823, 1824, 1822, 1822, 1822, 69, 1825, 1825, 
	1825, 1825, 69, 1823, 1824, 69, 69, 69, 
	69, 69, 1826, 1827, 1827, 1828, 1829, 1830, 
	1831, 1832, 1832, 1833, 1834, 69, 1835, 1835, 
	1836, 1837, 1838, 1839, 1840, 1840, 1841, 1842, 
	1826, 1826, 1826, 69, 1838, 1839, 69, 1843, 
	1843, 1843, 1843, 1637, 1844, 1845, 1844, 1844, 
	1844, 1845, 1845, 1845, 1637, 1847, 1846, 1848, 
	1848, 1849, 1850, 1852, 1853, 1851, 1846, 1854, 
	1854, 1855, 1856, 1858, 1859, 1860, 1861, 1857, 
	1846, 1862, 1862, 1863, 1864, 1866, 1867, 1865, 
	1846, 1868, 1868, 1869, 1870, 1858, 1859, 1872, 
	1873, 1871, 1846, 1874, 1874, 1875, 1876, 1877, 
	1879, 1880, 1878, 1846, 1881, 1881, 1882, 1883, 
	1858, 1859, 1884, 1885, 1846, 1886, 1886, 1887, 
	1888, 1889, 1890, 1846, 1886, 1886, 1887, 1888, 
	1889, 1890, 1891, 1846, 1892, 1892, 1893, 1894, 
	1896, 1897, 1895, 1846, 1898, 1898, 1899, 1900, 
	1902, 1903, 1904, 1905, 1901, 1846, 1906, 1906, 
	1907, 1908, 1910, 1911, 1909, 1846, 1912, 1912, 
	1913, 1914, 1902, 1903, 1916, 1917, 1915, 1846, 
	1918, 1918, 1919, 1920, 1921, 1923, 1924, 1922, 
	1846, 1925, 1925, 1926, 1927, 1902, 1903, 1928, 
	1929, 1846, 1930, 1930, 1931, 1932, 1933, 1934, 
	1846, 1930, 1930, 1931, 1932, 1935, 1933, 1934, 
	1936, 1846, 1937, 1846, 1938, 1938, 1939, 1940, 
	1941, 1943, 1944, 1945, 1946, 1946, 1947, 1948, 
	1942, 1846, 1949, 1949, 1950, 1951, 1953, 1954, 
	1955, 1955, 1956, 1957, 1952, 1846, 1958, 1958, 
	1959, 1960, 1961, 1963, 1964, 1965, 1966, 1966, 
	1967, 1968, 1962, 1846, 1969, 1969, 1970, 1971, 
	1973, 1974, 1955, 1955, 1975, 1976, 1972, 1846, 
	1977, 1977, 1978, 1979, 1980, 1982, 1983, 1984, 
	1985, 1985, 1986, 1987, 1981, 1846, 1988, 1988, 
	1989, 1990, 1992, 1993, 1955, 1955, 1994, 1995, 
	1991, 1846, 1996, 1996, 1997, 1998, 1999, 1996, 
	2001, 2002, 2003, 2003, 2004, 2005, 2000, 1846, 
	2006, 2006, 2007, 2008, 2009, 2010, 1955, 1955, 
	2011, 2012, 1846, 2009, 2010, 1846, 2013, 2013, 
	2014, 2015, 2013, 2017, 2018, 2019, 2019, 2020, 
	2021, 2016, 1846, 1996, 1996, 1997, 1998, 1996, 
	2001, 2002, 2003, 2003, 2004, 2005, 2022, 1846, 
	1992, 1993, 1846, 2023, 2023, 2024, 2025, 2027, 
	2028, 2029, 2030, 2030, 2031, 2032, 2026, 1846, 
	1977, 1977, 1978, 1979, 1982, 1983, 1984, 1985, 
	1985, 1986, 1987, 2033, 1846, 1988, 1988, 1989, 
	1990, 1992, 1993, 1955, 1955, 1994, 1995, 1846, 
	1973, 1974, 1846, 2034, 2034, 2035, 2036, 2038, 
	2039, 2040, 2041, 2041, 2042, 2043, 2037, 1846, 
	1958, 1958, 1959, 1960, 1963, 1964, 1965, 1966, 
	1966, 1967, 1968, 2044, 1846, 1969, 1969, 1970, 
	1971, 1973, 1974, 1955, 1955, 1975, 1976, 1846, 
	1953, 1954, 1846, 2045, 2045, 2046, 2047, 2049, 
	2050, 2051, 2052, 2052, 2053, 2054, 2048, 1846, 
	1938, 1938, 1939, 1940, 1943, 1944, 1945, 1946, 
	1946, 1947, 1948, 2055, 1846, 1949, 1949, 1950, 
	1951, 1953, 1954, 1955, 1955, 1956, 1957, 1846, 
	1933, 1934, 1846, 2056, 2056, 2057, 2058, 2059, 
	2060, 1846, 1928, 1929, 1846, 2061, 2061, 2062, 
	2063, 2065, 2066, 2064, 1846, 1918, 1918, 1919, 
	1920, 1923, 1924, 2067, 1846, 1916, 1917, 1846, 
	1904, 1905, 1846, 1889, 1890, 1846, 2068, 2068, 
	2069, 2070, 2071, 2072, 1846, 1884, 1885, 1846, 
	2073, 2073, 2074, 2075, 2077, 2078, 2076, 1846, 
	1874, 1874, 1875, 1876, 1879, 1880, 2079, 1846, 
	1872, 1873, 1846, 1860, 1861, 1846, 2080, 1660, 
	2081, 2081, 2082, 2083, 2085, 2086, 2084, 1660, 
	2087, 2087, 2088, 2089, 2091, 2092, 2090, 1660, 
	2093, 2093, 2094, 2095, 2097, 2098, 2096, 1660, 
	2099, 2099, 2100, 2101, 2103, 2104, 2102, 1660, 
	2105, 2105, 2106, 2107, 2109, 2110, 2108, 1660, 
	2111, 2111, 2112, 2113, 2114, 2114, 2114, 2115, 
	2116, 2114, 2114, 2114, 69, 2117, 2117, 2117, 
	2117, 69, 2115, 2116, 69, 2103, 2104, 69, 
	2091, 2092, 69, 2118, 1660, 2119, 2119, 2120, 
	2121, 2123, 2124, 2122, 1660, 2125, 2125, 2126, 
	2127, 2129, 2130, 2128, 1660, 2131, 2131, 2132, 
	2133, 2135, 2136, 2134, 1660, 2138, 2138, 2139, 
	2140, 2141, 2142, 2137, 2137, 2137, 69, 2143, 
	2143, 2144, 2145, 2146, 2147, 69, 2149, 2149, 
	2150, 2151, 2152, 2153, 2148, 2148, 2148, 69, 
	2154, 2154, 2155, 2156, 2157, 2158, 69, 2160, 
	2160, 2161, 2162, 2163, 2164, 2159, 2159, 2159, 
	69, 2165, 2165, 2166, 2167, 2168, 2169, 69, 
	2170, 2170, 2171, 2172, 2173, 2173, 2173, 2174, 
	2175, 2173, 2173, 2173, 69, 2176, 2176, 2176, 
	2176, 69, 2174, 2175, 69, 2163, 2164, 69, 
	2152, 2153, 69, 2141, 2142, 69, 2129, 2130, 
	69, 2177, 2177, 2177, 69, 2178, 2178, 2179, 
	2180, 2181, 2182, 69, 2178, 2178, 2179, 2180, 
	2181, 2182, 2183, 1660, 2184, 2184, 2185, 2186, 
	2188, 2189, 2187, 1660, 2190, 2190, 2191, 2192, 
	2194, 2195, 2193, 2193, 2193, 69, 2196, 2196, 
	2197, 2198, 2199, 2200, 69, 2196, 2196, 2197, 
	2198, 2201, 2199, 2200, 2201, 2201, 2201, 69, 
	2202, 2202, 69, 2199, 2200, 69, 2194, 2195, 
	69, 2181, 2182, 69, 2204, 2204, 2205, 2206, 
	2207, 2208, 2209, 2210, 2211, 2212, 2212, 2213, 
	2214, 2203, 2215, 2216, 2203, 2217, 2203, 2219, 
	2219, 2218, 2221, 2220, 2218, 2222, 2203, 2223, 
	2223, 2224, 2225, 2227, 2228, 2229, 2229, 2230, 
	2231, 2226, 2203, 2210, 2211, 69, 2232, 2203, 
	2233, 2233, 2233, 2233, 2218, 2235, 2234, 2234, 
	2234, 2218, 2236, 1660, 2237, 2237, 2238, 2239, 
	2241, 2242, 2240, 1660, 2243, 2243, 2244, 2245, 
	2247, 2248, 2246, 2246, 2246, 69, 2249, 2249, 
	2250, 2251, 2252, 2253, 69, 2249, 2249, 2250, 
	2251, 2252, 2253, 2254, 1660, 2255, 2255, 2256, 
	2257, 2259, 2260, 2258, 1660, 2262, 2262, 2263, 
	2264, 2266, 2267, 2265, 2265, 2265, 2261, 2268, 
	2268, 2268, 2261, 2269, 2269, 2270, 2271, 2272, 
	2273, 2274, 2274, 2275, 2276, 2265, 2265, 2265, 
	2261, 2272, 2273, 2261, 2266, 2267, 69, 2252, 
	2253, 69, 2247, 2248, 69, 2277, 1660, 2278, 
	2278, 2279, 2280, 2282, 2283, 2281, 1660, 2284, 
	2284, 2285, 2286, 2288, 2289, 2287, 1660, 2290, 
	2290, 2291, 2292, 2294, 2295, 2293, 1660, 2296, 
	2296, 2297, 2298, 2300, 2301, 2299, 2299, 2299, 
	2261, 2302, 2302, 2302, 2261, 2303, 2303, 2304, 
	2305, 2306, 2307, 2308, 2308, 2309, 2310, 2299, 
	2299, 2299, 2261, 2306, 2307, 2261, 2300, 2301, 
	69, 2288, 2289, 69, 2311, 1660, 2312, 2312, 
	2313, 2314, 2316, 2317, 2315, 1660, 2319, 2319, 
	2320, 2321, 2322, 2323, 2324, 2325, 2326, 2327, 
	2318, 2328, 2328, 2329, 2330, 2331, 2332, 2318, 
	2328, 2328, 2329, 2330, 2331, 2332, 2334, 2333, 
	2335, 2335, 2336, 2337, 2339, 2340, 2338, 2333, 
	2341, 2341, 2342, 2343, 2344, 2345, 2346, 2318, 
	2348, 2348, 2349, 2350, 2351, 2351, 2352, 2353, 
	2354, 2354, 2355, 2356, 2357, 2357, 2357, 2357, 
	2358, 2359, 2357, 2357, 2360, 2361, 2362, 2363, 
	2357, 2357, 2364, 2365, 2366, 2367, 2357, 2357, 
	2368, 2369, 2347, 2348, 2348, 2349, 2350, 2371, 
	2358, 2359, 2371, 2371, 2371, 2370, 2357, 2357, 
	2370, 2358, 2359, 2370, 2351, 2351, 2352, 2353, 
	2357, 2357, 2357, 2357, 2362, 2363, 2357, 2357, 
	2364, 2365, 2370, 2362, 2363, 2370, 2371, 2348, 
	2348, 2349, 2350, 2351, 2351, 2352, 2353, 2354, 
	2354, 2355, 2356, 2357, 2357, 2357, 2357, 2358, 
	2359, 2357, 2357, 2360, 2361, 2362, 2363, 2357, 
	2357, 2364, 2365, 2366, 2367, 2357, 2357, 2368, 
	2369, 2371, 2371, 2371, 2370, 2358, 2362, 2366, 
	2359, 2363, 2367, 2370, 2345, 2346, 2318, 2331, 
	2332, 2318, 2372, 2372, 2373, 2374, 2375, 2376, 
	2318, 2372, 2372, 2373, 2374, 2375, 2376, 2377, 
	2333, 2378, 2378, 2379, 2380, 2382, 2383, 2381, 
	2333, 2385, 2385, 2386, 2387, 2388, 2389, 2390, 
	2388, 2384, 2392, 2393, 2393, 2394, 2395, 2396, 
	2396, 2397, 2398, 2399, 2399, 2400, 2401, 2402, 
	2402, 2402, 2402, 2403, 2404, 2402, 2402, 2405, 
	2406, 2407, 2408, 2402, 2402, 2409, 2410, 2411, 
	2412, 2402, 2402, 2413, 2414, 2392, 2391, 2389, 
	2390, 2318, 2375, 2376, 2318, 2415, 2415, 2416, 
	2417, 2418, 2419, 2318, 2415, 2415, 2416, 2417, 
	2418, 2419, 2420, 2333, 2421, 2421, 2422, 2423, 
	2425, 2426, 2424, 2333, 2427, 2427, 2428, 2429, 
	2430, 2431, 2432, 2430, 2430, 2430, 2384, 2433, 
	2434, 2434, 2435, 2436, 2437, 2437, 2438, 2439, 
	2440, 2440, 2441, 2442, 2443, 2443, 2443, 2443, 
	2444, 2445, 2443, 2443, 2446, 2447, 2448, 2449, 
	2443, 2443, 2450, 2451, 2452, 2453, 2443, 2443, 
	2454, 2455, 2433, 2433, 2433, 2391, 2431, 2432, 
	2318, 2418, 2419, 2318, 2456, 2456, 2457, 2458, 
	2459, 2460, 2318, 2456, 2456, 2457, 2458, 2459, 
	2460, 2461, 2333, 2462, 2462, 2463, 2464, 2466, 
	2467, 2465, 2333, 2468, 2468, 2469, 2470, 2471, 
	2471, 2471, 2472, 2473, 2471, 2471, 2471, 2318, 
	2474, 2474, 2475, 2476, 2477, 2477, 2478, 2479, 
	2480, 2480, 2481, 2482, 2483, 2483, 2483, 2483, 
	2484, 2485, 2483, 2483, 2486, 2487, 2488, 2489, 
	2483, 2483, 2490, 2491, 2492, 2493, 2483, 2483, 
	2494, 2495, 2347, 2472, 2473, 2318, 2459, 2460, 
	2318, 2326, 2327, 69, 2497, 2498, 2499, 2500, 
	2501, 2502, 2503, 2504, 2505, 2506, 2507, 2508, 
	2509, 2510, 2497, 2498, 2499, 2500, 2501, 2502, 
	2503, 2504, 2505, 2506, 2507, 2508, 2509, 2510, 
	2496, 2511, 2511, 2512, 2513, 2514, 2515, 2516, 
	2514, 2515, 2516, 2517, 2518, 2496, 2519, 2519, 
	2520, 2521, 2523, 2524, 2522, 2522, 2522, 69, 
	2525, 2525, 2526, 2527, 2528, 2529, 69, 2525, 
	2525, 2526, 2527, 2528, 2529, 2530, 1660, 2531, 
	2531, 2532, 2533, 2535, 2536, 2534, 1660, 2537, 
	2537, 2538, 2539, 2541, 2542, 2540, 1660, 2543, 
	2543, 2544, 2545, 2547, 2548, 2546, 1660, 2550, 
	2550, 2551, 2552, 2554, 2555, 2553, 2549, 2556, 
	2556, 2557, 2558, 2560, 2561, 2559, 2549, 2562, 
	2562, 2563, 2564, 2566, 2567, 2565, 2549, 2568, 
	2568, 2569, 2570, 2572, 2573, 2571, 2549, 2574, 
	2574, 2575, 2576, 2578, 2579, 2577, 1660, 2580, 
	2580, 2581, 2582, 2584, 2585, 2583, 1660, 2586, 
	2586, 2587, 2588, 2589, 2589, 2589, 2590, 2591, 
	2589, 2589, 2589, 69, 2592, 2592, 2593, 2594, 
	2595, 2596, 69, 2597, 2597, 2598, 2599, 2600, 
	2601, 2602, 2600, 2600, 2600, 69, 2603, 2603, 
	69, 2601, 2602, 69, 2590, 2591, 69, 2578, 
	2579, 69, 2566, 2567, 69, 2554, 2555, 69, 
	2541, 2542, 69, 2528, 2529, 69, 2523, 2524, 
	69, 2604, 2604, 2496, 2605, 2605, 2496, 2606, 
	2606, 2607, 2608, 2609, 2610, 2496, 2611, 2611, 
	2496, 2612, 2612, 2496, 2613, 2613, 2496, 2614, 
	2614, 2615, 2616, 2617, 2618, 2496, 2619, 2619, 
	2496, 2620, 2620, 2621, 2622, 2623, 2624, 2496, 
	2625, 2626, 2625, 2626, 2496, 2627, 2627, 2496, 
	2628, 2628, 2496, 2629, 2629, 2630, 2631, 2632, 
	2633, 2496, 2634, 2634, 2496, 2635, 2635, 2496, 
	2636, 2636, 2496, 2637, 2637, 2638, 2639, 2640, 
	2641, 2496, 2642, 2643, 2644, 2642, 2643, 2644, 
	2496, 2645, 2645, 2496, 2646, 2646, 2496, 2647, 
	2647, 2496, 2648, 2648, 2649, 2650, 2651, 2652, 
	2496, 2653, 2654, 2653, 2654, 2496, 2655, 2655, 
	2496, 2656, 2656, 2496, 2657, 2657, 2658, 2659, 
	2660, 2661, 2496, 2662, 2662, 2496, 2663, 2663, 
	2496, 2664, 2664, 2496, 2665, 2665, 2666, 2667, 
	2668, 2669, 2496, 2670, 2670, 2671, 2672, 2673, 
	2674, 2496, 2675, 2675, 2496, 2676, 2676, 2496, 
	2677, 2678, 2496, 2679, 2496, 2680, 2680, 2681, 
	2682, 2683, 2684, 2496, 2685, 2496, 2686, 2686, 
	2687, 2688, 2689, 2690, 2496, 2691, 2691, 2496, 
	2692, 2692, 2496, 2693, 2693, 2496, 2694, 2694, 
	2496, 2695, 2695, 2696, 2697, 2698, 2699, 2496, 
	2700, 2700, 2496, 2701, 2701, 2496, 2702, 2702, 
	2496, 2703, 2703, 2496, 2704, 2704, 2496, 2705, 
	2705, 2496, 2706, 2706, 2496, 2707, 2707, 2708, 
	2709, 2710, 2711, 2496, 2712, 2713, 2712, 2713, 
	2496, 2714, 2714, 2496, 2715, 2715, 2716, 2717, 
	2718, 2719, 2496, 2720, 2720, 2721, 2722, 2723, 
	2724, 2496, 2725, 2726, 2727, 2728, 2727, 2728, 
	2496, 2729, 2496, 2730, 2730, 2731, 2732, 2733, 
	2734, 2496, 2735, 2496, 2736, 2736, 2737, 2738, 
	2739, 2740, 2496, 2741, 2741, 2496, 2742, 2742, 
	2743, 2744, 2745, 2746, 2496, 2747, 2747, 2748, 
	2749, 2750, 2751, 2496, 2752, 2753, 2752, 2753, 
	2496, 2754, 2754, 2496, 2755, 2755, 2496, 2756, 
	2756, 2496, 2757, 2757, 2758, 2759, 2760, 2761, 
	2496, 2762, 2762, 2763, 2764, 2765, 2766, 2496, 
	2767, 2768, 2769, 2767, 2768, 2769, 2496, 2770, 
	2770, 2496, 2771, 2771, 2496, 2772, 2772, 2496, 
	2773, 2773, 2774, 2775, 2776, 2777, 2496, 2778, 
	2778, 2496, 2779, 2779, 2780, 2781, 2782, 2783, 
	2496, 2784, 2784, 2785, 2786, 2787, 2787, 2788, 
	2789, 2496, 2790, 2790, 2496, 2791, 2791, 2792, 
	2793, 2794, 2795, 2796, 2496, 2797, 2797, 2798, 
	2799, 2800, 2800, 2801, 2802, 2496, 2803, 2803, 
	2496, 2804, 2804, 2496, 2805, 2805, 2496, 2806, 
	2806, 2496, 2807, 2807, 2808, 2809, 2810, 2811, 
	2496, 2812, 2812, 2496, 2813, 2813, 2496, 2814, 
	2814, 2815, 2816, 2817, 2818, 2496, 2819, 2820, 
	2821, 2819, 2820, 2821, 2496, 2822, 2822, 2823, 
	2824, 2825, 2826, 2496, 2827, 2827, 2496, 2828, 
	2828, 2496, 2829, 2829, 2496, 2830, 2830, 2831, 
	2832, 2833, 2834, 2496, 2835, 2835, 2836, 2837, 
	2838, 2839, 2496, 2840, 2841, 2842, 2843, 2840, 
	2841, 2842, 2843, 2496, 2844, 2844, 2496, 2845, 
	2845, 2846, 2847, 2848, 2849, 2496, 2850, 2850, 
	2496, 2851, 2851, 2852, 2853, 2854, 2855, 2496, 
	2856, 2856, 2496, 2857, 2857, 2858, 2859, 2860, 
	2861, 2496, 2862, 2862, 2496, 2863, 2863, 2496, 
	2864, 2864, 2496, 2865, 2865, 2866, 2867, 2868, 
	2869, 2496, 2870, 2871, 2872, 2870, 2871, 2872, 
	2496, 2873, 2873, 2496, 2874, 2874, 2496, 2875, 
	2875, 2876, 2877, 2878, 2879, 2496, 2880, 2880, 
	2496, 2881, 2881, 2882, 2883, 2884, 2885, 2496, 
	2886, 2886, 2496, 2887, 2887, 2496, 2889, 2888, 
	2890, 2890, 2891, 2892, 2894, 2895, 2893, 2888, 
	2896, 2896, 2896, 2896, 2896, 2896, 69, 2897, 
	2897, 2897, 2897, 69, 2898, 2898, 2898, 2898, 
	69, 2899, 1660, 2900, 2900, 2901, 2902, 2904, 
	2905, 2903, 1660, 2906, 2906, 2907, 2908, 2910, 
	2911, 2909, 1660, 2912, 2912, 2913, 2914, 2916, 
	2917, 2915, 1660, 2918, 2918, 2919, 2920, 2922, 
	2923, 2921, 2921, 2921, 69, 2924, 2924, 2925, 
	2926, 2927, 2928, 69, 2924, 2924, 2925, 2926, 
	2929, 2927, 2928, 2929, 2929, 2929, 69, 2930, 
	2930, 69, 2927, 2928, 69, 2922, 2923, 69, 
	2910, 2911, 69, 2931, 2931, 2931, 2931, 69, 
	2932, 2932, 69, 2933, 1660, 2934, 2934, 2935, 
	2936, 2938, 2939, 2937, 1660, 2940, 2940, 2941, 
	2942, 2944, 2945, 2943, 1660, 2946, 2946, 2947, 
	2948, 2950, 2951, 2949, 1660, 2952, 2952, 2953, 
	2954, 2956, 2957, 2955, 1660, 2958, 2958, 2959, 
	2960, 2962, 2963, 2961, 1660, 2964, 2964, 2965, 
	2966, 2967, 2969, 2970, 2968, 2968, 2968, 2261, 
	2971, 2971, 2972, 2973, 2974, 2975, 2261, 2977, 
	2977, 2978, 2979, 2981, 2982, 2980, 2980, 2980, 
	2976, 2983, 2983, 2983, 2976, 2985, 2984, 2984, 
	2984, 2976, 2986, 2986, 2986, 2976, 2988, 2987, 
	2987, 2987, 2976, 2990, 2989, 2989, 2989, 2976, 
	2991, 2991, 2991, 2976, 2993, 2992, 2992, 2992, 
	2976, 2994, 2994, 2994, 2994, 2995, 2995, 2995, 
	2976, 2996, 2996, 2996, 2996, 69, 2997, 2976, 
	2993, 2976, 2990, 2976, 2998, 2976, 2988, 2976, 
	2981, 2982, 69, 2999, 2999, 2999, 2261, 2971, 
	2971, 2972, 2973, 2974, 2975, 3000, 3000, 3000, 
	2261, 2969, 2970, 69, 2956, 2957, 69, 2944, 
	2945, 69, 3001, 1660, 3002, 3002, 3003, 3004, 
	3006, 3007, 3005, 1660, 3008, 3008, 3009, 3010, 
	3012, 3013, 3011, 1660, 3014, 3014, 3015, 3016, 
	3018, 3019, 3017, 1660, 3020, 3020, 3021, 3022, 
	3024, 3025, 3023, 1660, 3026, 3026, 3027, 3028, 
	3030, 3031, 3029, 1660, 3032, 3032, 3033, 3034, 
	3035, 3037, 3038, 3036, 3036, 3036, 2261, 3039, 
	3039, 3039, 3039, 2261, 3040, 3040, 3040, 2261, 
	3039, 3039, 3039, 3039, 3041, 3041, 3041, 2261, 
	3037, 3038, 69, 3024, 3025, 69, 3012, 3013, 
	69, 3042, 1660, 3043, 3043, 3044, 3045, 3047, 
	3048, 3046, 1660, 3049, 3049, 3050, 3051, 3053, 
	3054, 3052, 1660, 3055, 3055, 3056, 3057, 3059, 
	3060, 3058, 1660, 3061, 3061, 3062, 3063, 3065, 
	3066, 3064, 1660, 3067, 3067, 3068, 3069, 3071, 
	3072, 3070, 1660, 3073, 3073, 3074, 3075, 3077, 
	3078, 3076, 3076, 3076, 2261, 3079, 3079, 3079, 
	2261, 3080, 3080, 3081, 3082, 3083, 3084, 3085, 
	3085, 3086, 3087, 3076, 3076, 3076, 2261, 3083, 
	3084, 2261, 3077, 3078, 69, 3065, 3066, 69, 
	3053, 3054, 69, 3088, 1660, 3089, 3089, 3090, 
	3091, 3093, 3094, 3092, 1660, 3095, 3095, 3096, 
	3097, 3098, 3099, 3100, 3098, 1637, 3101, 3102, 
	3101, 3101, 3101, 3102, 1637, 3099, 3100, 69, 
	3103, 1660, 3104, 3104, 3105, 3106, 3108, 3109, 
	3107, 1660, 3110, 3110, 3111, 3112, 3114, 3115, 
	3113, 3113, 3113, 2261, 3116, 3116, 3116, 2261, 
	3117, 3117, 3117, 2261, 3118, 3118, 3118, 2261, 
	3120, 3119, 3121, 3121, 3121, 2261, 3122, 3122, 
	3122, 2261, 3123, 3123, 3123, 2261, 3124, 3124, 
	3124, 2261, 3126, 3120, 3126, 3126, 3126, 3125, 
	3114, 3115, 69, 3127, 3127, 3127, 2261, 3128, 
	3128, 3128, 2261, 3130, 3129, 3131, 3131, 3131, 
	2261, 3132, 3132, 3132, 2261, 3133, 3130, 3133, 
	3133, 3133, 3129, 3134, 3134, 3134, 2261, 3135, 
	3135, 3135, 2261, 3136, 3129, 3137, 3137, 3137, 
	2261, 3138, 3138, 3138, 2261, 3139, 3136, 3139, 
	3139, 3139, 3129, 3141, 3141, 3142, 3143, 3144, 
	3145, 3145, 3145, 19, 20, 3146, 3147, 3145, 
	3145, 3145, 3140, 3148, 3148, 3149, 3150, 3151, 
	3152, 67, 3153, 3152, 19, 20, 3154, 3155, 
	115, 116, 3156, 3157, 67, 67, 3152, 67, 
	3152, 67, 3152, 67, 691, 3159, 3159, 3142, 
	3160, 3161, 3145, 3162, 3163, 3164, 3165, 3166, 
	3167, 3168, 3169, 3170, 3171, 3172, 3173, 3174, 
	3175, 3145, 3145, 3162, 3163, 3164, 3165, 3166, 
	3167, 3168, 3169, 3170, 3171, 3172, 3173, 3174, 
	3175, 19, 20, 3176, 3177, 3145, 3145, 3145, 
	3158, 3159, 3159, 3142, 3160, 3161, 3145, 3180, 
	3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 
	3189, 3190, 3191, 3192, 3193, 3145, 3145, 3180, 
	3181, 3182, 3183, 3184, 3185, 3186, 3187, 3188, 
	3189, 3190, 3191, 3192, 3193, 19, 20, 3176, 
	3177, 3145, 3179, 3145, 3145, 3178, 3195, 3195, 
	3196, 3197, 3198, 3199, 3199, 3199, 3200, 3201, 
	3202, 3203, 3199, 3199, 3199, 3194, 3141, 3141, 
	3142, 3204, 3205, 3145, 3180, 3181, 3182, 3183, 
	3184, 3185, 3186, 3187, 3188, 3189, 3190, 3191, 
	3192, 3193, 3145, 3145, 3180, 3181, 3182, 3183, 
	3184, 3185, 3186, 3187, 3188, 3189, 3190, 3191, 
	3192, 3193, 19, 20, 3206, 3147, 3145, 3179, 
	3145, 3145, 3178, 3207, 3207, 3142, 3208, 3209, 
	3145, 3145, 3145, 81, 82, 3210, 3211, 3145, 
	3145, 3145, 691, 3212, 3212, 3149, 3213, 3214, 
	3152, 67, 3153, 3152, 81, 82, 3210, 3215, 
	90, 91, 3210, 3216, 67, 67, 3152, 67, 
	3152, 67, 3152, 67, 691, 3148, 3148, 3149, 
	3150, 3151, 3152, 67, 3219, 3220, 3221, 3222, 
	3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 
	3231, 3232, 3153, 3152, 3219, 3220, 3221, 3222, 
	3223, 3224, 3225, 3226, 3227, 3228, 3229, 3230, 
	3231, 3232, 19, 20, 3154, 3155, 115, 116, 
	3156, 3157, 67, 67, 3152, 3218, 67, 3152, 
	67, 3152, 67, 3217, 3212, 3212, 3149, 3213, 
	3214, 3152, 67, 3219, 3220, 3221, 3222, 3223, 
	3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 
	3232, 3153, 3152, 3219, 3220, 3221, 3222, 3223, 
	3224, 3225, 3226, 3227, 3228, 3229, 3230, 3231, 
	3232, 81, 82, 3233, 3215, 90, 91, 3233, 
	3216, 67, 67, 3152, 3218, 67, 3152, 67, 
	3152, 67, 3217, 613, 613, 613, 806, 803, 
	806, 805, 806, 803, 807, 803, 802, 613, 
	613, 613, 943, 943, 944, 945, 913, 915, 
	916, 914, 946, 947, 914, 914, 914, 920, 
	613, 954, 954, 955, 956, 949, 957, 958, 
	959, 959, 960, 961, 949, 949, 949, 948, 
	957, 958, 948, 613, 971, 971, 972, 973, 
	974, 975, 976, 977, 978, 979, 980, 981, 
	982, 983, 984, 985, 986, 987, 974, 975, 
	976, 977, 978, 979, 980, 981, 982, 983, 
	984, 985, 986, 987, 988, 989, 990, 990, 
	991, 992, 962, 988, 989, 962, 613, 1431, 
	1431, 1432, 1433, 1434, 1435, 1436, 1436, 1437, 
	1438, 1427, 1427, 1427, 1423, 1434, 1435, 1423, 
	1453, 1453, 1454, 1455, 1457, 1458, 1456, 1456, 
	1456, 1423, 1409, 1460, 1460, 1461, 1462, 1463, 
	1464, 1465, 1465, 1466, 1467, 1456, 1456, 1456, 
	1423, 1463, 1464, 1423, 1457, 1458, 1409, 613, 
	613, 613, 613, 613, 613, 613, 613, 613, 
	1835, 1835, 1836, 1837, 1838, 1839, 1840, 1840, 
	1841, 1842, 1826, 1826, 1826, 69, 1838, 1839, 
	69, 613, 613, 2006, 2006, 2007, 2008, 2009, 
	2010, 1955, 1955, 2011, 2012, 1846, 2009, 2010, 
	1846, 1988, 1988, 1989, 1990, 1992, 1993, 1955, 
	1955, 1994, 1995, 1991, 1846, 1992, 1993, 1846, 
	1969, 1969, 1970, 1971, 1973, 1974, 1955, 1955, 
	1975, 1976, 1972, 1846, 1973, 1974, 1846, 1949, 
	1949, 1950, 1951, 1953, 1954, 1955, 1955, 1956, 
	1957, 1952, 1846, 1953, 1954, 1846, 613, 613, 
	613, 613, 2204, 2204, 2205, 2206, 2207, 2208, 
	2209, 2210, 2211, 2212, 2212, 2213, 2214, 2203, 
	2210, 2211, 69, 613, 2269, 2269, 2270, 2271, 
	2272, 2273, 2274, 2274, 2275, 2276, 2265, 2265, 
	2265, 2261, 2272, 2273, 2261, 613, 2303, 2303, 
	2304, 2305, 2306, 2307, 2308, 2308, 2309, 2310, 
	2299, 2299, 2299, 2261, 2306, 2307, 2261, 613, 
	2351, 2351, 2352, 2353, 2357, 2357, 2357, 2357, 
	2362, 2363, 2357, 2357, 2364, 2365, 2370, 2362, 
	2363, 2370, 2348, 2348, 2349, 2350, 2371, 2358, 
	2359, 2371, 2371, 2371, 2370, 2358, 2359, 2370, 
	2371, 2348, 2348, 2349, 2350, 2351, 2351, 2352, 
	2353, 2354, 2354, 2355, 2356, 2357, 2357, 2357, 
	2357, 2358, 2359, 2357, 2357, 2360, 2361, 2362, 
	2363, 2357, 2357, 2364, 2365, 2366, 2367, 2357, 
	2357, 2368, 2369, 2371, 2371, 2371, 2370, 2358, 
	2362, 2366, 2359, 2363, 2367, 2370, 613, 613, 
	613, 613, 613, 613, 613, 3080, 3080, 3081, 
	3082, 3083, 3084, 3085, 3085, 3086, 3087, 3076, 
	3076, 3076, 2261, 3083, 3084, 2261, 613, 613, 
	613, 613, 0
};

static const short _zone_scanner_trans_targs[] = {
	0, 1, 1, 1, 2, 4, 16, 25, 
	39, 46, 129, 62, 66, 74, 80, 96, 
	99, 106, 117, 1059, 137, 1064, 242, 0, 
	3, 3, 3, 2, 152, 152, 152, 152, 
	152, 3, 155, 0, 3, 3, 3, 4, 
	16, 25, 39, 46, 51, 62, 66, 74, 
	80, 96, 99, 106, 117, 3, 155, 0, 
	5, 5, 5, 213, 216, 220, 5, 10, 
	6, 11, 20, 6, 5, 0, 5, 5, 
	9, 5, 10, 11, 20, 0, 7, 7, 
	7, 1059, 8, 0, 6, 6, 11, 11, 
	11, 12, 1065, 162, 226, 1060, 13, 0, 
	13, 13, 14, 158, 165, 166, 167, 168, 
	169, 170, 171, 172, 173, 174, 175, 176, 
	177, 1067, 222, 1065, 223, 1068, 225, 0, 
	15, 15, 15, 1061, 245, 0, 15, 15, 
	15, 1061, 245, 17, 21, 18, 19, 5, 
	5, 5, 5, 10, 6, 11, 20, 20, 
	20, 20, 22, 23, 24, 5, 5, 5, 
	5, 10, 6, 11, 20, 26, 30, 38, 
	27, 28, 29, 5, 5, 5, 5, 10, 
	6, 11, 20, 31, 34, 32, 33, 5, 
	5, 5, 5, 10, 6, 11, 20, 35, 
	36, 37, 5, 5, 5, 5, 10, 6, 
	11, 20, 5, 5, 5, 5, 10, 6, 
	11, 20, 40, 41, 42, 44, 43, 5, 
	5, 5, 5, 10, 6, 11, 20, 45, 
	5, 5, 5, 5, 10, 6, 11, 20, 
	47, 48, 49, 50, 5, 5, 5, 5, 
	10, 6, 11, 20, 52, 55, 53, 53, 
	53, 53, 127, 53, 53, 53, 4, 16, 
	25, 39, 46, 54, 62, 66, 74, 80, 
	96, 99, 106, 117, 53, 127, 56, 57, 
	58, 59, 60, 61, 5, 5, 5, 5, 
	10, 6, 11, 20, 63, 65, 64, 5, 
	5, 5, 5, 10, 6, 11, 20, 5, 
	5, 5, 5, 10, 6, 11, 20, 67, 
	69, 71, 73, 68, 5, 5, 5, 5, 
	10, 6, 11, 20, 70, 5, 5, 5, 
	5, 10, 6, 11, 20, 72, 5, 5, 
	5, 5, 10, 6, 11, 20, 5, 5, 
	5, 5, 10, 6, 11, 20, 75, 79, 
	76, 77, 78, 5, 5, 5, 5, 10, 
	6, 11, 20, 5, 5, 5, 5, 10, 
	6, 11, 20, 81, 85, 87, 82, 83, 
	84, 5, 5, 5, 5, 10, 6, 11, 
	20, 86, 5, 5, 5, 5, 10, 6, 
	11, 20, 5, 5, 5, 88, 5, 10, 
	6, 11, 20, 89, 5, 5, 5, 90, 
	5, 10, 6, 11, 20, 5, 5, 5, 
	91, 5, 10, 6, 11, 20, 92, 93, 
	94, 95, 5, 5, 5, 5, 10, 6, 
	11, 20, 97, 98, 5, 5, 5, 5, 
	10, 6, 11, 20, 100, 101, 105, 5, 
	5, 5, 5, 10, 6, 11, 20, 102, 
	103, 104, 5, 5, 5, 5, 10, 6, 
	11, 20, 5, 5, 5, 5, 10, 6, 
	11, 20, 107, 109, 111, 113, 108, 5, 
	5, 5, 5, 10, 6, 11, 20, 110, 
	5, 5, 5, 5, 10, 6, 11, 20, 
	112, 5, 5, 5, 5, 10, 6, 11, 
	20, 114, 115, 116, 5, 5, 5, 5, 
	10, 6, 11, 20, 118, 121, 123, 119, 
	120, 5, 5, 5, 5, 10, 6, 11, 
	20, 122, 5, 5, 5, 5, 10, 6, 
	11, 20, 124, 125, 0, 126, 0, 5, 
	5, 5, 126, 5, 10, 6, 11, 20, 
	128, 128, 128, 1062, 243, 130, 131, 131, 
	131, 131, 136, 131, 131, 131, 132, 4, 
	16, 25, 39, 46, 54, 62, 66, 74, 
	80, 96, 99, 106, 117, 131, 136, 53, 
	53, 53, 132, 133, 133, 133, 133, 133, 
	53, 127, 134, 134, 135, 135, 135, 135, 
	135, 53, 53, 53, 134, 53, 127, 0, 
	139, 180, 186, 140, 141, 142, 143, 144, 
	145, 1063, 0, 147, 147, 147, 147, 148, 
	147, 147, 147, 147, 148, 0, 0, 150, 
	150, 150, 150, 151, 150, 150, 150, 150, 
	151, 153, 153, 154, 154, 154, 154, 154, 
	3, 3, 3, 153, 3, 155, 0, 157, 
	157, 157, 157, 227, 164, 228, 157, 157, 
	157, 157, 227, 164, 228, 0, 159, 159, 
	159, 1060, 12, 1065, 162, 1066, 226, 159, 
	159, 159, 1066, 160, 160, 160, 1064, 161, 
	163, 163, 163, 164, 164, 164, 178, 212, 
	1067, 0, 181, 182, 183, 184, 185, 1063, 
	187, 188, 1063, 0, 160, 160, 160, 1064, 
	161, 0, 191, 191, 191, 1062, 192, 191, 
	191, 191, 1062, 192, 0, 194, 194, 194, 
	1067, 178, 1068, 212, 194, 194, 194, 1068, 
	209, 0, 209, 209, 209, 210, 211, 211, 
	211, 212, 212, 212, 214, 215, 5, 5, 
	5, 5, 10, 6, 11, 20, 217, 218, 
	219, 5, 5, 5, 5, 10, 6, 11, 
	20, 221, 5, 5, 5, 5, 10, 6, 
	11, 20, 224, 224, 224, 225, 225, 225, 
	226, 226, 226, 228, 228, 228, 0, 247, 
	1069, 247, 1069, 0, 249, 251, 256, 257, 
	252, 1070, 250, 1070, 249, 251, 252, 251, 
	252, 253, 0, 254, 255, 1070, 251, 250, 
	252, 1070, 0, 259, 264, 260, 1071, 1072, 
	261, 0, 262, 263, 1071, 259, 1072, 260, 
	264, 265, 266, 0, 267, 268, 269, 264, 
	265, 266, 0, 271, 271, 271, 271, 278, 
	0, 272, 0, 273, 273, 273, 272, 275, 
	275, 275, 275, 275, 1073, 274, 0, 276, 
	0, 276, 277, 277, 277, 277, 277, 273, 
	273, 273, 276, 1073, 274, 0, 280, 280, 
	280, 280, 291, 281, 285, 290, 286, 282, 
	0, 283, 283, 283, 281, 285, 286, 1074, 
	284, 283, 283, 283, 1074, 284, 285, 286, 
	287, 0, 288, 289, 285, 282, 286, 293, 
	293, 293, 293, 309, 294, 0, 0, 295, 
	295, 295, 1075, 299, 306, 0, 295, 295, 
	295, 296, 300, 305, 301, 1075, 299, 306, 
	0, 297, 298, 298, 298, 296, 300, 301, 
	1075, 299, 298, 298, 298, 300, 301, 302, 
	0, 303, 304, 300, 297, 301, 1076, 307, 
	307, 307, 307, 308, 0, 311, 312, 313, 
	316, 314, 314, 314, 314, 314, 315, 1077, 
	1078, 1079, 0, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 318, 318, 318, 319, 324, 
	332, 346, 353, 358, 366, 370, 378, 384, 
	400, 403, 410, 421, 318, 323, 1080, 1081, 
	1082, 318, 318, 318, 320, 431, 435, 318, 
	323, 1080, 1081, 1082, 321, 322, 318, 318, 
	318, 318, 323, 1080, 1081, 1082, 325, 328, 
	326, 327, 318, 318, 318, 318, 323, 1080, 
	1081, 1082, 329, 330, 331, 318, 318, 318, 
	318, 323, 1080, 1081, 1082, 333, 337, 345, 
	334, 335, 336, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 338, 341, 339, 340, 318, 
	318, 318, 318, 323, 1080, 1081, 1082, 342, 
	343, 344, 318, 318, 318, 318, 323, 1080, 
	1081, 1082, 318, 318, 318, 318, 323, 1080, 
	1081, 1082, 347, 348, 349, 351, 350, 318, 
	318, 318, 318, 323, 1080, 1081, 1082, 352, 
	318, 318, 318, 318, 323, 1080, 1081, 1082, 
	354, 355, 356, 357, 318, 318, 318, 318, 
	323, 1080, 1081, 1082, 359, 360, 361, 362, 
	363, 364, 365, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 367, 369, 368, 318, 318, 
	318, 318, 323, 1080, 1081, 1082, 318, 318, 
	318, 318, 323, 1080, 1081, 1082, 371, 373, 
	375, 377, 372, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 374, 318, 318, 318, 318, 
	323, 1080, 1081, 1082, 376, 318, 318, 318, 
	318, 323, 1080, 1081, 1082, 318, 318, 318, 
	318, 323, 1080, 1081, 1082, 379, 383, 380, 
	381, 382, 318, 318, 318, 318, 323, 1080, 
	1081, 1082, 318, 318, 318, 318, 323, 1080, 
	1081, 1082, 385, 389, 391, 386, 387, 388, 
	318, 318, 318, 318, 323, 1080, 1081, 1082, 
	390, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 318, 318, 318, 392, 318, 323, 1080, 
	1081, 1082, 393, 318, 318, 318, 394, 318, 
	323, 1080, 1081, 1082, 318, 318, 318, 395, 
	318, 323, 1080, 1081, 1082, 396, 397, 398, 
	399, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 401, 402, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 404, 405, 409, 318, 318, 
	318, 318, 323, 1080, 1081, 1082, 406, 407, 
	408, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 411, 413, 415, 417, 412, 318, 318, 
	318, 318, 323, 1080, 1081, 1082, 414, 318, 
	318, 318, 318, 323, 1080, 1081, 1082, 416, 
	318, 318, 318, 318, 323, 1080, 1081, 1082, 
	418, 419, 420, 318, 318, 318, 318, 323, 
	1080, 1081, 1082, 422, 425, 427, 423, 424, 
	318, 318, 318, 318, 323, 1080, 1081, 1082, 
	426, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 428, 429, 430, 318, 318, 318, 430, 
	318, 323, 1080, 1081, 1082, 432, 433, 434, 
	318, 318, 318, 318, 323, 1080, 1081, 1082, 
	436, 318, 318, 318, 318, 323, 1080, 1081, 
	1082, 0, 438, 438, 438, 438, 445, 0, 
	439, 440, 440, 440, 439, 440, 444, 0, 
	440, 440, 440, 441, 440, 444, 442, 442, 
	442, 442, 442, 443, 1083, 1084, 1085, 447, 
	447, 447, 447, 455, 448, 454, 1086, 1086, 
	1086, 454, 1087, 1086, 1090, 449, 449, 449, 
	450, 449, 453, 451, 451, 451, 451, 451, 
	452, 1087, 1088, 1089, 449, 449, 449, 449, 
	453, 0, 457, 458, 473, 503, 511, 524, 
	1091, 457, 459, 460, 1091, 461, 1091, 462, 
	463, 464, 465, 466, 467, 468, 469, 470, 
	471, 472, 1091, 474, 475, 481, 476, 477, 
	478, 479, 480, 1091, 482, 483, 484, 485, 
	494, 486, 487, 488, 489, 490, 491, 492, 
	493, 1091, 495, 496, 497, 498, 499, 500, 
	501, 502, 1091, 504, 505, 506, 507, 508, 
	509, 510, 1091, 512, 513, 514, 515, 516, 
	517, 518, 521, 519, 520, 1091, 522, 523, 
	1091, 525, 526, 527, 530, 528, 529, 1091, 
	531, 532, 533, 545, 548, 1091, 534, 535, 
	536, 537, 538, 539, 540, 541, 542, 543, 
	544, 1091, 546, 547, 1091, 549, 550, 1091, 
	0, 552, 553, 559, 576, 579, 585, 589, 
	1092, 552, 554, 555, 556, 557, 558, 1092, 
	560, 566, 572, 561, 562, 563, 564, 565, 
	1092, 567, 569, 568, 1092, 570, 571, 1092, 
	573, 574, 575, 1092, 577, 578, 1092, 580, 
	582, 581, 1092, 583, 584, 1092, 586, 587, 
	588, 1092, 590, 591, 1092, 0, 593, 1093, 
	593, 595, 1094, 597, 598, 598, 598, 598, 
	627, 598, 598, 598, 599, 598, 627, 600, 
	600, 600, 600, 626, 0, 600, 600, 600, 
	601, 600, 626, 602, 602, 602, 601, 602, 
	625, 602, 602, 602, 603, 602, 625, 0, 
	604, 604, 604, 603, 622, 622, 622, 622, 
	622, 604, 621, 604, 604, 604, 605, 604, 
	621, 606, 606, 606, 605, 618, 618, 618, 
	618, 618, 606, 617, 606, 606, 606, 607, 
	606, 617, 608, 608, 608, 607, 614, 614, 
	614, 614, 614, 608, 613, 608, 608, 608, 
	609, 608, 613, 1095, 609, 610, 610, 610, 
	610, 610, 611, 611, 612, 612, 612, 612, 
	612, 1095, 611, 615, 615, 616, 616, 616, 
	616, 616, 608, 608, 608, 615, 608, 613, 
	619, 619, 620, 620, 620, 620, 620, 606, 
	606, 606, 619, 606, 617, 623, 623, 624, 
	624, 624, 624, 624, 604, 604, 604, 623, 
	604, 621, 629, 630, 630, 630, 630, 632, 
	631, 630, 630, 630, 630, 632, 1096, 634, 
	635, 635, 635, 635, 637, 635, 635, 635, 
	636, 635, 637, 1097, 639, 640, 640, 640, 
	639, 640, 642, 640, 640, 640, 641, 640, 
	642, 1098, 644, 645, 645, 645, 645, 646, 
	1099, 1100, 1101, 645, 645, 645, 645, 646, 
	1099, 1100, 1101, 648, 1102, 648, 0, 650, 
	651, 651, 651, 650, 651, 703, 651, 651, 
	651, 652, 656, 698, 651, 703, 653, 653, 
	653, 652, 653, 702, 653, 653, 653, 654, 
	653, 702, 655, 655, 655, 700, 654, 655, 
	699, 655, 655, 655, 655, 699, 657, 657, 
	657, 657, 697, 658, 659, 659, 659, 658, 
	659, 696, 659, 659, 659, 660, 664, 691, 
	659, 696, 661, 661, 661, 660, 661, 695, 
	661, 661, 661, 662, 661, 695, 663, 663, 
	663, 693, 662, 663, 692, 663, 663, 663, 
	663, 692, 665, 665, 665, 665, 690, 666, 
	667, 667, 668, 668, 668, 687, 667, 689, 
	668, 686, 1103, 1110, 1111, 668, 668, 668, 
	669, 668, 686, 1103, 1110, 1111, 670, 670, 
	670, 683, 669, 685, 670, 682, 1103, 1108, 
	1109, 670, 670, 670, 671, 670, 682, 1108, 
	1109, 672, 672, 672, 679, 671, 681, 672, 
	678, 1103, 1106, 1107, 672, 672, 672, 673, 
	672, 678, 1106, 1107, 674, 674, 674, 676, 
	673, 674, 675, 1103, 1104, 1105, 674, 674, 
	674, 674, 675, 1104, 1105, 674, 674, 674, 
	677, 674, 675, 1103, 1104, 1105, 677, 672, 
	672, 672, 680, 681, 672, 678, 1103, 1106, 
	1107, 680, 670, 670, 670, 684, 685, 670, 
	682, 1103, 1108, 1109, 684, 668, 668, 668, 
	688, 689, 668, 686, 1103, 1110, 1111, 688, 
	665, 665, 665, 665, 690, 663, 663, 663, 
	694, 663, 692, 694, 657, 657, 657, 657, 
	697, 655, 655, 655, 701, 655, 699, 701, 
	705, 706, 706, 706, 705, 706, 714, 706, 
	706, 706, 707, 706, 714, 708, 708, 708, 
	707, 708, 713, 708, 708, 708, 709, 708, 
	713, 710, 710, 710, 709, 710, 712, 710, 
	710, 710, 711, 710, 712, 1112, 716, 717, 
	717, 717, 716, 717, 731, 717, 717, 717, 
	718, 717, 731, 719, 719, 719, 718, 719, 
	730, 720, 719, 719, 719, 719, 730, 721, 
	721, 721, 721, 729, 722, 721, 721, 721, 
	721, 729, 723, 723, 723, 723, 728, 724, 
	723, 723, 723, 723, 728, 725, 725, 725, 
	725, 727, 725, 725, 725, 726, 725, 727, 
	1113, 733, 734, 734, 734, 734, 742, 735, 
	736, 736, 736, 735, 736, 741, 736, 736, 
	736, 737, 736, 741, 738, 738, 738, 738, 
	740, 739, 1114, 0, 743, 744, 743, 743, 
	745, 751, 743, 750, 1115, 1116, 1117, 745, 
	751, 746, 0, 747, 747, 748, 749, 743, 
	743, 743, 749, 743, 750, 1115, 1116, 1117, 
	752, 753, 753, 748, 755, 756, 756, 756, 
	755, 756, 766, 756, 756, 756, 757, 756, 
	766, 758, 758, 758, 758, 765, 759, 760, 
	760, 760, 759, 760, 764, 0, 760, 760, 
	760, 761, 760, 764, 762, 762, 762, 762, 
	762, 763, 1118, 1119, 1120, 768, 769, 769, 
	769, 768, 769, 776, 769, 769, 769, 770, 
	769, 776, 771, 771, 771, 770, 771, 775, 
	771, 771, 771, 772, 771, 775, 773, 773, 
	773, 773, 773, 774, 1121, 1122, 1123, 778, 
	779, 779, 779, 778, 779, 815, 0, 779, 
	779, 779, 780, 794, 801, 808, 779, 815, 
	781, 781, 781, 781, 793, 0, 782, 783, 
	783, 783, 782, 783, 792, 783, 783, 783, 
	784, 783, 792, 0, 785, 785, 785, 788, 
	788, 788, 790, 790, 790, 1124, 785, 787, 
	1127, 1128, 788, 789, 1125, 1126, 790, 791, 
	1129, 1130, 0, 786, 795, 795, 795, 795, 
	800, 796, 797, 797, 797, 796, 797, 799, 
	0, 797, 797, 797, 798, 797, 799, 0, 
	798, 785, 785, 785, 788, 788, 788, 790, 
	790, 790, 1124, 785, 787, 1127, 1128, 788, 
	789, 1125, 1126, 790, 791, 1129, 1130, 802, 
	802, 802, 802, 807, 803, 804, 804, 804, 
	803, 804, 806, 804, 804, 804, 805, 804, 
	806, 805, 785, 785, 785, 788, 788, 788, 
	790, 790, 790, 1124, 785, 787, 1127, 1128, 
	788, 789, 1125, 1126, 790, 791, 1129, 1130, 
	809, 809, 809, 809, 814, 810, 811, 811, 
	811, 810, 811, 813, 811, 811, 811, 812, 
	811, 813, 785, 785, 785, 788, 788, 788, 
	790, 790, 790, 1124, 785, 787, 1127, 1128, 
	788, 789, 1125, 1126, 790, 791, 1129, 1130, 
	0, 817, 851, 859, 873, 880, 885, 893, 
	897, 905, 911, 927, 930, 937, 948, 818, 
	818, 818, 842, 845, 849, 818, 841, 818, 
	818, 818, 819, 818, 841, 820, 820, 820, 
	820, 840, 821, 822, 822, 822, 821, 822, 
	839, 822, 822, 822, 823, 822, 839, 824, 
	824, 824, 823, 824, 838, 0, 824, 824, 
	824, 825, 824, 838, 826, 826, 826, 825, 
	826, 837, 826, 826, 826, 827, 826, 837, 
	828, 828, 828, 827, 828, 836, 828, 828, 
	828, 829, 828, 836, 830, 830, 830, 829, 
	830, 835, 830, 830, 830, 831, 830, 835, 
	832, 832, 832, 832, 834, 832, 832, 832, 
	833, 832, 834, 1131, 843, 844, 818, 818, 
	818, 818, 841, 846, 847, 848, 818, 818, 
	818, 818, 841, 850, 818, 818, 818, 818, 
	841, 852, 855, 853, 854, 818, 818, 818, 
	818, 841, 856, 857, 858, 818, 818, 818, 
	818, 841, 860, 864, 872, 861, 862, 863, 
	818, 818, 818, 818, 841, 865, 868, 866, 
	867, 818, 818, 818, 818, 841, 869, 870, 
	871, 818, 818, 818, 818, 841, 818, 818, 
	818, 818, 841, 874, 875, 876, 878, 877, 
	818, 818, 818, 818, 841, 879, 818, 818, 
	818, 818, 841, 881, 882, 883, 884, 818, 
	818, 818, 818, 841, 886, 887, 888, 889, 
	890, 891, 892, 818, 818, 818, 818, 841, 
	894, 896, 895, 818, 818, 818, 818, 841, 
	818, 818, 818, 818, 841, 898, 900, 902, 
	904, 899, 818, 818, 818, 818, 841, 901, 
	818, 818, 818, 818, 841, 903, 818, 818, 
	818, 818, 841, 818, 818, 818, 818, 841, 
	906, 910, 907, 908, 909, 818, 818, 818, 
	818, 841, 818, 818, 818, 818, 841, 912, 
	916, 918, 913, 914, 915, 818, 818, 818, 
	818, 841, 917, 818, 818, 818, 818, 841, 
	818, 818, 818, 919, 818, 841, 920, 818, 
	818, 818, 921, 818, 841, 818, 818, 818, 
	922, 818, 841, 923, 924, 925, 926, 818, 
	818, 818, 818, 841, 928, 929, 818, 818, 
	818, 818, 841, 931, 932, 936, 818, 818, 
	818, 818, 841, 933, 934, 935, 818, 818, 
	818, 818, 841, 818, 818, 818, 818, 841, 
	938, 940, 942, 944, 939, 818, 818, 818, 
	818, 841, 941, 818, 818, 818, 818, 841, 
	943, 818, 818, 818, 818, 841, 945, 946, 
	947, 818, 818, 818, 818, 841, 949, 952, 
	954, 950, 951, 818, 818, 818, 818, 841, 
	953, 818, 818, 818, 818, 841, 955, 956, 
	0, 957, 818, 818, 818, 957, 818, 841, 
	959, 960, 1132, 962, 963, 963, 963, 962, 
	963, 971, 963, 963, 963, 964, 963, 971, 
	965, 965, 965, 964, 965, 970, 965, 965, 
	965, 966, 965, 970, 967, 967, 967, 967, 
	969, 968, 1133, 973, 1134, 975, 976, 976, 
	976, 975, 976, 1002, 976, 976, 976, 977, 
	976, 1002, 978, 978, 978, 977, 978, 1001, 
	978, 978, 978, 979, 978, 1001, 980, 980, 
	980, 979, 980, 1000, 980, 980, 980, 981, 
	998, 980, 1000, 982, 982, 982, 982, 997, 
	0, 982, 982, 982, 983, 982, 997, 984, 
	985, 995, 986, 987, 994, 988, 992, 989, 
	990, 990, 991, 983, 1135, 993, 996, 999, 
	998, 1004, 1005, 1005, 1005, 1004, 1005, 1015, 
	1005, 1005, 1005, 1006, 1005, 1015, 1007, 1007, 
	1007, 1006, 1007, 1014, 1007, 1007, 1007, 1008, 
	1007, 1014, 1009, 1009, 1009, 1008, 1009, 1013, 
	1009, 1009, 1009, 1010, 1011, 1009, 1013, 1136, 
	1012, 1011, 1017, 1018, 1018, 1018, 1017, 1018, 
	1028, 1018, 1018, 1018, 1019, 1018, 1028, 1020, 
	1020, 1020, 1019, 1020, 1027, 1020, 1020, 1020, 
	1021, 1020, 1027, 1022, 1022, 1022, 1021, 1022, 
	1026, 1022, 1022, 1022, 1023, 1022, 1026, 1024, 
	1024, 1024, 1024, 1024, 1025, 1137, 1138, 1139, 
	1030, 1031, 1031, 1031, 1030, 1031, 1033, 1031, 
	1031, 1031, 1032, 1031, 1033, 1140, 1032, 1035, 
	1036, 1036, 1036, 1035, 1036, 1046, 1036, 1036, 
	1036, 1037, 1036, 1046, 1038, 1039, 1040, 0, 
	1041, 1042, 1043, 1044, 1045, 0, 1141, 1048, 
	1049, 0, 1050, 1051, 1052, 1142, 1054, 1055, 
	1056, 1057, 1058, 1143, 0, 1, 138, 1, 
	1, 146, 1064, 242, 13, 179, 13, 13, 
	189, 208, 1067, 222, 1068, 225, 0, 128, 
	128, 128, 156, 229, 230, 231, 232, 244, 
	234, 235, 236, 237, 238, 239, 240, 241, 
	1062, 243, 0, 149, 156, 229, 230, 231, 
	232, 233, 234, 235, 236, 237, 238, 239, 
	240, 241, 0, 1, 138, 1, 1, 146, 
	1059, 137, 1064, 242, 1, 1, 1064, 160, 
	160, 160, 1064, 161, 164, 164, 164, 178, 
	212, 0, 190, 193, 195, 196, 197, 198, 
	199, 200, 201, 202, 203, 204, 205, 206, 
	207, 1064
};

static const short _zone_scanner_trans_actions[] = {
	1806, 0, 5, 7, 492, 642, 642, 642, 
	642, 642, 642, 642, 642, 642, 642, 642, 
	642, 642, 642, 3, 0, 3, 0, 1934, 
	101, 657, 660, 41, 55, 53, 51, 0, 
	57, 654, 101, 343, 0, 5, 7, 95, 
	95, 95, 95, 95, 95, 95, 95, 95, 
	95, 95, 95, 95, 95, 3, 0, 1566, 
	1569, 2298, 2302, 0, 0, 0, 2294, 1569, 
	2306, 3371, 2306, 339, 0, 337, 5, 7, 
	0, 3, 0, 354, 339, 1551, 0, 5, 
	7, 1683, 0, 9, 1560, 341, 0, 5, 
	7, 0, 2286, 339, 339, 357, 0, 2981, 
	5, 7, 1794, 2018, 2018, 2018, 2018, 2018, 
	2018, 2018, 2018, 2018, 2018, 2018, 2018, 2018, 
	2018, 3, 0, 1554, 339, 354, 339, 3644, 
	101, 657, 660, 2022, 101, 1563, 0, 5, 
	7, 357, 0, 0, 0, 0, 0, 1623, 
	2586, 2590, 2582, 1623, 2594, 3466, 2594, 0, 
	5, 7, 0, 0, 0, 1575, 2330, 2334, 
	2326, 1575, 2338, 3386, 2338, 0, 0, 0, 
	0, 0, 0, 1650, 2730, 2734, 2726, 1650, 
	2738, 3511, 2738, 0, 0, 0, 0, 1626, 
	2602, 2606, 2598, 1626, 2610, 3471, 2610, 0, 
	0, 0, 1647, 2714, 2718, 2710, 1647, 2722, 
	3506, 2722, 1632, 2634, 2638, 2630, 1632, 2642, 
	3481, 2642, 0, 0, 0, 0, 0, 1677, 
	2874, 2878, 2870, 1677, 2882, 3556, 2882, 0, 
	1680, 2890, 2894, 2886, 1680, 2898, 3561, 2898, 
	0, 0, 0, 0, 1584, 2378, 2382, 2374, 
	1584, 2386, 3401, 2386, 0, 0, 99, 648, 
	651, 645, 99, 0, 5, 7, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 3, 0, 0, 0, 
	0, 0, 0, 0, 1638, 2666, 2670, 2662, 
	1638, 2674, 3491, 2674, 0, 0, 0, 1605, 
	2490, 2494, 2486, 1605, 2498, 3436, 2498, 1620, 
	2570, 2574, 2566, 1620, 2578, 3461, 2578, 0, 
	0, 0, 0, 0, 1668, 2826, 2830, 2822, 
	1668, 2834, 3541, 2834, 0, 1671, 2842, 2846, 
	2838, 1671, 2850, 3546, 2850, 0, 1611, 2522, 
	2526, 2518, 1611, 2530, 3446, 2530, 1674, 2858, 
	2862, 2854, 1674, 2866, 3551, 2866, 0, 0, 
	0, 0, 0, 1587, 2394, 2398, 2390, 1587, 
	2402, 3406, 2402, 1590, 2410, 2414, 2406, 1590, 
	2418, 3411, 2418, 0, 0, 0, 0, 0, 
	0, 1617, 2554, 2558, 2550, 1617, 2562, 3456, 
	2562, 0, 1665, 2810, 2814, 2806, 1665, 2818, 
	3536, 2818, 1572, 2314, 2318, 0, 2310, 1572, 
	2322, 3381, 2322, 0, 1644, 2698, 2702, 0, 
	2694, 1644, 2706, 3501, 2706, 1653, 2746, 2750, 
	0, 2742, 1653, 2754, 3516, 2754, 0, 0, 
	0, 0, 1656, 2762, 2766, 2758, 1656, 2770, 
	3521, 2770, 0, 0, 1581, 2362, 2366, 2358, 
	1581, 2370, 3396, 2370, 0, 0, 0, 1596, 
	2442, 2446, 2438, 1596, 2450, 3421, 2450, 0, 
	0, 0, 1641, 2682, 2686, 2678, 1641, 2690, 
	3496, 2690, 1602, 2474, 2478, 2470, 1602, 2482, 
	3431, 2482, 0, 0, 0, 0, 0, 1578, 
	2346, 2350, 2342, 1578, 2354, 3391, 2354, 0, 
	1662, 2794, 2798, 2790, 1662, 2802, 3531, 2802, 
	0, 1614, 2538, 2542, 2534, 1614, 2546, 3451, 
	2546, 0, 0, 0, 1635, 2650, 2654, 2646, 
	1635, 2658, 3486, 2658, 0, 0, 0, 0, 
	0, 1659, 2778, 2782, 2774, 1659, 2786, 3526, 
	2786, 0, 1593, 2426, 2430, 2422, 1593, 2434, 
	3416, 2434, 0, 0, 504, 492, 1810, 561, 
	1914, 1918, 41, 1910, 561, 1922, 3111, 1922, 
	0, 5, 7, 3, 0, 0, 99, 648, 
	651, 645, 99, 0, 5, 7, 492, 97, 
	97, 97, 97, 97, 97, 97, 97, 97, 
	97, 97, 97, 97, 97, 3, 0, 101, 
	657, 660, 41, 55, 53, 51, 0, 57, 
	654, 101, 1938, 41, 55, 53, 51, 0, 
	57, 582, 1962, 1966, 3121, 1958, 582, 93, 
	89, 89, 89, 0, 0, 0, 0, 0, 
	0, 87, 37, 33, 444, 447, 441, 33, 
	0, 5, 7, 3, 0, 0, 2966, 450, 
	1758, 1762, 1750, 450, 0, 5, 7, 3, 
	0, 1938, 41, 55, 53, 51, 0, 57, 
	582, 1962, 1966, 3121, 1958, 582, 1786, 1770, 
	2941, 2951, 2931, 1770, 3572, 2956, 0, 5, 
	7, 3, 0, 354, 339, 2290, 1569, 2298, 
	2302, 3376, 1569, 3668, 2306, 3662, 2306, 0, 
	5, 7, 1718, 0, 5, 7, 357, 0, 
	0, 5, 7, 0, 5, 7, 0, 339, 
	357, 2278, 0, 0, 0, 0, 0, 79, 
	0, 0, 75, 1778, 33, 444, 447, 1754, 
	33, 3681, 450, 1758, 1762, 2921, 450, 0, 
	5, 7, 357, 0, 2971, 1770, 2941, 2951, 
	3578, 1770, 3674, 2956, 0, 5, 7, 1718, 
	1766, 465, 2936, 2946, 2926, 1766, 0, 5, 
	7, 0, 5, 7, 0, 0, 1608, 2506, 
	2510, 2502, 1608, 2514, 3441, 2514, 0, 0, 
	0, 1599, 2458, 2462, 2454, 1599, 2466, 3426, 
	2466, 0, 1629, 2618, 2622, 2614, 1629, 2626, 
	3476, 2626, 0, 5, 7, 0, 5, 7, 
	0, 5, 7, 0, 5, 7, 468, 372, 
	2906, 11, 1726, 27, 1742, 1742, 25, 0, 
	411, 1730, 17, 396, 375, 375, 13, 15, 
	0, 378, 387, 19, 19, 2911, 381, 384, 
	21, 408, 603, 65, 0, 0, 1, 600, 
	606, 1974, 69, 69, 609, 612, 1970, 71, 
	65, 0, 0, 67, 606, 69, 69, 612, 
	71, 71, 73, 0, 5, 7, 3, 0, 
	495, 492, 3116, 0, 5, 7, 41, 55, 
	53, 51, 0, 57, 1978, 0, 366, 1938, 
	1926, 41, 55, 53, 51, 0, 57, 59, 
	573, 576, 3121, 3126, 59, 77, 0, 5, 
	7, 3, 0, 3131, 3131, 615, 1982, 17, 
	369, 23, 399, 402, 375, 375, 13, 2916, 
	23, 0, 5, 7, 1986, 0, 15, 0, 
	378, 390, 19, 19, 381, 384, 21, 0, 
	5, 7, 3, 0, 618, 83, 627, 81, 
	621, 624, 3136, 81, 81, 633, 0, 5, 
	7, 3141, 3141, 630, 1990, 1994, 0, 0, 
	85, 17, 405, 1734, 1738, 375, 375, 13, 
	3566, 405, 0, 5, 7, 15, 0, 378, 
	393, 19, 19, 381, 384, 21, 1698, 0, 
	5, 7, 3, 0, 125, 117, 119, 121, 
	0, 123, 0, 5, 7, 3, 0, 1, 
	348, 1, 309, 307, 1476, 1479, 1473, 307, 
	2250, 3366, 2250, 0, 5, 7, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 3, 0, 1482, 1706, 
	1482, 231, 1134, 1137, 0, 0, 0, 1131, 
	231, 2098, 3176, 2098, 0, 0, 257, 1251, 
	1254, 1248, 257, 2150, 3241, 2150, 0, 0, 
	0, 0, 267, 1296, 1299, 1293, 267, 2170, 
	3266, 2170, 0, 0, 0, 235, 1152, 1155, 
	1149, 235, 2106, 3186, 2106, 0, 0, 0, 
	0, 0, 0, 285, 1377, 1380, 1374, 285, 
	2206, 3311, 2206, 0, 0, 0, 0, 269, 
	1305, 1308, 1302, 269, 2174, 3271, 2174, 0, 
	0, 0, 283, 1368, 1371, 1365, 283, 2202, 
	3306, 2202, 273, 1323, 1326, 1320, 273, 2182, 
	3281, 2182, 0, 0, 0, 0, 0, 303, 
	1458, 1461, 1455, 303, 2242, 3356, 2242, 0, 
	305, 1467, 1470, 1464, 305, 2246, 3361, 2246, 
	0, 0, 0, 0, 241, 1179, 1182, 1176, 
	241, 2118, 3201, 2118, 0, 0, 0, 0, 
	0, 0, 0, 277, 1341, 1344, 1338, 277, 
	2190, 3291, 2190, 0, 0, 0, 255, 1242, 
	1245, 1239, 255, 2146, 3236, 2146, 265, 1287, 
	1290, 1284, 265, 2166, 3261, 2166, 0, 0, 
	0, 0, 0, 297, 1431, 1434, 1428, 297, 
	2230, 3341, 2230, 0, 299, 1440, 1443, 1437, 
	299, 2234, 3346, 2234, 0, 259, 1260, 1263, 
	1257, 259, 2154, 3246, 2154, 301, 1449, 1452, 
	1446, 301, 2238, 3351, 2238, 0, 0, 0, 
	0, 0, 243, 1188, 1191, 1185, 243, 2122, 
	3206, 2122, 245, 1197, 1200, 1194, 245, 2126, 
	3211, 2126, 0, 0, 0, 0, 0, 0, 
	263, 1278, 1281, 1275, 263, 2162, 3256, 2162, 
	0, 295, 1422, 1425, 1419, 295, 2226, 3336, 
	2226, 233, 1143, 1146, 0, 1140, 233, 2102, 
	3181, 2102, 0, 281, 1359, 1362, 0, 1356, 
	281, 2198, 3301, 2198, 287, 1386, 1389, 0, 
	1383, 287, 2210, 3316, 2210, 0, 0, 0, 
	0, 289, 1395, 1398, 1392, 289, 2214, 3321, 
	2214, 0, 0, 239, 1170, 1173, 1167, 239, 
	2114, 3196, 2114, 0, 0, 0, 249, 1215, 
	1218, 1212, 249, 2134, 3221, 2134, 0, 0, 
	0, 279, 1350, 1353, 1347, 279, 2194, 3296, 
	2194, 253, 1233, 1236, 1230, 253, 2142, 3231, 
	2142, 0, 0, 0, 0, 0, 237, 1161, 
	1164, 1158, 237, 2110, 3191, 2110, 0, 293, 
	1413, 1416, 1410, 293, 2222, 3331, 2222, 0, 
	261, 1269, 1272, 1266, 261, 2158, 3251, 2158, 
	0, 0, 0, 275, 1332, 1335, 1329, 275, 
	2186, 3286, 2186, 0, 0, 0, 0, 0, 
	291, 1404, 1407, 1401, 291, 2218, 3326, 2218, 
	0, 247, 1206, 1209, 1203, 247, 2130, 3216, 
	2130, 0, 0, 492, 229, 1125, 1128, 41, 
	1122, 229, 2094, 3171, 2094, 0, 0, 0, 
	251, 1224, 1227, 1221, 251, 2138, 3226, 2138, 
	0, 271, 1314, 1317, 1311, 271, 2178, 3276, 
	2178, 323, 0, 5, 7, 3, 0, 498, 
	492, 49, 567, 570, 41, 564, 49, 2082, 
	0, 5, 7, 113, 3, 0, 115, 0, 
	5, 7, 3, 0, 693, 1702, 693, 0, 
	5, 7, 3, 0, 492, 492, 345, 1690, 
	1694, 41, 1, 1686, 345, 0, 5, 7, 
	113, 3, 0, 115, 0, 5, 7, 3, 
	0, 693, 1702, 693, 49, 567, 570, 564, 
	49, 329, 492, 0, 0, 0, 0, 0, 
	525, 41, 0, 0, 702, 0, 705, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 711, 0, 0, 0, 0, 0, 
	0, 0, 0, 723, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 726, 0, 0, 0, 0, 0, 0, 
	0, 0, 729, 0, 0, 0, 0, 0, 
	0, 0, 732, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 735, 0, 0, 
	738, 0, 0, 0, 0, 0, 0, 699, 
	0, 0, 0, 0, 0, 708, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 714, 0, 0, 717, 0, 0, 720, 
	331, 492, 0, 0, 0, 0, 0, 0, 
	537, 41, 0, 0, 0, 0, 0, 759, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	762, 0, 0, 0, 756, 0, 0, 750, 
	0, 0, 0, 753, 0, 0, 768, 0, 
	0, 0, 747, 0, 0, 741, 0, 0, 
	0, 744, 0, 0, 765, 666, 663, 2034, 
	103, 471, 474, 471, 39, 480, 483, 477, 
	39, 0, 5, 7, 471, 3, 0, 39, 
	480, 483, 477, 39, 501, 0, 5, 7, 
	492, 3, 0, 47, 555, 558, 41, 552, 
	47, 0, 5, 7, 492, 3, 0, 1930, 
	47, 555, 558, 41, 55, 53, 51, 0, 
	57, 552, 47, 0, 5, 7, 492, 3, 
	0, 47, 555, 558, 41, 55, 53, 51, 
	0, 57, 552, 47, 0, 5, 7, 492, 
	3, 0, 47, 555, 558, 41, 55, 53, 
	51, 0, 57, 552, 47, 0, 5, 7, 
	492, 3, 0, 549, 41, 55, 53, 51, 
	0, 57, 1938, 41, 55, 53, 51, 0, 
	57, 1942, 3121, 1938, 41, 55, 53, 51, 
	0, 57, 579, 1950, 1954, 3121, 1946, 579, 
	1938, 41, 55, 53, 51, 0, 57, 579, 
	1950, 1954, 3121, 1946, 579, 1938, 41, 55, 
	53, 51, 0, 57, 579, 1950, 1954, 3121, 
	1946, 579, 414, 31, 429, 432, 426, 31, 
	414, 0, 5, 7, 3, 0, 423, 471, 
	39, 480, 483, 477, 39, 0, 5, 7, 
	471, 3, 0, 474, 492, 45, 543, 546, 
	41, 540, 45, 0, 5, 7, 471, 3, 
	0, 474, 414, 31, 429, 432, 426, 31, 
	423, 1746, 423, 0, 5, 7, 3, 0, 
	1, 348, 1, 663, 2050, 103, 1542, 2254, 
	311, 1488, 1491, 41, 1485, 311, 0, 5, 
	7, 492, 0, 0, 3, 0, 315, 1506, 
	1509, 41, 1503, 315, 0, 5, 7, 3106, 
	3, 0, 507, 1842, 1846, 0, 41, 1838, 
	507, 0, 5, 7, 3, 0, 0, 5, 
	7, 3, 0, 492, 313, 1497, 1500, 41, 
	1494, 313, 0, 5, 7, 492, 0, 0, 
	3, 0, 317, 1515, 1518, 41, 1512, 317, 
	0, 5, 7, 3106, 3, 0, 510, 1854, 
	1858, 0, 41, 1850, 510, 0, 5, 7, 
	3, 0, 0, 5, 7, 3, 0, 0, 
	3101, 3656, 513, 1866, 1870, 0, 41, 513, 
	1862, 513, 3081, 3620, 3081, 0, 5, 7, 
	3101, 3, 0, 1539, 1710, 1539, 516, 1878, 
	1882, 0, 41, 516, 1874, 516, 3086, 3626, 
	3086, 0, 5, 7, 3101, 3, 0, 1710, 
	1539, 519, 1890, 1894, 0, 41, 519, 1886, 
	519, 3091, 3632, 3091, 0, 5, 7, 3101, 
	3, 0, 1710, 1539, 522, 1902, 1906, 0, 
	41, 1898, 522, 3096, 3638, 3096, 0, 5, 
	7, 3, 0, 1710, 1539, 1834, 3071, 3076, 
	2986, 3066, 1834, 3614, 3709, 3614, 489, 1830, 
	3056, 3061, 2986, 1830, 3051, 1830, 3608, 3702, 
	3608, 489, 1826, 3041, 3046, 2986, 1826, 3036, 
	1826, 3602, 3695, 3602, 489, 1822, 3026, 3031, 
	2986, 1822, 3021, 1822, 3596, 3688, 3596, 489, 
	321, 1533, 1536, 1530, 321, 1818, 3011, 3016, 
	2986, 3006, 1818, 489, 319, 1524, 1527, 1521, 
	319, 1814, 2996, 3001, 2986, 2991, 1814, 489, 
	492, 45, 543, 546, 41, 540, 45, 0, 
	5, 7, 492, 3, 0, 45, 543, 546, 
	41, 540, 45, 0, 5, 7, 492, 3, 
	0, 45, 543, 546, 41, 540, 45, 0, 
	5, 7, 471, 3, 0, 474, 492, 45, 
	543, 546, 41, 540, 45, 0, 5, 7, 
	492, 3, 0, 45, 543, 546, 41, 540, 
	45, 414, 0, 5, 7, 3, 0, 31, 
	429, 432, 426, 31, 414, 0, 5, 7, 
	3, 0, 31, 429, 432, 426, 31, 414, 
	0, 5, 7, 3, 0, 31, 429, 432, 
	426, 31, 0, 5, 7, 471, 3, 0, 
	474, 335, 0, 5, 7, 3, 0, 492, 
	45, 543, 546, 41, 540, 45, 0, 5, 
	7, 333, 3, 0, 0, 5, 7, 3, 
	0, 127, 1, 687, 0, 675, 5, 7, 
	678, 681, 3, 0, 1, 348, 1, 109, 
	111, 0, 2026, 663, 103, 105, 492, 684, 
	2074, 2078, 41, 2070, 684, 2066, 3166, 2066, 
	0, 663, 103, 107, 492, 45, 543, 546, 
	41, 540, 45, 0, 5, 7, 333, 3, 
	0, 0, 5, 7, 3, 0, 492, 43, 
	531, 534, 41, 528, 43, 690, 0, 5, 
	7, 113, 3, 0, 115, 0, 5, 7, 
	3, 0, 1, 348, 1, 492, 43, 531, 
	534, 41, 528, 43, 0, 5, 7, 492, 
	3, 0, 43, 531, 534, 41, 528, 43, 
	0, 5, 7, 113, 3, 0, 115, 0, 
	5, 7, 3, 0, 1, 348, 1, 492, 
	43, 531, 534, 41, 528, 43, 771, 0, 
	5, 7, 145, 147, 149, 151, 3, 0, 
	0, 5, 7, 3, 0, 1798, 492, 43, 
	531, 534, 41, 528, 43, 0, 5, 7, 
	0, 3, 0, 2090, 0, 5, 7, 0, 
	5, 7, 0, 5, 7, 1, 3, 0, 
	348, 1, 3, 0, 348, 1, 3, 0, 
	348, 1, 774, 127, 0, 5, 7, 3, 
	0, 492, 43, 531, 534, 41, 528, 43, 
	2030, 0, 5, 7, 663, 3, 0, 3151, 
	103, 669, 2042, 2046, 669, 2042, 2046, 669, 
	2042, 2046, 2034, 2038, 669, 3156, 2034, 2038, 
	669, 3156, 2034, 2038, 669, 3156, 2034, 0, 
	5, 7, 3, 0, 492, 43, 531, 534, 
	41, 528, 43, 0, 5, 7, 663, 3, 
	0, 103, 672, 2058, 2062, 672, 2058, 2062, 
	672, 2058, 2062, 2050, 2054, 672, 3161, 2050, 
	2054, 672, 3161, 2050, 2054, 672, 3161, 2050, 
	0, 5, 7, 3, 0, 492, 43, 531, 
	534, 41, 528, 43, 0, 5, 7, 471, 
	3, 0, 39, 480, 483, 39, 480, 483, 
	39, 480, 483, 474, 477, 39, 1790, 474, 
	477, 39, 1790, 474, 477, 39, 1790, 474, 
	777, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 153, 
	783, 786, 0, 0, 0, 780, 153, 0, 
	5, 7, 333, 3, 0, 0, 5, 7, 
	3, 0, 492, 43, 531, 534, 41, 528, 
	43, 0, 5, 7, 492, 3, 0, 47, 
	555, 558, 41, 552, 47, 597, 0, 5, 
	7, 585, 3, 0, 63, 591, 594, 61, 
	588, 63, 0, 5, 7, 585, 3, 0, 
	63, 591, 594, 61, 588, 63, 0, 5, 
	7, 492, 3, 0, 45, 543, 546, 41, 
	540, 45, 0, 5, 7, 471, 3, 0, 
	39, 480, 483, 477, 39, 0, 5, 7, 
	127, 3, 0, 1, 0, 0, 179, 900, 
	903, 897, 179, 0, 0, 0, 173, 873, 
	876, 870, 173, 0, 193, 963, 966, 960, 
	193, 0, 0, 0, 0, 189, 945, 948, 
	942, 189, 0, 0, 0, 157, 801, 804, 
	798, 157, 0, 0, 0, 0, 0, 0, 
	207, 1026, 1029, 1023, 207, 0, 0, 0, 
	0, 191, 954, 957, 951, 191, 0, 0, 
	0, 205, 1017, 1020, 1014, 205, 195, 972, 
	975, 969, 195, 0, 0, 0, 0, 0, 
	225, 1107, 1110, 1104, 225, 0, 227, 1116, 
	1119, 1113, 227, 0, 0, 0, 0, 163, 
	828, 831, 825, 163, 0, 0, 0, 0, 
	0, 0, 0, 199, 990, 993, 987, 199, 
	0, 0, 0, 177, 891, 894, 888, 177, 
	187, 936, 939, 933, 187, 0, 0, 0, 
	0, 0, 219, 1080, 1083, 1077, 219, 0, 
	221, 1089, 1092, 1086, 221, 0, 181, 909, 
	912, 906, 181, 223, 1098, 1101, 1095, 223, 
	0, 0, 0, 0, 0, 165, 837, 840, 
	834, 165, 167, 846, 849, 843, 167, 0, 
	0, 0, 0, 0, 0, 185, 927, 930, 
	924, 185, 0, 217, 1071, 1074, 1068, 217, 
	155, 792, 795, 0, 789, 155, 0, 203, 
	1008, 1011, 0, 1005, 203, 209, 1035, 1038, 
	0, 1032, 209, 0, 0, 0, 0, 211, 
	1044, 1047, 1041, 211, 0, 0, 161, 819, 
	822, 816, 161, 0, 0, 0, 171, 864, 
	867, 861, 171, 0, 0, 0, 201, 999, 
	1002, 996, 201, 175, 882, 885, 879, 175, 
	0, 0, 0, 0, 0, 159, 810, 813, 
	807, 159, 0, 215, 1062, 1065, 1059, 215, 
	0, 183, 918, 921, 915, 183, 0, 0, 
	0, 197, 981, 984, 978, 197, 0, 0, 
	0, 0, 0, 213, 1053, 1056, 1050, 213, 
	0, 169, 855, 858, 852, 169, 0, 0, 
	1802, 492, 45, 543, 546, 41, 540, 45, 
	471, 486, 1, 492, 45, 543, 546, 41, 
	540, 45, 0, 5, 7, 492, 3, 0, 
	43, 531, 534, 41, 528, 43, 0, 5, 
	7, 333, 3, 0, 0, 5, 7, 3, 
	0, 127, 1, 127, 1, 492, 43, 531, 
	534, 41, 528, 43, 0, 5, 7, 492, 
	3, 0, 43, 531, 534, 41, 528, 43, 
	0, 5, 7, 492, 3, 0, 45, 543, 
	546, 41, 540, 45, 0, 5, 7, 29, 
	417, 3, 0, 31, 429, 432, 426, 31, 
	696, 0, 5, 7, 420, 3, 0, 131, 
	133, 0, 135, 137, 0, 139, 0, 141, 
	143, 0, 435, 129, 1, 0, 0, 115, 
	113, 492, 43, 531, 534, 41, 528, 43, 
	0, 5, 7, 492, 3, 0, 43, 531, 
	534, 41, 528, 43, 0, 5, 7, 492, 
	3, 0, 45, 543, 546, 41, 540, 45, 
	0, 5, 7, 29, 417, 3, 0, 423, 
	115, 113, 492, 43, 531, 534, 41, 528, 
	43, 0, 5, 7, 492, 3, 0, 43, 
	531, 534, 41, 528, 43, 0, 5, 7, 
	492, 3, 0, 43, 531, 534, 41, 528, 
	43, 0, 5, 7, 113, 3, 0, 115, 
	0, 5, 7, 3, 0, 1, 348, 1, 
	492, 45, 543, 546, 41, 540, 45, 0, 
	5, 7, 663, 3, 0, 2034, 103, 492, 
	45, 543, 546, 41, 540, 45, 0, 5, 
	7, 1548, 3, 0, 115, 113, 115, 2086, 
	327, 113, 115, 113, 115, 2274, 2270, 1545, 
	115, 2266, 325, 113, 115, 2258, 1545, 115, 
	325, 113, 115, 2262, 462, 35, 0, 456, 
	459, 438, 453, 35, 35, 339, 360, 363, 
	2282, 438, 351, 35, 1714, 1557, 1782, 35, 
	360, 363, 2014, 2014, 2014, 2014, 2014, 2014, 
	2014, 2014, 2014, 2014, 2014, 2014, 2014, 2014, 
	351, 35, 2961, 2976, 3146, 3146, 3146, 3146, 
	3146, 3146, 3146, 3146, 3146, 3146, 3146, 3146, 
	3146, 3146, 1774, 639, 91, 2006, 2010, 1998, 
	636, 91, 2002, 639, 360, 363, 351, 35, 
	360, 363, 2902, 35, 35, 360, 363, 35, 
	1557, 3584, 3590, 3650, 3650, 3650, 3650, 3650, 
	3650, 3650, 3650, 3650, 3650, 3650, 3650, 3650, 
	3650, 1722
};

static const short _zone_scanner_eof_actions[] = {
	0, 1806, 1934, 343, 1566, 337, 1551, 9, 
	9, 337, 337, 1551, 1551, 2981, 3644, 1563, 
	343, 343, 343, 1566, 1551, 343, 343, 343, 
	1566, 343, 343, 343, 343, 1566, 343, 343, 
	343, 1566, 343, 343, 343, 1566, 1566, 343, 
	343, 343, 343, 1566, 343, 1566, 343, 343, 
	343, 343, 1566, 343, 343, 343, 343, 343, 
	343, 343, 343, 343, 343, 1566, 343, 343, 
	1566, 1566, 343, 343, 1566, 343, 1566, 343, 
	1566, 1566, 343, 343, 343, 343, 1566, 1566, 
	343, 343, 343, 343, 1566, 343, 1566, 1566, 
	343, 1566, 1566, 343, 343, 343, 343, 1566, 
	343, 343, 1566, 343, 1566, 343, 343, 343, 
	1566, 1566, 343, 343, 1566, 343, 1566, 343, 
	1566, 343, 343, 343, 1566, 343, 343, 343, 
	1566, 343, 1566, 343, 343, 504, 1810, 343, 
	1806, 343, 343, 504, 1934, 1934, 1934, 1934, 
	343, 9, 93, 93, 93, 93, 93, 93, 
	93, 93, 37, 504, 0, 2966, 504, 343, 
	1934, 1934, 1934, 343, 1786, 1810, 2290, 1551, 
	1806, 9, 1551, 9, 2981, 2290, 2290, 2290, 
	2290, 2290, 2290, 2290, 2290, 2290, 2290, 2290, 
	2290, 2290, 1551, 2278, 93, 93, 93, 93, 
	93, 93, 93, 93, 93, 1778, 3681, 1806, 
	1563, 2971, 2981, 2971, 2971, 2971, 2971, 2971, 
	2971, 2971, 2971, 2971, 2971, 2971, 2971, 2971, 
	465, 2981, 1551, 9, 1551, 343, 343, 1566, 
	343, 343, 343, 1566, 343, 1566, 1551, 1551, 
	9, 1551, 1551, 337, 1551, 468, 468, 468, 
	468, 468, 468, 468, 468, 468, 468, 468, 
	468, 468, 9, 1563, 468, 1563, 0, 0, 
	27, 27, 27, 27, 387, 387, 387, 387, 
	27, 27, 603, 603, 1974, 1974, 1974, 1974, 
	603, 67, 1974, 1974, 1974, 1974, 73, 495, 
	3116, 366, 366, 3116, 1926, 3116, 73, 77, 
	77, 77, 369, 369, 369, 77, 390, 390, 
	390, 390, 369, 77, 0, 83, 627, 633, 
	85, 633, 9, 9, 85, 393, 393, 393, 
	393, 633, 9, 85, 0, 0, 125, 125, 
	125, 125, 125, 125, 125, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 309, 309, 309, 
	309, 309, 309, 309, 309, 323, 498, 498, 
	2082, 2082, 2082, 2082, 323, 323, 323, 498, 
	498, 2082, 2082, 2082, 2082, 323, 498, 323, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 329, 
	329, 329, 329, 329, 329, 329, 329, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	331, 331, 331, 331, 331, 331, 331, 331, 
	666, 666, 337, 337, 337, 337, 337, 337, 
	501, 501, 501, 1930, 501, 1930, 501, 1930, 
	501, 1930, 1930, 1930, 1930, 337, 1930, 1930, 
	1930, 337, 1930, 1930, 1930, 337, 1930, 1930, 
	1930, 337, 337, 337, 337, 337, 337, 337, 
	337, 337, 337, 337, 337, 337, 501, 501, 
	337, 337, 337, 337, 337, 337, 337, 666, 
	666, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	1542, 1542, 1542, 1542, 1542, 1542, 1542, 1542, 
	501, 501, 501, 501, 501, 501, 337, 337, 
	337, 337, 337, 501, 501, 501, 501, 337, 
	337, 337, 337, 337, 337, 337, 337, 337, 
	337, 337, 337, 337, 337, 337, 501, 501, 
	337, 337, 337, 337, 337, 337, 337, 687, 
	687, 687, 2026, 2026, 687, 687, 337, 687, 
	2026, 2026, 501, 501, 337, 337, 501, 501, 
	690, 690, 690, 690, 337, 337, 337, 501, 
	501, 501, 501, 690, 690, 690, 690, 337, 
	337, 501, 501, 771, 771, 1798, 1798, 771, 
	2090, 774, 774, 774, 774, 774, 774, 774, 
	771, 771, 771, 1798, 1798, 2030, 3151, 771, 
	771, 771, 1798, 1798, 2030, 3151, 771, 771, 
	771, 1798, 1798, 771, 2090, 771, 771, 337, 
	777, 777, 337, 337, 501, 501, 501, 501, 
	597, 597, 597, 597, 501, 501, 337, 337, 
	337, 337, 337, 337, 337, 337, 337, 337, 
	337, 337, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 777, 777, 777, 777, 
	777, 777, 777, 777, 1802, 1802, 337, 337, 
	337, 501, 501, 501, 501, 337, 337, 337, 
	337, 337, 337, 337, 337, 337, 501, 501, 
	501, 501, 501, 501, 690, 690, 696, 696, 
	696, 696, 696, 696, 696, 696, 696, 337, 
	696, 696, 696, 696, 696, 337, 690, 690, 
	337, 337, 337, 501, 501, 501, 501, 501, 
	501, 690, 690, 690, 690, 337, 337, 337, 
	501, 501, 501, 501, 501, 501, 690, 690, 
	690, 690, 337, 337, 337, 501, 501, 666, 
	666, 337, 501, 501, 690, 690, 690, 690, 
	2086, 690, 690, 690, 690, 2274, 337, 690, 
	690, 2266, 690, 690, 2266, 690, 690, 2266, 
	690, 690, 2266, 0, 0, 0, 0, 91, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0
};

static const int zone_scanner_start = 1059;
static const int zone_scanner_first_final = 1059;
static const int zone_scanner_error = 0;

static const int zone_scanner_en_err_line = 246;
static const int zone_scanner_en_dname_ = 248;
static const int zone_scanner_en_text_ = 258;
static const int zone_scanner_en_default_ttl_ = 270;
static const int zone_scanner_en_zone_origin_ = 279;
static const int zone_scanner_en_include_file_ = 292;
static const int zone_scanner_en_base64_ = 310;
static const int zone_scanner_en_bitmap_ = 317;
static const int zone_scanner_en_nonempty_hex_r_data = 437;
static const int zone_scanner_en_hex_r_data = 446;
static const int zone_scanner_en_dns_alg_ = 456;
static const int zone_scanner_en_cert_type_ = 551;
static const int zone_scanner_en_r_data_a = 592;
static const int zone_scanner_en_r_data_ns = 594;
static const int zone_scanner_en_r_data_soa = 596;
static const int zone_scanner_en_r_data_hinfo = 628;
static const int zone_scanner_en_r_data_minfo = 633;
static const int zone_scanner_en_r_data_mx = 638;
static const int zone_scanner_en_r_data_txt = 643;
static const int zone_scanner_en_r_data_aaaa = 647;
static const int zone_scanner_en_r_data_loc = 649;
static const int zone_scanner_en_r_data_srv = 704;
static const int zone_scanner_en_r_data_naptr = 715;
static const int zone_scanner_en_r_data_cert = 732;
static const int zone_scanner_en_r_data_apl = 743;
static const int zone_scanner_en_r_data_ds = 754;
static const int zone_scanner_en_r_data_sshfp = 767;
static const int zone_scanner_en_r_data_ipseckey = 777;
static const int zone_scanner_en_r_data_rrsig = 816;
static const int zone_scanner_en_r_data_nsec = 958;
static const int zone_scanner_en_r_data_dnskey = 961;
static const int zone_scanner_en_r_data_dhcid = 972;
static const int zone_scanner_en_r_data_nsec3 = 974;
static const int zone_scanner_en_r_data_nsec3param = 1003;
static const int zone_scanner_en_r_data_tlsa = 1016;
static const int zone_scanner_en_r_data_l32 = 1029;
static const int zone_scanner_en_r_data_l64 = 1034;
static const int zone_scanner_en_r_data_eui48 = 1047;
static const int zone_scanner_en_r_data_eui64 = 1053;
static const int zone_scanner_en_main = 1059;


#line 90 "./scanner.rl"


scanner_t* scanner_create(const char     *file_name,
                          const char     *origin,
                          const uint16_t rclass,
                          const uint32_t ttl,
                          void (*process_record)(const scanner_t *),
                          void (*process_error)(const scanner_t *),
                          void *data)
{
	char settings[1024];

	scanner_t *s = calloc(1, sizeof(scanner_t));
	if (s == NULL) {
		return NULL;
	}

	if (file_name != NULL) {
		// Get absolute path of the zone file.
		if (realpath(file_name, (char*)(s->buffer)) != NULL) {
			char *full_name = strdup((char*)(s->buffer));
			s->path = strdup(dirname(full_name));
			free(full_name);
		} else {
			free(s);
			return NULL;
		}

		s->file_name = strdup(file_name);
	} else {
		s->path = strdup(".");
		s->file_name = strdup("<NULL>");
	}

	// Nonzero initial scanner state.
	s->cs = zone_scanner_start;

	// Disable processing during parsing of settings.
	s->process_record = &noop;
	s->process_error = &noop;

	// Create ORIGIN directive and parse it using scanner to set up origin.
	const char *format;
	size_t origin_len = strlen(origin);
	if (origin_len > 0 && origin[origin_len - 1] != '.') {
		format = "$ORIGIN %s.\n";
	} else {
		format = "$ORIGIN %s\n";
	}
	int ret = snprintf(settings, sizeof(settings), format, origin);
	if (ret <= 0 || (size_t)ret >= sizeof(settings) ||
	    scanner_process(settings, settings + ret, true, s) != 0) {
		scanner_free(s);
		return NULL;
	}

	// Set scanner defaults.
	s->default_class = rclass;
	s->default_ttl = ttl;
	s->process_record = process_record ? process_record : &noop;
	s->process_error = process_error ? process_error : &noop;
	s->data = data;
	s->line_counter = 1;

	return s;
}

void scanner_free(scanner_t *s)
{
	if (s != NULL) {
		free(s->file_name);
		free(s->path);
		free(s);
	}
}

int scanner_process(const char *start,
                    const char *end,
                    const bool is_complete,
                    scanner_t  *s)
{
	// Necessary scanner variables.
	const char *p = start;
	const char *pe = end;
	char       *eof = NULL;
	int        stack[RAGEL_STACK_SIZE];

	// Auxiliary variables which are used in scanner body.
	struct in_addr  addr4;
	struct in6_addr addr6;
	uint32_t timestamp;
	int16_t  window;
	int      ret;

	// Next 2 variables are for better performance.
	// Restoring r_data pointer to next free space.
	uint8_t *rdata_tail = s->r_data + s->r_data_tail;
	// Initialization of the last r_data byte.
	uint8_t *rdata_stop = s->r_data + MAX_RDATA_LENGTH - 1;

	// Restoring scanner states.
	int cs  = s->cs;
	int top = s->top;
	memcpy(stack, s->stack, sizeof(stack));

	// End of file check.
	if (is_complete == true) {
		eof = (char *)pe;
	}

	// Writing scanner body (in C).
	
#line 5042 "scanner.c"
	{
	int _klen;
	unsigned int _trans;
	short _widec;
	const short *_acts;
	unsigned int _nacts;
	const short *_keys;

	if ( p == pe )
		goto _test_eof;
	if ( cs == 0 )
		goto _out;
_resume:
	_widec = (*p);
	_klen = _zone_scanner_cond_lengths[cs];
	_keys = _zone_scanner_cond_keys + (_zone_scanner_cond_offsets[cs]*2);
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				switch ( _zone_scanner_cond_spaces[_zone_scanner_cond_offsets[cs] + ((_mid - _keys)>>1)] ) {
	case 0: {
		_widec = (short)(640 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		break;
	}
	case 1: {
		_widec = (short)(1152 + ((*p) - -128));
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 256;
		break;
	}
	case 2: {
		_widec = (short)(128 + ((*p) - -128));
		if ( 
#line 555 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		break;
	}
	case 3: {
		_widec = (short)(2688 + ((*p) - -128));
		if ( 
#line 1174 "./scanner_body.rl"
 s->number64 != 0  ) _widec += 256;
		break;
	}
	case 4: {
		_widec = (short)(4224 + ((*p) - -128));
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 256;
		break;
	}
	case 5: {
		_widec = (short)(1664 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		break;
	}
	case 6: {
		_widec = (short)(3200 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1174 "./scanner_body.rl"
 s->number64 != 0  ) _widec += 512;
		break;
	}
	case 7: {
		_widec = (short)(4736 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 512;
		break;
	}
	case 8: {
		_widec = (short)(5760 + ((*p) - -128));
		if ( 
#line 1174 "./scanner_body.rl"
 s->number64 != 0  ) _widec += 256;
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 512;
		break;
	}
	case 9: {
		_widec = (short)(12928 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 1024;
		break;
	}
	case 10: {
		_widec = (short)(6784 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 1174 "./scanner_body.rl"
 s->number64 != 0  ) _widec += 512;
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 1024;
		break;
	}
	case 11: {
		_widec = (short)(8832 + ((*p) - -128));
		if ( 
#line 54 "./scanner_body.rl"
 s->multiline  ) _widec += 256;
		if ( 
#line 63 "./scanner_body.rl"
 !s->multiline  ) _widec += 512;
		if ( 
#line 1174 "./scanner_body.rl"
 s->number64 != 0  ) _widec += 1024;
		if ( 
#line 1175 "./scanner_body.rl"
 s->number64 == 0  ) _widec += 2048;
		break;
	}
				}
				break;
			}
		}
	}

	_keys = _zone_scanner_trans_keys + _zone_scanner_key_offsets[cs];
	_trans = _zone_scanner_index_offsets[cs];

	_klen = _zone_scanner_single_lengths[cs];
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + _klen - 1;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + ((_upper-_lower) >> 1);
			if ( _widec < *_mid )
				_upper = _mid - 1;
			else if ( _widec > *_mid )
				_lower = _mid + 1;
			else {
				_trans += (unsigned int)(_mid - _keys);
				goto _match;
			}
		}
		_keys += _klen;
		_trans += _klen;
	}

	_klen = _zone_scanner_range_lengths[cs];
	if ( _klen > 0 ) {
		const short *_lower = _keys;
		const short *_mid;
		const short *_upper = _keys + (_klen<<1) - 2;
		while (1) {
			if ( _upper < _lower )
				break;

			_mid = _lower + (((_upper-_lower) >> 1) & ~1);
			if ( _widec < _mid[0] )
				_upper = _mid - 2;
			else if ( _widec > _mid[1] )
				_lower = _mid + 2;
			else {
				_trans += (unsigned int)((_mid - _keys)>>1);
				goto _match;
			}
		}
		_trans += _klen;
	}

_match:
	_trans = _zone_scanner_indicies[_trans];
	cs = _zone_scanner_trans_targs[_trans];

	if ( _zone_scanner_trans_actions[_trans] == 0 )
		goto _again;

	_acts = _zone_scanner_actions + _zone_scanner_trans_actions[_trans];
	_nacts = (unsigned int) *_acts++;
	while ( _nacts-- > 0 )
	{
		switch ( *_acts++ )
		{
	case 0:
#line 20 "./scanner_body.rl"
	{
		p--; {cs = stack[--top]; goto _again;}
	}
	break;
	case 1:
#line 25 "./scanner_body.rl"
	{
		s->line_counter++;
	}
	break;
	case 2:
#line 29 "./scanner_body.rl"
	{
		if (s->multiline == true) {
			ERR(ZSCANNER_ELEFT_PARENTHESIS);
			p--; {cs = 246; goto _again;}
		}
		s->multiline = true;
	}
	break;
	case 3:
#line 36 "./scanner_body.rl"
	{
		if (s->multiline == false) {
			ERR(ZSCANNER_ERIGHT_PARENTHESIS);
			p--; {cs = 246; goto _again;}
		}
		s->multiline = false;
	}
	break;
	case 4:
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 5:
#line 67 "./scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 6:
#line 70 "./scanner_body.rl"
	{
		if ((*p) == '\r') {
			ERR(ZSCANNER_DOS_NEWLINE);
		}

		if (s->buffer_length < sizeof(s->buffer) - 1) {
			s->buffer[s->buffer_length++] = (*p);
		}
	}
	break;
	case 7:
#line 79 "./scanner_body.rl"
	{
		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Error counter incrementation.
		s->error_counter++;

		// Initialization of fcall stack.
		top = 0;

		// Process error message.
		s->process_error(s);

		// Reset.
		s->error_code = ZSCANNER_OK;
		s->multiline = false;

		// In case of serious error, stop scanner.
		if (s->stop == true) {
			return -1;
		}
	}
	break;
	case 8:
#line 104 "./scanner_body.rl"
	{ {cs = 1059; goto _again;} }
	break;
	case 9:
#line 108 "./scanner_body.rl"
	{
		s->item_length = 0;
		s->item_length_position = s->dname_tmp_length++;
	}
	break;
	case 10:
#line 112 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length++] = (*p);
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 11:
#line 121 "./scanner_body.rl"
	{
		if (s->dname_tmp_length < MAX_DNAME_LENGTH) {
			(s->dname)[s->item_length_position] =
				(uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 12:
#line 131 "./scanner_body.rl"
	{
		if (s->item_length < MAX_LABEL_LENGTH) {
			(s->dname)[s->dname_tmp_length] = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ELABEL_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 13:
#line 140 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length] *= 10;
		(s->dname)[s->dname_tmp_length] += digit_to_num[(uint8_t)(*p)];
	}
	break;
	case 14:
#line 144 "./scanner_body.rl"
	{
		s->dname_tmp_length++;
	}
	break;
	case 15:
#line 147 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 16:
#line 166 "./scanner_body.rl"
	{
		(s->dname)[s->dname_tmp_length++] = 0;
	}
	break;
	case 17:
#line 169 "./scanner_body.rl"
	{
		memcpy(s->dname + s->dname_tmp_length,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length += s->zone_origin_length;

		if (s->dname_tmp_length > MAX_DNAME_LENGTH) {
			WARN(ZSCANNER_EDNAME_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 18:
#line 181 "./scanner_body.rl"
	{
		memcpy(s->dname,
		       s->zone_origin,
		       s->zone_origin_length);

		s->dname_tmp_length = s->zone_origin_length;
	}
	break;
	case 19:
#line 189 "./scanner_body.rl"
	{
		s->item_length_position = 0;
		s->dname_tmp_length = 0;
	}
	break;
	case 20:
#line 193 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 21:
#line 205 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 248; goto _again;} }
	break;
	case 22:
#line 209 "./scanner_body.rl"
	{
		s->item_length_location = rdata_tail++;
	}
	break;
	case 23:
#line 212 "./scanner_body.rl"
	{
		s->item_length = rdata_tail - s->item_length_location - 1;

		if (s->item_length <= MAX_ITEM_LENGTH) {
			*(s->item_length_location) = (uint8_t)(s->item_length);
		} else {
			WARN(ZSCANNER_EITEM_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 24:
#line 225 "./scanner_body.rl"
	{
		s->dname = s->r_owner;
		s->r_owner_length = 0;
	}
	break;
	case 25:
#line 229 "./scanner_body.rl"
	{
		s->r_owner_length = s->dname_tmp_length;
	}
	break;
	case 26:
#line 232 "./scanner_body.rl"
	{
		if (s->r_owner_length == 0) {
			WARN(ZSCANNER_EBAD_PREVIOUS_OWNER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 27:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 28:
#line 250 "./scanner_body.rl"
	{
		s->dname = rdata_tail;
	}
	break;
	case 29:
#line 253 "./scanner_body.rl"
	{
		rdata_tail += s->dname_tmp_length;
	}
	break;
	case 30:
#line 261 "./scanner_body.rl"
	{
		// Overflow check: 10*(s->number64) + fc - ASCII_0 <= UINT64_MAX
		if ((s->number64 < (UINT64_MAX / 10)) ||   // Dominant fast check.
			((s->number64 == (UINT64_MAX / 10)) && // Marginal case.
			 ((uint8_t)(*p) <= (UINT64_MAX % 10) + ASCII_0)
			)
		   ) {
			s->number64 *= 10;
			s->number64 += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER64_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 31:
#line 278 "./scanner_body.rl"
	{
		s->number64 = 0;
	}
	break;
	case 32:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 33:
#line 289 "./scanner_body.rl"
	{
		s->decimal_counter = 0;
	}
	break;
	case 34:
#line 292 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
	break;
	case 35:
#line 295 "./scanner_body.rl"
	{
		s->decimal_counter++;
	}
	break;
	case 36:
#line 299 "./scanner_body.rl"
	{
		if (s->decimal_counter == 0 && s->number64 < UINT32_MAX) {
			s->number64 *= pow(10, s->decimals);
		} else if (s->decimal_counter <= s->decimals &&
				 s->number64_tmp < UINT32_MAX) {
			s->number64 *= pow(10, s->decimals - s->decimal_counter);
			s->number64 += s->number64_tmp * pow(10, s->decimals);
		} else {
			WARN(ZSCANNER_EFLOAT_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 37:
#line 316 "./scanner_body.rl"
	{
		s->decimals = 2;
	}
	break;
	case 38:
#line 319 "./scanner_body.rl"
	{
		s->decimals = 3;
	}
	break;
	case 39:
#line 328 "./scanner_body.rl"
	{
		if (s->number64 <= UINT8_MAX) {
			*rdata_tail = (uint8_t)(s->number64);
			rdata_tail += 1;
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 40:
#line 337 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			*((uint16_t *)rdata_tail) = htons((uint16_t)(s->number64));
			rdata_tail += 2;
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 41:
#line 346 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			*((uint32_t *)rdata_tail) = htonl((uint32_t)(s->number64));
			rdata_tail += 4;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 42:
#line 356 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_type = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 43:
#line 365 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			s->r_data_length = (uint16_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 44:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 45:
#line 389 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 60)) {
	                  s->number64 *= 60;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 246; goto _again;}
	              }
	            }
	break;
	case 46:
#line 396 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 3600)) {
	                  s->number64 *= 3600;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 246; goto _again;}
	              }
	            }
	break;
	case 47:
#line 403 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 86400)) {
	                  s->number64 *= 86400;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 246; goto _again;}
	              }
	            }
	break;
	case 48:
#line 410 "./scanner_body.rl"
	{ if (s->number64 <= (UINT32_MAX / 604800)) {
	                  s->number64 *= 604800;
	              } else {
	                  WARN(ZSCANNER_ENUMBER32_OVERFLOW);
	                  p--; {cs = 246; goto _again;}
	              }
	            }
	break;
	case 49:
#line 420 "./scanner_body.rl"
	{
		s->number64_tmp = s->number64;
	}
	break;
	case 50:
#line 423 "./scanner_body.rl"
	{
		if (s->number64 + s->number64_tmp < UINT32_MAX) {
			s->number64 += s->number64_tmp;
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 51:
#line 441 "./scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 52:
#line 444 "./scanner_body.rl"
	{
		if (s->buffer_length < MAX_RDATA_LENGTH) {
			s->buffer[s->buffer_length++] = (*p);
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 53:
#line 452 "./scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (s->buffer_length == 14) { // Date; 14 = len("YYYYMMDDHHmmSS").
			ret = date_to_timestamp(s->buffer, &timestamp);

			if (ret == ZSCANNER_OK) {
				*((uint32_t *)rdata_tail) = htonl(timestamp);
				rdata_tail += 4;
			} else {
				WARN(ret);
				p--; {cs = 246; goto _again;}
			}
		} else if (s->buffer_length <= 10) { // Timestamp format.
			char *end;

			s->number64 = strtoull((char *)(s->buffer), &end,  10);

			if (end == (char *)(s->buffer) || *end != '\0') {
				WARN(ZSCANNER_EBAD_TIMESTAMP);
				p--; {cs = 246; goto _again;}
			}

			if (s->number64 <= UINT32_MAX) {
				*((uint32_t *)rdata_tail) = htonl((uint32_t)s->number64);
				rdata_tail += 4;
			} else {
				WARN(ZSCANNER_ENUMBER32_OVERFLOW);
				p--; {cs = 246; goto _again;}
			}
		} else {
			WARN(ZSCANNER_EBAD_TIMESTAMP_LENGTH);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 54:
#line 487 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIMESTAMP_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 55:
#line 497 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*(rdata_tail++) = (*p);
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 56:
#line 505 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 57:
#line 509 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 58:
#line 514 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = 0;
			s->item_length++;
		} else {
			WARN(ZSCANNER_ETEXT_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 59:
#line 523 "./scanner_body.rl"
	{
		if ((*rdata_tail < (UINT8_MAX / 10)) ||   // Dominant fast check.
			((*rdata_tail == (UINT8_MAX / 10)) && // Marginal case.
			 ((*p) <= (UINT8_MAX % 10) + ASCII_0)
			)
		   ) {
			*rdata_tail *= 10;
			*rdata_tail += digit_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ENUMBER8_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 60:
#line 536 "./scanner_body.rl"
	{
		rdata_tail++;
	}
	break;
	case 61:
#line 539 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 62:
#line 561 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 258; goto _again;} }
	break;
	case 63:
#line 571 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->default_ttl = (uint32_t)(s->number64);
		} else {
			ERR(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 64:
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 65:
#line 586 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 270; goto _again;} }
	break;
	case 66:
#line 590 "./scanner_body.rl"
	{
		s->dname = s->zone_origin;
	}
	break;
	case 67:
#line 593 "./scanner_body.rl"
	{
		s->zone_origin_length = s->dname_tmp_length;
	}
	break;
	case 68:
#line 596 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_ORIGIN);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 69:
#line 603 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 279; goto _again;} }
	break;
	case 70:
#line 607 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	break;
	case 71:
#line 610 "./scanner_body.rl"
	{
		*rdata_tail = 0; // Ending filename string.
		strncpy((char*)(s->include_filename), (char*)(s->r_data),
		        sizeof(s->include_filename));

		// Check for correct string copy.
		if (strlen(s->include_filename) !=
		    (size_t)(rdata_tail - s->r_data)) {
			ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
			p--; {cs = 246; goto _again;}
		}

		// For detection whether origin is not present.
		s->dname = NULL;
	}
	break;
	case 72:
#line 625 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 73:
#line 630 "./scanner_body.rl"
	{
		s->dname = s->r_data;
	}
	break;
	case 74:
#line 633 "./scanner_body.rl"
	{
		s->r_data_length = s->dname_tmp_length;
	}
	break;
	case 75:
#line 636 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 76:
#line 641 "./scanner_body.rl"
	{
		char text_origin[4 * MAX_DNAME_LENGTH]; // Each char as \DDD.

		// Origin conversion from wire to text form.
		if (s->dname == NULL) { // Use current origin.
			wire_dname_to_str(s->zone_origin,
			                  s->zone_origin_length,
			                  text_origin);
		} else { // Use specified origin.
			wire_dname_to_str(s->r_data,
			                  s->r_data_length,
			                  text_origin);
		}

		// Relative file path.
		if (s->include_filename[0] != '/') {
			snprintf((char*)(s->buffer), sizeof(s->buffer),
			         "%s/%s", s->path, s->include_filename);
		} else {
			strncpy((char*)(s->buffer), (char*)(s->include_filename),
			        sizeof(s->buffer));
		}

		// Create new file loader for included zone file.
		file_loader_t *fl = file_loader_create((char*)(s->buffer),
		                                       text_origin,
		                                       s->default_class,
		                                       s->default_ttl,
		                                       s->process_record,
		                                       s->process_error,
		                                       s->data);
		if (fl != NULL) {
			// Process included zone file.
			ret = file_loader_process(fl);
			file_loader_free(fl);

			if (ret != 0) {
				ERR(ZSCANNER_EUNPROCESSED_INCLUDE);
				p--; {cs = 246; goto _again;}
			}
		} else {
			ERR(ZSCANNER_EUNOPENED_INCLUDE);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 77:
#line 694 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 292; goto _again;} }
	break;
	case 78:
#line 700 "./scanner_body.rl"
	{
		s->stop = true;
	}
	break;
	case 79:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
	break;
	case 80:
#line 707 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 81:
#line 719 "./scanner_body.rl"
	{
		s->r_class = s->default_class;
	}
	break;
	case 82:
#line 723 "./scanner_body.rl"
	{
		s->r_ttl = s->default_ttl;
	}
	break;
	case 83:
#line 727 "./scanner_body.rl"
	{
		s->r_class = KNOT_CLASS_IN;
	}
	break;
	case 84:
#line 731 "./scanner_body.rl"
	{
		if (s->number64 <= UINT32_MAX) {
			s->r_ttl = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_ENUMBER32_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 85:
#line 746 "./scanner_body.rl"
	{
		s->buffer_length = 0;
	}
	break;
	case 86:
#line 749 "./scanner_body.rl"
	{
		if (s->buffer_length < MAX_RDATA_LENGTH) {
			s->buffer[s->buffer_length++] = (*p);
		}
		else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 87:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 88:
#line 763 "./scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (inet_pton(AF_INET, (char *)s->buffer, &addr4) <= 0) {
			WARN(ZSCANNER_EBAD_IPV4);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 89:
#line 771 "./scanner_body.rl"
	{
		memcpy(rdata_tail, &(addr4.s_addr), INET4_ADDR_LENGTH);
		rdata_tail += INET4_ADDR_LENGTH;
	}
	break;
	case 90:
#line 776 "./scanner_body.rl"
	{
		s->buffer[s->buffer_length] = 0;

		if (inet_pton(AF_INET6, (char *)s->buffer, &addr6) <= 0) {
			WARN(ZSCANNER_EBAD_IPV6);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 91:
#line 784 "./scanner_body.rl"
	{
		memcpy(rdata_tail, &(addr6.s6_addr), INET6_ADDR_LENGTH);
		rdata_tail += INET6_ADDR_LENGTH;
	}
	break;
	case 92:
#line 801 "./scanner_body.rl"
	{
		memset(&(s->apl), 0, sizeof(s->apl));
	}
	break;
	case 93:
#line 804 "./scanner_body.rl"
	{
		s->apl.excl_flag = 128; // dec 128  = bin 10000000.
	}
	break;
	case 94:
#line 807 "./scanner_body.rl"
	{
		s->apl.addr_family = 1;
	}
	break;
	case 95:
#line 810 "./scanner_body.rl"
	{
		s->apl.addr_family = 2;
	}
	break;
	case 96:
#line 813 "./scanner_body.rl"
	{
		if ((s->apl.addr_family == 1 && s->number64 <= 32) ||
		    (s->apl.addr_family == 2 && s->number64 <= 128)) {
			s->apl.prefix_length = (uint8_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_APL);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 97:
#line 822 "./scanner_body.rl"
	{
		// Write address family.
		*((uint16_t *)rdata_tail) = htons(s->apl.addr_family);
		rdata_tail += 2;
		// Write prefix length in bites.
		*(rdata_tail) = s->apl.prefix_length;
		rdata_tail += 1;
		// Copy address to buffer.
		uint8_t len;
		switch (s->apl.addr_family) {
		case 1:
			len = INET4_ADDR_LENGTH;
			memcpy(s->buffer, &(addr4.s_addr), len);
			break;
		case 2:
			len = INET6_ADDR_LENGTH;
			memcpy(s->buffer, &(addr6.s6_addr), len);
			break;
		default:
			WARN(ZSCANNER_EBAD_APL);
			p--; {cs = 246; goto _again;}
		}
		// Find prefix without trailing zeroes.
		while (len > 0) {
			if ((s->buffer[len - 1] & 255) != 0) {
				break;
			}
			len--;
		}
		// Write negation flag + prefix length in bytes.
		*(rdata_tail) = len + s->apl.excl_flag;
		rdata_tail += 1;
		// Write address prefix non-null data.
		memcpy(rdata_tail, s->buffer, len);
		rdata_tail += len;
	}
	break;
	case 98:
#line 858 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_APL);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 99:
#line 876 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 100:
#line 884 "./scanner_body.rl"
	{
		*rdata_tail += second_hex_to_num[(uint8_t)(*p)];
		rdata_tail++;
	}
	break;
	case 101:
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 102:
#line 902 "./scanner_body.rl"
	{
		if ((rdata_tail - s->r_data) != s->r_data_length) {
			WARN(ZSCANNER_EBAD_RDATA_LENGTH);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 103:
#line 909 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 104:
#line 919 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 105:
#line 927 "./scanner_body.rl"
	{
		*(rdata_tail++) += second_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = second_right_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 106:
#line 937 "./scanner_body.rl"
	{
		*(rdata_tail++) += third_left_base64_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = third_right_base64_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 107:
#line 947 "./scanner_body.rl"
	{
		*(rdata_tail++) += fourth_base64_to_num[(uint8_t)(*p)];
	}
	break;
	case 108:
#line 951 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE64_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 109:
#line 973 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 310; goto _again;} }
	break;
	case 110:
#line 977 "./scanner_body.rl"
	{
		if (rdata_tail <= rdata_stop) {
			*rdata_tail = first_base32hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 111:
#line 985 "./scanner_body.rl"
	{
		*(rdata_tail++) += second_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = second_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 112:
#line 995 "./scanner_body.rl"
	{
		*rdata_tail += third_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 113:
#line 998 "./scanner_body.rl"
	{
		*(rdata_tail++) += fourth_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = fourth_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 114:
#line 1008 "./scanner_body.rl"
	{
		*(rdata_tail++) += fifth_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = fifth_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 115:
#line 1018 "./scanner_body.rl"
	{
		*rdata_tail += sixth_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 116:
#line 1021 "./scanner_body.rl"
	{
		*(rdata_tail++) += seventh_left_base32hex_to_num[(uint8_t)(*p)];

		if (rdata_tail <= rdata_stop) {
			*rdata_tail = seventh_right_base32hex_to_num[(uint8_t)(*p)];
		} else {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 117:
#line 1031 "./scanner_body.rl"
	{
		*(rdata_tail++) += eighth_base32hex_to_num[(uint8_t)(*p)];
	}
	break;
	case 118:
#line 1035 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE32HEX_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 119:
#line 1070 "./scanner_body.rl"
	{
		*(rdata_tail++) = 0;
	}
	break;
	case 120:
#line 1073 "./scanner_body.rl"
	{
		*(rdata_tail++) = 1;
	}
	break;
	case 121:
#line 1076 "./scanner_body.rl"
	{
		*(rdata_tail++) = 2;
	}
	break;
	case 122:
#line 1079 "./scanner_body.rl"
	{
		*(rdata_tail++) = 3;
	}
	break;
	case 123:
#line 1082 "./scanner_body.rl"
	{
		*(rdata_tail++) = 5;
	}
	break;
	case 124:
#line 1085 "./scanner_body.rl"
	{
		*(rdata_tail++) = 6;
	}
	break;
	case 125:
#line 1088 "./scanner_body.rl"
	{
		*(rdata_tail++) = 7;
	}
	break;
	case 126:
#line 1091 "./scanner_body.rl"
	{
		*(rdata_tail++) = 8;
	}
	break;
	case 127:
#line 1094 "./scanner_body.rl"
	{
		*(rdata_tail++) = 10;
	}
	break;
	case 128:
#line 1097 "./scanner_body.rl"
	{
		*(rdata_tail++) = 12;
	}
	break;
	case 129:
#line 1100 "./scanner_body.rl"
	{
		*(rdata_tail++) = 13;
	}
	break;
	case 130:
#line 1103 "./scanner_body.rl"
	{
		*(rdata_tail++) = 14;
	}
	break;
	case 131:
#line 1106 "./scanner_body.rl"
	{
		*(rdata_tail++) = 252;
	}
	break;
	case 132:
#line 1109 "./scanner_body.rl"
	{
		*(rdata_tail++) = 253;
	}
	break;
	case 133:
#line 1112 "./scanner_body.rl"
	{
		*(rdata_tail++) = 254;
	}
	break;
	case 134:
#line 1116 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(1);
		rdata_tail += 2;
	}
	break;
	case 135:
#line 1120 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(2);
		rdata_tail += 2;
	}
	break;
	case 136:
#line 1124 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(3);
		rdata_tail += 2;
	}
	break;
	case 137:
#line 1128 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(4);
		rdata_tail += 2;
	}
	break;
	case 138:
#line 1132 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(5);
		rdata_tail += 2;
	}
	break;
	case 139:
#line 1136 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(6);
		rdata_tail += 2;
	}
	break;
	case 140:
#line 1140 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(7);
		rdata_tail += 2;
	}
	break;
	case 141:
#line 1144 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(8);
		rdata_tail += 2;
	}
	break;
	case 142:
#line 1148 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(253);
		rdata_tail += 2;
	}
	break;
	case 143:
#line 1152 "./scanner_body.rl"
	{
		*((uint16_t *)rdata_tail) = htons(254);
		rdata_tail += 2;
	}
	break;
	case 144:
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 145:
#line 1163 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 146:
#line 1181 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 147:
#line 1187 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_A, &rdata_tail); }
	break;
	case 148:
#line 1188 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NS, &rdata_tail); }
	break;
	case 149:
#line 1189 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_CNAME, &rdata_tail); }
	break;
	case 150:
#line 1190 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SOA, &rdata_tail); }
	break;
	case 151:
#line 1191 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_PTR, &rdata_tail); }
	break;
	case 152:
#line 1192 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_HINFO, &rdata_tail); }
	break;
	case 153:
#line 1193 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_MINFO, &rdata_tail); }
	break;
	case 154:
#line 1194 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_MX, &rdata_tail); }
	break;
	case 155:
#line 1195 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_TXT, &rdata_tail); }
	break;
	case 156:
#line 1196 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RP, &rdata_tail); }
	break;
	case 157:
#line 1197 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_AFSDB, &rdata_tail); }
	break;
	case 158:
#line 1198 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RT, &rdata_tail); }
	break;
	case 159:
#line 1199 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_KEY, &rdata_tail); }
	break;
	case 160:
#line 1200 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_AAAA, &rdata_tail); }
	break;
	case 161:
#line 1201 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_LOC, &rdata_tail); }
	break;
	case 162:
#line 1202 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SRV, &rdata_tail); }
	break;
	case 163:
#line 1203 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NAPTR, &rdata_tail); }
	break;
	case 164:
#line 1204 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_KX, &rdata_tail); }
	break;
	case 165:
#line 1205 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_CERT, &rdata_tail); }
	break;
	case 166:
#line 1206 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DNAME, &rdata_tail); }
	break;
	case 167:
#line 1207 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_APL, &rdata_tail); }
	break;
	case 168:
#line 1208 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DS, &rdata_tail); }
	break;
	case 169:
#line 1209 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SSHFP, &rdata_tail); }
	break;
	case 170:
#line 1210 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_IPSECKEY, &rdata_tail); }
	break;
	case 171:
#line 1211 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_RRSIG, &rdata_tail); }
	break;
	case 172:
#line 1212 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC, &rdata_tail); }
	break;
	case 173:
#line 1213 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DNSKEY, &rdata_tail); }
	break;
	case 174:
#line 1214 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_DHCID, &rdata_tail); }
	break;
	case 175:
#line 1215 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC3, &rdata_tail); }
	break;
	case 176:
#line 1216 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NSEC3PARAM, &rdata_tail); }
	break;
	case 177:
#line 1217 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_TLSA, &rdata_tail); }
	break;
	case 178:
#line 1218 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_SPF, &rdata_tail); }
	break;
	case 179:
#line 1219 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_NID, &rdata_tail); }
	break;
	case 180:
#line 1220 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_L32, &rdata_tail); }
	break;
	case 181:
#line 1221 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_L64, &rdata_tail); }
	break;
	case 182:
#line 1222 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_LP, &rdata_tail); }
	break;
	case 183:
#line 1223 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_EUI48, &rdata_tail); }
	break;
	case 184:
#line 1224 "./scanner_body.rl"
	{ type_num(KNOT_RRTYPE_EUI64, &rdata_tail); }
	break;
	case 185:
#line 1230 "./scanner_body.rl"
	{
		if (s->number64 <= UINT16_MAX) {
			window_add_bit(s->number64, s);
		} else {
			WARN(ZSCANNER_ENUMBER16_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 186:
#line 1243 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_A, s); }
	break;
	case 187:
#line 1244 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NS, s); }
	break;
	case 188:
#line 1245 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CNAME, s); }
	break;
	case 189:
#line 1246 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SOA, s); }
	break;
	case 190:
#line 1247 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_PTR, s); }
	break;
	case 191:
#line 1248 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_HINFO, s); }
	break;
	case 192:
#line 1249 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MINFO, s); }
	break;
	case 193:
#line 1250 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_MX, s); }
	break;
	case 194:
#line 1251 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TXT, s); }
	break;
	case 195:
#line 1252 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RP, s); }
	break;
	case 196:
#line 1253 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AFSDB, s); }
	break;
	case 197:
#line 1254 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RT, s); }
	break;
	case 198:
#line 1255 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KEY, s); }
	break;
	case 199:
#line 1256 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_AAAA, s); }
	break;
	case 200:
#line 1257 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LOC, s); }
	break;
	case 201:
#line 1258 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SRV, s); }
	break;
	case 202:
#line 1259 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NAPTR, s); }
	break;
	case 203:
#line 1260 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_KX, s); }
	break;
	case 204:
#line 1261 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_CERT, s); }
	break;
	case 205:
#line 1262 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNAME, s); }
	break;
	case 206:
#line 1263 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_APL, s); }
	break;
	case 207:
#line 1264 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DS, s); }
	break;
	case 208:
#line 1265 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SSHFP, s); }
	break;
	case 209:
#line 1266 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_IPSECKEY, s); }
	break;
	case 210:
#line 1267 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_RRSIG, s); }
	break;
	case 211:
#line 1268 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC, s); }
	break;
	case 212:
#line 1269 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DNSKEY, s); }
	break;
	case 213:
#line 1270 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_DHCID, s); }
	break;
	case 214:
#line 1271 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3, s); }
	break;
	case 215:
#line 1272 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NSEC3PARAM, s); }
	break;
	case 216:
#line 1273 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_TLSA, s); }
	break;
	case 217:
#line 1274 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_SPF, s); }
	break;
	case 218:
#line 1275 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_NID, s); }
	break;
	case 219:
#line 1276 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L32, s); }
	break;
	case 220:
#line 1277 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_L64, s); }
	break;
	case 221:
#line 1278 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_LP, s); }
	break;
	case 222:
#line 1279 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI48, s); }
	break;
	case 223:
#line 1280 "./scanner_body.rl"
	{ window_add_bit(KNOT_RRTYPE_EUI64, s); }
	break;
	case 224:
#line 1284 "./scanner_body.rl"
	{
		memset(s->windows, 0, sizeof(s->windows));
		s->last_window = -1;
	}
	break;
	case 225:
#line 1288 "./scanner_body.rl"
	{
		for (window = 0; window <= s->last_window; window++) {
			if ((s->windows[window]).length > 0) {
				if (rdata_tail + 2 + (s->windows[window]).length <= rdata_stop)
				{
					// Window number.
					*rdata_tail = (uint8_t)window;
					rdata_tail += 1;
					// Bitmap length.
					*rdata_tail = (s->windows[window]).length;
					rdata_tail += 1;
					// Copying bitmap.
					memcpy(rdata_tail,
					       (s->windows[window]).bitmap,
					       (s->windows[window]).length);
					rdata_tail += (s->windows[window]).length;
				} else {
					WARN(ZSCANNER_ERDATA_OVERFLOW);
					p--; {cs = 246; goto _again;}
				}
			}
		}
	}
	break;
	case 226:
#line 1311 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BITMAP);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 227:
#line 1319 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 317; goto _again;} }
	break;
	case 228:
#line 1323 "./scanner_body.rl"
	{
		if (s->number64 <= 90) {
			s->loc.d1 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 229:
#line 1331 "./scanner_body.rl"
	{
		if (s->number64 <= 180) {
			s->loc.d2 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 230:
#line 1339 "./scanner_body.rl"
	{
		if (s->number64 <= 59) {
			s->loc.m1 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 231:
#line 1347 "./scanner_body.rl"
	{
		if (s->number64 <= 59) {
			s->loc.m2 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 232:
#line 1355 "./scanner_body.rl"
	{
		if (s->number64 <= 59999) {
			s->loc.s1 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 233:
#line 1363 "./scanner_body.rl"
	{
		if (s->number64 <= 59999) {
			s->loc.s2 = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 234:
#line 1371 "./scanner_body.rl"
	{
		if ((s->loc.alt_sign ==  1 && s->number64 <= 4284967295) ||
		    (s->loc.alt_sign == -1 && s->number64 <=   10000000))
		{
			s->loc.alt = (uint32_t)(s->number64);
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 235:
#line 1381 "./scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.siz = s->number64;
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 236:
#line 1389 "./scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.hp = s->number64;
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 237:
#line 1397 "./scanner_body.rl"
	{
		if (s->number64 <= 9000000000ULL) {
			s->loc.vp = s->number64;
		} else {
			WARN(ZSCANNER_EBAD_NUMBER);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 238:
#line 1405 "./scanner_body.rl"
	{
		s->loc.lat_sign = -1;
	}
	break;
	case 239:
#line 1408 "./scanner_body.rl"
	{
		s->loc.long_sign = -1;
	}
	break;
	case 240:
#line 1411 "./scanner_body.rl"
	{
		s->loc.alt_sign = -1;
	}
	break;
	case 241:
#line 1428 "./scanner_body.rl"
	{
		memset(&(s->loc), 0, sizeof(s->loc));
		// Defaults.
		s->loc.siz = 100;
		s->loc.vp  = 1000;
		s->loc.hp  = 1000000;
		s->loc.lat_sign  = 1;
		s->loc.long_sign = 1;
		s->loc.alt_sign  = 1;
	}
	break;
	case 242:
#line 1438 "./scanner_body.rl"
	{
		// Write version.
		*(rdata_tail) = 0;
		rdata_tail += 1;
		// Write size.
		*(rdata_tail) = loc64to8(s->loc.siz);
		rdata_tail += 1;
		// Write horizontal precision.
		*(rdata_tail) = loc64to8(s->loc.hp);
		rdata_tail += 1;
		// Write vertical precision.
		*(rdata_tail) = loc64to8(s->loc.vp);
		rdata_tail += 1;
		// Write latitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_LAT_ZERO + s->loc.lat_sign *
			(3600000 * s->loc.d1 + 60000 * s->loc.m1 + s->loc.s1));
		rdata_tail += 4;
		// Write longitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_LONG_ZERO + s->loc.long_sign *
			(3600000 * s->loc.d2 + 60000 * s->loc.m2 + s->loc.s2));
		rdata_tail += 4;
		// Write altitude.
		*((uint32_t *)rdata_tail) = htonl(LOC_ALT_ZERO + s->loc.alt_sign *
			(s->loc.alt));
		rdata_tail += 4;
	}
	break;
	case 243:
#line 1464 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_LOC_DATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 244:
#line 1477 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 245:
#line 1495 "./scanner_body.rl"
	{
		s->item_length = 0;
	}
	break;
	case 246:
#line 1498 "./scanner_body.rl"
	{
		s->item_length++;
	}
	break;
	case 247:
#line 1501 "./scanner_body.rl"
	{
		if (s->item_length != 6) {
			WARN(ZSCANNER_EBAD_EUI_LENGTH);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 248:
#line 1507 "./scanner_body.rl"
	{
		if (s->item_length != 8) {
			WARN(ZSCANNER_EBAD_EUI_LENGTH);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 249:
#line 1513 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_DASH);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 250:
#line 1528 "./scanner_body.rl"
	{
		s->item_length = 0;
	}
	break;
	case 251:
#line 1531 "./scanner_body.rl"
	{
		s->item_length++;
	}
	break;
	case 252:
#line 1534 "./scanner_body.rl"
	{
		if (s->item_length != 4) {
			WARN(ZSCANNER_EBAD_L64_LENGTH);
			p--; {cs = 246; goto _again;}
		}
	}
	break;
	case 253:
#line 1540 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_COLON);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 254:
#line 1553 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ALGORITHM);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 255:
#line 1557 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CERT_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 256:
#line 1579 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 456; goto _again;} }
	break;
	case 257:
#line 1594 "./scanner_body.rl"
	{ p--; {stack[top++] = cs; cs = 551; goto _again;} }
	break;
	case 258:
#line 1598 "./scanner_body.rl"
	{
		rdata_tail = s->r_data;
	}
	break;
	case 259:
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 260:
#line 1719 "./scanner_body.rl"
	{
		p--;
		switch (s->r_type) {
		case KNOT_RRTYPE_A:
			{stack[top++] = cs; cs = 592; goto _again;}
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
			{stack[top++] = cs; cs = 594; goto _again;}
		case KNOT_RRTYPE_SOA:
			{stack[top++] = cs; cs = 596; goto _again;}
		case KNOT_RRTYPE_HINFO:
			{stack[top++] = cs; cs = 628; goto _again;}
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_RP:
			{stack[top++] = cs; cs = 633; goto _again;}
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_LP:
			{stack[top++] = cs; cs = 638; goto _again;}
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
			{stack[top++] = cs; cs = 643; goto _again;}
		case KNOT_RRTYPE_AAAA:
			{stack[top++] = cs; cs = 647; goto _again;}
		case KNOT_RRTYPE_LOC:
			{stack[top++] = cs; cs = 649; goto _again;}
		case KNOT_RRTYPE_SRV:
			{stack[top++] = cs; cs = 704; goto _again;}
		case KNOT_RRTYPE_NAPTR:
			{stack[top++] = cs; cs = 715; goto _again;}
		case KNOT_RRTYPE_CERT:
			{stack[top++] = cs; cs = 732; goto _again;}
		case KNOT_RRTYPE_APL:
			{stack[top++] = cs; cs = 743; goto _again;}
		case KNOT_RRTYPE_DS:
			{stack[top++] = cs; cs = 754; goto _again;}
		case KNOT_RRTYPE_SSHFP:
			{stack[top++] = cs; cs = 767; goto _again;}
		case KNOT_RRTYPE_IPSECKEY:
			{stack[top++] = cs; cs = 777; goto _again;}
		case KNOT_RRTYPE_RRSIG:
			{stack[top++] = cs; cs = 816; goto _again;}
		case KNOT_RRTYPE_NSEC:
			{stack[top++] = cs; cs = 958; goto _again;}
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
			{stack[top++] = cs; cs = 961; goto _again;}
		case KNOT_RRTYPE_DHCID:
			{stack[top++] = cs; cs = 972; goto _again;}
		case KNOT_RRTYPE_NSEC3:
			{stack[top++] = cs; cs = 974; goto _again;}
		case KNOT_RRTYPE_NSEC3PARAM:
			{stack[top++] = cs; cs = 1003; goto _again;}
		case KNOT_RRTYPE_TLSA:
			{stack[top++] = cs; cs = 1016; goto _again;}
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L64:
			{stack[top++] = cs; cs = 1034; goto _again;}
		case KNOT_RRTYPE_L32:
			{stack[top++] = cs; cs = 1029; goto _again;}
		case KNOT_RRTYPE_EUI48:
			{stack[top++] = cs; cs = 1047; goto _again;}
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = cs; cs = 1053; goto _again;}
		default:
			WARN(ZSCANNER_ECANNOT_TEXT_DATA);
			{cs = 246; goto _again;}
		}
	}
	break;
	case 261:
#line 1792 "./scanner_body.rl"
	{
		switch (s->r_type) {
		// Next types must not have empty rdata.
		case KNOT_RRTYPE_A:
		case KNOT_RRTYPE_NS:
		case KNOT_RRTYPE_CNAME:
		case KNOT_RRTYPE_PTR:
		case KNOT_RRTYPE_DNAME:
		case KNOT_RRTYPE_SOA:
		case KNOT_RRTYPE_HINFO:
		case KNOT_RRTYPE_MINFO:
		case KNOT_RRTYPE_MX:
		case KNOT_RRTYPE_AFSDB:
		case KNOT_RRTYPE_RT:
		case KNOT_RRTYPE_KX:
		case KNOT_RRTYPE_TXT:
		case KNOT_RRTYPE_SPF:
		case KNOT_RRTYPE_RP:
		case KNOT_RRTYPE_AAAA:
		case KNOT_RRTYPE_LOC:
		case KNOT_RRTYPE_SRV:
		case KNOT_RRTYPE_NAPTR:
		case KNOT_RRTYPE_CERT:
		case KNOT_RRTYPE_DS:
		case KNOT_RRTYPE_SSHFP:
		case KNOT_RRTYPE_IPSECKEY:
		case KNOT_RRTYPE_RRSIG:
		case KNOT_RRTYPE_NSEC:
		case KNOT_RRTYPE_KEY:
		case KNOT_RRTYPE_DNSKEY:
		case KNOT_RRTYPE_DHCID:
		case KNOT_RRTYPE_NSEC3:
		case KNOT_RRTYPE_NSEC3PARAM:
		case KNOT_RRTYPE_TLSA:
		case KNOT_RRTYPE_NID:
		case KNOT_RRTYPE_L32:
		case KNOT_RRTYPE_L64:
		case KNOT_RRTYPE_LP:
		case KNOT_RRTYPE_EUI48:
		case KNOT_RRTYPE_EUI64:
			{stack[top++] = cs; cs = 437; goto _again;}
		// Next types can have empty rdata.
		case KNOT_RRTYPE_APL:
		default:
			{stack[top++] = cs; cs = 446; goto _again;}
		}
	}
	break;
	case 262:
#line 1843 "./scanner_body.rl"
	{ p--; }
	break;
	case 263:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 264:
#line 1856 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_A; }
	break;
	case 265:
#line 1857 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NS; }
	break;
	case 266:
#line 1858 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CNAME; }
	break;
	case 267:
#line 1859 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SOA; }
	break;
	case 268:
#line 1860 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_PTR; }
	break;
	case 269:
#line 1861 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_HINFO; }
	break;
	case 270:
#line 1862 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MINFO; }
	break;
	case 271:
#line 1863 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_MX; }
	break;
	case 272:
#line 1864 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TXT; }
	break;
	case 273:
#line 1865 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RP; }
	break;
	case 274:
#line 1866 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AFSDB; }
	break;
	case 275:
#line 1867 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RT; }
	break;
	case 276:
#line 1868 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KEY; }
	break;
	case 277:
#line 1869 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_AAAA; }
	break;
	case 278:
#line 1870 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LOC; }
	break;
	case 279:
#line 1871 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SRV; }
	break;
	case 280:
#line 1872 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NAPTR; }
	break;
	case 281:
#line 1873 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_KX; }
	break;
	case 282:
#line 1874 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_CERT; }
	break;
	case 283:
#line 1875 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNAME; }
	break;
	case 284:
#line 1876 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_APL; }
	break;
	case 285:
#line 1877 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DS; }
	break;
	case 286:
#line 1878 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SSHFP; }
	break;
	case 287:
#line 1879 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_IPSECKEY; }
	break;
	case 288:
#line 1880 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_RRSIG; }
	break;
	case 289:
#line 1881 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC; }
	break;
	case 290:
#line 1882 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DNSKEY; }
	break;
	case 291:
#line 1883 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_DHCID; }
	break;
	case 292:
#line 1884 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3; }
	break;
	case 293:
#line 1885 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NSEC3PARAM; }
	break;
	case 294:
#line 1886 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_TLSA; }
	break;
	case 295:
#line 1887 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_SPF; }
	break;
	case 296:
#line 1888 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_NID; }
	break;
	case 297:
#line 1889 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L32; }
	break;
	case 298:
#line 1890 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_L64; }
	break;
	case 299:
#line 1891 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_LP; }
	break;
	case 300:
#line 1892 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI48; }
	break;
	case 301:
#line 1893 "./scanner_body.rl"
	{ s->r_type = KNOT_RRTYPE_EUI64; }
	break;
	case 302:
#line 1899 "./scanner_body.rl"
	{
		if (rdata_tail - s->r_data > UINT16_MAX) {
			WARN(ZSCANNER_ERDATA_OVERFLOW);
			p--; {cs = 246; goto _again;}
		}
		s->r_data_length = rdata_tail - s->r_data;

		s->process_record(s);
	}
	break;
#line 7494 "scanner.c"
		}
	}

_again:
	if ( cs == 0 )
		goto _out;
	if ( ++p != pe )
		goto _resume;
	_test_eof: {}
	if ( p == eof )
	{
	const short *__acts = _zone_scanner_actions + _zone_scanner_eof_actions[cs];
	unsigned int __nacts = (unsigned int) *__acts++;
	while ( __nacts-- > 0 ) {
		switch ( *__acts++ ) {
	case 4:
#line 44 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_REST);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 15:
#line 147 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 20:
#line 193 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_DNAME_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 27:
#line 238 "./scanner_body.rl"
	{
		s->r_owner_length = 0;
		WARN(ZSCANNER_EBAD_OWNER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 32:
#line 281 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 44:
#line 382 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIME_UNIT);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 54:
#line 487 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TIMESTAMP_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 56:
#line 505 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 57:
#line 509 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_TEXT);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 61:
#line 539 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_NUMBER);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 64:
#line 579 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_TTL);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 68:
#line 596 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_ORIGIN);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 72:
#line 625 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_FILENAME);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 75:
#line 636 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_INCLUDE_ORIGIN);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 79:
#line 704 "./scanner_body.rl"
	{
		s->stop = false;
	}
	break;
	case 80:
#line 707 "./scanner_body.rl"
	{
		ERR(ZSCANNER_EBAD_DIRECTIVE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 87:
#line 758 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ADDRESS_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 98:
#line 858 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_APL);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 101:
#line 888 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 103:
#line 909 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 108:
#line 951 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE64_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 118:
#line 1035 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BASE32HEX_CHAR);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 144:
#line 1159 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 145:
#line 1163 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_GATEWAY_KEY);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 146:
#line 1181 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 226:
#line 1311 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_BITMAP);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 243:
#line 1464 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_LOC_DATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 244:
#line 1477 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_HEX_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 249:
#line 1513 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_DASH);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 253:
#line 1540 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CHAR_COLON);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 254:
#line 1553 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_ALGORITHM);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 255:
#line 1557 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_CERT_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 259:
#line 1601 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EBAD_RDATA);
		p--; {cs = 246; goto _again;}
	}
	break;
	case 263:
#line 1850 "./scanner_body.rl"
	{
		WARN(ZSCANNER_EUNSUPPORTED_TYPE);
		p--; {cs = 246; goto _again;}
	}
	break;
#line 7748 "scanner.c"
		}
	}
	}

	_out: {}
	}

#line 202 "./scanner.rl"

	// Check if scanner state machine is in uncovered state.
	if (cs == zone_scanner_error) {
		ERR(ZSCANNER_UNCOVERED_STATE);
		s->error_counter++;

		// Fill error context data.
		for (s->buffer_length = 0;
		     ((p + s->buffer_length) < pe) &&
		     (s->buffer_length < sizeof(s->buffer) - 1);
		     s->buffer_length++)
		{
			// Only rest of the current line.
			if (*(p + s->buffer_length) == '\n') {
				break;
			}
			s->buffer[s->buffer_length] = *(p + s->buffer_length);
		}

		// Ending string in buffer.
		s->buffer[s->buffer_length++] = 0;

		// Processing error.
		s->process_error(s);

		return -1;
	}

	// Check unclosed multiline record.
	if (is_complete && s->multiline) {
		ERR(ZSCANNER_UNCLOSED_MULTILINE);
		s->error_counter++;
		s->process_error(s);
	}

	// Storing scanner states.
	s->cs  = cs;
	s->top = top;
	memcpy(s->stack, stack, sizeof(stack));

	// Storing r_data pointer.
	s->r_data_tail = rdata_tail - s->r_data;

	// Check if any errors has occured.
	if (s->error_counter > 0) {
		return -1;
	}

	return 0;
}
