/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */
/* ibus - The Input Bus
 * Copyright (C) 2013 Peng Huang <shawn.p.huang@gmail.com>
 * Copyright (C) 2013 Takao Fujiwara <takao.fujiwara1@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301
 * USA
 */

#include <glib.h>
#include <glib/gprintf.h>
#include <locale.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>

#include "ibuserror.h"
#include "ibuskeys.h"
#include "ibuskeysyms.h"
#include "ibustypes.h"
#include "gtkimcontextsimpleseqs.h"

#define HEADER \
"/* -*- mode: C; c-basic-offset: 4; indent-tabs-mode: nil; -*- */\n"          \
"/* vim:set et sts=4: */\n"                                                   \
"/* ibus - The Input Bus\n"                                                   \
" * Copyright (C) 2013 Peng Huang <shawn.p.huang@gmail.com>\n"                \
" * Copyright (C) 2013 Takao Fujiwara <takao.fujiwara1@gmail.com>\n"          \
" *\n"                                                                        \
" * This library is free software; you can redistribute it and/or\n"          \
" * modify it under the terms of the GNU Lesser General Public\n"             \
" * License as published by the Free Software Foundation; either\n"           \
" * version 2.1 of the License, or (at your option) any later version.\n"     \
" *\n"                                                                        \
" * This library is distributed in the hope that it will be useful,\n"        \
" * but WITHOUT ANY WARRANTY; without even the implied warranty of\n"         \
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU\n"      \
" * Lesser General Public License for more details.\n"                        \
" *\n"                                                                        \
" * You should have received a copy of the GNU Lesser General Public\n"       \
" * License along with this library; if not, write to the Free Software\n"    \
" * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301\n" \
" * USA\n"                                                                    \
" */\n"                                                                       \
"\n"                                                                          \
"#ifndef __IBUS_COMPOSETABLE_H_\n"                                            \
"#define __IBUS_COMPOSETABLE_H_\n"                                            \
"\n"                                                                          \
"/* The table is generated by ibus/src/gencomposetable\n"                     \
" *\n"                                                                        \
" * if ibus_compose_seqs_a[N - 1] is an outputed compose character,\n"        \
" * ibus_compose_seqs_a[N * 2 - 1] is also an outputed compose character.\n"  \
" * and ibus_compose_seqs_a[0] to ibus_compose_seqs_a[0 + N - 3] are the\n"   \
" * sequences and call ibus_engine_simple_add_table:\n"                       \
" * ibus_engine_simple_add_table(engine, ibus_compose_seqs_a,\n"              \
" *                              N - 2, G_N_ELEMENTS(ibus_compose_seqs_a) / N)\n" \
" * The compose sequences are allowed within G_MAXUINT16 */\n"                \
"\n"                                                                          \
"typedef struct _IBusComposeTable IBusComposeTable;\n"                        \
"struct _IBusComposeTable\n"                                                  \
"{\n"                                                                         \
"    const guint16 *data;\n"                                                  \
"    gint max_seq_len;\n"                                                     \
"    gint n_seqs;\n"                                                          \
"};\n"                                                                        \
"\n"                                                                          \
"typedef struct _IBusComposeTableLocaleList IBusComposeTableLocaleList;\n"    \
"struct _IBusComposeTableLocaleList\n"                                        \
"{\n"                                                                         \
"    const gchar *locale;\n"                                                  \
"    const IBusComposeTable *table;\n"                                        \
"};\n"                                                                        \
""

#define FOOTER \
"#endif"

#define AM_ET_ARRAY \
"static const guint16 ibus_compose_seqs_am_et[] = {\n"                        \
"  /* Not sure if am_ET compose file is useful. It uses ASCII key\n"          \
"   * at first but not compose keys.\n"                                       \
"   * Comment out am_compose_seqs until get the request. */\n"                \
"  0\n"                                                                       \
"};\n"                                                                        \
"\n"                                                                          \
"static const IBusComposeTable ibus_compose_table_am_et = {\n"                \
"    ibus_compose_seqs_am_et,\n"                                              \
"    0,\n"                                                                    \
"    0\n"                                                                     \
"};\n"                                                                        \
""

#define PRINT_IBUS_COMPOSE_TABLE_LOCALE(locale, len) \
    g_printf (                                                                \
"static const IBusComposeTable ibus_compose_table_" #locale " = {\n"          \
"    ibus_compose_seqs_" #locale ",\n"                                        \
"    %d,\n"                                                                   \
"    G_N_ELEMENTS (ibus_compose_seqs_" #locale ") / (%d + 2)\n"               \
"};\n"                                                                        \
"\n"                                                                          \
"", len, len)

#define PRINT_LOCALE_AND_IBUS_COMPOSE_TABLE(locale) \
    g_printf (                                                                \
"    { \"" #locale "\", &ibus_compose_table_" #locale " },\n")                \

#define IS_DEAD_KEY(k) \
    ((k) >= IBUS_KEY_dead_grave && (k) <= (IBUS_KEY_dead_dasia + 1))

#define GTK_MAX_COMPOSE_LEN 7

int MAX_COMPOSE_LEN = 0;
int N_INDEX_STRIDE = 0;

const gchar *COMPOSE_FILES[] = {
    "/usr/share/X11/locale/am_ET.UTF-8/Compose",
    "/usr/share/X11/locale/el_GR.UTF-8/Compose",
    "/usr/share/X11/locale/fi_FI.UTF-8/Compose",
    "/usr/share/X11/locale/pt_BR.UTF-8/Compose",
    NULL
};

typedef struct _IBusComposeTableCompact IBusComposeTableCompact;
struct _IBusComposeTableCompact
{
    const guint16 *data;
    gint max_seq_len;
    gint n_index_size;
    gint n_index_stride;
};

static const IBusComposeTableCompact ibus_compose_table_compact = {
    gtk_compose_seqs_compact,
    5,
    25,
    6
};

GHashTable *en_keysym_table = NULL;

static gboolean
is_codepoint (const gchar *str)
{
    gboolean retval = (strlen (str) > 1 && *str == 'U');
    int i;

    if (!retval)
        return FALSE;

    for (i = 1; str[i] != '\0'; i++) {
        if (g_ascii_isxdigit (str[i]))
            continue;
        else
            return FALSE;
    }

    return TRUE;
}

static gint
get_codepoint (const gchar *str)
{
    if (g_str_has_prefix (str, "IBUS_KEY_"))
        return GPOINTER_TO_INT (g_hash_table_lookup (en_keysym_table, str));
    if (*str == '0' && *(str + 1) == '\0')
        return 0;
    if (*str == '0' && *(str + 1) == 'x')
        return (gint) g_ascii_strtoll (str, NULL, 16);

    g_assert_not_reached ();
    return 0;
}

static GHashTable *
parse_keysym_file (const gchar *keysym_file)
{
    gchar *contents = NULL;
    gchar **lines = NULL;
    gsize length = 0;
    GError *error = NULL;
    GHashTable *table = NULL;
    int i;

    if (!g_file_get_contents (keysym_file, &contents, &length, &error)) {
        g_error ("Specify $IBUS_KEYSYMS for the path of ibuskeysyms.h: %s",
                 error->message);
        g_error_free (error);
    }

    table = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, NULL);

    lines = g_strsplit (contents, "\n", -1);
    g_free (contents);

    for (i = 0; lines[i] != NULL; i++) {
        const gchar *line = lines[i];
        gchar **words;
        gint length;
        const gchar *key;
        gint64 code;

        if (!g_str_has_prefix (line, "#define"))
            continue;

        words = g_strsplit (line , " ", -1);
        length = g_strv_length (words);

        g_assert (length >= 2);

        key = words[1];

        /* In case of #define __IBUS_KEYSYMS_H__ */
        if (length == 2) {
            g_assert (!g_str_has_prefix (key, "IBUS_KEY"));
            continue;
        }

        code = g_ascii_strtoll (words[2], NULL, 16);
        g_hash_table_insert (table, g_strdup (key), GINT_TO_POINTER (code));
        g_strfreev (words);
    }

    g_strfreev (lines);
    return table;
}

static gboolean
parse_compose_value (GArray *array, const gchar *val, GError **error)
{
    gchar **words = g_strsplit (val, "\"", 3);
    gchar *result;
    gunichar uch;

    g_assert (g_strv_length (words) >= 3);

    uch = g_utf8_get_char (words[1]);

    if (uch == 0) {
         g_set_error (error,
                     IBUS_ERROR,
                     IBUS_ERROR_FAILED,
                     "Invalid value: %s", val);
        g_strfreev (words);
        return FALSE;
    }
    else if (uch == '\\') {
        uch = words[1][1];

        /* The escaped string "\"" is separated with '\\' and '"'. */
        if (uch == '\0' && words[2][0] == '"')
            uch = '"';
        /* The escaped octal */
        else if (uch >= '0' && uch <= '8')
            uch = g_ascii_strtoll(words[1] + 1, 0, 8);
        /* If we need to handle other escape sequences. */
        else if (uch != '\\')
            g_assert_not_reached ();
    }
    else if (g_utf8_get_char (g_utf8_next_char (words[1])) > 0) {
         g_set_error (error,
                     IBUS_ERROR,
                     IBUS_ERROR_FAILED,
                     "GTK+ supports to output one char only: %s", val);
        g_strfreev (words);
        return FALSE;
    }


    result = g_strdup_printf ("0x%04X", uch);
    g_array_append_val (array, result);

    if (uch == '"')
        result = g_strdup (g_strstrip (words[2] + 1));
    else
        result = g_strdup (g_strstrip (words[2]));

    g_array_append_val (array, result);
    g_strfreev (words);

    return TRUE;
}

static void
parse_compose_sequence (GArray *array, const gchar *seq)
{
    gchar **words = g_strsplit (seq, "<", -1);
    gchar *result;
    int i;
    int n = 0;

    g_assert (g_strv_length (words) >= 2);

    for (i = 1; words[i] != NULL; i++) {
        gchar *start = words[i];
        gchar *end = index (words[i], '>');
        gchar *match;

        if (words[i][0] == '\0')
             continue;

        g_assert (start != NULL && end != NULL && end > start);

        match = g_strndup (start, end - start);

        if (is_codepoint (match)) {
            gint64 code = g_ascii_strtoll (match + 1, NULL, 16);
            result = g_strdup_printf ("0x%04X", (unsigned int) code);
        } else
            result = g_strdup_printf ("IBUS_KEY_%s", match);

        g_free (match);
        g_array_append_val (array, result);
        n++;
    }

    g_strfreev (words);
    g_assert (0 < n && n < GTK_MAX_COMPOSE_LEN);

    result = g_strdup ("0");
    g_array_append_val (array, result);
}

static void
clear_char_array (gpointer data)
{
    gchar **array = data;
    g_free (*array);
}

static void
parse_compose_line (GList **compose_table, const gchar *line)
{
    gchar **components = NULL;
    GArray *array;
    GError *error = NULL;

    if (line[0] == '\0' || line[0] == '#')
        return;

    if (g_str_has_prefix (line, "include "))
        return;

    components = g_strsplit (line, ":", 2);

    if (components[1] == NULL) {
        g_warning ("No delimiter ':': %s", line);
        g_strfreev (components);
        return;
    }

    array = g_array_new (TRUE, TRUE, sizeof (gchar *));
    g_array_set_clear_func (array, clear_char_array);

    parse_compose_sequence (array, g_strstrip (components[0]));

    if (!parse_compose_value (array, g_strstrip (components[1]), &error)) {
        g_warning ("%s: %s", error->message, line);
        g_clear_error (&error);
        g_strfreev (components);
        g_array_unref (array);
        return;
    }

    g_strfreev (components);

    *compose_table = g_list_append (*compose_table, array);
}

static GList *
parse_compose_file (const gchar *compose_file)
{
    gchar *contents = NULL;
    gchar **lines = NULL;
    gsize length = 0;
    GError *error = NULL;
    GList *compose_table = NULL;
    int i;

    if (!g_file_get_contents (compose_file, &contents, &length, &error)) {
        g_error ("%s", error->message);
        g_error_free (error);
    }

    lines = g_strsplit (contents, "\n", -1);
    g_free (contents);
    for (i = 0; lines[i] != NULL; i++)
        parse_compose_line (&compose_table, lines[i]);
    g_strfreev (lines);

    return compose_table;
}

static int
compare_seq_index (const void *key, const void *value)
{
    const guint *keysyms = key;
    const guint16 *seq = value;

    if (keysyms[0] < seq[0])
        return -1;
    else if (keysyms[0] > seq[0])
        return 1;
    return 0;
}

static int
compare_seq (const void *key, const void *value)
{
    int i = 0;
    const guint *keysyms = key;
    const guint16 *seq = value;

    while (keysyms[i]) {
        if (keysyms[i] < seq[i])
            return -1;
        else if (keysyms[i] > seq[i])
            return 1;

        i++;
    }

    return 0;
}

/* Implement check_compact_table() in ibus/src/ibusenginesimple.c
 */
static gboolean
check_compact_table (const guint                   *compose_buffer,
                     const IBusComposeTableCompact *table,
                     gint                           n_compose)
{
    gint row_stride;
    guint16 *seq_index;
    guint16 *seq;
    gint i;

    seq_index = bsearch (compose_buffer,
                         table->data,
                         table->n_index_size,
                         sizeof (guint16) *  table->n_index_stride,
                         compare_seq_index);

    if (seq_index == NULL) {
        // g_debug ("compact: no\n");
        return FALSE;
    }

    seq = NULL;
    i = n_compose - 1;

    if (i >= table->max_seq_len) {
        return FALSE;
    }

    row_stride = i + 1;

    if (seq_index[i + 1] <= seq_index[i]) {
        return FALSE;
    }

    seq = bsearch (compose_buffer + 1,
                   table->data + seq_index[i],
                   (seq_index[i + 1] - seq_index[i]) / row_stride,
                   sizeof (guint16) * row_stride,
                   compare_seq);
    // g_debug ("seq = %p", seq);

    if (!seq) {
        // g_debug ("no\n");
        return FALSE;
    }
    else {
        gunichar value = seq[row_stride - 1];
        // g_debug ("U+%04X\n", value);
        if (compose_buffer[n_compose + 1] == value)
            return TRUE;
        else
            return FALSE;
    }
}

static gboolean
check_normalize_nfc (gunichar* combination_buffer, gint n_compose)
{
    gunichar combination_buffer_temp[GTK_MAX_COMPOSE_LEN];
    gchar *combination_utf8_temp = NULL;
    gchar *nfc_temp = NULL;
    gint n_combinations;
    gunichar temp_swap;
    gint i;

    n_combinations = 1;

    for (i = 1; i < n_compose; i++ )
        n_combinations *= i;

    if (combination_buffer[0] >= 0x390 && combination_buffer[0] <= 0x3FF) {
        for (i = 1; i < n_compose; i++ )
            if (combination_buffer[i] == 0x303)
                combination_buffer[i] = 0x342;
    }

    memcpy (combination_buffer_temp, combination_buffer,
            GTK_MAX_COMPOSE_LEN * sizeof (gunichar) );

    for (i = 0; i < n_combinations; i++ ) {
        g_unicode_canonical_ordering (combination_buffer_temp, n_compose);
        combination_utf8_temp = g_ucs4_to_utf8 (combination_buffer_temp,
                                                -1, NULL, NULL, NULL);
        nfc_temp = g_utf8_normalize (combination_utf8_temp, -1,
                                     G_NORMALIZE_NFC);

        if (g_utf8_strlen (nfc_temp, -1) == 1) {
            memcpy (combination_buffer,
                    combination_buffer_temp,
                    GTK_MAX_COMPOSE_LEN * sizeof (gunichar) );

            g_free (combination_utf8_temp);
            g_free (nfc_temp);

            return TRUE;
        }

        g_free (combination_utf8_temp);
        g_free (nfc_temp);

        if (n_compose > 2) {
            temp_swap = combination_buffer_temp[i % (n_compose - 1) + 1];
            combination_buffer_temp[i % (n_compose - 1) + 1] =
             combination_buffer_temp[(i+1) % (n_compose - 1) + 1];
            combination_buffer_temp[(i+1) % (n_compose - 1) + 1] = temp_swap;
        }
        else
            break;
    }

  return FALSE;
}

/* Implement check_algorithmically() in ibus/src/ibusenginesimple.c
 */
static gboolean
check_algorithmically (const guint *compose_buffer,
                       gint         n_compose)
{
    int i = 0;
    gunichar combination_buffer[GTK_MAX_COMPOSE_LEN];
    gchar *combination_utf8, *nfc;

    if (n_compose >= GTK_MAX_COMPOSE_LEN)
        return FALSE;

    for (i = 0; i < n_compose && IS_DEAD_KEY (compose_buffer[i]); i++)
        ;
    if (i == n_compose)
        return FALSE;

    if (i > 0 && i == n_compose - 1) {
        combination_buffer[0] = ibus_keyval_to_unicode (compose_buffer[i]);
        combination_buffer[n_compose] = 0;
        i--;

        while (i >= 0) {
            switch (compose_buffer[i]) {
#define CASE(keysym, unicode) \
            case IBUS_KEY_dead_##keysym: combination_buffer[i+1] = unicode; \
            break

            CASE (grave, 0x0300);
            CASE (acute, 0x0301);
            CASE (circumflex, 0x0302);
            CASE (tilde, 0x0303);
            CASE (macron, 0x0304);
            CASE (breve, 0x0306);
            CASE (abovedot, 0x0307);
            CASE (diaeresis, 0x0308);
            CASE (hook, 0x0309);
            CASE (abovering, 0x030A);
            CASE (doubleacute, 0x030B);
            CASE (caron, 0x030C);
            CASE (abovecomma, 0x0313);
            CASE (abovereversedcomma, 0x0314);
            CASE (horn, 0x031B);
            CASE (belowdot, 0x0323);
            CASE (cedilla, 0x0327);
            CASE (ogonek, 0x0328);
            CASE (iota, 0x0345);
            CASE (voiced_sound, 0x3099);
            CASE (semivoiced_sound, 0x309A);

            /* The following cases are to be removed once xkeyboard-config,
             * xorg are fully updated.
             */
            /* Workaround for typo in 1.4.x xserver-xorg */
            case 0xfe66: combination_buffer[i+1] = 0x314; break;
            /* CASE (dasia, 0x314); */
            /* CASE (perispomeni, 0x342); */
            /* CASE (psili, 0x343); */
#undef CASE
            default:
                combination_buffer[i+1] =
                        ibus_keyval_to_unicode (compose_buffer[i]);
            }
            i--;
        }

        if (check_normalize_nfc (combination_buffer, n_compose)) {
            gunichar value;

            combination_utf8 = g_ucs4_to_utf8 (combination_buffer,
                                               -1, NULL, NULL, NULL);
            nfc = g_utf8_normalize (combination_utf8, -1, G_NORMALIZE_NFC);

            value = g_utf8_get_char (nfc);
            g_free (combination_utf8);
            g_free (nfc);

            if (compose_buffer[n_compose + 1] == value)
                return TRUE;
        }
    }

    return FALSE;
}

static GList *
check_duplicated_compose (GList *compose_table)
{
    GList *table;
    GList *removed_table = NULL;

    for (table = compose_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        static guint keysyms[GTK_MAX_COMPOSE_LEN + 2];
        int i;
        int n_compose = 0;

        for (i = 0; i <= GTK_MAX_COMPOSE_LEN + 2; i++) {
            keysyms[i] = 0;
        }

        for (i = 0; i < array->len; i++) {
            const gchar *data = g_array_index (array, const gchar *, i);
            guint codepoint = (guint) get_codepoint (data);

            keysyms[i] = codepoint;

            if (codepoint == 0) {
                data = g_array_index (array, const gchar *, i + 1);
                codepoint = (guint) get_codepoint (data);
                keysyms[i + 1] = codepoint;
                break;
            }

            n_compose++;
        }

        if (check_compact_table (keysyms,
                                 &ibus_compose_table_compact,
                                 n_compose))
            removed_table = g_list_append (removed_table, array);

        else if (check_algorithmically (keysyms, n_compose))
            removed_table = g_list_append (removed_table, array);
    }

    for (table = removed_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        compose_table = g_list_remove (compose_table, array);
        g_array_unref (array);
    }

    g_list_free (removed_table);

    return compose_table;
}

static GList *
check_uint16 (GList *compose_table)
{
    GList *table;
    GList *removed_table = NULL;

    for (table = compose_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        int i;

        for (i = 0; i < array->len; i++) {
            const gchar *data = g_array_index (array, const gchar *, i);
            guint codepoint = (guint) get_codepoint (data);

            if (codepoint == 0)
                break;

            if (codepoint > 0xffff) {
                removed_table = g_list_append (removed_table, array);
                break;
            }
        }
    }

    for (table = removed_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        compose_table = g_list_remove (compose_table, array);
        g_array_unref (array);
    }

    g_list_free (removed_table);

    return compose_table;
}

static GList *
format_for_gtk (GList *compose_table)
{
    GList *table;
    GList *new_table = NULL;
    int i;
    int j;

    for (table = compose_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;

        for (i = 0; i < array->len; i++) {
            const gchar *data = g_array_index (array, const gchar *, i);
            guint codepoint = (guint) get_codepoint (data);

            if (codepoint == 0) {
                if (MAX_COMPOSE_LEN < i)
                    MAX_COMPOSE_LEN = i;
                break;
            }
        }
    }

    N_INDEX_STRIDE = MAX_COMPOSE_LEN + 2;

    for (table = compose_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        for (i = 0; i < array->len; i++) {
            const gchar *data = g_array_index (array, const gchar *, i);
            guint codepoint = (guint) get_codepoint (data);

            if (codepoint == 0) {
                gchar *value = g_strdup (g_array_index (array, const gchar *,
                                                        i + 1));
                gchar *comment = g_strdup (g_array_index (array, const gchar *,
                                                          i + 2));
                gchar *result;

                g_array_remove_range (array, i, array->len - i);

                for (j = i; j < MAX_COMPOSE_LEN; j++) {
                    result = g_strdup ("0");
                    g_array_append_val (array, result);
                }

                codepoint = (guint) get_codepoint (value);
                g_free (value);

                if (codepoint > 0xffff) {
                    result = g_strdup_printf ("0x%04X", codepoint / 0x10000);
                    g_array_append_val (array, result);
                    result = g_strdup_printf ("0x%04X",
                            codepoint - codepoint / 0x10000 * 0x10000);
                    g_array_append_val (array, result);
                } else {
                    result = g_strdup ("0");
                    g_array_append_val (array, result);
                    result = g_strdup_printf ("0x%04X", codepoint);
                    g_array_append_val (array, result);
                }

                g_array_append_val (array, comment);
                new_table = g_list_append (new_table, array);
                break;
            }
        }
    }

    return new_table;
}

static gint
compare_array (gpointer a, gpointer b)
{
    GArray *array_a = (GArray *) a;
    GArray *array_b = (GArray *) b;
    int i;
    for (i = 0; i < MAX_COMPOSE_LEN; i++) {
        const gchar *data_a = g_array_index (array_a, const gchar *, i);
        const gchar *data_b = g_array_index (array_b, const gchar *, i);
        gint code_a = get_codepoint (data_a);
        gint code_b = get_codepoint (data_b);

        if (code_a != code_b)
            return code_a - code_b;
    }
    return 0;
}

static void
print_compose_table (GList *compose_table)
{
    GList *table;
    int i;
    int TOTAL_SIZE = 0;

    for (table = compose_table; table != NULL; table = table->next) {
        GArray *array = (GArray *) table->data;
        const gchar *data;
        const gchar *upper;
        const gchar *lower;
        const gchar *comment;

        g_assert (array->len >= MAX_COMPOSE_LEN + 2);

        g_printf ("  ");
        for (i = 0; i < MAX_COMPOSE_LEN; i++) {
            data = g_array_index (array, const gchar *, i);

            if (i == MAX_COMPOSE_LEN -1)
                g_printf ("%s,\n", data);
            else
                g_printf ("%s, ", data);
        }
        upper = g_array_index (array, const gchar *, i);
        lower = g_array_index (array, const gchar *, i + 1);
        comment = g_array_index (array, const gchar *, i + 2);

        if (table == g_list_last (compose_table))
            g_printf ("    %s, %s  /* %s */\n", upper, lower, comment);
        else
            g_printf ("    %s, %s, /* %s */\n", upper, lower, comment);

        TOTAL_SIZE += N_INDEX_STRIDE;
    }

    g_printerr ("TOTAL_SIZE: %d\nMAX_COMPOSE_LEN: %d\nN_INDEX_STRIDE: %d\n",
                TOTAL_SIZE, MAX_COMPOSE_LEN, N_INDEX_STRIDE);
}

static void
print_compose_table_locale_list (void) {
    int i;

#define HEADER_IBUS_COMPOSE_TABLE_LOCLE_LIST \
  "static const IBusComposeTableLocaleList ibus_compose_table_locale_list[] = {"

    g_printf ("%s\n", HEADER_IBUS_COMPOSE_TABLE_LOCLE_LIST);

    for (i = 0; COMPOSE_FILES[i] != NULL; i++) {
        const gchar *compose_file = COMPOSE_FILES[i];

        if (g_strstr_len (compose_file, -1, "am_ET") != NULL)
            continue;
        else if (g_strstr_len (compose_file, -1, "el_GR") != NULL)
            PRINT_LOCALE_AND_IBUS_COMPOSE_TABLE (el_gr);
        else if (g_strstr_len (compose_file, -1, "fi_FI") != NULL)
            PRINT_LOCALE_AND_IBUS_COMPOSE_TABLE (fi_fi);
        else if (g_strstr_len (compose_file, -1, "pt_BR") != NULL)
            PRINT_LOCALE_AND_IBUS_COMPOSE_TABLE (pt_br);
        else
            g_assert_not_reached ();
    }

    g_printf ("    { NULL, NULL }\n};\n\n");
};

int
main (int argc, char *argv[])
{
    int i;
    const gchar *en_keysym_file;
    const gchar *compose_file;
    GList *compose_table = NULL;

    setlocale (LC_ALL, "");

    en_keysym_file = g_getenv ("IBUS_KEYSYMS");

    if (en_keysym_file == NULL) {
        en_keysym_file = "ibuskeysyms.h";
    }

    en_keysym_table = parse_keysym_file (en_keysym_file);

    g_printf ("%s\n", HEADER);

    for (i = 0; COMPOSE_FILES[i] != NULL; i++) {
        compose_file = COMPOSE_FILES[i];

        MAX_COMPOSE_LEN = 0;
        N_INDEX_STRIDE = 0;

        if (g_strstr_len (compose_file, -1, "am_ET") != NULL) {
            g_printf ("%s\n", AM_ET_ARRAY);
            continue;
        }
        else if (g_strstr_len (compose_file, -1, "el_GR") != NULL) {
            g_printerr ("Generating ibus_compose_seqs_el_gr ...\n");
            g_printf ("static const guint16 ibus_compose_seqs_el_gr[] = {\n");
        }
        else if (g_strstr_len (compose_file, -1, "fi_FI") != NULL) {
            g_printerr ("Generating ibus_compose_seqs_fi_fi ...\n");
            g_printf ("static const guint16 ibus_compose_seqs_fi_fi[] = {\n");
        }
        else if (g_strstr_len (compose_file, -1, "pt_BR") != NULL) {
            g_printerr ("Generating ibus_compose_seqs_pt_br ...\n");
            g_printf ("static const guint16 ibus_compose_seqs_pt_br[] = {\n");
        }
        else
            g_assert_not_reached ();

        compose_table = parse_compose_file (compose_file);
        compose_table = check_duplicated_compose (compose_table);
        compose_table = check_uint16 (compose_table);
        compose_table = format_for_gtk (compose_table);
        compose_table = g_list_sort (compose_table,
                                     (GCompareFunc) compare_array);

        print_compose_table (compose_table);

        g_printf ("};\n\n");

        if (g_strstr_len (compose_file, -1, "el_GR") != NULL)
            PRINT_IBUS_COMPOSE_TABLE_LOCALE (el_gr, MAX_COMPOSE_LEN);
        else if (g_strstr_len (compose_file, -1, "fi_FI") != NULL)
            PRINT_IBUS_COMPOSE_TABLE_LOCALE (fi_fi, MAX_COMPOSE_LEN);
        else if (g_strstr_len (compose_file, -1, "pt_BR") != NULL)
            PRINT_IBUS_COMPOSE_TABLE_LOCALE (pt_br, MAX_COMPOSE_LEN);
        else
            g_assert_not_reached ();

        g_list_free_full (compose_table, (GDestroyNotify) g_array_unref);
    }

    print_compose_table_locale_list ();

    g_printf ("%s\n", FOOTER);

    g_hash_table_destroy (en_keysym_table);

    return 0;
}
