/*	$Id: dialog.c,v 1.3 2003/04/24 14:13:45 ad Exp $	*/

/*
 * Copyright (c) 2001, 2003 Andrew Doran.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by Andrew Doran.
 * 4. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/param.h>
#include <sys/ioctl.h>
#include <sys/uio.h>
#include <sys/device.h>
#include <sys/mman.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <ctype.h>

#include <dev/i2o/i2o.h>
#include <dev/i2o/iopio.h>

#include "extern.h"

#define	BUFSIZE		32768
#define	NUMIOP		8

static void	cell(int, const char *, ...);
static void	checkreq(const char *req);
static void	getpage(int, int, int, const char *);
static void	i2oerr(struct i2o_reply *);
static void	i2ofail(struct i2o_fault_notify *);
static void	indexpage(void);
static void	indexpage0(const char *, int);
static int	openit(const char *);
static int	parsepath(const char *, int *, int *, int *);
static void	unixerr(const char *, int);

int	main(int, char **);

static const char html_hdr[] =
    "<html>\n<head>\n<title>i2ocfg</title>\n</head>\n<body>\n";

static const char html_ftr[] =
    "\n</body>\n</html>";

static const char html_baduri[] =
    "<h1>i2ocfg: invalid URI</h1>\n<p>\n"
    "URIs should be of the form "
    "/<strong>iop</strong>/<strong>tid</strong>/<strong>page</strong>.<br>\n"
    "<br>"
    "<strong>iop</strong>: IOP number.<br>\n"
    "<strong>tid</strong>: Target ID of device to configure.<br>\n"
    "<strong>page</strong>: Configuration page. The index is page 0.<br>\n"
    "<br>Example: http://myhost/cgi-bin/i2ocfg/0/8/0?var1=val1&var2=val2\n";

static const char html_i2oerr_hdr[] =
    "<h1>i2ocfg: I2O protocol error</h1>\n<p>\n"
    "<p><h5>Reply received:</h5><p>\n"
    "<table cellspacing=0 cellpadding=0><tr>\n";
static const char html_i2oerr_ftr[] =
    "<p><h5>Possbile causes:</h5><p>\n"
    "The target or IOP does not support configuration dialogs.<br>\n"
    "An invalid TID or invalid page was specified.<br>\n"
    "The request was malformed.<br>\n";

static const char html_i2ofail_hdr[] =
    "<h1>i2ocfg: I2O message transport failure</h1>\n<p>\n"
    "<p><h5>Reply received:</h5><p>\n"
    "<table cellspacing=0 cellpadding=0><tr>\n";

static const char	*pathfudge = "";

int
main(int argc, char **argv)
{
	const char *path, *req1;
	int iop, tid, page;

	chdir("/tmp");

	if ((path = getenv("PATH_INFO")) == NULL || *path == '\0') {
		path = "/";
		pathfudge = "/";
	}

	if (path[0] == '/' && path[1] == '\0') {
		indexpage();
		return (EXIT_SUCCESS);
	} else if (parsepath(path, &iop, &tid, &page) != 0) {
		printf("%s\n%s\n%s\n", html_hdr, html_baduri, html_ftr);
		return (EXIT_SUCCESS);
	}

	if ((req1 = getenv("REQUEST_URI")) != NULL) {
		for (; *req1 != '\0'; req1++)
			if (*req1 == '?')
				break;
		if (*req1 != '\0') {
			req1++;
			checkreq(req1);
		} else
			req1 = NULL;
	}

	getpage(iop, tid, page, req1);
	return (EXIT_SUCCESS);
}

/*
 * XXX Kludge to permit exclusive access to the IOP control device for pages
 * that utilise frames or embedded images when using eager browsers.
 */
static int
openit(const char *fn)
{
	int fd, tries;

	for (tries = 50;; tries--) {
		if ((fd = open(fn, O_RDWR)) >= 0)
			break;

		if (tries == 0 || errno != EBUSY)
			unixerr(fn, errno);

		usleep(50000);
	}

	return (fd);
}

static int
parsepath(const char *path, int *iop, int *tid, int *page)
{
	char *p;

	if (path[0] != '/' || !isdigit(path[1]))
		return (-1);

	*iop = (int)strtol(path + 1, &p, 0);
	if (p[0] != '/' || !isdigit(p[1]))
		return (-1);

	*tid = (int)strtol(p + 1, &p, 0);
	if (p[0] != '/' || !isdigit(p[1]))
		return (-1);

	*page = (int)strtol(p + 1, &p, 0);
	if (p[0] != '\0')
		return (-1);

	return (0);
}

static void
checkreq(const char *req)
{
	enum {
		WH_IDENT,
		WH_VAL,
		WH_EQU,
		WH_AMP,
	} where;

	for (where = WH_IDENT; *req != '\0'; req++) {
		if (*req == '&') {
			if (where != WH_VAL)
				goto bad;
			where = WH_AMP;
			continue;
		}

		if (*req == '=') {
			if (where != WH_IDENT)
				goto bad;
			where = WH_EQU;
			continue;
		}

		if (*req == '?')
			goto bad;

		if (where == WH_EQU) {
			if (*req == '&')
				goto bad;
			where = WH_VAL;
			continue;
		}

		if (where == WH_AMP) {
			if (*req == '=')
				goto bad;
			where = WH_IDENT;
			continue;
		}
	}

	if (where == WH_VAL)
		return;

 bad:
	printf("%s\n%s\n%s\n", html_hdr, html_baduri, html_ftr);
	exit(EXIT_SUCCESS);
}

static void
cell(int strong, const char *fmt, ...)
{
	va_list va;

	printf(strong ? "<td><strong>" : "<td>");
	va_start(va, fmt);
	vprintf(fmt, va);
	va_end(va);
	printf(strong ? "</strong></td>" : "</td>");
}

static void
indexpage(void)
{
	const char *srvname;
	char fn[16], map[NUMIOP];
	struct stat sb;
	uid_t euid;
	gid_t egid;
	int iop;

	euid = geteuid();
	egid = getegid();

	for (iop = 0; iop < NUMIOP; iop++) {
		map[iop] = 0;
		sprintf(fn, "/dev/iop%d", iop);
		if (access(fn, F_OK))
			continue;
		if (stat(fn, &sb))
			unixerr(fn, errno);

		if ((sb.st_uid == euid &&
		    (sb.st_mode & (S_IRUSR|S_IWUSR)) == (S_IRUSR|S_IWUSR)) ||
		    (sb.st_gid == egid &&
		    (sb.st_mode & (S_IRGRP|S_IWGRP)) == (S_IRGRP|S_IWGRP)) ||
		    (sb.st_mode & (S_IROTH|S_IWOTH)) == (S_IROTH|S_IWOTH))
			map[iop] = 1;
		else
			unixerr(fn, EPERM);
	}

	if ((srvname = getenv("SERVER_NAME")) == NULL)
		srvname = "<unknown>";

	printf("Content-type: text/html\n\n");
	printf("%s\n", html_hdr);
	printf("<h1>i2ocfg on <i>%s</i></h1>\n", srvname);

	for (iop = 0; iop < NUMIOP; iop++)
		if (map[iop]) {
			sprintf(fn, "/dev/iop%d", iop);
			indexpage0(fn, iop);
		}

	printf("%s\n", html_ftr);
}

static void
indexpage0(const char *fn, int iop)
{
	struct i2o_lct *lct;
	struct iop_tidmap *it;
	struct i2o_lct_entry *ent;
	struct iovec iov;
	int fd, i, tid, classid, nent;

	if ((it = malloc(BUFSIZE)) == NULL)
		unixerr("malloc", errno);
	if ((lct = malloc(BUFSIZE)) == NULL)
		unixerr("malloc", errno);

	fd = openit(fn);

	iov.iov_len = BUFSIZE;
	iov.iov_base = lct;
	if (ioctl(fd, IOPIOCGLCT, &iov))
		unixerr("ioctl(IOPIOCGLCT)", errno);

	iov.iov_len = BUFSIZE;
	iov.iov_base = it;
	if (ioctl(fd, IOPIOCGTIDMAP, &iov))
		unixerr("ioctl(IOPIOCGTIDMAP)", errno);

	close(fd);

	printf("<p><h5>Devices on iop%d:</h5>", iop);
	printf("<p><table border=1 cellspacing=0 cellpadding=2>\n");
	printf("<tr>\n");
	cell(1, "TID");
	cell(1, "Device");
	cell(1, "Class");
	cell(1, "Description");
	printf("</tr>\n");

	ent = lct->entry;
	nent = ((le16toh(lct->tablesize) << 2) -
	    sizeof(struct i2o_lct) + sizeof(struct i2o_lct_entry)) /
	    sizeof(struct i2o_lct_entry);

	for (i = 0; i < nent; i++, it++, ent++) {
		printf("<tr>\n");

		tid = le32toh(ent->localtid) & 4095;
		classid = le32toh(ent->classid) & 4095;

		cell(0, "<a href=\"%s%d/%d/0\">%d</a>",
		    pathfudge, iop, tid, tid);

		if (tid == 0)
			cell(0, "iop%d", iop);
		else if ((it->it_flags & IT_CONFIGURED) != 0)
			cell(0, "%s", it->it_dvname);
		else
			cell(0, "-");

		cell(0, "0x%04x", classid);
		cell(0, "%s", i2o_xlat_class(classid));

		printf("</tr>\n");
	}

	printf("</table>\n");

	free(it);
	free(lct);
}

static void
i2oerr(struct i2o_reply *rf)
{

	printf("Content-type: text/html\n\n");
	printf("%s\n%s\n", html_hdr, html_i2oerr_hdr);

	printf("<td width=150>Status code:</td><td>0x%02x (%s)</td>\n",
	    rf->reqstatus, i2o_xlat_status(rf->reqstatus));
	printf("</tr><tr>\n");
	printf("<td>Detailed status code:</td><td>0x%04x (%s)</td>\n",
	    le16toh(rf->detail), i2o_xlat_detail(rf->detail));
	printf("</tr></table>\n");

	printf("%s\n%s\n", html_i2oerr_ftr, html_ftr);

	exit(EXIT_SUCCESS);
}

static void
i2ofail(struct i2o_fault_notify *rf)
{

	printf("Content-type: text/html\n\n");
	printf("%s\n%s\n", html_hdr, html_i2ofail_hdr);

	printf("<td width=150>Lowest version:</td><td>0x%02x</td>\n",
	    rf->lowestver);
	printf("</tr><tr>\n");

	printf("<td width=150>Highest version:</td><td>0x%02x</td>\n",
	    rf->highestver);
	printf("</tr><tr>\n");

	printf("<td width=150>Severity:</td><td>0x%02x</td>\n",
	    rf->severity);
	printf("</tr><tr>\n");

	printf("<td width=150>Failure code:</td><td>0x%02x</td>\n",
	    rf->failurecode);
	printf("</tr><tr>\n");

	printf("<td width=150>Failing IOP:</td><td>%d</td>\n",
	    le16toh(rf->failingiop));
	printf("</tr><tr>\n");

	printf("<td width=150>Failing host unit:</td><td>%d</td>\n",
	    le16toh(rf->failinghostunit));

	printf("</tr></table>\n");

	printf("%s\n%s\n", html_i2oerr_ftr, html_ftr);

	exit(EXIT_SUCCESS);
}

static void
unixerr(const char *bitch, int _errno)
{

	printf("Content-type: text/html\n\n");
	printf("%s\n", html_hdr);
	printf("<h1>i2ocfg: host OS error</h1>\n<p>\n");

	printf("<p><h5>Returned error:</h5><p>\n");

	printf("i2ocfg: %s: %s\n", bitch, strerror(_errno));

	printf("%s\n", html_ftr);

	exit(EXIT_SUCCESS);
}

void
getpage(int iop, int tid, int page, const char *req1)
{
	struct ioppt pt;
	struct i2o_util_config_dialog mf;
	struct i2o_reply rf;
	static char posted[BUFSIZE];
	char *p, *buf;
	int fd, sz;

	mf.msgflags = I2O_MSGFLAGS(i2o_util_config_dialog);
	mf.msgfunc = I2O_MSGFUNC(tid, I2O_UTIL_CONFIG_DIALOG);
	mf.pageno = page;

	pt.pt_msg = &mf;
	pt.pt_msglen = sizeof(mf);
	pt.pt_reply = &rf;
	pt.pt_replylen = sizeof(rf);
	pt.pt_timo = 10000;

	if (req1 != NULL) {
		strcpy(posted, req1);
		sz = strlen(req1);
		posted[sz++] = '&';
	} else
		sz = 0;
	sz += (int)fread(posted + sz, 1, sizeof(posted) - sz, stdin);
	if (sz > 0) {
		pt.pt_nbufs = 2;
		pt.pt_bufs[1].ptb_data = (void *)&posted;
		pt.pt_bufs[1].ptb_datalen = sz;
		pt.pt_bufs[1].ptb_out = 1;
	} else
		pt.pt_nbufs = 1;

	buf = mmap(NULL, BUFSIZE, PROT_READ | PROT_WRITE, MAP_ANON, -1, 0);
	if (buf == MAP_FAILED)
		unixerr("mmap()", errno);

	pt.pt_bufs[0].ptb_data = buf;
	pt.pt_bufs[0].ptb_datalen = BUFSIZE - 1;
	pt.pt_bufs[0].ptb_out = 0;

	sprintf(buf, "/dev/iop%d", iop);
	fd = openit(buf);

	if (ioctl(fd, IOPIOCPT, &pt) < 0)
		unixerr("ioctl(IOPIOCPT)", errno);

	close(fd);

	if ((le32toh(rf.msgflags) & I2O_MSGFLAGS_FAIL) != 0)
		i2ofail((struct i2o_fault_notify *)&rf);

	if (rf.reqstatus != 0)
		i2oerr(&rf);

	buf[BUFSIZE - 1] = '\0';
	p = memchr(buf, '\0', BUFSIZE);
	fwrite(buf, p - buf, 1, stdout);
	fflush(stdout);
}
