/***********************************************************************

Copyright (c) 2001,2002 Fritz Ganter <ganter@ganter.at>

Website: www.gpsdrive.de

Disclaimer: Please do not use for navigation.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*********************************************************************/
/*
$Log: battery.c,v $
Revision 1.10  2002/11/02 12:38:55  ganter
changed website to www.gpsdrive.de

Revision 1.9  2002/09/01 18:30:06  ganter
fixed segfault on no apm computers

Revision 1.8  2002/08/31 13:10:51  ganter
v1.26 release

Mapblast server works again (they changed the URL).
Bugfix for -a option.
Added -i option to ignore NMEA checksum (for broken GPS receivers).
Added "j" key to switch to next waypoint on route mode.
Added support for festival lite (flite) speech output.

Revision 1.7  2002/07/30 20:49:54  ganter
1.26pre3
added support for festival lite (flite)
changed http request to HTTP1.1 and added correct servername

Revision 1.6  2002/07/01 00:45:00  ganter
added trip info (in settings menu)
ACPI fixes (close battery fd)

Revision 1.5  2002/06/29 21:47:38  ganter
v1.23

Revision 1.4  2002/06/29 00:23:17  ganter
added ACPI support for battery meter

Revision 1.3  2002/06/23 17:09:34  ganter
v1.23pre9
now PDA mode looks good.

Revision 1.2  2002/05/23 09:12:13  ganter
added  Marco Molteni

*/

/*
 *
 * Power-related (battery, APM) functions.
 *
 * This module exports an interface that is operating system independent.
 * The supported OSes are Linux and FreeBSD.
 *
 * Code modularization and support for FreeBSD by Marco Molteni
 * <molter@gufi.org>.
 */


#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <glib.h>
#include <gdk/gdktypes.h>
#include <gdk/gdk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gtk/gtk.h>

#ifdef __FreeBSD__
#include <fcntl.h>
#include <machine/apm_bios.h>
#endif /* __FreeBSD__ */

#include "../config.h"
#include "battery.h"
#include "power.h"


extern gint debug;
extern enum
{ english, german, spanish }
voicelang;
extern gint speech_out_speek (char *text);
extern gint havebattery;
gint batlevel, batlevel_old = 125;	/* battery level, range 0..100 */
gint batloading, batloading_old = FALSE;	/* is the battery charging? */


#ifdef __linux__
/*
 * Return TRUE on success, FALSE on error.
 */
static int
battery_get_values_linux (int *blevel, int *bloading)
{
  FILE *battery;
  gint i, e, ret, v1, v2, v1a, v2a;
  gchar b[100], t[100], t2[100], t3[100];

  ret = TRUE;
  battery = fopen ("/proc/apm", "r");
  if (battery != NULL)
    {
      fscanf (battery, "%s %s %s %x %s %x %d%% %s %s",
	      b, b, b, bloading, b, &i, blevel, b, b);
      /*     1.16 1.2 0x03 0x01      0x00 0x01 99%      -1  ?    */
      fclose (battery);

      /*
       * Bit 7 is set if we have a battery (laptop). If it isn't set,
       * (desktop) then we don't want to display the battery.
       */
      if ((i & 0x80) != 0)
	ret = FALSE;
      return ret;
    }
  else
    {
/* we try if we have acpi */
      v1 = v2 = v1a = v2a = 0;

      battery = fopen ("/proc/acpi/battery/0/info", "r");
      if (battery == NULL)
	battery = fopen ("/proc/acpi/battery/BAT0/info", "r");
      if (battery != NULL)
	do
	  {
	    e = fscanf (battery, "%s %s %s %s %[^\n]", t, t2, b, t3, b);
	    if (((strstr (t, "ast")) != NULL)
		&& ((strstr (t2, "ull")) != NULL))
	      {
		sscanf (t3, "\n%d\n", &v1);
		ret = TRUE;
	      }
	  }
	while (e != EOF);
      if (battery != NULL)
	fclose (battery);

      battery = fopen ("/proc/acpi/battery/1/info", "r");
      if (battery == NULL)
	battery = fopen ("/proc/acpi/battery/BAT1/info", "r");
      if (battery != NULL)
	do
	  {
	    e = fscanf (battery, "%s %s %s %s %[^\n]", t, t2, b, t3, b);
	    if (((strstr (t, "ast")) != NULL)
		&& ((strstr (t2, "ull")) != NULL))
	      {
		sscanf (t3, "\n%d\n", &v1a);
		ret = TRUE;
	      }
	  }
	while (e != EOF);
      if (battery != NULL)
	fclose (battery);
      v1 += v1a;

      battery = fopen ("/proc/acpi/battery/0/status", "r");
      if (battery == NULL)
	battery = fopen ("/proc/acpi/battery/BAT0/state", "r");
      if (battery != NULL)
	do
	  {
	    e = fscanf (battery, "%s %s %s %s %[^\n]", t, t2, t3, b, b);
	    if (((strstr (t, "emaining")) != NULL)
		&& ((strstr (t2, "apacity")) != NULL))
	      {
		sscanf (t3, "\n%d\n", &v2);
		ret = TRUE;
	      }
	  }
	while (e != EOF);
      if (battery != NULL)
	fclose (battery);

      battery = fopen ("/proc/acpi/battery/1/status", "r");
      if (battery == NULL)
	battery = fopen ("/proc/acpi/battery/BAT1/state", "r");
      if (battery != NULL)
	do
	  {
	    e = fscanf (battery, "%s %s %s %s %[^\n]", t, t2, t3, b, b);
	    if (((strstr (t, "emaining")) != NULL)
		&& ((strstr (t2, "apacity")) != NULL))
	      {
		sscanf (t3, "\n%d\n", &v2a);
		ret = TRUE;
	      }
	  }
	while (e != EOF);
      if (battery != NULL)
	fclose (battery);
      v2 += v2a;

/*       g_print("\nv1: %d, v2:%d",v1,v2); */
      if (v2 != 0)
	*blevel = (int) (((double) v2 / v1) * 100.0);


      battery = fopen ("/proc/acpi/ac_adapter/0/status", "r");
      if (battery == NULL)
	battery = fopen ("/proc/acpi/ac_adapter/ADP0/state", "r");
      if (battery == NULL)
	return FALSE;
      do
	{
	  e = fscanf (battery, "%s%[^\n]", t, t2);
	  if ((strstr (t, "Status:")) != NULL)
	    {
	      if ((strstr (t2, "on-line")) != NULL)
		*bloading = TRUE;
	      else
		*bloading = FALSE;
	      ret = TRUE;
	    }
	  if ((strstr (t, "state:")) != NULL)
	    {
	      if ((strstr (t2, "on-line")) != NULL)
		*bloading = TRUE;
	      else
		*bloading = FALSE;
	      ret = TRUE;
	    }
	}
      while (e != EOF);
      if (battery != NULL)
	fclose (battery);

    }
  return ret;
}
#endif /* Linux */


#ifdef __FreeBSD__
/*
 * Return TRUE on success, FALSE on error.
 */
static int
battery_get_values_fbsd (int *blevel, int *bloading)
{
  int fd;
  struct apm_info ai;

  *blevel = -1;
  *bloading = FALSE;

  if ((fd = open ("/dev/apm", O_RDONLY)) == -1)
    {
      fprintf (stderr, "gpsdrive: open(/dev/apm): %s\n", strerror (errno));
      return FALSE;
    }
  if (ioctl (fd, APMIO_GETINFO, &ai) == -1)
    {
      fprintf (stderr, "gpsdrive: ioctl(APMIO_GETINFO): %s\n",
	       strerror (errno));
      close (fd);
      return FALSE;
    }

  /*
   * Battery level. If unknown or error we fail.
   */
  if (ai.ai_batt_life >= 0 && ai.ai_batt_life <= 100)
    {
      *blevel = ai.ai_batt_life;
    }
  else
    {
      if (ai.ai_batt_life == 255)
	{
	  fprintf (stderr, "gpsdrive: battery level is unknown\n");
	}
      else
	{
	  fprintf (stderr, "gpsdrive: battery level is invalid\n");
	}
      close (fd);
      return FALSE;
    }

  /*
   * Is the battery charging? If unknown or error we fail.
   */
  if (ai.ai_acline == 1)
    {				/* on-line */
      *bloading = TRUE;
    }
  else if (ai.ai_acline == 0)
    {				/* off-line */
      *bloading = FALSE;
    }
  else
    {
      if (ai.ai_acline == 255)
	{			/* unknown */
	  fprintf (stderr, "gpsdrive: battery charging status is unknown\n");
	}
      else
	{			/* error */
	  fprintf (stderr, "gpsdrive: battery charging status is invalid\n");
	}
      close (fd);
      return FALSE;
    }
  close (fd);
  return TRUE;
}
#endif /* __FreeBSD__ */


/*
 * Return TRUE on success, FALSE on error.
 */
int
battery_get_values (void)
{
  if (disableapm)
    {
      return FALSE;
    }
#if defined(__linux__)
  return battery_get_values_linux (&batlevel, &batloading);
#elif defined(__FreeBSD__)
  return battery_get_values_fbsd (&batlevel, &batloading);
#else
  /* add support for your favourite OS here */
  return FALSE;
#endif
}


/*
 * display battery meter
 */
void
expose_display_battery ()
{
  gchar bbuf[200];
  GdkGC *kontext;
  GdkDrawable *mydrawable;

  extern GtkWidget *drawing_battery;
  extern GdkColor mygray;
  extern GdkColor black;
  extern GdkColor green;
  extern GdkColor yellow;
  extern GdkColor red;
  extern GdkPixbuf *batimage;


  if (!havebattery)
      return;

  /* XXX What to do if the reading fails? */
  battery_get_values ();

  mydrawable = drawing_battery->window;
  kontext = gdk_gc_new (mydrawable);

  gdk_gc_set_foreground (kontext, &mygray);
  gdk_draw_rectangle (mydrawable, kontext, 1, 0, 0, 25, 50);
  gdk_gc_set_foreground (kontext, &black);
  gdk_draw_rectangle (mydrawable, kontext, 0, 19, 0, 6, 50);

  if (batlevel > 25)
    gdk_gc_set_foreground (kontext, &green);
  else
    {
      if (batlevel > 15)
	gdk_gc_set_foreground (kontext, &yellow);
      else
	gdk_gc_set_foreground (kontext, &red);
    }
  gdk_draw_rectangle (mydrawable, kontext, 1, 20, 50 - batlevel / 2, 5,
		      batlevel / 2);

  if (batloading)
    batimage = gdk_pixbuf_new_from_xpm_data ((const char **) powercord_xpm);
  else
    batimage = gdk_pixbuf_new_from_xpm_data ((const char **) battery_xpm);

  gdk_gc_set_function (kontext, GDK_AND);
  gdk_pixbuf_render_to_drawable_alpha (batimage, mydrawable, 0, 0, 0, 0,
				       17, 50, GDK_PIXBUF_ALPHA_BILEVEL,
				       255, GDK_RGB_DITHER_NONE, 0, 0);
  gdk_gc_set_function (kontext, GDK_COPY);

  gdk_pixbuf_unref (batimage);


  if (((batlevel - 1) / 10 != (batlevel_old - 1) / 10) && (!batloading))
    {
      if (debug)
	g_print ("\nBattery: %d%%\n", batlevel);

      /* This is for Festival, so we cannot use gettext() for i18n */
      switch (voicelang)
	{
	case english:
	  sprintf (bbuf, "Remaining battery: %d%%", batlevel);
	  break;
	case spanish:
	  sprintf (bbuf, "Batera restante: %d%%", batlevel);
	  break;
	case german:
	  sprintf (bbuf, "Batterieladung: %d%%", batlevel);
	}
      speech_out_speek (bbuf);
      batlevel_old = batlevel;
    }
}
