/*Module support for GNU Pascal
  Copyright (C) 1994-2002, Free Software Foundation, Inc.

  Authors: Jukka Virtanen <jtv@hut.fi>
           Peter Gerwinski <peter@gerwinski.de>
           Alexei Volokhov <voh@ispras.ru>
           Jan-Jaap van der Heijden <j.j.vanderheijden@student.utwente.nl>
           Frank Heckenbach <frank@pascal.gnu.de>

  This file is part of GNU Pascal.

  GNU Pascal is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation; either version 2, or (at your
  option) any later version.

  GNU Pascal is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with GNU Pascal; see the file COPYING. If not, write to the
  Free Software Foundation, 59 Temple Place - Suite 330, Boston, MA
  02111-1307, USA. */

#include "gbe.h"

#include "gpc-defs.h"
#include "objects.h"
#include "util.h"
#include "module.h"
#include "types.h"
#include "p/version.h"

#ifdef __EMX__
# include <process.h>
# include <stdlib.h>
# include <errno.h>
#endif

static tree null_tree_node = NULL_TREE;
#include "gpi.h"
#define NUM_SPECIAL_NODES (sizeof (special_nodes) / sizeof (*special_nodes) - 1)

#define GPM_MAGIC "GNU Pascal module version "
#define GPM_HEADER GPM_MAGIC GPC_VERSION_STRING "\n"

/* Code USE_GPI_DEBUG_KEY into the header since GPI files with and without it are
   not compatible. Also add the GCC version (in gpi_version_string) */
#ifdef USE_GPI_DEBUG_KEY
#define GPI_VERSION_PREFIX GPC_VERSION_STRING " D "
#else
#define GPI_VERSION_PREFIX GPC_VERSION_STRING " "
#endif
static char *gpi_version_string = NULL_PTR;

/* HASH_FUNC must avoid negative values. */
#define MAX_HASH_TABLE 8191
#define HASH_FUNC(NODE) (abs ((NODE) - NULL_TREE) % MAX_HASH_TABLE)

typedef HOST_WIDE_INT gpi_int;

static const gpi_int endianness_marker = GPI_ENDIANNESS_MARKER;
static const gpi_int inverse_endianness_marker = GPI_INVERSE_ENDIANNESS_MARKER;

/* Codes used in the chunks of GPI files */
#define GPI_CHUNK(ID, REQUIRED, UNIQUE, NAME) ID
enum { GPI_CHUNKS, NUM_GPI_CHUNKS };
#undef GPI_CHUNK
#define GPI_CHUNK(ID, REQUIRED, UNIQUE, NAME) NAME
static const char *gpi_chunk_names[NUM_GPI_CHUNKS] = { GPI_CHUNKS };

/* A list of all modules contained in this source file. */
module_t module_list = NULL_MODULE;

/* We are currently compiling this module. */
module_t current_module;

/* Files to be linked seen outside of any module. */
static string_list *pending_link_files = NULL_PTR;

/* A list of all exported names in all modules seen so far.
 *
 * TREE_LIST of exported interfaces we know at the current point
 * of compilation: TREE_VALUE is an IDENTIFIER_NODE of an exported
 * interface name TREE_PURPOSE is a TREE_LIST chain of the names
 * (decls?) exported by this interface.
 */
static tree exported_interface_list;

/* A separate list of the names of the interfaces which survives the end
 * of the current module.
 */
static tree internal_exported_interface_list = NULL_TREE;

/* Required module interfaces StandardInput and StandardOutput;
 * StandardError is a GPC extension.
 */
tree standard_interface_input;
tree standard_interface_output;
tree standard_interface_error;

/* Export "all" mark. */
tree export_all;

/* Nonzero means print automake debugging information to stderr. */
int flag_debug_automake;

/* Flag for debugging the GPI mechanism. */
int flag_debug_gpi = 0;

/* Nonzero means to do an AutoMake:
 * 1 = AutoLink (default), 2 = AutoMake, 3 = AutoBuild. */
int flag_automake = 1;

/* `gpc' program to use in AutoMake. */
char *automake_gpc = NULL;

/* Options to pass to child gpc processes in AutoMake. */
char *automake_gpc_options = NULL;

/* Name of a file to store information about targets in. */
char *automake_temp_filename = NULL;

/* Search path for GPI and unit source files. */
char *unit_path = NULL;

/* Search path for object and non-Pascal source files. */
char *object_path = NULL;

/* Holds the name (an LEX_ID node) of the interface
 * currently being loaded from a GPI file, or NULL_TREE if we
 * are not loading a GPI file. */
tree loading_gpi_file = NULL_TREE;

/* A table holding the nodes imported from all GPI files
 * together with their UIDs, so duplicates can be identified. */
static struct interface_table_t
{
  tree interface_name, module_name;
  gpi_int gpi_checksum;
  tree interface_name_node;
  tree initializers;  /* Same as in module_t */
  int count;
  tree *nodes;
  int *hashlist_next;
  int hash_table[MAX_HASH_TABLE];
  struct interface_table_t *next;
} *interface_table = NULL_PTR;

/* @@ There are too many separate hash tables. Not a big problem,
      but putting them together would be a little nicer. -- Frank */
static struct export_hash_t
{
  int count, size;
  tree *nodes;
  int *hashlist_next;
  int hash_table[MAX_HASH_TABLE];
} *export_hash = NULL_PTR;

/* Memory-buffered files */

typedef struct
{
  char *filename;
  int size, curpos;
  unsigned char *buffer;
} MEMFILE;

static MEMFILE *mopen_read PARAMS ((const char *));
static MEMFILE *
mopen_read (name)
     const char *name;
{
  struct stat finfo;
  MEMFILE *s = NULL_PTR;
  FILE *the_file = fopen (name, "rb");
  if (the_file)
    {
      if (fstat (fileno (the_file), &finfo) == 0)
        {
          s = (MEMFILE *) xmalloc (sizeof (MEMFILE));
          s->size = finfo.st_size;
          s->curpos = 0;
          s->buffer = (char *) xmalloc (s->size);
          if (fread (s->buffer, 1, s->size, the_file) == s->size)
            s->filename = save_string (name);
          else
            {
              free (s->buffer);
              free (s);
              s = NULL_PTR;
            }
        }
      fclose (the_file);
    }
  return s;
}

static void mread1 PARAMS ((MEMFILE *, void *, int));
static void
mread1 (F, P, S)
     MEMFILE *F;
     void *P;
     int S;
{
  if (F->curpos + S > F->size)
    fatal ("unexpected end of file in `%s'", F->filename);
  memcpy (P, F->buffer + F->curpos, S);
  F->curpos += S;
}

static void mseek PARAMS ((MEMFILE *, int));
static void
mseek (F, O)
     MEMFILE *F;
     int O;
{
  assert (O >= 0 && O <= F->size);
  F->curpos = O;
}

#define mptr(F, O) ((F)->buffer + (O))
#define meof(F) ((F)->curpos >= (F)->size)
#define mtell(F) ((F)->curpos)
#define mclose(F) (assert (F), free ((F)->buffer), free ((F)->filename), free (F))

/* Declare local utility functions with prototypes. */

static void add_to_export_hash PARAMS ((tree));
static char *quote_arg PARAMS ((char *));
static int execute PARAMS ((char *, char *));
static char *locate_file_1 PARAMS ((char *, char *));
static void append_string_list PARAMS ((string_list **, char *));
static char *get_automake_switches PARAMS ((int));
static struct interface_table_t *get_interface_table PARAMS ((tree, tree, gpi_int));
static gpi_int compute_checksum PARAMS ((unsigned char *, gpi_int));
static int itab_check_gpi_checksum PARAMS ((tree, gpi_int, int));
static char *file_basename PARAMS ((char *));
static int file_is_being_compiled PARAMS ((char *, int));
static char *locate_object_file PARAMS ((char *));
static int module_must_be_recompiled PARAMS ((tree, char *, char *, tree));
static char *locate_interface_source PARAMS ((char *, char *, char *));
static MEMFILE *gpi_open PARAMS ((tree, char *, char *, int, gpi_int *, gpi_int *, gpi_int *));
static tree load_gpi_file PARAMS ((tree, char *, char *));
static void import_node PARAMS ((tree, tree, tree, int));
static module_t find_module PARAMS ((tree, int));
static void module_expand_exported_ranges PARAMS ((tree));

static char *load_string PARAMS ((MEMFILE *));
static void store_length PARAMS ((void *, int));
static void start_chunk PARAMS ((FILE *, unsigned char, gpi_int));
static void store_string_chunk PARAMS ((FILE *, unsigned char, char *));
static void store_string PARAMS ((char *));
static void store_tree PARAMS ((tree, FILE *, tree, int));
static int get_node_id PARAMS ((tree));
static void store_node PARAMS ((tree));
static void store_node_fields PARAMS ((tree, int));

static tree load_tree PARAMS ((MEMFILE *, gpi_int, gpi_int));
static tree load_node PARAMS ((void));
static tree mark_node_loaded PARAMS ((tree, int));
static void itab_store_node PARAMS ((tree, tree, gpi_int, tree));

/* Scan the program/module parameter list for entries of FILE_TYPE.
 * If this is at top level, and they are variables of file type,
 * flag the files as external.
 *
 * Since the order of declarations are relaxed, this is checked
 * before every routine.
 *
 * All nodes already handled are marked.
 */
void
associate_external_objects (external_name_list)
     tree external_name_list;
{
  tree link;

  if (external_name_list && top_level_p ())
    for (link = external_name_list;
         link;
         link = TREE_CHAIN (link))

      {
        tree id = TREE_VALUE (link);

        if (id == identifier_output)
          current_module->output_file_node = global_output_file_node;
        else if (id == identifier_input)
          current_module->input_file_node = global_input_file_node;
        else if (id == identifier_stderr)
          current_module->error_file_node = global_error_file_node;
        else if (! TREE_PURPOSE (link))
          {
            tree name = lookup_name (id);

            if (name
                && TREE_CODE (name) == VAR_DECL
                && TREE_CODE (TREE_TYPE (name)) == FILE_TYPE)
              {
                PASCAL_EXTERNAL_OBJECT (name) = 1;
                TREE_PURPOSE (link) = error_mark_node;
              }
          }
      }
}

/* Check if all the names in program/module param list have
 * been declared.
 *
 * If not, give a warning.
 */
void
check_external_objects (idlist)
     tree idlist;
{
  char *what  = current_module->main_program ? "program" : "module";

  for (; idlist; idlist = TREE_CHAIN (idlist))
    {
      tree id = TREE_VALUE (idlist);

      if (id != identifier_output
          && id != identifier_input
          && id != identifier_stderr)
        {
          tree name = lookup_name (id);

          if (name == NULL_TREE)
            warning ("identifier `%s' in %s heading is undefined",
                     IDENTIFIER_POINTER (id), what);
          else if (TREE_CODE (name) != VAR_DECL
                   || TREE_CODE (TREE_TYPE (name)) != FILE_TYPE)
            warning ("identifier `%s' in %s heading is not a variable of file type",
                     IDENTIFIER_POINTER (id), what);
        }
    }
}

/* Possibly add the `static' qualifier.
 * This makes unexported stuff invisible from outside the module.
 */
tree
maybe_make_static (qualifiers)
     tree qualifiers;
{
  int make_static = 1;

  /* @@@@ Need to verify if the function was declared with a FORWARD directive. */
  if (qualifiers)
    {
      tree scan;
      for (scan = qualifiers; scan; scan = TREE_CHAIN (scan))
        if (TREE_VALUE (scan) == extern_id)
          {
            make_static = 0;
            break;
          }
    }

  if (make_static)
    qualifiers = chainon (qualifiers, build_tree_list (NULL_TREE, static_id));

  return qualifiers;
}

/* Locate a module by its NAME.
 * If CREATE is nonzero, create a new module if an old one is not found.
 */
module_t
find_module (name, create)
     tree name;
     int create;
{
  module_t curr;

  for (curr = module_list; curr; curr = curr->next)
    if (curr->name == name)
      return curr;

  if (create)
    {
#ifdef EGCS97
      curr = (module_t) xmalloc (sizeof (struct module));
#else
      curr = (module_t) obstack_alloc (&permanent_obstack,
                                       sizeof (struct module));
#endif
      /* Initialize */
      memset ((void *) curr, 0, sizeof (struct module));
      curr->name = name;
      curr->next = module_list;
      curr->link_files = pending_link_files;
      pending_link_files = NULL_PTR;
      module_list = curr;
    }
  return curr;
}

/* Allocate and initialize a new structure for a new module|unit|program
 * named ID (an IDENTIFIER_NODE).  MAIN_PROGRAM is nonzero if this module
 * is the main program.
 */
void
initialize_module (id, main_program)
     tree id;
     int main_program;
{
  char *initializer;
  extra_inits_used = 0;
  current_module = find_module (id, 1);
  current_module->main_program = main_program;
  if (current_module->main_program)
    {
      string_list *link_file;
      for (link_file = current_module->link_files; link_file;
           link_file = link_file->next)
        add_to_automake_temp_file (link_file->string);
      initializer = "init_pascal_main_program";
    }
  else
    initializer = concat ("init_", IDENTIFIER_POINTER (current_module->name), NULL_PTR);
  current_module->initializers = build_tree_list (NULL_TREE, get_identifier (initializer));
}

void
set_module_asmname (name)
     tree name;
{
  current_module->asmname = name;
  current_module->initializers = build_tree_list (NULL_TREE, get_identifier (
    concat ("init_", TREE_STRING_POINTER (name), "_", IDENTIFIER_POINTER (current_module->name), NULL_PTR)));
}

tree
get_module_destructor_name (module)
     module_t module;
{
  if (module->main_program)
    return get_identifier ("fini_pascal_main_program");
  else if (module->asmname)
    return get_identifier (concat ("fini_", TREE_STRING_POINTER (module->asmname), "_",
                           IDENTIFIER_POINTER (module->name), NULL_PTR));
  else
    return get_identifier (concat ("fini_", IDENTIFIER_POINTER (module->name), NULL_PTR));
}

/* Destroy the info about all identifiers in MOD which will
   be released with the next poplevel(). */
void
finalize_module (mod)
    struct module *mod;
{
  mod->imports = NULL_TREE;
  mod->exports = NULL_TREE;
  /* mod->pending_decls = NULL_TREE; */
  mod->autoexport = NULL_TREE;
  mod->autoexport_suspended = 0;
  mod->implementation = 0;
  mod->interface = 0;
  mod->output_file_node = NULL_TREE;
  mod->input_file_node = NULL_TREE;
  if (export_hash)
    {
      free (export_hash->nodes);
      free (export_hash->hashlist_next);
      free (export_hash);
      export_hash = NULL_PTR;
    }
  exported_interface_list = NULL_TREE;
  while (interface_table)
    {
      struct interface_table_t *next_interface = interface_table->next;
      free (interface_table->nodes);
      free (interface_table->hashlist_next);
      free (interface_table);
      interface_table = next_interface;
    }
}

#if 0

/* Perform the interface declarations of the current module. */
void
handle_pending_decls ()
{
  if (current_module->pending_decls)
    {
      tree scan;
      for (scan = current_module->pending_decls; scan; scan = TREE_CHAIN (scan))
        {
          if (TREE_PURPOSE (scan) == void_type_node)
            grok_directive (TREE_VALUE (TREE_VALUE (TREE_VALUE (scan))),
                            TREE_PURPOSE (TREE_VALUE (TREE_VALUE (scan))),
                            TREE_PURPOSE (TREE_VALUE (scan)),
                            0);
          else if (TREE_PURPOSE (scan) == NULL_TREE)
            declare_vars (TREE_VALUE (TREE_VALUE (scan)),                /* names */
                          TREE_PURPOSE (TREE_VALUE (scan)),              /* type */
                          TREE_PURPOSE (TREE_CHAIN (TREE_VALUE (scan))), /* init */
                          TREE_VALUE (TREE_CHAIN (TREE_VALUE (scan))),   /* qual */
                          0);
          else
            assert (0);
        }
      current_module->pending_decls = NULL_TREE;
    }
}

#endif

/* Open a new interface NAME for the current module.  EXPORT_LIST
 * contains the names that should be exported by this interface.
 * In a unit, this is called exactly once with EXPORT_LIST having
 * the special value export_all.
 *
 * Later, after the interface lists have been filled, they will be
 * written to GPI files by create_gpi_files() below.
 *
 * NAME is an IDENTIFIER_NODE of the exported interface name,
 * or a TREE_LIST if exporting a range (TREE_PURPOSE..TREE_VALUE).
 * EXPORT_LIST:
 *   TREE_LIST
 *       TREE_PURPOSE: Export renaming (new name) or NULL_TREE.
 *       TREE_VALUE: IDENTIFIER_NODE of the exported name
 */
void
export_interface (name, export_list)
     tree name;
     tree export_list;
{
  tree exported;
  tree exported_names = NULL_TREE;
  tree nscan, new_export_list;

  for (exported = exported_interface_list; exported; exported = TREE_CHAIN (exported))
    if (TREE_VALUE (exported) == name)
      {
        error ("interface `%s' has already been exported",
               IDENTIFIER_POINTER (name));
        return;
      }

  if (export_list == export_all)
    exported_names = build_tree_list (NULL_TREE, NULL_TREE);
  else
    for (nscan = export_list; nscan; nscan = TREE_CHAIN (nscan))
      {
        exported_names = chainon (exported_names,
          build_tree_list (TREE_PURPOSE (nscan), TREE_VALUE (nscan)));
        add_to_export_hash (TREE_VALUE (nscan));
      }

  current_module->exports = chainon (current_module->exports,
                                     build_tree_list (exported_names, name));

  new_export_list = build_tree_list (exported_names, name);
  exported_interface_list = chainon (exported_interface_list, new_export_list);
  internal_exported_interface_list = chainon (internal_exported_interface_list,
                                              build_tree_list (NULL_TREE, name));
  if (export_list == export_all)
    current_module->autoexport = chainon (current_module->autoexport,
                                          build_tree_list (exported_names, new_export_list));
}

/* Import module/unit interfaces specified on the command line
 * via `--uses=...'.
 */
void
do_extra_import ()
{
  if (extra_imports)
    {
      char *p = extra_imports;
      while (*p)
        {
          char *buffer = xmalloc (strlen (p) + 1);
          char *q = buffer;
          tree interface_name, file_name;
          while (*p &&
                 ((*p >= 'A' && *p <= 'Z')
                   || (*p >= 'a' && *p <= 'z')
                   || (*p >= '0' && *p <= '9')
                   || (*p == '_')))
            {
              if (q == buffer)
                *q = toupper (*p);
              else
                *q = tolower (*p);
              p++;
              q++;
            }
          *q = 0;
          interface_name = get_identifier (buffer);
          if (*p == '(')
            {
              q = buffer;
              p++;
              while (*p && *p != ')')
                {
                  *q = *p;
                  p++;
                  q++;
                }
              *q = 0;
              if (*p == ')')
                p++;
              else
                warning ("missing `)' in `--uses' parameter");
              file_name = build_string (q - buffer + 1, buffer);
            }
          else
            file_name = NULL_TREE;

          import_interface (interface_name, NULL_TREE, 0, file_name);
          while (*p
                 && !((*p >= 'A' && *p <= 'Z')
                     || (*p >= 'a' && *p <= 'z')
                     || (*p >= '0' && *p <= '9')
                     || (*p == '_')))
            {
              if (*p != ',')
                warning ("missing `,' in `--uses' parameter");
              p++;
            }
          free (buffer);
        }
    }
}

static void
add_to_export_hash (node)
     tree node;
{
  int n, h = HASH_FUNC (node);
  if (!export_hash)
    {
      export_hash = (struct export_hash_t *) xmalloc (sizeof (*export_hash));
      export_hash->count = 0;
      export_hash->size = 64;
      export_hash->nodes = (tree *) xmalloc (export_hash->size * sizeof (tree));
      export_hash->hashlist_next = (int *) xmalloc (export_hash->size * sizeof (int));
      for (n = 0; n < MAX_HASH_TABLE; n++)
        export_hash->hash_table[n] = -1;
    }
  if (export_hash->count >= export_hash->size)
    {
      export_hash->size *= 2;
      export_hash->nodes = (tree *) xrealloc (export_hash->nodes, export_hash->size * sizeof (tree));
      export_hash->hashlist_next = (int *) xrealloc (export_hash->hashlist_next, export_hash->size * sizeof (int));
    }
  n = export_hash->count++;
  export_hash->nodes[n] = node;
  export_hash->hashlist_next[n] = export_hash->hash_table[h];
  export_hash->hash_table[h] = n;
}

int
name_exported_p (node)
     tree node;
{
  /* Exported ranges are not taken into account here
     because name_exported_p() is never called for them. */
#if 0
  /* @@@@@@@@ Responsible for quadratic behaviour. Use a hash table or something. */
  tree scan;
  for (scan = current_module->exports; scan; scan = TREE_CHAIN (scan))
    {
      tree id_chain;

      for (id_chain = TREE_PURPOSE (scan); id_chain;
           id_chain = TREE_CHAIN (id_chain))
        {
          if (node == TREE_VALUE (id_chain))
            return 1;
        }
    }
  return 0;
#else
  int n;
  if (export_hash)
    for (n = export_hash->hash_table[HASH_FUNC (node)]; n >= 0; n = export_hash->hashlist_next[n])
      {
        assert (n < export_hash->count);
        if (export_hash->nodes[n] == node)
          return 1;
      }
  return 0;
#endif
}

/* Export names of a Borland Pascal Unit.
 * This also handles `export foo = all' clauses
 * and perhaps, one day, PXSC modules.
 */
void
handle_autoexport (name)
     tree name;
{
  tree e;
  if (!loading_gpi_file && !current_module->autoexport_suspended)
    {
      for (e = current_module->autoexport; e; e = TREE_CHAIN (e))
        {
          /* TREE_PURPOSE (e) serves as a pointer to the tail of
             the list stored in TREE_PURPOSE (TREE_VALUE (e)), to
             allow for fast appending at the end.
             The export list may contain duplicates now (see
             store_node_fields(), case TREE_LIST), so don't bother
             checking (which was O(n^2)). Though I don't think there
             were any at this point, anyway. -- Frank */
          tree t = module_export_clause (name, NULL_TREE, 0);
          TREE_CHAIN (TREE_PURPOSE (e)) = t;
          TREE_PURPOSE (e) = t;
        }
      add_to_export_hash (name);
    }
}

/* Suspend or resume autoexport */
void
suspend_autoexport (suspend)
     int suspend;
{
  if (pedantic || flag_what_pascal)
    error ("`public' and `private' directives for interfaces are a GNU Pascal extension");
  if (!current_module->autoexport)
    {
      warning ("`public' and `private' directives for interfaces make only sense");
      warning (" in modules with `export Foo = all;' and in units");
    }
  if (current_module->autoexport_suspended == suspend)
    warning ("redundant `%s' interface directive", suspend ? "private" : "public");
  current_module->autoexport_suspended = suspend;
}

/* The AutoMake facility
 *
 * When invoked with an `--automake' or `--autobuild' option,
 * GPC can do an implicit `make' for preprocessing, compiling,
 * and linking.  A Makefile is not needed; everything is
 * extracted from the Pascal source.
 *
 * Read the `AutoMake' section of the Texinfo documentation
 * for more information about this.
 *
 * This will be replaced by an external utility `gp' in some
 * future release.
 *
 * -- PG
 */

/* Print an error message */
char *
my_strerror (e)
     int e;
{
#ifdef HAVE_STRERROR
  return strerror (e);
#else
  static char buffer[30];
  if (!e)
    return "cannot access";
  if (e > 0 && e < sys_nerr)
    return sys_errlist[e];
  sprintf (buffer, "Unknown error %d", e);
  return buffer;
#endif
}

/* Also in gpc.c :-( */
static char *
quote_arg (str)
     char *str;
{
  int need_quoting = 0;
  char *p = str, *q, *r;
  for (q = p; *q && !need_quoting; q++)
    need_quoting = *q == ' ' || *q == '\\';
  if (need_quoting)
    {
      r = p = xmalloc (2 * strlen (str) + 1);
      for (q = str; *q; q++)
        {
          if (*q == ' ' || *q == '\\')
            *r++ = '\\';
          *r++ = *q;
        }
      *r++ = 0;
    }
  else
    p = str;
  return p;
}

/* JJ 970809: rewritten to utilize standard pexecute() and pwait() calls
 *
 * For the AutoMake facility we need to execute gpc as a child
 * process. execute() acts as an interface to the underlying
 * pexecute() and pwait() functions borrowed from GCC or libiberty.
 *
 * Return 1 on success, 0 otherwise.
 */
static int
execute (what, args)
     char *what;
     char *args;
{
  char *s;
  int pid, wait_status;
  int i;
  char **argv;
  char *errmsg_fmt, *errmsg_arg;
  char *temp_base = choose_temp_base ();

  /* With `--debug-automake', print what we are about to do, and maybe query.  */
  if (flag_debug_automake)
    {
      fprintf (stderr, "GPC automake: %s %s\n", what, args);
#ifdef DEBUG
      fprintf (stderr, "\nGo ahead? (y or n) ");
      fflush (stderr);
      i = getchar ();
      if (i != '\n')
        while (getchar () != '\n') ;
      if (i != 'y' && i != 'Y')
        return 0;
#endif
    }

  /* Count the args (upper estimate) */
  i = 0;
  for (s = args; *s; s++)
    if (*s == ' ')
      i++;
  i++;
  argv = alloca (sizeof (char *) * (i + 3));

  i = 0;
  argv[i++] = what;

  s = args;
  while (1)
    {
      argv[i++] = s;
      for (; *s && *s != ' '; s++)
        if (*s == '\\' && s[1])
          {
            /* remove `\'; s will step over the following character if it's a space! */
            /* `strcpy (s, s + 1);' might not work since strings overlap */
            char *a;
            for (a = s; *a /* sic! */ ; a++)
              *a = a[1];
          }
      if (*s == 0)
        break;
      *s++ = 0;
    }
  argv[i++] = 0;

  pid = pexecute (argv[0], (char * const *) argv, progname,
                  temp_base, &errmsg_fmt, &errmsg_arg,
                  PEXECUTE_FIRST | PEXECUTE_LAST | PEXECUTE_SEARCH);

  if (pid == -1)
    {
      int errno_val = errno;
      fprintf (stderr, "%s: ", progname);
      fprintf (stderr, errmsg_fmt, errmsg_arg);
      fprintf (stderr, ": %s\n", my_strerror (errno_val));
      exit (FATAL_EXIT_CODE);
    }

  pid = pwait (pid, &wait_status, 0);
  if (flag_debug_automake)
    fprintf (stderr, "GPC automake: done\n");
  if (pid == -1)
    {
      fprintf (stderr, "%s: wait: %s\n", progname, my_strerror (errno));
      exit (FATAL_EXIT_CODE);
    }
  else if (WIFSIGNALED (wait_status))
    {
      fprintf (stderr, "%s: subprocess got fatal signal %d\n",
               progname, WTERMSIG (wait_status));
      exit (FATAL_EXIT_CODE);
    }
  else if (WIFEXITED (wait_status))
    {
      if (WEXITSTATUS (wait_status) != 0)
        {
          fprintf (stderr, "%s: %s exited with status %d\n",
                   progname, what, WEXITSTATUS (wait_status));
          exit (WEXITSTATUS (wait_status));
        }
      return 1;
    }
  assert (0);
}

/* Subroutine of locate_file */
static char *
locate_file_1 (filename, p)
     char *filename, *p;
{
  if (p)
    while (*p)
      {
        char *pathname = p, *q = p, *new_filename;
        int l_pathname;
        while (*q && *q != PATH_SEPARATOR)
          q++;
        if (*q)
          p = q + 1;
        else
          p = q;
        if (q > pathname + 1 && IS_DIR_SEPARATOR (q[-1]))
          q--;
        l_pathname = q - pathname;
        if (l_pathname == 0) l_pathname = 1;
        new_filename = xmalloc (l_pathname + 1 + strlen (filename) + 1);
        if (q == pathname)
          new_filename[0] = '.';
        else
          strncpy (new_filename, pathname, l_pathname);
        new_filename[l_pathname] = DIR_SEPARATOR;
        strcpy (new_filename + l_pathname + 1, filename);
        if (access (new_filename, R_OK) != -1)
          return new_filename;
        else
          free (new_filename);
      }
  return NULL;
}

/* Locate the file FILENAME in the relevant directories.
 * Return a newly allocated char * holding the result, or NULL
 * if the file is not accessible.
 */
char *
locate_file (filename, kind)
     char *filename;
     locate_file_t kind;
{
  char *q, *r = NULL;
  if (!filename)
    return NULL;
  for (q = filename; *q && !IS_DIR_SEPARATOR (*q); q++);
  if (!*q)
    switch (kind)
      {
        /* We must search for object files compiled from units etc.
           first in unit_destination_path because, e.g., a intl.o
           (different from the Intl unit's one) exists also in the
           build directory which is in the default unit_path when
           running a freshly build xgpc in the build directory. For
           the same reason use `.' if no destination path given. */
        case LF_OBJECT:
          if (!r) r = locate_file_1 (filename, object_path);
          if (!r) r = locate_file_1 (filename, object_destination_path ? object_destination_path : ".");
          /* FALLTHROUGH */
        case LF_UNIT:
          if (!r) r = locate_file_1 (filename, unit_path);
          if (!r) r = locate_file_1 (filename, unit_destination_path ? unit_destination_path : ".");
          break;
        case LF_COMPILED_OBJECT:
          if (!r) r = locate_file_1 (filename, object_destination_path ? object_destination_path : ".");
          if (!r) r = locate_file_1 (filename, object_path);
          /* FALLTHROUGH */
        case LF_COMPILED_UNIT:
          if (!r) r = locate_file_1 (filename, unit_destination_path ? unit_destination_path : ".");
          if (!r) r = locate_file_1 (filename, unit_path);
          break;
        default:
          assert (0);
      }
  if (!r && access (filename, R_OK) != -1)
    r = save_string (filename);
  return r;
}

/* Append P to the string containing the AutoMake GPC options. */
void
add_automake_gpc_options (p)
     char *p;
{
  if (! automake_gpc_options)
    automake_gpc_options = save_string (p);
  else
    {
      char *q = concat (automake_gpc_options, " ", p, NULL_PTR);
      free (automake_gpc_options);
      automake_gpc_options = q;
    }
}

/* Add a line to the AutoMake temporary file.
 * The contents of that file will be passed to the linker
 * and to recursive calls of the compiler.
 */
void
add_to_automake_temp_file (line)
     char *line;
{
  if (flag_debug_automake)
    fprintf (stderr, "GPC automake: adding to automake temp file: %s\n", line);
  assert (automake_temp_filename);
  if (line)
    {
      FILE *automake_temp_file = fopen (automake_temp_filename, "at");
      if (!automake_temp_file)
        fatal ("cannot append to automake temp file `%s'", automake_temp_filename);
      fprintf (automake_temp_file, "%s\n", line);
      fclose (automake_temp_file);
    }
}

static void
append_string_list (list, str)
     string_list **list;
     char *str;
{
  string_list *p;
  /* Check whether str already is in the list. */
  for (p = *list; p; p = p->next)
    if (strcmp (p->string, str) == 0)
      return;
  p = (string_list *) xmalloc (sizeof (string_list));
  p->string = save_string (str);
  p->next = NULL_PTR;
  while (*list)
    list = &((*list)->next);
  *list = p;
}

/* Remember a FILENAME to be linked.
 * This information will be written into the GPI file.
 * From there, it will be read and written to the AutoMake temp file.
 */
void
add_to_link_file_list (filename)
     char *filename;
{
  if (current_module && current_module->main_program)
    add_to_automake_temp_file (filename);
  else if (current_module)
    append_string_list (&current_module->link_files, filename);
  else
    append_string_list (&pending_link_files, filename);
}

/* Store the name of the executable to be produced in
 * the AutoMake temporary file.
 */
void
store_executable_name ()
{
  if (executable_file_name)
    {
      /* Store the name of the program in the automake temp
       * file, so the executable can be named after it.
       */
      char *name;
      if (*executable_file_name)
        ; /* accept it */
      else if (main_input_filename)
        {
          char *p;
          executable_file_name = save_string (main_input_filename);
          p = executable_file_name + strlen (executable_file_name) - 1;
          while (p > executable_file_name && *p != '.')
            p--;
          if (*p == '.')
            *p = 0;
        }
      else  /* Should not happen */
#ifdef HAVE_EXECUTABLE_SUFFIX
        executable_file_name = "a";
#else
        executable_file_name = "a.out";
#endif

      if (executable_path)
        {
          char *p = executable_file_name + strlen (executable_file_name);
          do
            p--;
          while (p >= executable_file_name && ! IS_DIR_SEPARATOR (*p));
          p++;
          name = concat ("-o ", executable_path, p,
#ifdef HAVE_EXECUTABLE_SUFFIX
                         EXECUTABLE_SUFFIX,
#endif
                         NULL_PTR);
        }
      else
        name = concat ("-o ", executable_file_name,
#ifdef HAVE_EXECUTABLE_SUFFIX
                       EXECUTABLE_SUFFIX,
#endif
                       NULL_PTR);
      add_to_automake_temp_file (name);
      free (name);
    }
  else if (executable_path)
    {
#ifdef HAVE_EXECUTABLE_SUFFIX
      char *name = concat ("-o ", executable_path, "a", EXECUTABLE_SUFFIX, NULL_PTR);
#else
      char *name = concat ("-o ", executable_path, "a.out", NULL_PTR);
#endif
      add_to_automake_temp_file (name);
      free (name);
    }
}

/* Return common flags to the AutoMake command line.
 * PASCAL_SOURCE is nonzero if GPC-specific options should be
 * passed, 0 if only general GCC options are allowed.
 * This function is shared by module_must_be_compiled
 * and compile_module.
 */
static char *
get_automake_switches (pascal_source)
     int pascal_source;
{
  char *cmd_line = "";
  if (pascal_source)
    {
      if (flag_automake == 1)
        cmd_line = "--autolink ";
      else if (flag_automake == 3)
        cmd_line = "--autobuild ";
      else
        cmd_line = "--automake ";
      if (automake_gpc_options)
        cmd_line = concat (cmd_line, automake_gpc_options, " ", NULL_PTR);
      if (automake_temp_filename)
        cmd_line = concat (cmd_line, "--amtmpfile=", quote_arg (automake_temp_filename), " ", NULL_PTR);
    }
  else if (automake_gpc_options)
    {
      /* Filter out Pascal-specific options when compiling non-Pascal
       * source. If the `p' subdirectory was present when `cc1' was
       * compiled, this is not necessary, because then `cc1' will know
       * about those options and ignore them. But we cannot rely on
       * this.
       */
      char *p = automake_gpc_options, *q;
      while (*p)
        {
          int pass_option = 1;
          while (*p == ' ' || *p == '\t')
            p++;
          /* Find the end of this option. */
          q = p;
          while (*q && *q != ' ' && *q != '\t')
            {
              if (*q == '\\' && q[1])
                q++;
              q++;
            }

          if (*p == '-' && (p[1] == '-' || p[1] == 'f'))
            {
              /* This is a long option. Filter out GPC specifica. */
              int j;
              char *option;
              p[1] = 'f';
              j = 0;
              while ((option = gpc_options[j].name) != NULL_PTR && strncmp (p, option, strlen (option)))
                j++;
              if (option)
                pass_option = 0;
            }
          if (pass_option)
            {
              /* This is a short or a general long GCC option. Pass it. */
              char tmp = *q;
              *q = 0;
              cmd_line = concat (cmd_line, p, " ", NULL_PTR);
              *q = tmp;
            }
          p = q;
        }
    }
  return cmd_line;
}

static char*
file_basename (filename)
     char *filename;
{
  char *f = filename;
  while (*f) f++;
  while (f > filename && !IS_DIR_SEPARATOR (f[-1])) f--;
  return f;
}

static int
file_is_being_compiled (filename, times)
     char *filename;
     int times;
{
  FILE *automake_temp_file;
  char *fn = file_basename (filename);
  /* File is being compiled --> avoid a cycle.
     This also avoids compiling one source file twice with `--autobuild'. */
  assert (automake_temp_filename);
  automake_temp_file = fopen (automake_temp_filename, "rt");
  if (!automake_temp_file)
    fatal ("cannot read automake temp file `%s'", automake_temp_filename);
  while (! feof (automake_temp_file))
    {
      char s[2048];
      if (fgets (s, 2048, automake_temp_file))
        {
          int l = strlen (s) - 1;
          if (s[l] == '\n')
            s[l] = 0;
          if (strncmp (s, "#compiling: ", 12) == 0 && strcmp (file_basename (s + 12), fn) == 0)
            times--;
        }
    }
  fclose (automake_temp_file);
  return times <= 0;
}

static char*
locate_object_file (source)
     char *source;
{
  char *p = file_basename (source), *q = source + strlen (source) - 1;
  while (q > p && *q != '.') q--;
  if (q > p && q[1])
    {
      char *object_filename = save_string (p);
      q = object_filename + (q - p);
      q[1] = 'o';
      q[2] = 0;
      p = locate_file (object_filename, LF_COMPILED_UNIT);
      free (object_filename);
      return p;
    }
  else
    return NULL;
}

/* Check whether a module must be recompiled (for AutoMake). */
static int
module_must_be_recompiled (interface_name, gpi_filename, source_name, import_list)
     tree interface_name;
     char *gpi_filename;
     char *source_name;
     tree import_list;
{
  char *gpc_args, *dep_filename;
  tree interface;
  FILE *dep_file;
  struct stat gpi_status, source_status;
  char *object_filename, dep_line[2048];

  /* Module is in this file --> cannot recompile. */
  for (interface = internal_exported_interface_list;
       interface; interface = TREE_CHAIN (interface))
    if (TREE_VALUE (interface) == interface_name)
      return 0;

  /* `--autobuild' given --> recompile. */
  if (flag_automake > 2)
    return 1;

  stat (gpi_filename, &gpi_status);
  stat (source_name, &source_status);

  /* Check the name of the object file. If it doesn't exist, we must
     compile. If it is older than the GPI file, update the time. */
  object_filename = locate_object_file (source_name);
  if (object_filename)
    {
      struct stat object_status;
      stat (object_filename, &object_status);
      if (gpi_status.st_mtime > object_status.st_mtime)
        gpi_status.st_mtime = object_status.st_mtime;
      free (object_filename);
    }
  else if (flag_automake > 1)  /* not with `--autolink' */
    return 1;

  /* GPI or object file older than source --> recompile. */
  if (gpi_status.st_mtime < source_status.st_mtime && flag_automake > 1)  /* not with `--autolink' */
    return 1;

  /* GPI file older than imported GPI files --> recompile. */
  for (interface = import_list; interface; interface = TREE_CHAIN (interface))
    {
      char *other_gpi_name = (char *) xmalloc (strlen (IDENTIFIER_POINTER (TREE_VALUE (interface))) + 5), *p, *q = other_gpi_name;
      p = IDENTIFIER_POINTER (TREE_VALUE (interface));
      if (p)
        while (*p)
          *q++ = tolower (*p++);
      strcpy (q, ".gpi");
      p = locate_file (other_gpi_name, LF_COMPILED_UNIT);
      free (other_gpi_name);
      /* One needed GPI does not exist. Let a recursive AutoMake generate it.
         @@@ Do we need to go deeper?
             I hope not, with the new checksum mechanism. -- Frank */
      if (!p && flag_automake > 1)  /* not with `--autolink' */
        return 1;
      if (p)
        {
          struct stat other_gpi_status;
          stat (p, &other_gpi_status);
          free (p);
          if (gpi_status.st_mtime < other_gpi_status.st_mtime && flag_automake > 1)  /* not with `--autolink' */
            return 1;
        }
    }

  if (flag_automake <= 1)  /* `--autolink' */
    return 0;

  /* Run the preprocessor with `-M' for creating a dependency file.
   * @@@ Should do this when compiling the module and store the
   * information in the GPI file. That would be faster. */
  dep_filename = choose_temp_base ();

  /* Touch dep_filename, so it won't be chosen as the name for another temp file */
  close (open (dep_filename, O_WRONLY | O_CREAT, 0666));

  /* Run the AutoMake command. */
  gpc_args = concat (get_automake_switches (1),
                     "-M -o ", quote_arg (dep_filename),
                     " ", quote_arg (source_name), NULL_PTR);
  execute (automake_gpc ? automake_gpc : "gpc", gpc_args);
  free (gpc_args);

  /* Parse the `.d' file and check if the object file or the GPI file
   * is older than one of the source files.  If yes, recompile.
   */
  dep_file = fopen (dep_filename, "rt");
  if (!dep_file)
    fatal ("cannot read dependency file `%s'", dep_filename);

  while (! feof (dep_file))
    {
      if (fscanf (dep_file, "%2047s", dep_line) > 0
          && strcmp (dep_line, "\\") != 0
          && dep_line [strlen (dep_line) - 1] != ':')
        {
          /* This is the name of a source file. If it is younger than the GPI file, recompile. */
          struct stat tmp_status;
          stat (dep_line, &tmp_status);
          if (gpi_status.st_mtime < tmp_status.st_mtime)
            {
              fclose (dep_file);
              unlink (dep_filename);
              free (dep_filename);
              return 1;
            }
        }
    }
  fclose (dep_file);
  unlink (dep_filename);
  free (dep_filename);
  return 0;  /* All possible checks passed --> no recompilation. */
}

/* Compile a module during an AutoMake.
 * Return 0 on success, nonzero otherwise.
 */
int
compile_module (filename, destination_path)
     char *filename, *destination_path;
{
  char *gpc_name, *gpc_args;
  int result, pascal_source;
  char *plain_filename, *object_filename, *p;

  filename = locate_file (filename, LF_UNIT);
  /* @@@@@@ Kludge. Allow two simultaneous compilations of the same module
            (fjf40[57].pas), but not infinitely many (sven18[a-c].pas).
            The number 2 is arbitrary. */
  if (!filename || file_is_being_compiled (filename, 2))
    {
      if (filename)
        free (filename);
      return -1;
    }

  /* Build the AutoMake command line. */
  if (automake_gpc)
    gpc_name = automake_gpc;
  else
    gpc_name = "gpc";

  /* Pass automake GPC options only if Pascal source. */
  p = filename + strlen (filename) - 1;
  while (p > filename && *p != '.')
    p--;
  pascal_source = strcmp (p, ".pas") == 0 || strcmp (p, ".p") == 0
                  || strcmp (p, ".pp") == 0 || strcmp (p, ".dpr") == 0;

  gpc_args = concat (get_automake_switches (pascal_source), "-c ", NULL_PTR);

  /* Create the object file in a special directory if one was specified. */
  if (destination_path)
    {
      /* p still points to the file name extension. */
      char *q = p;
      while (q >= filename && ! IS_DIR_SEPARATOR (*q))
        q--;
      q++;
      if (p > q)
        {
          char tmp = *p;
          *p = 0;
          gpc_args = concat (gpc_args, "-o ", quote_arg (destination_path), quote_arg (q), ".o ", NULL_PTR);
          *p = tmp;
          if (pascal_source)
            gpc_args = concat (gpc_args, "--gpi-destination-path=", quote_arg (destination_path), " ", NULL_PTR);
        }
    }

  gpc_args = concat (gpc_args, quote_arg (filename), NULL_PTR);

#ifndef EGCS
  {
    /* @@ KLUDGE: gcc-2.8.1 can't compile modern C++ files because of
       changes in the C++ standard. So invoke `gcc' (hoping it's a newer
       version) rather than `gpc'. Since I'm probably the only one who
       uses a GPC based on gcc-2.8.1, it's ok if it works on my system. ;-)
       -- Frank. */
    char *ext = strrchr (filename, '.');
    if (!strcmp (gpc_name, "gpc")
        && (!strcmp (ext, ".cc")
            || !strcmp (ext, ".cpp")
            || !strcmp (ext, ".C")
            || !strcmp (ext, ".c++")
            || !strcmp (ext, ".cxx")))
      gpc_name = "gcc";
  }
#endif

  /* Document what we are doing in the automake temp file. */
  p = concat ("#compiling: ", filename, NULL_PTR);
  add_to_automake_temp_file (p);
  free (p);

  if (execute (gpc_name, gpc_args) == 1)
    result = 0;
  else
    result = -1;
  free (gpc_args);

  /* Tell the linker about the object file. */
  plain_filename = filename + strlen (filename) - 1;
  while (plain_filename > filename && ! IS_DIR_SEPARATOR (*plain_filename))
    plain_filename--;
  if (IS_DIR_SEPARATOR (*plain_filename))
    plain_filename++;
  object_filename = xmalloc (strlen (plain_filename)
                             + strlen (OBJECT_SUFFIX) + 1);
  strcpy (object_filename, plain_filename);
  p = object_filename + strlen (plain_filename) - 1;
  while (p > object_filename && *p != '.')
    p--;
  *p = 0;
  strcat (object_filename, OBJECT_SUFFIX);
  p = locate_file (object_filename, LF_COMPILED_UNIT);
  if (p)
    {
      add_to_link_file_list (p);
      free (p);
    }
  else
    error ("file `%s' not found", object_filename);

  free (filename);
  free (object_filename);

  return result;
}

/* GPI file handling -- storing and retrieving tree nodes in an
 * implementation-dependent (but not *too* implementation-dependent ;-)
 * "GNU Pascal Interface" binary file.
 *
 * For an introduction, read the `GPI' section of the Texinfo documentation.
 *
 * @@@ This should be generalized to store arbitrary tree structures
 *     (which should go into tree.c).
 *
 * This would have the following advantages:
 *
 *   - It would be cleaner (see `@@@@@@' in gpc-typeck.c).
 *
 *   - GPI would become suitable for other GNU languages, too
 *     ("C precompiled headers").
 *
 *   - The GPI format would not need to be changed so often as it is
 *     necessary now.
 *
 *   - load_node() below would no more need to re-do everything the
 *     parser already did. This would simplify and speed up load_node()
 *     simultaneously.
 *
 * - PG
 */

/* In principle, wb and rb should be arguments to the various
   load/store routines. But they would have to be passed around
   unchanged to each recursive invocation which would be quite
   inefficient. But because it would be unchanged, and there are no
   other concurrent invocations of these routines, it's ok to make
   them static. */

static struct
{
  int size, count;
  int autoexport_flag;  /* If non-null, store_node_fields() puts here the
                           enum values it finds to export them later.
                           If null, this is no autoexport module, and
                           we don't want to autoexport enum values. */
  tree storing_gpi_file;
  tree append_additional_globals_to_export;
  int main_list_flag;
  tree *nodes;
  gpi_int *offsets;
  char *main_exported;
  int *hashlist_next;
  int hash_table[MAX_HASH_TABLE];
  int outbufsize, outbufcount;
  unsigned char *outbuf;
} wb;

static struct
{
  gpi_int *offsets;
  tree *nodes;
  MEMFILE *infile;
} rb;

#define STORE_LENGTH_F(F, X, L) (assert (fwrite ((X), (L), 1, (F)) == 1))
#define STORE_ANY_F(F, X) STORE_LENGTH_F ((F), &(X), sizeof (X))
#define STORE_ANY(X) store_length (&(X), sizeof (X))

static void
store_length (buf, size)
     void *buf;
     int size;
{
  if (wb.outbufcount + size > wb.outbufsize)
    {
      while (wb.outbufcount + size > wb.outbufsize)
        wb.outbufsize *= 2;
      wb.outbuf = (char *) xrealloc (wb.outbuf, wb.outbufsize);
    }
  memcpy (wb.outbuf + wb.outbufcount, buf, size);
  wb.outbufcount += size;
}

#define LOAD_LENGTH_F mread1
#define LOAD_ANY_F(F, X) LOAD_LENGTH_F ((F), &(X), sizeof (X))
#define LOAD_LENGTH(X, L) LOAD_LENGTH_F (rb.infile, (X), (L))
#define LOAD_ANY(X) LOAD_LENGTH (&(X), sizeof (X))

static void
start_chunk (s, code, size)
     FILE *s;
     unsigned char code;
     gpi_int size;
{
  STORE_ANY_F (s, code);
  STORE_ANY_F (s, size);
}

static void
store_string_chunk (s, code, str)
     FILE *s;
     unsigned char code;
     char *str;
{
  gpi_int l = strlen (str);
  start_chunk (s, code, l);
  if (l > 0)
    STORE_LENGTH_F (s, str, l);
  if (flag_debug_gpi)
    fprintf (stderr, "GPI storing %s: %s\n", gpi_chunk_names[code], str);
}

static void
store_string (str)
     char *str;
{
  gpi_int l = strlen (str);
  STORE_ANY (l);
  if (l > 0)
    store_length (str, l);
}

static void
store_tree (name, s, main_node, autoexport_flag)
     tree name;
     FILE *s;
     tree main_node;
     int autoexport_flag;
{
  int n, offset_size;
  gpi_int main_node_id, checksum;
  tree main_node_end, t;

  /* How many nodes will be stored (approx.; will be extended when necessary) */
  wb.size = list_length (main_node) + 1024 + NUM_SPECIAL_NODES;
  wb.nodes = (tree *) xmalloc (wb.size * sizeof (tree));
  wb.offsets = (gpi_int *) xmalloc (wb.size * sizeof (gpi_int));
  wb.main_exported = (char *) xmalloc (wb.size * sizeof (char));
  wb.hashlist_next = (int *) xmalloc (wb.size * sizeof (int));
  wb.outbuf = (char *) xmalloc ((wb.outbufsize = 1024));
  wb.outbufcount = 0;
  wb.main_list_flag = 0;
  wb.storing_gpi_file = name;
  for (n = 0; n < MAX_HASH_TABLE; n++)
    wb.hash_table[n] = -1;

  assert (LAST_AND_UNUSED_TREE_CODE < 255);  /* If this ever fails, the type for storing
                                                tree codes in GPI files must be enlarged. */

  /* Put the special nodes in the hash table.
     The reason for the backward loop is only that the "more common"
     nodes (e.g., integer_type_node vs. ptrdiff_type_node if they
     are the same) are processed last and therefore inserted at the
     front of the hash list and thus actually used in GPIs. */
  for (wb.count = NUM_SPECIAL_NODES - 1; wb.count >= 0; wb.count--)
    {
      tree special_node = *(special_nodes[wb.count]);
      int h = HASH_FUNC (special_node);
      wb.nodes[wb.count] = special_node;
      wb.offsets[wb.count] = -1;
      wb.main_exported[wb.count] = 0;
      wb.hashlist_next[wb.count] = wb.hash_table[h];
      wb.hash_table[h] = wb.count;
    }
  wb.count = n = NUM_SPECIAL_NODES;

  /* Tell store_node_fields() if and where to store additional identifiers to export */
  wb.autoexport_flag = autoexport_flag;
  for (main_node_end = main_node; TREE_CHAIN (main_node_end); main_node_end = TREE_CHAIN (main_node_end)) ;
  wb.append_additional_globals_to_export = main_node_end;

  /* Put the elements of the main list into the nodes buffer */
  t = main_node;
  assert (TREE_CODE (t) == TREE_LIST);
  while (t)
    {
      get_node_id (TREE_PURPOSE (t)),
      get_node_id (TREE_VALUE (t));

      /* Successively store all nodes from the tree buffer.
         Note that wb.count will grow while the loop runs.
         Furthermore, this may put more nodes into the main
         list, so this loop must be within the other loop. */
      #define FLUSH_NODES \
      { \
        wb.offsets[n] = wb.outbufcount; \
        store_node_fields (wb.nodes[n], n); \
        n++; \
      }
      while (wb.count > n)
        FLUSH_NODES;

      t = TREE_CHAIN (t);
    }

  /* Store main node */
  main_node_id = get_node_id (main_node);
  wb.main_list_flag = 1;
  FLUSH_NODES;
  assert (wb.count == n && n == main_node_id + 1);

  /* Remove appended values from main list again (probably not necessary) */
  if (main_node_end)
    {
      if (flag_debug_gpi && TREE_CHAIN (main_node_end))
        {
          fprintf (stderr, "GPI summary: Additionally exported identifiers (enum values and VMT variables):\n");
          for (t = TREE_CHAIN (main_node_end); t; t = TREE_CHAIN (t))
            if (TREE_CODE (TREE_VALUE (t)) == IDENTIFIER_NODE)
              fprintf (stderr, "  %s\n", IDENTIFIER_POINTER (TREE_VALUE (t)));
            else
              error ("internal error: additionally exported object is not an identifier");
        }
      TREE_CHAIN (main_node_end) = NULL_TREE;
    }

  /* Just some statistics for those who want to check the quality of the hash function */
  if (flag_debug_gpi)
    {
      #define MAX_COLLISION_COUNT 1024
      int collision_count[MAX_COLLISION_COUNT], h, i;
      for (i = 0; i < MAX_COLLISION_COUNT; i++)
        collision_count[i] = 0;
      for (h = 0; h < MAX_HASH_TABLE; h++)
        {
          i = 0;
          for (n = wb.hash_table[h]; n >= 0; n = wb.hashlist_next[n])
            i++;
          if (i >= MAX_COLLISION_COUNT)
            i = MAX_COLLISION_COUNT - 1;
          collision_count[i]++;
        }
      fprintf (stderr, "GPI summary: Hash distribution:\nCollisions  Times\n");
      for (i = 1; i < MAX_COLLISION_COUNT; i++)
        if (collision_count[i] != 0)
          fprintf (stderr, "%10i%c %i\n", i - 1, i == MAX_COLLISION_COUNT - 1 ? '+' : ' ',
                   collision_count[i]);
    }

  /* Write chunk to file */
  checksum = compute_checksum (wb.outbuf, wb.outbufcount);
  itab_check_gpi_checksum (name, checksum, 1);
  start_chunk (s, GPI_CHUNK_NODES, wb.outbufcount + sizeof (checksum));
  STORE_LENGTH_F (s, wb.outbuf, wb.outbufcount);
  STORE_ANY_F (s, checksum);

  /* Write offset table, excluding the special nodes */
  offset_size = (wb.count - NUM_SPECIAL_NODES) * sizeof (gpi_int);
  start_chunk (s, GPI_CHUNK_OFFSETS, offset_size + sizeof (main_node_id));
  STORE_LENGTH_F (s, wb.offsets + NUM_SPECIAL_NODES, offset_size);
  STORE_ANY_F (s, main_node_id);

  wb.storing_gpi_file = NULL_TREE;
  free (wb.nodes);
  free (wb.offsets);
  free (wb.main_exported);
  free (wb.hashlist_next);
  free (wb.outbuf);
}

static int
get_node_id (node)
     tree node;
{
  int n, h = HASH_FUNC (node);
  for (n = wb.hash_table[h]; n >= 0; n = wb.hashlist_next[n])
    {
      assert (n < wb.count);
      if (wb.nodes[n] == node)
        break;
    }
  if (n < 0)
    {
      /* New node. Allocate new number and maintain the hash table */
      if (wb.count >= wb.size)
        {
          wb.size *= 2;
          wb.nodes = (tree *) xrealloc (wb.nodes, wb.size * sizeof (tree));
          wb.offsets = (gpi_int *) xrealloc (wb.offsets, wb.size * sizeof (gpi_int));
          wb.main_exported = (char *) xrealloc (wb.main_exported, wb.size * sizeof (char));
          wb.hashlist_next = (int *) xrealloc (wb.hashlist_next, wb.size * sizeof (int));
        }
      n = wb.count++;
      wb.nodes[n] = node;
      wb.main_exported[n] = 0;
      wb.hashlist_next[n] = wb.hash_table[h];
      wb.hash_table[h] = n;
    }
  return n;
}

/* Put node into hash table if not there already and write an index in the GPI file */
static void
store_node (node)
     tree node;
{
  gpi_int n = get_node_id (node);
  STORE_ANY (n);
}

static tree
load_tree (s, start_of_nodes, size_of_offsets)
     MEMFILE *s;
     gpi_int start_of_nodes, size_of_offsets;
{
  tree result;
  int n, nodes_count;
  size_of_offsets -= sizeof (gpi_int);  /* main node id */
  nodes_count = (size_of_offsets / sizeof (gpi_int)) + NUM_SPECIAL_NODES;

  /* Read offset table */
  rb.infile = s;
  rb.offsets = (gpi_int *) xmalloc (nodes_count * sizeof (gpi_int));
  LOAD_LENGTH (rb.offsets + NUM_SPECIAL_NODES, size_of_offsets);
  for (n = NUM_SPECIAL_NODES; n < nodes_count; n++)
    rb.offsets[n] += start_of_nodes;

  /* Predefine the standard nodes */
  rb.nodes = (tree *) xmalloc (nodes_count * sizeof (tree));
  memset (rb.nodes, 0, nodes_count * sizeof (tree));
  for (n = 0; n < NUM_SPECIAL_NODES; n++)
    {
      rb.offsets[n] = -1;
      rb.nodes[n] = *(special_nodes[n]);
      if (n != 0 && !rb.nodes[n])
        fatal ("special node #%i is NULL", n);
    }

  /* Load and return the tree */
#ifndef EGCS97
  push_obstacks_nochange ();
  end_temporary_allocation ();
#endif /* not EGCS97 */
  result = load_node ();
#ifndef EGCS97
  pop_obstacks ();
#endif /* not EGCS97 */

  free (rb.offsets);
  free (rb.nodes);
  return result;
}

/* Mark node as loaded */
static tree
mark_node_loaded (node, n)
     tree node;
     int n;
{
  assert (node);
  assert (n);
  if (rb.nodes[n])
    /* @@ This happens, AFAICS, with recursive tree structures. The effect
          seems to be that building of the same node might be attempted
          several times until one succeeds, and the other waiting ones are
          discarded!? Can this be avoided? -- Frank */
    assert (TREE_CODE (rb.nodes[n]) == TREE_CODE (node));
  else
    rb.nodes[n] = node;
  return rb.nodes[n];
}

static void
itab_store_node (interface_name, module_name, uid, t)
  tree interface_name, module_name;
  gpi_int uid;
  tree t;
{
  struct interface_table_t *itab = get_interface_table (interface_name, module_name, uid);
  /* When loading, itab->nodes[original_uid] may have been set in the
     meantime, cf. mark_node_loaded(). When storing, the node may have
     been loaded already indirectly via another interface in the case
     of cyclic dependencides. */
  if (!itab->nodes[uid])
    {
      int hash = HASH_FUNC (t);
      itab->nodes[uid] = t;
      itab->hashlist_next[uid] = itab->hash_table[hash];
      itab->hash_table[hash] = uid;
    }
}

/* Store/load a tree node's type main variant if not equal to itself */
#define store_main_variant(t) \
  store_node (TYPE_MAIN_VARIANT (t) == t ? NULL_TREE : TYPE_MAIN_VARIANT (t));

static void load_main_variant PARAMS ((tree));
static void load_main_variant (t)
    tree t;
{
  tree tmp = load_node ();
  if (!tmp)
    TYPE_MAIN_VARIANT (t) = t;
  else
    {
      TYPE_MAIN_VARIANT (t) = tmp;
      TYPE_NEXT_VARIANT (t) = TYPE_NEXT_VARIANT (tmp);
      TYPE_NEXT_VARIANT (tmp) = t;
    }
}

/* @@@@ Store/load a tree node's flags */
static void store_flags PARAMS ((tree));
static void
store_flags (t)
     tree t;
{
  /* @@ If we don't set TREE_ASM_WRITTEN in the GPI file in those cases
        where it refers to debug info (see ../tree.h), we get a segfault
        under DJGPP when compiling a program that uses a unit first
        without, then with `-g'. */
  int save = TREE_ASM_WRITTEN (t);
  switch (TREE_CODE (t))
    {
      case RECORD_TYPE:
      case UNION_TYPE:
      case QUAL_UNION_TYPE:
      case ENUMERAL_TYPE:
        TREE_ASM_WRITTEN (t) = 1;
        break;
      default:
        break;
    }
  store_length ((tree *) t + 2, 4);
  TREE_ASM_WRITTEN (t) = save;
}
#define load_flags(t) LOAD_LENGTH ((tree *) t + 2, 4)
#define copy_flags(t, f) memcpy ((tree *) t + 2, f, 4)

/* @@@@ Store/load a *_TYPE tree node's extra flags */
#define store_type_flags(t) store_length (&TYPE_UID (t), sizeof (TYPE_UID (t)) + 4 + sizeof (TYPE_ALIGN (t)))
#define load_type_flags(t)  LOAD_LENGTH  (&TYPE_UID (t), sizeof (TYPE_UID (t)) + 4 + sizeof (TYPE_ALIGN (t)))
#define copy_type_flags(t0, t) memcpy (&TYPE_UID (t), &TYPE_UID (t0), sizeof (TYPE_UID (t)) + 4 + sizeof (TYPE_ALIGN (t)))

/* @@@@ Store/load a *_DECL tree node's extra flags */
#define store_decl_flags1(t) (STORE_ANY (DECL_UID (t)), store_length ((&DECL_SIZE (t)) + 1, 4))
#define load_decl_flags1(t)  (LOAD_ANY  (DECL_UID (t)), LOAD_LENGTH  ((&DECL_SIZE (t)) + 1, 4))
#ifdef EGCS97
#error Check these! (see source comment)
  /*
  This was previously there (above) and was wrong for 2.8.1 already (martin2a.pas).
  I don't have "EGCS97" here, so I don't know what the types look like there. -- Frank
  #define store_decl_flags1(t) store_length (&DECL_UID (t), sizeof (DECL_UID (t)) + 4)
  #define load_decl_flags1(t)  LOAD_LENGTH  (&DECL_UID (t), sizeof (DECL_UID (t)) + 4)
  */
#define store_decl_flags(t) (store_decl_flags1(t), STORE_ANY (t->decl.u1.i))
#define load_decl_flags(t)  (load_decl_flags1(t),  LOAD_ANY  (t->decl.u1.i))
#else
#define store_decl_flags(t) (store_decl_flags1(t), STORE_ANY (DECL_FRAME_SIZE (t)))
#define load_decl_flags(t)  (load_decl_flags1(t),  LOAD_ANY  (DECL_FRAME_SIZE (t)))
#endif

/* Load a string out of a memfile */
static char *
load_string (s)
     MEMFILE *s;
{
  char *str;
  gpi_int l;
  LOAD_ANY_F (s, l);
  str = (char *) xmalloc ((int) l + 1);
  if (l > 0)
    LOAD_LENGTH_F (s, str, l);
  str[l] = 0;
  return str;
}

/* Store the fields of a node in a stream. */
static void
store_node_fields (t, uid)
     tree t;
     int uid;
{
  unsigned char code;

  #ifdef USE_GPI_DEBUG_KEY
  gpi_int debug_key = GPI_DEBUG_KEY;
  STORE_ANY (debug_key);
  #endif

  /* UID hashing does not work for IDENTIFIER_NODEs because their
     fields can change between interfaces. */
  if (TREE_CODE (t) == INTERFACE_NAME_NODE || TREE_CODE (t) == IDENTIFIER_NODE)
    store_node (NULL_TREE);
  else
    {
      struct interface_table_t *itab;
      int hash = HASH_FUNC (t);
      gpi_int n = -1;
      for (itab = interface_table; itab; itab = itab->next)
        {
          for (n = itab->hash_table[hash]; n >= 0; n = itab->hashlist_next[n])
            {
              assert (n < itab->count);
              if (itab->nodes[n] == t)
                break;
            }
          if (n >= 0)
            break;
        }
      if (itab)
        {
          assert (itab->interface_name_node);
          store_node (itab->interface_name_node);
          STORE_ANY (n);
        }
      else
        {
          store_node (NULL_TREE);
          itab_store_node (wb.storing_gpi_file, current_module->name, uid, t);
        }
    }

  code = (unsigned char) TREE_CODE (t);
  STORE_ANY (code);
  if (flag_debug_gpi)
    {
      fprintf (stderr, "GPI storing <%i>:\n", uid);
      if (code == INTERFACE_NAME_NODE)
        fprintf (stderr, " <interface_name_node %x %s module %s checksum %x>\n",
                 (int) t,
                 IDENTIFIER_POINTER (TREE_VALUE (t)),
                 IDENTIFIER_POINTER (TREE_PURPOSE (t)),
                 ((struct interface_table_t *) TREE_TYPE (t))->gpi_checksum);
      else
        debug_tree (t);
    }
  if (code != INTERFACE_NAME_NODE && code != TREE_LIST)
    store_flags (t);
  switch (code)
    {
      case VOID_TYPE:
      case REAL_TYPE:
      case COMPLEX_TYPE:
      case BOOLEAN_TYPE:
      case CHAR_TYPE:
      case LANG_TYPE:
      case INTEGER_TYPE:
      case ENUMERAL_TYPE:
      case SET_TYPE:
      case POINTER_TYPE:
      case REFERENCE_TYPE:
      case FILE_TYPE:
      case RECORD_TYPE:
      case UNION_TYPE:
        store_type_flags (t);
        store_node (TYPE_SIZE (t));
#ifdef EGCS
        store_node (TYPE_SIZE_UNIT (t));
#endif
        break;
      default:
        break;
    }
  switch (code)
    {
      case INTERFACE_NAME_NODE:
        {
          assert (TREE_VALUE (t) && TREE_PURPOSE (t));
          store_string (IDENTIFIER_POINTER (TREE_VALUE (t)));
          store_string (IDENTIFIER_POINTER (TREE_PURPOSE (t)));
          STORE_ANY (((struct interface_table_t *) TREE_TYPE (t))->gpi_checksum);
          break;
        }

      case IDENTIFIER_NODE:
        {
          store_string (IDENTIFIER_POINTER (t));
          if (IDENTIFIER_LOCAL_VALUE (t))
            store_node (IDENTIFIER_LOCAL_VALUE (t));
          else if (IDENTIFIER_GLOBAL_VALUE (t)
                   && (TREE_CODE_CLASS (TREE_CODE (IDENTIFIER_GLOBAL_VALUE (t))) != 'd'
                       || ! PASCAL_REDEFINABLE_DECL (IDENTIFIER_GLOBAL_VALUE (t))))
            store_node (IDENTIFIER_GLOBAL_VALUE (t));
          else
            store_node (IDENTIFIER_LIMBO_VALUE (t));
          break;
        }

      case TREE_LIST:
        {
          gpi_int n, m = -1, i;
          for (i = 1; i <= 2; i++)  /* first count, then store */
            {
              tree t0;
              n = 0;
              for (t0 = t; t0; t0 = TREE_CHAIN (t0))
                {
                  /* Due to the autoexporting of enum type values and VMT variables,
                     the main list may contain duplicates. Avoid them here.
                     @@ `n != 0' is probably a kludge. Apparently, the first item
                        of the list is the module name which may be identical to
                        another identifier. */
                  int flag = 1;
                  if (wb.main_list_flag && n != 0)
                    {
                      int id1 = get_node_id (TREE_VALUE (t0)),
                          id2 = get_node_id (TREE_PURPOSE (t0));
                      if (wb.main_exported[id1] >= i && wb.main_exported[id2] >= i)
                        flag = 0;
                      wb.main_exported[id1] = i;
                      wb.main_exported[id2] = i;
                    }
                  if (flag)
                    n++;
                  if (i == 2)
                    {
                      if (flag)
                        {
                          store_flags (t0);
                          store_node (TREE_PURPOSE (t0));
                          store_node (TREE_VALUE (t0));
                        }
                      else if (flag_debug_gpi)
                        fprintf (stderr, "GPI ignoring duplicate identifier `%s'\n",
                                         IDENTIFIER_POINTER (TREE_VALUE (t0)));
                    }
                }
              if (i == 1)
                {
                  m = n;
                  STORE_ANY (n);
                }
            }
          assert (m == n);
          break;
        }

      case ENUMERAL_TYPE:
        /* For directly or indirectly exported enumerated types, mark their
           values here for exporting in the main list(!), but only in
           autoexported interfaces (EP, 6.11.2, note 2).
           Not storing TYPE_VALUES (t) directly causes the values not be
           connected with the type in an importing interface. Therefore, it
           will not automatically re-export the values again (BP behaves
           the same way). (fjf628.pas) */
        if (wb.autoexport_flag)
          {
            tree item;
            for (item = TYPE_VALUES (t); item; item = TREE_CHAIN (item))
              wb.append_additional_globals_to_export =
                TREE_CHAIN (wb.append_additional_globals_to_export) =
                  build_tree_list (NULL_TREE, TREE_PURPOSE (item));
          }
        /* FALLTHROUGH */
      case VOID_TYPE:
      case REAL_TYPE:
      case COMPLEX_TYPE:
      case BOOLEAN_TYPE:
      case CHAR_TYPE:
      case LANG_TYPE:
      case INTEGER_TYPE:
        {
          store_node (TREE_TYPE (t));
          store_node (TYPE_MIN_VALUE (t));
          store_node (TYPE_MAX_VALUE (t));
          store_main_variant (t);
          break;
        }

      case SET_TYPE:
        {
          store_node (TREE_TYPE (t));
          store_node (TYPE_DOMAIN (t));
          store_main_variant (t);
          break;
        }

      case POINTER_TYPE:
      case REFERENCE_TYPE:
        {
          store_node (TREE_TYPE (t));
          store_main_variant (t);
          break;
        }

      case FILE_TYPE:
        {
          /* Since FILE_TYPE is not documented in tree.def,
             I only can guess about its structure. */
          store_node (TREE_TYPE (t));
          store_node (TYPE_DOMAIN (t));
          store_main_variant (t);
          break;
        }

      case ARRAY_TYPE:
        {
          store_node (TREE_TYPE (t));
          store_node (TYPE_DOMAIN (t));
          store_type_flags (t);
          store_main_variant (t);
          break;
        }

      case RECORD_TYPE:
      case UNION_TYPE:
        {
          tree f;
          signed char lang_code = 0;
          if (! TYPE_LANG_SPECIFIC (t))
            TYPE_LANG_SPECIFIC (t) = allocate_type_lang_specific ();
          lang_code = TYPE_LANG_CODE (t);
          STORE_ANY (lang_code);
          store_node (TYPE_LANG_INFO (t));
          store_node (TYPE_LANG_BASE (t));
          /* Sanity check */
          assert (!TYPE_LANG_INFO (t)
                  || lang_code == PASCAL_LANG_VARIANT_RECORD
                  || lang_code == PASCAL_LANG_OBJECT
                  || lang_code == PASCAL_LANG_ABSTRACT_OBJECT
                  || lang_code == PASCAL_LANG_STRING);
          for (f = TYPE_FIELDS (t); f; f = TREE_CHAIN (f))
            {
              store_node (f);
              store_node (DECL_FIELD_BITPOS (f));
            }
          store_node (NULL_TREE);  /* end of field decls */
          store_main_variant (t);
          /* Even if an object type is only indirectly mentioned,
             its VMT variable must be exported. */
          if (PASCAL_TYPE_OBJECT (t) && TYPE_LANG_CODE (t) != PASCAL_LANG_ABSTRACT_OBJECT)
            wb.append_additional_globals_to_export =
              TREE_CHAIN (wb.append_additional_globals_to_export) =
                build_tree_list (NULL_TREE, get_vmt (t));
          break;
        }

      case FUNCTION_TYPE:  /* @@@ unused? */
        {
          tree a;
          store_node (TREE_TYPE (t));
          store_node (TYPE_ARG_TYPES (t));
          store_type_flags (t);
          for (a = TYPE_ATTRIBUTES (t); a; a = TREE_CHAIN (a))
            {
              store_node (TREE_VALUE (a));
              store_node (TREE_PURPOSE (a));
            }
          store_node (NULL_TREE);
          store_main_variant (t);
          break;
        }

      case INTEGER_CST:
        {
          STORE_ANY (TREE_INT_CST_LOW (t));
          STORE_ANY (TREE_INT_CST_HIGH (t));
          store_node (TREE_TYPE (t));
          break;
        }

      case REAL_CST:
        {
          STORE_ANY (TREE_REAL_CST (t));
          store_node (TREE_TYPE (t));
          break;
        }

      case COMPLEX_CST:
        {
          store_node (TREE_REALPART (t));
          store_node (TREE_IMAGPART (t));
          store_node (TREE_TYPE (t));
          break;
        }

      case STRING_CST:
        {
          STORE_ANY (TREE_STRING_LENGTH (t));
          store_length (TREE_STRING_POINTER (t), TREE_STRING_LENGTH (t));
          store_node (TREE_TYPE (t));
          break;
        }

      case FUNCTION_DECL:
        {
          tree a;
          store_node (DECL_NAME (t));  /* name of function */
          store_string (IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (t)));
          store_node (TREE_TYPE (TREE_TYPE (t)));  /* return type */

          /* Arguments. */
          if (! DECL_LANG_SPECIFIC (t))
            DECL_LANG_SPECIFIC (t) = allocate_decl_lang_specific ();
          for (a = DECL_LANG_PARMS (t); a; a = TREE_CHAIN (a))
            if (TREE_CODE (a) == PARM_DECL)
              store_node (a);

          /* Store NULL_TREE (ellipsis) or void_type_node (no ellipsis). */
          for (a = TYPE_ARG_TYPES (TREE_TYPE (t)); a; a = TREE_CHAIN (a))
            if (TREE_VALUE (a) == void_type_node)
              break;
          if (a)
            store_node (void_type_node);
          else
            store_node (NULL_TREE);

          /* Machine attributes. */
          for (a = TYPE_ATTRIBUTES (TREE_TYPE (t)); a; a = TREE_CHAIN (a))
            {
              /* Both may be NULL_TREE. */
              store_node (TREE_VALUE (a));
              store_node (TREE_PURPOSE (a));
            }
          store_node (error_mark_node);
          /* This might be a constructor or a virtual method. */
          break;
        }

      case LABEL_DECL:
      case PARM_DECL:
      case RESULT_DECL:
        {
          store_node (DECL_NAME (t));
          store_node (TREE_TYPE (t));
          break;
        }

      case FIELD_DECL:
        {
          enum machine_mode mode = DECL_MODE (t);
          tree f;
          store_decl_flags (t);
          STORE_ANY (mode);
          store_node (DECL_NAME (t));
          store_node (TREE_TYPE (t));
          store_node (DECL_SIZE (t));
          store_node (DECL_FIELD_BITPOS (t));
          f = PASCAL_DECL_FIXUPLIST (t);
          /* A little optimization */
          if (f && TREE_CHAIN (f) && !TREE_VALUE (f) && !TREE_PURPOSE (f))
            f = TREE_CHAIN (f);
          store_node (f);
          break;
        }

      case CONST_DECL:
      case TYPE_DECL:
        {
          store_node (DECL_NAME (t));
          store_node (TREE_TYPE (t));
          store_node (DECL_INITIAL (t));
          break;
        }

      case VAR_DECL:
        {
          store_node (DECL_NAME (t));
          store_string (IDENTIFIER_POINTER (DECL_ASSEMBLER_NAME (t)));
          store_node (TREE_TYPE (t));
          /* Only store DECL_INITIAL if this is an `absolute' variable. */
          if (DECL_INITIAL (t)
              && PASCAL_ABSOLUTE_CLAUSE (DECL_INITIAL (t)))
            store_node (DECL_INITIAL (t));
          else
            store_node (NULL_TREE);
          break;
        }

      case CONSTRUCTOR:
        {
          store_node (TREE_OPERAND (t, 1));
          store_node (TREE_TYPE (t));
          break;
        }

      /* 3 operands */
      case BIT_FIELD_REF:
      case COND_EXPR:
      case METHOD_CALL_EXPR:
        {
          store_node (TREE_OPERAND (t, 0));
          store_node (TREE_OPERAND (t, 1));
          store_node (TREE_OPERAND (t, 2));
          store_node (TREE_TYPE (t));
          break;
        }
      /* two operands */
      case COMPONENT_REF:
      case ARRAY_REF:
      case COMPOUND_EXPR:
      case MODIFY_EXPR:
      case INIT_EXPR:
      case CALL_EXPR:
      case PLUS_EXPR:
      case MINUS_EXPR:
      case MULT_EXPR:
      case TRUNC_DIV_EXPR:
      case CEIL_DIV_EXPR:
      case FLOOR_DIV_EXPR:
      case ROUND_DIV_EXPR:
      case TRUNC_MOD_EXPR:
      case FLOOR_MOD_EXPR:
      case CEIL_MOD_EXPR:
      case ROUND_MOD_EXPR:
      case RDIV_EXPR:
      case EXACT_DIV_EXPR:
      case MIN_EXPR:
      case MAX_EXPR:
      case LSHIFT_EXPR:
      case RSHIFT_EXPR:
      case LROTATE_EXPR:
      case RROTATE_EXPR:
      case BIT_IOR_EXPR:
      case BIT_XOR_EXPR:
      case BIT_AND_EXPR:
      case BIT_ANDTC_EXPR:
      case TRUTH_ANDIF_EXPR:
      case TRUTH_ORIF_EXPR:
      case TRUTH_AND_EXPR:
      case TRUTH_OR_EXPR:
      case TRUTH_XOR_EXPR:
      case LT_EXPR:
      case LE_EXPR:
      case GT_EXPR:
      case GE_EXPR:
      case EQ_EXPR:
      case NE_EXPR:
      case IN_EXPR:
      case SET_LE_EXPR:
      case RANGE_EXPR:
      case COMPLEX_EXPR:
        {
          store_node (TREE_OPERAND (t, 0));
          store_node (TREE_OPERAND (t, 1));
          store_node (TREE_TYPE (t));
          break;
        }
      /* one operand */
      case INDIRECT_REF:
      case BUFFER_REF:
      case FIX_TRUNC_EXPR:
      case FIX_CEIL_EXPR:
      case FIX_FLOOR_EXPR:
      case FIX_ROUND_EXPR:
      case FLOAT_EXPR:
      case NEGATE_EXPR:
      case ABS_EXPR:
      case FFS_EXPR:
      case BIT_NOT_EXPR:
      case TRUTH_NOT_EXPR:
      case CARD_EXPR:
      case CONVERT_EXPR:
      case NOP_EXPR:
      case PLACEHOLDER_EXPR:
      case NON_LVALUE_EXPR:
      case SAVE_EXPR:
      case UNSAVE_EXPR:
      case ADDR_EXPR:
      case REFERENCE_EXPR:
      case ENTRY_VALUE_EXPR:
      case CONJ_EXPR:
      case REALPART_EXPR:
      case IMAGPART_EXPR:
        {
          store_node (TREE_OPERAND (t, 0));
          store_node (TREE_TYPE (t));
          break;
        }
      default:
        assert (0);
    }
}

/* Return an interface table (newly allocated if not existing yet)
   and make sure that it can hold nodes up to at least MAX_UID. */
static struct interface_table_t *
get_interface_table (interface_name, module_name, max_uid)
     tree interface_name, module_name;
     gpi_int max_uid;
{
  struct interface_table_t *itab;
  for (itab = interface_table; itab; itab = itab->next)
    if (itab->interface_name == interface_name)
      break;
  if (!itab)
    {
      gpi_int n;
      itab = (struct interface_table_t *) xmalloc (sizeof (struct interface_table_t));
      memset ((void *) itab, 0, sizeof (struct interface_table_t));
      itab->count = 0;
      itab->interface_name = interface_name;
      itab->module_name = module_name;
      itab->gpi_checksum = 0;
      itab->interface_name_node = NULL_TREE;
      itab->initializers = NULL_TREE;
      for (n = 0; n < MAX_HASH_TABLE; n++)
        itab->hash_table[n] = -1;
      itab->next = interface_table;
      interface_table = itab;
    }
  else if (!itab->module_name)
    {
      itab->module_name = module_name;
      if (itab->interface_name_node)
        TREE_PURPOSE (itab->interface_name_node) = module_name;
    }
  else if (module_name && module_name != itab->module_name)
    {
      error ("interface `%s' in both modules `%s' and `%s'",
             IDENTIFIER_POINTER (interface_name),
             IDENTIFIER_POINTER (itab->module_name),
             IDENTIFIER_POINTER (module_name));
      exit (FATAL_EXIT_CODE);
    }
  if (max_uid >= itab->count)
    {
      gpi_int n, old_count = itab->count;
      if (itab->count == 0) itab->count = 64;
      while (itab->count <= max_uid) itab->count *= 2;  /* avoid too many reallocations */
      itab->nodes = (tree *) xrealloc (itab->nodes, itab->count * sizeof (tree));
      itab->hashlist_next = (int *) xrealloc (itab->hashlist_next, itab->count * sizeof (int));
      for (n = old_count; n < itab->count; n++)
        {
          itab->nodes[n] = NULL_TREE;
          itab->hashlist_next[n] = -1;
        }
    }
  return itab;
}

/* Compute a checksum for GPI files.
   @@ Simple weighted sum. Perhaps we should use MD5 or something. */
static gpi_int
compute_checksum (buf, size)
     unsigned char *buf;
     gpi_int size;
{
  gpi_int sum = 0, n;
  for (n = 0; n < size; n++)
    sum += n * buf[n];
  return sum;
}

/* If update_flag is 0, the function checks if the checksum matches
   the one already stored (if any), returns 1 if so, 0 otherwise,
   but never sets it. If update_flag is 1, it updates the checksum,
   fails fatally on a mismatch, otherwise always returns 1 and sets
   it if it wasn't stored before. */
static int
itab_check_gpi_checksum (interface_name, gpi_checksum, update_flag)
     tree interface_name;
     gpi_int gpi_checksum;
     int update_flag;
{
  struct interface_table_t *itab;
  tree t;
  for (itab = interface_table; itab; itab = itab->next)
    if (itab->interface_name == interface_name)
      break;
  assert (itab);
  if (itab->interface_name_node)
    {
      if (itab->gpi_checksum == gpi_checksum)
        return 1;
      if (update_flag)
        {
          error ("checksum mismatch for interface `%s' (recompile interfaces in the right order)",
                 IDENTIFIER_POINTER (interface_name));
          exit (FATAL_EXIT_CODE);
        }
    return 0;
    }
  if (!update_flag)
    return 1;
  itab->gpi_checksum = gpi_checksum;
  /* @@ misuse of TREE_LIST and TREE_TYPE(1) */
  t = build_tree_list (itab->module_name, itab->interface_name);
  TREE_CODE (t) = INTERFACE_NAME_NODE;
  TREE_TYPE (t) = (tree) itab;
  itab->interface_name_node = t;
  return 1;
}

/* Return the name of the module the interface named NAME is in. */
tree
itab_get_initializers (name)
     tree name;
{
  struct interface_table_t *itab;
  for (itab = interface_table; itab; itab = itab->next)
    if (itab->interface_name == name)
      break;
  assert (itab);
  return itab->initializers;
}

/* Load (parts of) a tree out of a stream.
 * (A generalized version of this should go into tree.c.) */
static tree
load_node ()
{
  gpi_int uid, original_uid;
  unsigned char code;
  int save_pos;
  tree t = NULL_TREE;
  char flags[4];
  struct interface_table_t *itab;
  tree interface_node, interface_name, module_name;

  /* Check whether the node number n has already been loaded
     (this includes the special nodes). */
  LOAD_ANY (uid);
  if (uid == 0 || rb.nodes[uid])
    return rb.nodes[uid];

  /* If not, seek file for reading the node */
  save_pos = mtell (rb.infile);
  mseek (rb.infile, rb.offsets[uid]);
  #ifdef USE_GPI_DEBUG_KEY
  {
    gpi_int key;
    LOAD_ANY (key);
    assert (key == GPI_DEBUG_KEY);
  }
  #endif

  interface_node = load_node ();
  if (interface_node)
    {
      LOAD_ANY (original_uid);
      interface_name = TREE_VALUE (interface_node);
      module_name = TREE_PURPOSE (interface_node);
    }
  else
    {
      original_uid = uid;
      interface_name = loading_gpi_file;
      module_name = NULL_TREE;
    }
  for (itab = interface_table; itab; itab = itab->next)
    if (itab->interface_name == interface_name
        && (!module_name || itab->module_name == module_name)
        && original_uid < itab->count && itab->nodes[original_uid])
      {
        t = itab->nodes[original_uid];
        assert (mark_node_loaded (t, uid) == t);
        mseek (rb.infile, save_pos);
        return t;
      }

  LOAD_ANY (code);
  switch (code)
    {
      case INTERFACE_NAME_NODE:
        {
          gpi_int gpi_checksum;
          tree i, m;
          char *id = load_string (rb.infile);
          i = get_identifier (id);
          free (id);
          id = load_string (rb.infile);
          m = get_identifier (id);
          free (id);
          /* @@ misuse of TREE_LIST */
          t = build_tree_list (m, i);
          TREE_CODE (t) = INTERFACE_NAME_NODE;
          assert (mark_node_loaded (t, uid) == t);
          LOAD_ANY (gpi_checksum);
          get_interface_table (i, m, 1);
          itab_check_gpi_checksum (i, gpi_checksum, 1);
          mseek (rb.infile, save_pos);
          if (flag_debug_gpi)
            fprintf (stderr, "GPI loaded <%i>:\n"
                     " <interface_name_node %x %s module %s checksum %x>\n",
                     uid, (int) t,
                     IDENTIFIER_POINTER (TREE_VALUE (t)),
                     IDENTIFIER_POINTER (TREE_PURPOSE (t)),
                     gpi_checksum);
          return t;
        }

      case TREE_LIST:
        break;
      case IDENTIFIER_NODE:
      case ARRAY_TYPE:
      case FUNCTION_TYPE:
      case FUNCTION_DECL:
      case LABEL_DECL:
      case PARM_DECL:
      case RESULT_DECL:
      case VAR_DECL:
        LOAD_ANY (flags);
        break;
      default:
        t = make_node (code);
        assert (mark_node_loaded (t, uid) == t);
        load_flags (t);
        break;
    }
  switch (code)
    {
      case VOID_TYPE:
      case REAL_TYPE:
      case COMPLEX_TYPE:
      case BOOLEAN_TYPE:
      case CHAR_TYPE:
      case LANG_TYPE:
      case INTEGER_TYPE:
      case ENUMERAL_TYPE:
      case SET_TYPE:
      case POINTER_TYPE:
      case REFERENCE_TYPE:
      case FILE_TYPE:
      case RECORD_TYPE:
      case UNION_TYPE:
        load_type_flags (t);
        TYPE_SIZE (t) = load_node ();
#ifdef EGCS
        TYPE_SIZE_UNIT (t) = load_node ();
#endif
        break;
      default:
        break;
    }
  switch (code)
    {
      case IDENTIFIER_NODE:
        {
          char *id = load_string (rb.infile);
          t = get_identifier (id);
          free (id);
          assert (mark_node_loaded (t, uid) == t);
          copy_flags (t, flags);
          IDENTIFIER_LIMBO_VALUE (t) = load_node ();
          break;
        }

      case TREE_LIST:
        {
          gpi_int n, i;
          tree last;
          LOAD_ANY (n);
          t = make_node (TREE_LIST);
          assert (mark_node_loaded (t, uid) == t);
          load_flags (t);
          TREE_PURPOSE (t) = load_node ();
          TREE_VALUE (t) = load_node ();
          last = t;
          for (i = 2; i <= n; i++)
            {
              TREE_CHAIN (last) = make_node (TREE_LIST);
              last = TREE_CHAIN (last);
              load_flags (last);
              TREE_PURPOSE (last) = load_node ();
              TREE_VALUE (last) = load_node ();
            }
          break;
        }

      case VOID_TYPE:
      case REAL_TYPE:
      case COMPLEX_TYPE:
      case BOOLEAN_TYPE:
      case CHAR_TYPE:
      case LANG_TYPE:
      case INTEGER_TYPE:
      case ENUMERAL_TYPE:
        {
          TREE_TYPE (t) = load_node ();
          TYPE_MIN_VALUE (t) = load_node ();
          TYPE_MAX_VALUE (t) = load_node ();
          load_main_variant (t);
          break;
        }

      case SET_TYPE:
        {
          TREE_TYPE (t) = load_node ();
          TYPE_DOMAIN (t) = load_node ();
          load_main_variant (t);
          break;
        }

      case POINTER_TYPE:
      case REFERENCE_TYPE:
        {
          TREE_TYPE (t) = error_mark_node;  /* Avoid crash in recursion */
          TREE_TYPE (t) = load_node ();
          load_main_variant (t);
          break;
        }

      case FILE_TYPE:
        {
          /* Since FILE_TYPE is not documented in tree.def,
             I only can guess about its structure. */
          TREE_TYPE (t) = load_node ();
          TYPE_DOMAIN (t) = load_node ();
          load_main_variant (t);
          break;
        }

      case ARRAY_TYPE:
        {
          tree type, domain, t1;
          union tree_node t0;
          int packed, save_flag_pack_struct = 0;

          type = load_node ();
          domain = load_node ();

          load_type_flags (&t0);
          packed = PASCAL_TYPE_PACKED (&t0);
          if (packed)
            {
              save_flag_pack_struct = flag_pack_struct;
              flag_pack_struct = 1;
            }
          /* In case this array already exists, we do not want to
             spoil its flags, thus we copy it. */
          t = build_type_copy (build_array_type (type, domain));
          if (packed)
            flag_pack_struct = save_flag_pack_struct;

          t1 = mark_node_loaded (t, uid);
          if (t1 != t)
            {
              mseek (rb.infile, save_pos);
              return t1;
            }
          copy_flags (t, flags);
          copy_type_flags (&t0, t);
          if (packed)
            t = grok_packed (t);
          load_main_variant (t);
          break;
        }

      case RECORD_TYPE:
      case UNION_TYPE:
        {
          tree f, f0 = NULL_TREE;
          tree current_type_name_save = current_type_name;
          signed char lang_code;
          TYPE_LANG_SPECIFIC (t) = allocate_type_lang_specific ();
          LOAD_ANY (lang_code);
          TYPE_LANG_CODE (t) = lang_code;
          TYPE_LANG_INFO (t) = load_node ();
          TYPE_LANG_BASE (t) = load_node ();
          if (lang_code == PASCAL_LANG_OBJECT || lang_code == PASCAL_LANG_ABSTRACT_OBJECT)
            current_type_name = TYPE_LANG_NAME (t);  /* == TYPE_LANG_INFO */
          do
            {
              f = load_node ();
              if (f)
                {
                  /* Don't use chainon: danger of circularity
                     if f0 was loaded via gpi_contents. */
                  if (f0)
                    TREE_CHAIN (f0) = f;
                  else
                    TYPE_FIELDS (t) = f;
                  /* Delete possibly invalid TREE_CHAIN info
                     if t was loaded via gpi_contents. */
                  TREE_CHAIN (f) = NULL_TREE;
                  f0 = f;
                  DECL_CONTEXT (f) = t;
                  DECL_FIELD_BITPOS (f) = load_node ();
                }
            }
          while (f);
          load_main_variant (t);
          current_type_name = current_type_name_save;  /* usually NULL_TREE */
          break;
        }

      case FUNCTION_TYPE:  /* @@@ unused? */
        {
          tree t1, type = load_node ();
          tree arg_types = load_node ();
          tree a;
          t = build_function_type (type, arg_types);
          t1 = mark_node_loaded (t, uid);
          if (t1 != t)
            {
              mseek (rb.infile, save_pos);
              return t1;
            }
          load_type_flags (t);
          a = load_node ();
          while (a)
            {
              TYPE_ATTRIBUTES (t) = chainon (TYPE_ATTRIBUTES (t),
                build_tree_list (load_node (), a));
              a = load_node ();
            }
          copy_flags (t, flags);
          load_main_variant (t);
          break;
        }

      case INTEGER_CST:
        {
          LOAD_ANY (TREE_INT_CST_LOW (t));
          LOAD_ANY (TREE_INT_CST_HIGH (t));
          TREE_TYPE (t) = load_node ();
          break;
        }

      case REAL_CST:
        {
          LOAD_ANY (TREE_REAL_CST (t));
          TREE_TYPE (t) = load_node ();
          break;
        }

      case COMPLEX_CST:
        {
          TREE_REALPART (t) = load_node ();
          TREE_IMAGPART (t) = load_node ();
          TREE_TYPE (t) = load_node ();
          break;
        }

      case STRING_CST:
        {
          LOAD_ANY (TREE_STRING_LENGTH (t));
#ifdef EGCS97
          TREE_STRING_POINTER (t) = xmalloc (TREE_STRING_LENGTH (t) + 9);
#else
          TREE_STRING_POINTER (t) = oballoc (TREE_STRING_LENGTH (t) + 9);
#endif
          LOAD_LENGTH (TREE_STRING_POINTER (t), TREE_STRING_LENGTH (t));
          if (TREE_STRING_POINTER (t) [TREE_STRING_LENGTH (t) - 1] != 0)
            fatal ("invalid string constant in GPI file");
          TREE_TYPE (t) = load_node ();
          break;
        }

      case FUNCTION_DECL:
        {
          tree t1, name, full_name, asmname, type, d, heading,
               save_local, save_global, save_limbo = error_mark_node;
          tree args, a, last_arg, argdecls;
          tree directives = NULL_TREE, attributes = NULL_TREE;
          int ellipsis;
          int in_parm_list = in_parm_level_p ();
          extern tree d_attribute;
          char *id;

          name = load_node ();
          save_local = IDENTIFIER_LOCAL_VALUE (name);
          save_global = IDENTIFIER_GLOBAL_VALUE (name);
          IDENTIFIER_GLOBAL_VALUE (name) = IDENTIFIER_LOCAL_VALUE (name) = NULL_TREE;

          if (current_type_name)
            {
              full_name = get_method_name (current_type_name, name);
              /* In a method don't overwrite the limbo value which may be used
                 for a global function of the same name in the same interface. */
              save_limbo = IDENTIFIER_LIMBO_VALUE (name);
            }
          else
            full_name = name;

          id = load_string (rb.infile);
          asmname = get_identifier (id);
          free (id);
          type = load_node ();

          if (in_parm_list)
            /* This is a method definition of an object reference inside the
               parameter list. Don't let external function declarations become
               global here. */
            pushlevel (0);

          immediate_size_expand = 0;
          size_volatile++;

          /* Arguments. */
          args = NULL_TREE;
          last_arg = NULL_TREE;
          a = load_node ();
          while (a != NULL_TREE && a != void_type_node)
            {
              if (last_arg)
                TREE_CHAIN (last_arg) = a;
              else
                args = a;
              last_arg = a;
              a = load_node ();
            }
          if (last_arg)
            TREE_CHAIN (last_arg) = NULL_TREE;
          ellipsis = (a == NULL_TREE);

          /* Machine attributes. */
          a = load_node ();
          while (a != error_mark_node)
            {
              tree b = load_node ();
              attributes = chainon (attributes, build_tree_list (b, a));
              a = load_node ();
            }
          if (attributes)
            directives = build_tree_list (attributes, d_attribute);

          type = tree_cons (NULL_TREE, type, NULL_TREE);
          type = tree_cons (NULL_TREE, extern_id, type);

          pushlevel (0);
          clear_parm_order ();
          declare_parm_level (1);
          for (a = args; a; a = TREE_CHAIN (a))
            {
              /* Do essentially the same as handle_formal_param_list().
               * We don't need to think about schemata etc. here because
               * they already were prediscriminated before they were
               * written into the GPI file.
               */
              tree name = DECL_NAME (a);
              d = build_tree_list (NULL_TREE, TREE_TYPE (a));
              d = build_tree_list (build_tree_list (d, name),
                                   build_tree_list (NULL_TREE, NULL_TREE));
              push_parm_decl (d);
            }
          argdecls = get_parm_info (! ellipsis);
          poplevel (0, 0, 0);

          size_volatile--;

          /* Avoid a warning in case this unit is in the same file
           * as the main program.
           */
          IDENTIFIER_LIMBO_VALUE (full_name) = NULL_TREE;

          heading = build_nt (CALL_EXPR, full_name, argdecls, NULL_TREE);
          if (! asmname || full_name == asmname)
            {
              directives = chainon (directives,
                                    build_tree_list (NULL_TREE, d_external));
              t = grok_directive (type, heading, directives, 1);
            }
          else
            {
              directives = chainon (directives,
                                    build_tree_list (asmname, d_asmname));
              t = grok_directive (type, heading, directives, 1);
            }

          /* This might be a constructor or a virtual method. */
          copy_flags (t, flags);

          if (current_type_name)
            {
              /* Normally, t will be ignored since grok_directive already
               * has done the job of an external function declaration. Only
               * if this is a method (i.e. current_type_name != NULL_TREE),
               * a copy of t with `name' instead of `full_name' will go as
               * a field into the object (a RECORD_TYPE node).
               */
              t = copy_node (t);
              DECL_NAME (t) = name;
            }
          t1 = mark_node_loaded (t, uid);
          if (t1 != t)
            {
              if (in_parm_list)
                poplevel (0, 0, 0);
              if (save_limbo != error_mark_node)
                IDENTIFIER_LIMBO_VALUE (name) = save_limbo;
              IDENTIFIER_LOCAL_VALUE (name) = save_local;
              IDENTIFIER_GLOBAL_VALUE (name) = save_global;
              mseek (rb.infile, save_pos);
              return t1;
            }

          if (in_parm_list)
            poplevel (0, 0, 0);
          if (save_limbo != error_mark_node)
            IDENTIFIER_LIMBO_VALUE (name) = save_limbo;
          /* Restore the original values in order to enable `import only'. */
          else if (current_module && current_module->main_program)
            IDENTIFIER_LIMBO_VALUE (name) = IDENTIFIER_GLOBAL_VALUE (name);
          else
            IDENTIFIER_LIMBO_VALUE (name) = IDENTIFIER_LOCAL_VALUE (name);
          IDENTIFIER_LOCAL_VALUE (name) = save_local;
          IDENTIFIER_GLOBAL_VALUE (name) = save_global;
          break;
        }

      case LABEL_DECL:
      case PARM_DECL:
      case RESULT_DECL:
        {
          tree t1;
          t = make_node (code);
          copy_flags (t, flags);
          DECL_NAME (t) = load_node ();
          TREE_TYPE (t) = load_node ();
          DECL_CONTEXT (t) = current_function_decl;
          t1 = mark_node_loaded (t, uid);
          if (t1 != t)
            {
              mseek (rb.infile, save_pos);
              return t1;
            }
          break;
        }

      case FIELD_DECL:
        {
          enum machine_mode mode;
          load_decl_flags (t);
          LOAD_ANY (mode);
          DECL_MODE (t) = mode;
          DECL_NAME (t) = load_node ();
          size_volatile++;
          TREE_TYPE (t) = load_node ();
          size_volatile--;
          DECL_SIZE (t) = load_node ();
          DECL_FIELD_BITPOS (t) = load_node ();
          PASCAL_DECL_FIXUPLIST (t) = load_node ();
          break;
        }

      case CONST_DECL:
      case TYPE_DECL:
        {
          DECL_NAME (t) = load_node ();
          TREE_TYPE (t) = load_node ();
          if (code == TYPE_DECL)
            TYPE_NAME (TREE_TYPE (t)) = DECL_NAME (t);
          DECL_INITIAL (t) = load_node ();
          DECL_CONTEXT (t) = current_function_decl;
          break;
        }

#ifdef NEW_DECLARE_VARS

      case VAR_DECL:
        {
          tree name = load_node ();
          char *id = load_string (rb.infile);
          tree type = load_node ();
          t = build_decl (code, name, type);
          assert (mark_node_loaded (t, uid) == t);
          DECL_ASSEMBLER_NAME (t) = get_identifier (id);
          free (id);
          DECL_INITIAL (t) = load_node ();
          if (PASCAL_ABSOLUTE_CLAUSE (DECL_INITIAL (t)))
            TREE_USED (DECL_NAME (t)) = 1;
          DECL_CONTEXT (t) = current_function_decl;
          copy_flags (t, flags);
          if (code == TYPE_DECL)
            TYPE_NAME (TREE_TYPE (t)) = DECL_NAME (t);
          pushdecl (t);
          rest_of_decl_compilation (t,
            asmname ? IDENTIFIER_POINTER (asmname) : NULL_PTR, 1, 1);
          expand_decl (t);
          break;
        }

#else /* not NEW_DECLARE_VARS */

      case VAR_DECL:
        {
          /* @@@@ We re-build the tree node rather than load it. */
          tree t1, name, asmname, type, initial;
          unsigned qualifiers = VQ_EXTERNAL;
          tree save_local, save_global;
          char *id;
          name = load_node ();
          save_local = IDENTIFIER_LOCAL_VALUE (name);
          save_global = IDENTIFIER_GLOBAL_VALUE (name);
          IDENTIFIER_GLOBAL_VALUE (name) = IDENTIFIER_LOCAL_VALUE (name) = NULL_TREE;
          id = load_string (rb.infile);
          asmname = get_identifier (id);
          free (id);
          type = load_node ();
          initial = load_node ();
          if (initial && PASCAL_ABSOLUTE_CLAUSE (initial))
            {
              t = build_decl (VAR_DECL, name, type);
              DECL_INITIAL (t) = initial;
              pushdecl (t);
              TREE_USED (DECL_NAME (t)) = 1;
            }
          else
            {
              /* Use a dummy tree node to test the volatile flag. */
              union tree_node t0;
              copy_flags (&t0, flags);
              if (TREE_THIS_VOLATILE (&t0))
                qualifiers |= VQ_VOLATILE;
              declare_vars (build_tree_list (NULL_TREE, name), type, NULL_TREE,
                            asmname ? IDENTIFIER_POINTER (asmname) : NULL_PTR,
                            qualifiers, 1, NULL_TREE /* attributes */);
              t = lookup_name (name);
            }
          copy_flags (t, flags);
          t1 = mark_node_loaded (t, uid);
          if (t1 != t)
            {
              IDENTIFIER_LOCAL_VALUE (name) = save_local;
              IDENTIFIER_GLOBAL_VALUE (name) = save_global;
              mseek (rb.infile, save_pos);
              return t1;
            }
          /* Restore the original local value in order to enable `import only'. */
          if (current_module && current_module->main_program)
            IDENTIFIER_LIMBO_VALUE (name) = IDENTIFIER_GLOBAL_VALUE (name);
          else
            IDENTIFIER_LIMBO_VALUE (name) = IDENTIFIER_LOCAL_VALUE (name);
          IDENTIFIER_LOCAL_VALUE (name) = save_local;
          IDENTIFIER_GLOBAL_VALUE (name) = save_global;
          break;
        }

#endif /* not NEW_DECLARE_VARS */

      case CONSTRUCTOR:
        {
          TREE_OPERAND (t, 1) = load_node ();
          TREE_TYPE (t) = load_node ();
          break;
        }

      /* 3 operands */
      case BIT_FIELD_REF:
      case COND_EXPR:
      case METHOD_CALL_EXPR:
        {
          TREE_OPERAND (t, 0) = load_node ();
          TREE_OPERAND (t, 1) = load_node ();
          TREE_OPERAND (t, 2) = load_node ();
          TREE_TYPE (t) = load_node ();
          break;
        }
      /* two operands */
      case COMPONENT_REF:
      case ARRAY_REF:
      case COMPOUND_EXPR:
      case MODIFY_EXPR:
      case INIT_EXPR:
      case CALL_EXPR:
      case PLUS_EXPR:
      case MINUS_EXPR:
      case MULT_EXPR:
      case TRUNC_DIV_EXPR:
      case CEIL_DIV_EXPR:
      case FLOOR_DIV_EXPR:
      case ROUND_DIV_EXPR:
      case TRUNC_MOD_EXPR:
      case FLOOR_MOD_EXPR:
      case CEIL_MOD_EXPR:
      case ROUND_MOD_EXPR:
      case RDIV_EXPR:
      case EXACT_DIV_EXPR:
      case MIN_EXPR:
      case MAX_EXPR:
      case LSHIFT_EXPR:
      case RSHIFT_EXPR:
      case LROTATE_EXPR:
      case RROTATE_EXPR:
      case BIT_IOR_EXPR:
      case BIT_XOR_EXPR:
      case BIT_AND_EXPR:
      case BIT_ANDTC_EXPR:
      case TRUTH_ANDIF_EXPR:
      case TRUTH_ORIF_EXPR:
      case TRUTH_AND_EXPR:
      case TRUTH_OR_EXPR:
      case TRUTH_XOR_EXPR:
      case LT_EXPR:
      case LE_EXPR:
      case GT_EXPR:
      case GE_EXPR:
      case EQ_EXPR:
      case NE_EXPR:
      case IN_EXPR:
      case SET_LE_EXPR:
      case RANGE_EXPR:
      case COMPLEX_EXPR:
        {
          TREE_OPERAND (t, 0) = load_node ();
          TREE_OPERAND (t, 1) = load_node ();
          TREE_TYPE (t) = load_node ();
          break;
        }
      /* one operand */
      case INDIRECT_REF:
      case BUFFER_REF:
      case FIX_TRUNC_EXPR:
      case FIX_CEIL_EXPR:
      case FIX_FLOOR_EXPR:
      case FIX_ROUND_EXPR:
      case FLOAT_EXPR:
      case NEGATE_EXPR:
      case ABS_EXPR:
      case FFS_EXPR:
      case BIT_NOT_EXPR:
      case TRUTH_NOT_EXPR:
      case CARD_EXPR:
      case CONVERT_EXPR:
      case NOP_EXPR:
      case PLACEHOLDER_EXPR:
      case NON_LVALUE_EXPR:
      case SAVE_EXPR:
      case UNSAVE_EXPR:
      case ADDR_EXPR:
      case REFERENCE_EXPR:
      case ENTRY_VALUE_EXPR:
      case CONJ_EXPR:
      case REALPART_EXPR:
      case IMAGPART_EXPR:
        {
          TREE_OPERAND (t, 0) = load_node ();
          TREE_TYPE (t) = load_node ();
          break;
        }
      default:
        assert (0);
    }
  itab_store_node (interface_name, module_name, original_uid, t);
  if (flag_debug_gpi)
    {
      fprintf (stderr, "GPI loaded <%i>:\n", uid);
      debug_tree (t);
    }
  mseek (rb.infile, save_pos);
  return t;
}

/* Create a GPM file (GNU Pascal Module) pointing to the GPI files.
 * This is needed when an implementation module is looking for the
 * interfaces.
 */
void
create_gpm_file ()
{
  tree escan;
  FILE *gpm_file;
  char *gpm_file_name, *p;
  if (flag_implementation_only)
    return;
  gpm_file_name = concat (IDENTIFIER_POINTER (current_module->name), ".gpm", NULL_PTR);
  for (p = gpm_file_name; *p; p++)
    *p = tolower (*p);
  if (gpi_destination_path)
    gpm_file_name = concat (gpi_destination_path, gpm_file_name, NULL_PTR);
  gpm_file = fopen (gpm_file_name, "wb");
  if (!gpm_file)
    {
      error ("cannot create GPM file `%s'", gpm_file_name);
      exit (FATAL_EXIT_CODE);
    }
  if (flag_debug_gpi)
    fprintf (stderr, "creating GPM file: %s\n", gpm_file_name);
  if (flag_debug_gpi)
    fprintf (stderr, "GPM storing header: %s", GPM_HEADER);
  fputs (GPM_HEADER, gpm_file);
  for (escan = current_module->exports; escan; escan = TREE_CHAIN (escan))
    {
      char *name = IDENTIFIER_POINTER (TREE_VALUE (escan));
      gpi_int l = strlen (name);
      STORE_ANY_F (gpm_file, l);
      STORE_LENGTH_F (gpm_file, name, l);
      if (flag_debug_gpi)
        fprintf (stderr, "GPM storing interface name %s\n", name);
    }
  fclose (gpm_file);
}

/* Create GPI files (GNU Pascal Interface) containing precompiled
 * export interfaces of a unit or module.
 */
void
create_gpi_files ()
{
  tree escan, purpose, *pscan, tmp;
  if (flag_implementation_only)
    return;
  for (escan = current_module->exports; escan; escan = TREE_CHAIN (escan))
    {
      FILE *s;
      char *plain_input_filename, *gpi_file_name, *p;
      int autoexport_flag = 0;
      tree name = TREE_VALUE (escan), iscan;
      gpi_file_name = concat (IDENTIFIER_POINTER (name), ".gpi", NULL_PTR);
      for (p = gpi_file_name; *p; p++)
        *p = tolower (*p);
      if (gpi_destination_path)
        gpi_file_name = concat (gpi_destination_path, gpi_file_name, NULL_PTR);
      s = fopen (gpi_file_name, "wb");
      if (!s)
        {
          error ("cannot create GPI file `%s'", gpi_file_name);
          exit (FATAL_EXIT_CODE);
        }
      if (flag_debug_gpi)
        fprintf (stderr, "creating GPI file: %s\n", gpi_file_name);
      if (flag_debug_gpi)
        fprintf (stderr, "GPI storing header: %s", GPI_HEADER);
      STORE_LENGTH_F (s, GPI_HEADER, strlen (GPI_HEADER));
      STORE_ANY_F (s, endianness_marker);
      if (!gpi_version_string)
        gpi_version_string = concat (GPI_VERSION_PREFIX, version_string, NULL_PTR);
      store_string_chunk (s, GPI_CHUNK_VERSION, gpi_version_string);
      store_string_chunk (s, GPI_CHUNK_MODULE_NAME, IDENTIFIER_POINTER (current_module->name));
      plain_input_filename = main_input_filename + strlen (main_input_filename) - 1;
      while (plain_input_filename >= main_input_filename && !IS_DIR_SEPARATOR (*plain_input_filename))
        plain_input_filename--;
      plain_input_filename++;
      store_string_chunk (s, GPI_CHUNK_SRCFILE, plain_input_filename);
      /* Store names of interfaces imported by this module */
      for (iscan = current_module->imports; iscan;
           iscan = TREE_CHAIN (iscan))
        if (! TREE_STATIC (iscan))
          {
            /* Store them as strings, not as tree nodes. */
            /* We don't yet want to use the gpi_contents mechanism. */
            tree iname = TREE_VALUE (iscan);
            char *name;
            gpi_int l;
            struct interface_table_t *itab;
            assert (iname && TREE_CODE (iname) == IDENTIFIER_NODE);
            itab = get_interface_table (iname, NULL_TREE, 1);
            assert (itab->interface_name_node);
            name = IDENTIFIER_POINTER (iname);
            l = strlen (name);
            start_chunk (s, GPI_CHUNK_IMPORT, l + sizeof (itab->gpi_checksum));
            if (l > 0)
              STORE_LENGTH_F (s, name, l);
            STORE_ANY_F (s, itab->gpi_checksum);
            if (flag_debug_gpi)
              fprintf (stderr, "GPI storing %s: %s (checksum: %i)\n", gpi_chunk_names[GPI_CHUNK_IMPORT], name, (int) itab->gpi_checksum);
          }

      autoexport_flag = 0;
      for (tmp = current_module->autoexport; tmp; tmp = TREE_CHAIN (tmp))
        if (TREE_PURPOSE (TREE_VALUE (tmp)) == TREE_PURPOSE (escan))
          autoexport_flag = 1;
      module_expand_exported_ranges (TREE_PURPOSE (escan));
      for (pscan = &TREE_PURPOSE (escan); *pscan;)
        if (TREE_VALUE (*pscan) || TREE_PURPOSE (*pscan))  /* both may be NULL_TREE */
          {
            assert (TREE_CODE (TREE_VALUE (*pscan)) == IDENTIFIER_NODE);
            if (lookup_name (TREE_VALUE (*pscan)) == NULL_TREE)
              error ("exported name `%s' undefined",
                     IDENTIFIER_POINTER (TREE_VALUE (*pscan)));
            pscan = &TREE_CHAIN (*pscan);
          }
        else
          *pscan = TREE_CHAIN (*pscan);
      purpose = chainon (build_tree_list (NULL_TREE, TREE_VALUE (escan)), TREE_PURPOSE (escan));
      store_tree (name, s, purpose, autoexport_flag);
      fclose (s);
      free (gpi_file_name);
    }
}

/* Extend GPI files to contain additional information from
 * the implementation part.
 */
void
extend_gpi_files ()
{
  tree escan;
  /* @@ Is this a good check? -- Frank */
  int is_implementation = !current_module->interface  /* unit */
                          || current_module->implementation  /* module implementation */;
  if (flag_implementation_only)
    {
      if (flag_automake)
        warning ("`--automake' together with `--implementation-only' can cause problems");
      return;
    }
  if (!(is_implementation || current_module->link_files || gpc_main))
    return;
  for (escan = current_module->exports; escan; escan = TREE_CHAIN (escan))
    {
      FILE *s;
      char *p;
      tree name = TREE_VALUE (escan);
      tree t;
      string_list *link_file;
      char *current_gpi_file_name = concat (IDENTIFIER_POINTER (name), ".gpi", NULL_PTR);
      for (p = current_gpi_file_name; *p; p++)
        *p = tolower (*p);
      if (gpi_destination_path)
        current_gpi_file_name = concat (gpi_destination_path, current_gpi_file_name, NULL_PTR);
      s = fopen (current_gpi_file_name, "ab");
      if (!s)
        {
          error ("cannot append to GPI file `%s'", current_gpi_file_name);
          exit (FATAL_EXIT_CODE);
        }
      if (flag_debug_gpi)
        fprintf (stderr, "extending GPI file: %s\n", current_gpi_file_name);
      free (current_gpi_file_name);
      if (is_implementation)
        store_string_chunk (s, GPI_CHUNK_IMPLEMENTATION, "");
      #if 0  /* @@ These seem to be implementation-imports, right?
                   The former code ignored them when loading GPIs,
                   so why store them at all? -- Frank */
      tree iscan;
      /* Store names of additional interfaces imported by this module */
      for (iscan = current_module->imports; iscan;
           iscan = TREE_CHAIN (iscan))
        if (TREE_STATIC (iscan))
          {
            /* Store them as strings, not as tree nodes. */
            /* We don't yet want to use the gpi_contents mechanism. */
            tree iname = TREE_VALUE (iscan);
            assert (iname && TREE_CODE (iname) == IDENTIFIER_NODE);
            store_string_chunk (s, GPI_CHUNK_IMPORT_IMPL, IDENTIFIER_POINTER (iname));
          }
      #endif
      /* Store additional names of files to be linked. */
      for (link_file = current_module->link_files; link_file;
           link_file = link_file->next)
        if (link_file->string[0] == '-')
          store_string_chunk (s, GPI_CHUNK_LIB, link_file->string);
        else
          store_string_chunk (s, GPI_CHUNK_LINK, file_basename (link_file->string));
      for (t = current_module->initializers; t; t = TREE_CHAIN (t))
        store_string_chunk (s, GPI_CHUNK_INITIALIZER, IDENTIFIER_POINTER (TREE_VALUE (t)));
      if (gpc_main)
        store_string_chunk (s, GPI_CHUNK_GPC_MAIN_NAME, gpc_main);
      fclose (s);
    }
}

/* Subroutine of gpi_open(): Search for the source of an interface. */
static char *
locate_interface_source (interface_name, explicit_name, gpi_stored_name)
     char *interface_name, *explicit_name, *gpi_stored_name;
{
  char *result = NULL_PTR;

  /* First try the name given in EXPLICIT_NAME.  The user may omit the
   * extension and/or use capital letters in the filename.
   */
  if (explicit_name)
    {
      char *module_filename = xmalloc (strlen (explicit_name) + 4 + 1), *mfn_end;
      strcpy (module_filename, explicit_name);  /* First, try the given name. */
      mfn_end = strchr (module_filename, 0);
      result = locate_file (module_filename, LF_UNIT);
      if (! result)
        {
          strcpy (mfn_end, ".pas");  /* Next, try extension `.pas' */
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".p");  /* Next, try extension `.p' */
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".pp");  /* Next, try extension `.pp' */
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".dpr");  /* Next, try extension `.dpr' */
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          /* No success yet.  But the user did specify the filename
           * by a string constant.  Try decapitalized version.
           */
          char *p = module_filename;
          *mfn_end = 0;
          while (*p)
            {
              *p = tolower (*p);
              p++;
            }
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".pas");
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".p");
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".pp");
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".dpr");
          result = locate_file (module_filename, LF_UNIT);
        }
      free (module_filename);
    }
  /* EXPLICIT_NAME not given.  Try to derive the source file name from
   * the INTERFACE_NAME which is the name of the `.gpi' file.
   */
  else if (interface_name)
    {
      char *module_filename = xmalloc (strlen (interface_name) + 4 + 1), *mfn_end;
      strcpy (module_filename, interface_name);
      mfn_end = strchr (module_filename, 0);
      /* Cut the extension `.gpi' */
      do
        mfn_end--;
      while (mfn_end > module_filename && *mfn_end != '.');
      *mfn_end = 0;
      strcpy (mfn_end, ".pas");
      result = locate_file (module_filename, LF_UNIT);
      if (! result)
        {
          strcpy (mfn_end, ".p");
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".pp");
          result = locate_file (module_filename, LF_UNIT);
        }
      if (! result)
        {
          strcpy (mfn_end, ".dpr");
          result = locate_file (module_filename, LF_UNIT);
        }
      free (module_filename);
    }
  /* Last resort: Try the name stored in the GPI file. */
  if (!result && gpi_stored_name)
    result = locate_file (gpi_stored_name, LF_UNIT);
  return result;
}

/* Open a GPI file for reading and process the header and trailer.
 * Do an AutoMake, if necessary and/or requested.
 * The file is closed via a normal mclose().
 */
static MEMFILE *
gpi_open (interface_name, name, source, automake_level, p_start_of_nodes, p_size_of_nodes, p_size_of_offsets)
     tree interface_name;
     char *name;
     char *source;
     int automake_level;
     gpi_int *p_start_of_nodes, *p_size_of_nodes, *p_size_of_offsets;
{
  MEMFILE *s = NULL;
  char *module_filename = NULL, *source_name = NULL, *temp_name;

  temp_name = locate_file (name, LF_COMPILED_UNIT);
  if (temp_name)
    s = mopen_read (temp_name);

  if (s)
    {
      char *errstr = NULL;
      int must_recompile = 0, version_set = 0, module_name_set = 0,
          start_of_nodes = -1, start_of_offsets = -1, source_found = 0;
      tree import_list = NULL_TREE, imported;
      string_list *link_list = NULL;
      char header[sizeof (GPI_HEADER)];
      gpi_int endianness_test_number_read;
      int implementation_flag = 0;

      if (flag_debug_gpi)
        fprintf (stderr, "opened GPI file: %s\n", name);
      LOAD_LENGTH_F (s, header, strlen (GPI_HEADER));
      header[strlen (GPI_HEADER)] = 0;
      LOAD_ANY_F (s, endianness_test_number_read);
      if (strcmp (header, GPI_HEADER) != 0)
        errstr = "header mismatch in GPI file `%s'";
      else if (endianness_test_number_read == inverse_endianness_marker)
        errstr = "GPI files are not portable between hosts with different endianness (`%s')";
      else if (endianness_test_number_read != endianness_marker)
        errstr = "invalid endianness marker in GPI file `%s'";
      else
        {
          int abort_flag = 0;
          struct interface_table_t *itab = NULL_PTR;
          if (flag_debug_gpi)
            fprintf (stderr, "GPI loaded header: %s", header);
          while (!abort_flag && !meof (s))
            {
              unsigned char code;
              gpi_int chunk_size;
              char *str = NULL;
              LOAD_ANY_F (s, code);
              switch (code)
                {
                  case GPI_CHUNK_VERSION:
                    str = load_string (s);
                    version_set = 1;
                    if (!gpi_version_string)
                      gpi_version_string = concat (GPI_VERSION_PREFIX, version_string, NULL_PTR);
                    if (strcmp (str, gpi_version_string) != 0)
                      must_recompile = 1;
                    break;
                  case GPI_CHUNK_MODULE_NAME:
                    str = load_string (s);
                    module_name_set = 1;
                    itab = get_interface_table (interface_name, get_identifier (str), 1);
                    break;
                  case GPI_CHUNK_SRCFILE:
                    str = module_filename = load_string (s);
                    source_name = locate_interface_source (temp_name, source, module_filename);
                    if (source_name)
                      source_found = 1;
                    else
                      source_name = save_string (module_filename);
                    break;
                  case GPI_CHUNK_IMPORT:
                    {
                      /* Keep names of imported interfaces for later use. */
                      tree imported, interface;
                      gpi_int l, checksum;
                      LOAD_ANY_F (s, l);
                      l -= sizeof (checksum);
                      str = (char *) xmalloc ((int) l + 1);
                      if (l > 0)
                        LOAD_LENGTH_F (s, str, l);
                      str[l] = 0;
                      LOAD_ANY_F (s, checksum);
                      interface = get_identifier (str);
                      (void) get_interface_table (interface, NULL_PTR, 1);
                      if (!itab_check_gpi_checksum (interface, checksum, 0))
                        must_recompile = 1;
                      for (imported = current_module->imports; imported; imported = TREE_CHAIN (imported))
                        if (interface == TREE_VALUE (imported))
                          break;
                      if (!imported)
                        for (imported = import_list; imported; imported = TREE_CHAIN (imported))
                          if (interface == TREE_VALUE (imported))
                            break;
                      if (!imported)
                        {
                          tree new_import = build_tree_list (NULL_TREE, interface);
                          /* @@ misuse of TREE_LIST and TREE_TYPE(2) */
                          TREE_CODE (new_import) = INTERFACE_NAME_NODE;
                          TREE_TYPE (new_import) = (tree) checksum;
                          import_list = chainon (import_list, new_import);
                        }
                      break;
                    }
                  case GPI_CHUNK_LINK:
                    {
                      /* Keep names of link files for later use. */
                      char *object_file_name;
                      str = load_string (s);
                      object_file_name = locate_file (str, LF_OBJECT);
                      if (object_file_name)
                        append_string_list (&link_list, object_file_name);
                      else
                        must_recompile = 1;
                      break;
                    }
                  case GPI_CHUNK_LIB:
                    str = load_string (s);
                    append_string_list (&link_list, str);
                    break;
                  case GPI_CHUNK_INITIALIZER:
                    str = load_string (s);
                    if (!itab)
                      errstr = "wrong order of chunks in GPI file `%s'";
                    else
                      itab->initializers = chainon (itab->initializers,
                        build_tree_list (NULL_TREE, get_identifier (str)));
                    break;
                  case GPI_CHUNK_GPC_MAIN_NAME:
                    gpc_main = load_string (s);
                    break;
                  case GPI_CHUNK_NODES:
                    LOAD_ANY_F (s, chunk_size);
                    start_of_nodes = mtell (s);
                    if (p_start_of_nodes)
                      *p_start_of_nodes = start_of_nodes;
                    if (p_size_of_nodes)
                      *p_size_of_nodes = chunk_size;
                    mseek (s, mtell (s) + chunk_size);
                    break;
                  case GPI_CHUNK_OFFSETS:
                    LOAD_ANY_F (s, chunk_size);
                    start_of_offsets = mtell (s);
                    if (p_size_of_offsets)
                      *p_size_of_offsets = chunk_size;
                    mseek (s, mtell (s) + chunk_size);
                    break;
                  case GPI_CHUNK_IMPLEMENTATION:
                    LOAD_ANY_F (s, chunk_size);
                    if (chunk_size != 0)
                      {
                        mseek (s, mtell (s) + chunk_size);
                        errstr = "implementation flag in GPI file `%s' contains unexpected data";
                      }
                    else if (implementation_flag)
                      errstr = "duplicate implementation flag in GPI file `%s'";
                    implementation_flag = 1;
                    break;
                  default:
                    errstr = "unknown chunk in GPI file `%s'";
                    abort_flag = 1;  /* file may be damaged, try to avoid crashing */
                }
              if (str && flag_debug_gpi)
                fprintf (stderr, "GPI loaded %s: %s\n", gpi_chunk_names[code], str);
            }
          if (errstr)
            ;
          else if (!version_set)
            errstr = "GPI file `%s' does not contain version chunk";
          else if (!module_name_set)
            errstr = "GPI file `%s' does not contain module name chunk";
          else if (!module_filename)
            errstr = "GPI file `%s' does not contain source file name chunk";
          else if (start_of_nodes < 0)
            errstr = "GPI file `%s' does not contain nodes chunk";
          else if (start_of_offsets < 0)
            errstr = "GPI file `%s' does not contain offsets chunk";
          else
            {
              mseek (s, start_of_offsets);  /* Position the file for a later load_tree() */

              if (automake_level > 1 && !must_recompile)
                /* Check modules used by the one being imported. */
                for (imported = import_list; imported && !must_recompile;
                     imported = TREE_CHAIN (imported))
                  {
                    MEMFILE *u;
                    char *p, *u_name = concat (IDENTIFIER_POINTER (TREE_VALUE (imported)), ".gpi", NULL_PTR);
                    for (p = u_name; *p; p++)
                      *p = tolower (*p);
                    u = gpi_open (TREE_VALUE (imported), u_name, NULL, automake_level, NULL, NULL, NULL);
                    if (u)
                      mclose (u);
                    else
                      must_recompile = 1;
                    free (u_name);
                  }
            }
        }
      if (errstr)
        {
          warning (errstr, name);
          module_filename = NULL;
          must_recompile = 1;
        }

      /* @@@@ The code of module_must_be_recompiled() should be moved to the right places
              within this function. E.g., autobuild should be checked at the beginning.
              But it must check if the file is already being compiled, so it should
              probably be near the actual recompilation code below (e.g., it's not good
              to call file_is_being_compiled() twice (see compile_module()). Since
              automake is going to disappear in the future, the trouble may not be
              worth doing it. -- Frank */
      if (!(automake_level > 1
            && !(source_found && file_is_being_compiled (source_name, 1))
            && (must_recompile || (source_found && module_must_be_recompiled (interface_name, temp_name, source_name, import_list)))))
        {
          char *object_filename;

          if (errstr)
            /* We get here if an error was just output, but recompilation is
               prevented by the user or the fact that we're retrying already,
               or the module is in a file already being compiled. */
            {
              error ("cannot recompile module");
              exit (FATAL_EXIT_CODE);
            }

          name = temp_name;  /* Don't do this until here when we know that the GPI file is valid */

          /* Tell the linker about the object file. */
          object_filename = locate_object_file (source_name);
          if (object_filename)
            {
              add_to_link_file_list (object_filename);
              free (object_filename);
            }
          /* else @@ wrong if in current file (mod9.pas)
            error ("object file `%s' not found", object_filename); */

          /* Record modules used by the one being imported as
             "implementation imports" for the one being compiled.
             Also mark them as being imported implicitly. */
          for (imported = import_list; imported; imported = TREE_CHAIN (imported))
            {
              itab_check_gpi_checksum (TREE_VALUE (imported), (gpi_int) TREE_TYPE (imported), 1);
#if 0         /* I think we don't need to keep track of indirectly imported interfaces.
                 @@ If this asssumption is right, we can probably remove everything
                    related to TREE_PRIVATE with `imports'. -- Frank */
              TREE_STATIC (imported) = 1;
              TREE_PRIVATE (imported) = 1;
            }
          current_module->imports = chainon (current_module->imports, import_list);
#else
            }
#endif

          /* Copy names of files to be linked to the AutoMake temp file. */
          while (link_list)
            {
              add_to_link_file_list (link_list->string);
              link_list = link_list->next;
            }
        }
      else
        {
          mclose (s);
          s = NULL;
          if (flag_debug_gpi)
            fprintf (stderr, "recompiling: %s -> %s\n", module_filename, name);
          /* "fallthrough" */
        }
    }
  if (!s && automake_level > 1)
    {
      int result = -1;
      if (!module_filename)
        {
          source_name = locate_interface_source (name, source, NULL_PTR);
          module_filename = save_string (name);
        }
      if (source_name)
        result = compile_module (source_name, unit_destination_path);
      if (result == 0)
        /* Module has been compiled. Reload the GPI file. Don't compile again. */
        /* @@ Replace recursive call with a loop */
        s = gpi_open (interface_name, name, source, 0, p_start_of_nodes, p_size_of_nodes, p_size_of_offsets);
      else if (source)
        {
          if (flag_automake > 1)
            {
              error ("module/unit `%s' could not be compiled", source);
              exit (FATAL_EXIT_CODE);
            }
        }
      else
        {
          /* Cut the extension `.gpi' */
          char *p = module_filename + strlen (module_filename);
          do
            p--;
          while (p > module_filename && *p != '.');
          *p = 0;
          if (flag_automake > 1)
            {
              error ("module/unit `%s' could not be compiled", module_filename);
              exit (FATAL_EXIT_CODE);
            }
        }
    }
  if (temp_name)
    free (temp_name);
  if (module_filename)
    free (module_filename);
  if (source_name)
    free (source_name);
  return s;
}

/* Try to load a GPI file and to extract an exported module interface. */
static tree
load_gpi_file (interface_name, name, source)
     tree interface_name;
     char *name;
     char *source;
{
  MEMFILE *gpi_file;
  char *p, *current_gpi_file_name;
  gpi_int start_of_nodes, size_of_nodes, size_of_offsets;
  tree export;
  tree we_were_loading_a_gpi_file = loading_gpi_file;
  loading_gpi_file = interface_name;
  current_gpi_file_name = concat (name, ".gpi", NULL_PTR);
  for (p = current_gpi_file_name; *p; p++)
    *p = tolower (*p);
  gpi_file = gpi_open (interface_name, current_gpi_file_name, source, flag_automake, &start_of_nodes, &size_of_nodes, &size_of_offsets);
  if (gpi_file)
    {
      tree temp;
      gpi_int checksum;
      int oldpos = mtell (gpi_file);
      mseek (gpi_file, start_of_nodes + size_of_nodes - sizeof (checksum));
      LOAD_ANY_F (gpi_file, checksum);
      mseek (gpi_file, oldpos);
      if (compute_checksum (mptr (gpi_file, start_of_nodes), size_of_nodes - sizeof (checksum)) != checksum)
        {
          error ("%s: checksum mismatch (GPI file corrupt)", current_gpi_file_name);
          exit (FATAL_EXIT_CODE);
        }
      itab_check_gpi_checksum (interface_name, checksum, 1);
      temp = load_tree (gpi_file, start_of_nodes, size_of_offsets);
      export = build_tree_list (TREE_CHAIN (temp), TREE_VALUE (temp));
      mclose (gpi_file);
    }
  else
    export = NULL_TREE;
  free (current_gpi_file_name);
  loading_gpi_file = we_were_loading_a_gpi_file;
  return export;
}

/* Activate a node during import */
static void
import_node (src, import_rename, export_rename, accept_readonly)
     tree src, import_rename, export_rename;
     int accept_readonly;
{
  tree t = IDENTIFIER_LIMBO_VALUE (src), dest;
  if (import_rename)
    dest = import_rename;
  else if (export_rename)
    dest = export_rename;
  else
    dest = src;
  if (accept_readonly && TREE_READONLY (src))
    TREE_READONLY (t) = 1;
  /* @@ Push only those decls that were *not* pushed already during
        load_node() to avoid duplicates. Finally, load_node() should
        not push any of them, so we should do it here unconditionally. */
#ifdef NEW_DECLARE_VARS
  if (TREE_CODE (t) != FUNCTION_DECL)
#else
  if (TREE_CODE (t) != VAR_DECL && TREE_CODE (t) != FUNCTION_DECL)
#endif
    pushdecl_current_nocheck (t);
  if (current_module && current_module->main_program)
    IDENTIFIER_GLOBAL_VALUE (dest) = t;
  else
    IDENTIFIER_LOCAL_VALUE (dest) = t;
  IDENTIFIER_LIMBO_VALUE (src) = NULL_TREE;
}

/* Try to load a GPM file and to extract the interface part of this module. */
int
load_gpm_file ()
{
  MEMFILE *gpm_file;
  char gpm_header_line[sizeof (GPM_HEADER)], *gpm_file_name, *temp_name, *p;

  gpm_file_name = concat (IDENTIFIER_POINTER (current_module->name), ".gpm", NULL_PTR);
  for (p = gpm_file_name; *p; p++)
    *p = tolower (*p);

  temp_name = locate_file (gpm_file_name, LF_COMPILED_UNIT);
  if (temp_name)
    {
      free (gpm_file_name);
      gpm_file_name = temp_name;
    }
  if (! gpm_file_name)
    return -1;

  gpm_file = mopen_read (gpm_file_name);
  if (! gpm_file)
    return -1;
  if (flag_debug_gpi)
    fprintf (stderr, "opened GPM file: %s\n", gpm_file_name);

  LOAD_LENGTH_F (gpm_file, gpm_header_line, strlen (GPM_HEADER));
  gpm_header_line[strlen (GPM_HEADER)] = 0;
  if (strcmp (gpm_header_line, GPM_HEADER) != 0)
    {
      /* Don't warn if only the version doesn't match. Let it be
         silently recompiled. */
      if (strncmp (gpm_header_line, GPM_HEADER, strlen (GPM_MAGIC)) != 0)
        warning ("GPM file header error");
      mclose (gpm_file);
      return -1;
    }
  if (flag_debug_gpi)
    fprintf (stderr, "GPM loaded header: %s", gpm_header_line);
  while (!meof (gpm_file))
    {
      tree exported;
      char *gpm_line = load_string (gpm_file);
      tree interface_name = get_identifier (gpm_line);
      if (flag_debug_gpi)
        fprintf (stderr, "GPM loaded interface name: %s\n", gpm_line);
      exported = load_gpi_file (interface_name, gpm_line, NULL);
      if (exported)
        {
          tree ename;
          for (ename = TREE_PURPOSE (exported); ename;
               ename = TREE_CHAIN (ename))
            /* "Activate" all names. */
            if (TREE_VALUE (ename))
              import_node (TREE_VALUE (ename), NULL_TREE, NULL_TREE, 0);
          export_interface (interface_name, TREE_PURPOSE (exported));
        }
      free (gpm_line);
    }
  mclose (gpm_file);
  return 0;
}

/* Import an interface of a unit or module.  Look up and read the
 * GPI file and import either everything or only the requested parts
 * exported by the interface.
 *
 * INTERFACE is an IDENTIFIER_NODE of the interface name.
 *
 * IMPORT_QUALIFIER:
 *     NULL_TREE if no qualifiers given.
 *     TREE_LIST:
 *       TREE_PURPOSE:
 *           NULL_TREE  -> no restricted_import_option given
 *           !NULL_TREE -> restricted_import_option (== `only') given
 *       TREE_VALUE:
 *           TREE_LIST
 *             TREE_PURPOSE: imported <name> from the interface
 *             TREE_VALUE:
 *                 NULL_TREE: name has not been renamed
 *                 identifier_node: new name of the renamed <name>
 *
 * QUALIFIED_IMPORT is 0 if unqualified references are allowed;
 *                     1 if qualified references are mandatory.
 *
 * UNIT_FILENAME is an optional IDENTIFIER_NODE holding the name of the
 * source file of this unit. (Not used with modules.)
 */
void
import_interface (interface, import_qualifier, qualified_import, unit_filename)
     tree interface;
     tree import_qualifier;
     int qualified_import;
     tree unit_filename;
{
  tree exported_name_list, imported, exported;

  /* Handle a special case of a circular dependency. */
  if (interface == current_module->name)
    {
      if (current_module->main_program)
        error ("program trying to import itself like a module or unit");
      else
        error ("self-dependent module or unit");
      return;
    }

  for (imported = current_module->imports; imported;
       imported = TREE_CHAIN (imported))
    if (interface == TREE_VALUE (imported) && !TREE_PRIVATE (imported))
      {
        error ("interface `%s' already has been imported",
               IDENTIFIER_POINTER (interface));
        return;
      }

  /* Mark whether this interface was imported in the interface or
   * implementation part.
   */
  if (! imported)
    {
      imported = build_tree_list (unit_filename, interface);
      current_module->imports = chainon (current_module->imports, imported);
    }
  TREE_STATIC (imported) = ! this_is_an_interface_module;
  TREE_PRIVATE (imported) = 0;

  if (interface == standard_interface_input)
    {
      current_module->input_file_node = global_input_file_node;
      exported_name_list = build_tree_list (NULL_TREE, identifier_input);
      (void) get_interface_table (interface, interface, 1);
    }
  else if (interface == standard_interface_output)
    {
      current_module->output_file_node = global_output_file_node;
      exported_name_list = build_tree_list (NULL_TREE, identifier_output);
      (void) get_interface_table (interface, interface, 1);
    }
  else if (interface == standard_interface_error)
    {
      if (pedantic || flag_what_pascal)
        warning ("ISO Pascal does not define `StandardError'");
      current_module->error_file_node = global_error_file_node;
      exported_name_list = build_tree_list (NULL_TREE, identifier_stderr);
      (void) get_interface_table (interface, interface, 1);
    }
  else
    {
      for (exported = exported_interface_list; exported; exported = TREE_CHAIN (exported))
        if (TREE_VALUE (exported) == interface)
          break;

      if (! exported)
        {
          char *gpi_source;
          if (unit_filename)
            gpi_source = TREE_STRING_POINTER (unit_filename);
          else
            gpi_source = NULL;
          exported = load_gpi_file (interface, IDENTIFIER_POINTER (interface),
                                    gpi_source);
          if (exported)
            {
              exported_interface_list = chainon (exported_interface_list,
                                                 exported);
              internal_exported_interface_list = chainon (internal_exported_interface_list,
                                                          build_tree_list (NULL_TREE, interface));
            }
          else
            {
              error ("module/unit interface `%s' could not be imported",
                     IDENTIFIER_POINTER (interface));
              exit (FATAL_EXIT_CODE);
              return;
            }
        }

      exported_name_list = TREE_PURPOSE (exported);
    }

  if (qualified_import)
    warning ("`qualified' not yet supported - ignored");

  /* EXPORTED is now the correct interface. */

  if (import_qualifier)
    {
      tree iname;
      for (iname = TREE_VALUE (import_qualifier); iname;
           iname = TREE_CHAIN (iname))
        {
          tree ename = exported_name_list;

          while (ename && TREE_VALUE (ename) != TREE_PURPOSE (iname))
            ename = TREE_CHAIN (ename);

          /* Activate this name and resolve import/export renaming. */
          if (ename && TREE_VALUE (ename) == TREE_PURPOSE (iname))
            import_node (TREE_VALUE (ename), TREE_VALUE (iname), TREE_PURPOSE (ename), 1);
          else
            error ("interface `%s' does not export `%s'",
                   IDENTIFIER_POINTER (interface),
                   IDENTIFIER_POINTER (TREE_PURPOSE (iname)));
        }
    }

  if (import_qualifier == NULL_TREE  /* without qualifiers */
      || TREE_PURPOSE (import_qualifier) == NULL_TREE)  /* without `only' */
    {
      tree ename;
      for (ename = exported_name_list; ename; ename = TREE_CHAIN (ename))
        /* "Activate" all remaining names and resolve export renaming. */
        if (IDENTIFIER_LIMBO_VALUE (TREE_VALUE (ename)))
          import_node (TREE_VALUE (ename), NULL_TREE, TREE_PURPOSE (ename), 1);
    }
}

/* Return a TREE_LIST to be chained to an interface export list
 * containing the name NAME (an IDENTIFIER_NODE) of the thing to be
 * exported plus, if given, the qualifiers RENAMED (an IDENTIFIER_NODE
 * holding the new name) and PROTECTED.
 *
 * We use TREE_READONLY of an IDENTIFIER_NODE to indicate `protected'.
 *
 * Only exportable names allowed as the NAME here:
 *        constant_name
 *      | type_name
 *      | schema_name
 *      | procedure_name
 *      | function_name
 *      | variable_name
 *      | `protected' variable_name
 */
tree
module_export_clause (name, renamed, protected)
     tree name;
     tree renamed;
     int  protected;
{
  tree rval = build_tree_list (renamed, name);
  TREE_READONLY (name) = protected;
  return rval;
}

/* Export a range of an enumerated type.  These need special care
 * because they contain identifiers in between which must be exported
 * as well. See module_expand_exported_ranges.
 */
tree
module_export_range (low, high)
     tree low;
     tree high;
{
  return module_export_clause (build_tree_list (low, high), 0, 0);
}

/* Replace the exported TREE_LISTs denoting ranges by the actual identifier nodes. */
static void
module_expand_exported_ranges (list)
     tree list;
{
  tree nscan;
  for (nscan = list; nscan; nscan = TREE_CHAIN (nscan))
    if (TREE_VALUE (nscan) && TREE_CODE (TREE_VALUE (nscan)) == TREE_LIST)
      {
        tree low = TREE_PURPOSE (TREE_VALUE (nscan));
        tree high = TREE_VALUE (TREE_VALUE (nscan));
        tree type = TREE_TYPE (lookup_name (low));
        tree exported = NULL_TREE;
        tree item;
        int export_it = 0;
        assert (TREE_CODE (type) == ENUMERAL_TYPE && TREE_TYPE (lookup_name (high)) == type);
        for (item = TYPE_VALUES (type); item; item = TREE_CHAIN (item))
          {
            if (TREE_PURPOSE (item) == low)
              export_it = 1;
            if (export_it)
              exported = chainon (exported,
                           build_tree_list (NULL_TREE, TREE_PURPOSE (item)));
            if (TREE_PURPOSE (item) == high)
              {
                if (! export_it)
                  error ("wrong order in exported range");
                export_it = 0;
              }
          }
        /* Chain EXPORTED to NSCAN. */
        if (exported)
          {
            tree new_nscan = exported;
            while (TREE_CHAIN (new_nscan))
              new_nscan = TREE_CHAIN (new_nscan);
            TREE_CHAIN (nscan) = chainon (exported, TREE_CHAIN (nscan));
            TREE_VALUE (nscan) = TREE_PURPOSE (nscan) = NULL_TREE;  /* Remove the range TREE_LIST */
            nscan = new_nscan;
          }
      }
}
