/* glickomania.c -- a puzzle game.

   Copyright (C) 2002 Kenneth Oksanen
   Copyright (C) 1992, 1995, 1996, 1997 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* AIX requires this to be the first thing in the file.  */
#if defined (_AIX) && !defined (__GNUC__)
 #pragma alloca
#endif

#include "includes.h"
#include "game_logic.h"


static GtkWidget *mainwindow, *box1, *draw_area, *statusbar;
static guint statusbar_context_id;


static void update_statusbar(void)
{
#define BUF_SIZE 512
  char buf[BUF_SIZE + 1];

  gtk_statusbar_pop(GTK_STATUSBAR(statusbar), statusbar_context_id);
  problem_stats(buf, BUF_SIZE);
  gtk_statusbar_push(GTK_STATUSBAR(statusbar), statusbar_context_id,
		     buf);
}


typedef enum {
  SEPARATE = 0,
  JOINED = 1,
  MERGED = 2,
  N_TILE_STYLES = 3
} tile_style_t;

static tile_style_t tile_style = JOINED;

static struct {
  unsigned char red, green, blue;
  enum { UNMADE_COLOR, OWN_COLOR, SHARED_COLOR } sharing;
  GdkColor color;
  GdkGC *gc;
} tile_color[MAX_N_TILES] = {
  { 255,   0,   0, UNMADE_COLOR },
  {   0, 255,   0, UNMADE_COLOR },
  {   0,   0, 255, UNMADE_COLOR },
  { 255, 255,   0, UNMADE_COLOR },
  {   0, 255, 255, UNMADE_COLOR },
  { 255,   0, 255, UNMADE_COLOR },
  { 153, 130, 178, UNMADE_COLOR },
#if 0
  { 178, 140,  45, UNMADE_COLOR },
  { 119, 168, 178, UNMADE_COLOR },
#endif
};

static void build_colors(void)
{
  int i;
  GdkGC *gc;
  GdkColor color;
  GdkColormap *colormap = gdk_colormap_get_system();
  
  /* Make background color back. */
  color.red = color.green = color.blue = 0;	/* Black */
  gdk_colormap_alloc_color(colormap, &color, FALSE, TRUE);
  gdk_window_set_background(draw_area->window, &color);

  for (i = 0; i < MAX_N_TILES; i++) {
    if (tile_color[i].sharing == SHARED_COLOR)
      gdk_colormap_free_colors(colormap, &tile_color[i].color, 1);
    tile_color[i].color.red = 256 * tile_color[i].red;
    tile_color[i].color.green = 256 * tile_color[i].green;
    tile_color[i].color.blue = 256 * tile_color[i].blue;
    if (gdk_colormap_alloc_color(colormap, &tile_color[i].color,
				 FALSE, TRUE))
      tile_color[i].sharing = SHARED_COLOR;
    else
      tile_color[i].sharing = OWN_COLOR;
    
    gc = gdk_gc_new(draw_area->window);
    gdk_gc_set_fill(gc, GDK_SOLID);
    gdk_gc_set_foreground(gc, &tile_color[i].color);
    
    tile_color[i].gc = gc;
  }
}


static int no_home_dir_is_warned = 0;

static FILE *open_rc_file(const char *mode)
{
  char buf[BUF_SIZE+1], *dirname;
  const char *basename = "/.glickomania";

  dirname = getenv("HOME");
  if (dirname == NULL) {
    struct passwd *pw = getpwuid(getuid());
    if (pw != NULL)
      dirname = pw->pw_dir;
  }
  if (dirname == NULL) {
    if (!no_home_dir_is_warned) {
      no_home_dir_is_warned = 1;
      msgbox("Could not locate ~/.glickomania");
    }
    return NULL;
  }

  strncpy(buf, dirname, BUF_SIZE - strlen(basename));
  strcat(buf, basename);

  return fopen(buf, mode);
}


static void save(void)
{
  int i;
  FILE *fp;

  fp = open_rc_file("w");
  if (fp == NULL) {
    static int rc_file_write_is_warned = 0;
    if (!no_home_dir_is_warned && !rc_file_write_is_warned) {
      no_home_dir_is_warned = 1;
      msgbox("Could write file ~/.glickomania");
    }
    return;
  }
  fprintf(fp, "GLICKOMANIA %u.%u\n",
	  GLICKOMANIA_VERSION_MAJOR, GLICKOMANIA_VERSION_MINOR);
  
  save_problems(fp);

  fprintf(fp, "TILE STYLE %u\n", (unsigned int) tile_style);
  for (i = 0; i < MAX_N_TILES; i++)
    fprintf(fp, "TILE COLOR %u %u %u\n", 
	    tile_color[i].red, tile_color[i].green, tile_color[i].blue);
  /* Other preferences may be implemented in future. */

  fclose(fp);
}


static void load(void)
{
  int i;
  FILE *fp;
  unsigned int version_major, version_minor;
  static int rc_file_corrupted_is_warned = 0;
  char buf[BUF_SIZE];

  fp = open_rc_file("r");
  if (fp == NULL)
    return;
  if (fscanf(fp, "GLICKOMANIA %u.%u\n",
	     &version_major, &version_minor) != 2) {
    if (!rc_file_corrupted_is_warned) {
      rc_file_corrupted_is_warned = 1;
      msgbox("~/.glickomania is corrupted");
    }
    goto barf;
  }
  if (version_major > GLICKOMANIA_VERSION_MAJOR
      || (version_major == GLICKOMANIA_VERSION_MAJOR
	  && version_minor > GLICKOMANIA_VERSION_MINOR)) {
    if (!rc_file_corrupted_is_warned) {
      rc_file_corrupted_is_warned = 1;
      sprintf(buf, "~/.glickomania written by a glickomania-%u.%u, which is "
	      " newer than my version %u.%u.  I can not read the file.\n\n"
	      "I shall, however, overwrite the file if you choose to continue "
	      " the game.",
	      version_major, version_minor,
	      GLICKOMANIA_VERSION_MAJOR, GLICKOMANIA_VERSION_MINOR);
      msgbox(buf);
    }
    goto barf;
  }

  if (!load_problems(fp)) {
    if (!rc_file_corrupted_is_warned) {
      rc_file_corrupted_is_warned = 1;
      msgbox("~/.glickomania is corrupted");
    }
    goto barf;
  }

  if (fscanf(fp, "TILE STYLE %u\n", (unsigned int *) &tile_style) != 1
      || tile_style >= N_TILE_STYLES) {
    tile_style = MERGED;
    if (!rc_file_corrupted_is_warned) {
      rc_file_corrupted_is_warned = 1;
      msgbox("~/.glickomania is corrupted");
    }
    goto barf;
  }
  for (i = 0; i < MAX_N_TILES; i++) {
    unsigned int r, g, b;

    if (fscanf(fp, "TILE COLOR %u %u %u\n", &r, &g, &b) != 3
	|| r > 255 || g > 255 || b > 255) {
      if (!rc_file_corrupted_is_warned) {
	rc_file_corrupted_is_warned = 1;
	msgbox("~/.glickomania is corrupted");
      }
      goto barf;
    }
    tile_color[i].red = r;
    tile_color[i].green = g;
    tile_color[i].blue = b;
  }
  /* Other preferences may come in future. */

 barf:
  fclose(fp);
}


/* XXX Make these adjustable parameters. */
#define TILE          40
#define JOINT_WIDTH   14
#define MARGIN        4
#define PLACE         (TILE + 2 * MARGIN)

#define WIDTH         (width * PLACE + 2 * MARGIN)
#define HEIGHT        (height * PLACE + 2 * MARGIN)


static void redraw_board(void)
{
  int x, y;
  int xc, yc;

  gdk_window_clear_area(draw_area->window,
			0, 0, WIDTH, HEIGHT);

  for (xc = MARGIN, x = 0; x < width; x++, xc += PLACE)
    for (yc = MARGIN, y = height - 1; y >= 0; y--, yc += PLACE) {
      tile_t tile = board[x][y];
      GdkGC *gc = tile_color[tile].gc;
      GdkDrawable *w = draw_area->window;
      int mrg = (tile_style == JOINED) ? (TILE-JOINT_WIDTH)/2 : 0;

      if (tile == EMPTY)
	gdk_window_clear_area(w, xc, yc, PLACE, PLACE);
      else {

	gdk_draw_rectangle(w, gc, TRUE,
			   xc + MARGIN, yc + MARGIN,
			   TILE, TILE);
	if (tile_style == JOINED || tile_style == MERGED) {
	  if (x > 0) {
	    if (board[x-1][y] == tile) {
	      gdk_draw_rectangle(w, gc, TRUE,
				 xc, yc + MARGIN + mrg,
				 MARGIN, TILE - 2*mrg);
	      if (tile_style == MERGED) {
		if (y < height-1
		    && board[x-1][y+1] == tile
		    && board[x][y+1] == tile)
		  gdk_draw_rectangle(w, gc, TRUE,
				     xc, yc,
				     MARGIN, MARGIN);
		if (y > 0
		    && board[x-1][y-1] == tile
		    && board[x][y-1] == tile)
		  gdk_draw_rectangle(w, gc, TRUE,
				     xc, yc + MARGIN + TILE,
				     MARGIN, MARGIN);
	      }
	    }
	  }
	  if (x < width-1) {
	    if (board[x+1][y] == tile) {
	      gdk_draw_rectangle(w, gc, TRUE,
				 xc + MARGIN + TILE, yc + MARGIN + mrg,
				 MARGIN, TILE - 2*mrg);
	      if (tile_style == MERGED) {
		if (y < height-1
		    && board[x+1][y+1] == tile
		    && board[x][y+1] == tile)
		  gdk_draw_rectangle(w, gc, TRUE,
				     xc + MARGIN + TILE, yc,
				     MARGIN, MARGIN);
		if (y > 0
		    && board[x+1][y-1] == tile
		    && board[x][y-1] == tile)
		  gdk_draw_rectangle(w, gc, TRUE,
				     xc + MARGIN + TILE, yc + MARGIN + TILE,
				     MARGIN, MARGIN);
	      }
	    }
	  }
	  if (y < height-1 && board[x][y+1] == tile)
	    gdk_draw_rectangle(w, gc, TRUE,
			       xc + MARGIN + mrg, yc,
			       TILE - 2*mrg, MARGIN);
	  if (y > 0 && board[x][y-1] == tile)
	    gdk_draw_rectangle(w, gc, TRUE,
			       xc + MARGIN + mrg, yc + MARGIN + TILE,
			       TILE - 2*mrg, MARGIN);
	}
      }
    }
}


static gint
drawing_area_event_handler(GtkWidget *widget, GdkEvent *event, void *ctx)
{
  int x, y;
  GdkEventExpose *e;

  switch (event->type) {
  case GDK_EXPOSE:
    redraw_board();
    return TRUE;

  case GDK_BUTTON_PRESS:
    gtk_widget_get_pointer(widget, &x, &y);
    x = x / PLACE;
    y = height - y / PLACE - 1;
    switch (make_move(x, y)) {
    case 0:
      /* Move not allowed. */
      break;
    case 3:
      {
	char buf[BUF_SIZE+1];

	snprintf(buf, BUF_SIZE,
		 "Congratulations!  You have solved all the problems "
		 "glickomania-%u.%u has to offer!\n\n"
		 "The author would be very interested to hear of your "
		 "gaming experience.  "
		 "Please mail a few words to <cessu@iki.fi>",
		 GLICKOMANIA_VERSION_MAJOR, GLICKOMANIA_VERSION_MINOR);
	msgbox(buf);
      }
      /*FALLTHROUGH*/
    case 2:
      /* The board got cleared. */
      save();
      update_statusbar();
      /*FALLTHROUGH*/
    default:
      redraw_board();
      break;
    }
    return TRUE;

  default:
    return FALSE;
  }
}


static void new_problem(void)
{
  gtk_drawing_area_size(GTK_DRAWING_AREA(draw_area),
			WIDTH, HEIGHT);
  redraw_board();
  update_statusbar();
}


static void init_board(void)
{
  int i;

  gtk_widget_push_visual(gdk_rgb_get_visual());
  gtk_widget_push_colormap(gdk_rgb_get_cmap());
  
  draw_area = gtk_drawing_area_new();
  
  gtk_widget_pop_colormap();
  gtk_widget_pop_visual();

  gtk_widget_set_events(draw_area,
			gtk_widget_get_events(draw_area)
			  | GDK_EXPOSURE_MASK | GDK_BUTTON_PRESS_MASK);
  gtk_box_pack_start_defaults(GTK_BOX(box1), draw_area);
  gtk_widget_realize(draw_area);
  gtk_widget_show(draw_area);

  gtk_signal_connect(GTK_OBJECT(draw_area), "event",
		     (GtkSignalFunc) drawing_area_event_handler, 0);
}


static void munmake_cb(GtkMenuItem *menuitem,
			   gpointer user_data)
{
  if (unmake_move())
    redraw_board();
}

static void mremake_cb(GtkMenuItem *menuitem,
			   gpointer user_data)
{
  if (remake_move())
    redraw_board();
}

static void mprev_unsolved_cb(GtkMenuItem *menuitem,
				  gpointer user_data)
{
  if (prev_unsolved_problem())
    new_problem();
}

static void mprev_cb(GtkMenuItem *menuitem,
			 gpointer user_data)
{
  if (prev_problem()) 
    new_problem();
}

static void mnext_cb(GtkMenuItem *menuitem,
			 gpointer user_data)
{
  if (next_problem())
    new_problem();
}

static void mnext_unsolved_cb(GtkMenuItem *menuitem,
				  gpointer user_data)
{
  if (next_unsolved_problem())
    new_problem();
}


static void mset_tile_separate_cb(GtkMenuItem *menuitem,
				  gpointer user_data)
{
  tile_style = SEPARATE;
  redraw_board();
}

static void mset_tile_joined_cb(GtkMenuItem *menuitem,
				gpointer user_data)
{
  tile_style = JOINED;
  redraw_board();
}

static void mset_tile_merged_cb(GtkMenuItem *menuitem,
				gpointer user_data)
{
  tile_style = MERGED;
  redraw_board();
}


static tile_t recolored_tile;
GtkColorSelectionDialog *color_selection_dialog = NULL;

void color_changed_cb(GtkWidget *widget, GtkColorSelection *cs)
{
  gdouble color[3];

  gtk_color_selection_get_color(cs, color);
  tile_color[recolored_tile].red =   (int) (color[0] * 255.0);
  tile_color[recolored_tile].green = (int) (color[1] * 255.0);
  tile_color[recolored_tile].blue =  (int) (color[2] * 255.0);
  build_colors();
  redraw_board();
}


void destroy_color_selection_dialog(GtkWidget *widget, gpointer dummy)
{
  gtk_widget_destroy(GTK_WIDGET(color_selection_dialog));
  save();
  color_selection_dialog = NULL;
}


static void colorsel(tile_t tile)
{
  GtkWidget *cs;
  char buf[BUF_SIZE + 1];

  if (color_selection_dialog != NULL) {
    /* This implies there is already an active color selector */
    msgbox("First finish your previous color selection.");
    return;
  }
    
  recolored_tile = tile;
  snprintf(buf, BUF_SIZE, "Select color for tile #%d", recolored_tile + 1);
  color_selection_dialog =
    GTK_COLOR_SELECTION_DIALOG(gtk_color_selection_dialog_new(buf));
  cs = color_selection_dialog->colorsel;
  gtk_signal_connect(GTK_OBJECT(cs), "color_changed",
		     (GtkSignalFunc) color_changed_cb, 
		     (gpointer) cs);
  gtk_widget_destroy(color_selection_dialog->cancel_button);
  gtk_widget_destroy(color_selection_dialog->help_button);
  gtk_signal_connect(GTK_OBJECT(color_selection_dialog->ok_button),
		     "clicked",
		     GTK_SIGNAL_FUNC(destroy_color_selection_dialog), NULL);
  gtk_widget_show(GTK_WIDGET(color_selection_dialog));
}

static void mcolorsel1_cb(GtkWidget *w, gpointer p) { colorsel(0); }
static void mcolorsel2_cb(GtkWidget *w, gpointer p) { colorsel(1); }
static void mcolorsel3_cb(GtkWidget *w, gpointer p) { colorsel(2); }
static void mcolorsel4_cb(GtkWidget *w, gpointer p) { colorsel(3); }
static void mcolorsel5_cb(GtkWidget *w, gpointer p) { colorsel(4); }
static void mcolorsel6_cb(GtkWidget *w, gpointer p) { colorsel(5); }
static void mcolorsel7_cb(GtkWidget *w, gpointer p) { colorsel(6); }

#if MAX_N_TILES != 7
#error Something is missing here...
#endif


static void mainwindow_destroy(GtkObject *object,
			       gpointer user_data)
{
  gtk_main_quit();
}


static void mquit_cb(GtkMenuItem *menuitem,
		     gpointer user_data)
{
  save();
  gtk_widget_destroy(mainwindow);
}


static void mreset_cb(GtkMenuItem *menuitem,
		      gpointer user_data)
{
  reset_problems();
  save();
  update_statusbar();
}


static void mabout_cb(GtkMenuItem *menuitem,
		      gpointer user_data)
{
  char buf[BUF_SIZE+1];

  snprintf(buf, BUF_SIZE,
	   "Glickomania version %d.%d\n\n"
	   "Written by Kenneth Oksanen\n"
	   "<cessu@iki.fi>\n\n"
	   "This software is distributed under\n"
	   "General Public License (GPL)",
	   GLICKOMANIA_VERSION_MAJOR, GLICKOMANIA_VERSION_MINOR);
  msgbox(buf);
}

static void mrules_cb(GtkMenuItem *menuitem,
		      gpointer user_data)
{
  msgbox("Click on any of two or more (vertically or horizontally) "
	 "adjacent tiles of same color to remove them.  Tiles fall "
	 "downwards to fill empty gaps, and from right to left to fill empty "
	 "columns. \n"
	 "\n"
	 "The objective is to clear the board of all tiles.  There is no "
	 "time limit and you can undo and redo without penalty.\n"
	 "\n"
	 "All problems have been verified to be solvable.");
}


/* (This comment is from GTK documentation.)

   This is the GtkItemFactoryEntry structure used to generate new
   menus:
   Item 1: The menu path. The letter after the underscore indicates an
           accelerator key once the menu is open.
   Item 2: The accelerator key for the entry
   Item 3: The callback function.
   Item 4: The callback action.  This changes the parameters with
           which the function is called.  The default is 0.
   Item 5: The item type, used to define what kind of an item it is.
           Here are the possible values:

     NULL               -> "<Item>"
     ""                 -> "<Item>"
     "<Title>"          -> create a title item
     "<Item>"           -> create a simple item
     "<CheckItem>"      -> create a check item
     "<ToggleItem>"     -> create a toggle item
     "<RadioItem>"      -> create a radio item
     <path>             -> path of a radio item to link against
     "<Separator>"      -> create a separator
     "<Branch>"         -> create an item to hold sub items (optional)
     "<LastBranch>"     -> create a right justified branch */

static GtkItemFactoryEntry menu_items[] = {
  { "/_Game", 
    NULL, NULL, 0, "<Branch>" },
  { "/Game/_Undo",
    "u", munmake_cb, 0, NULL },
  { "/Game/_Redo",
    "r", mremake_cb, 0, NULL },
  { "/Game/sep1",
    NULL, NULL, 0, "<Separator>" },
  { "/Game/Previous",
    "<shift>p", mprev_cb, 0, NULL },
  { "/Game/Previous Unsolved",
    "p", mprev_unsolved_cb, 0, NULL },
  { "/Game/Next Unsolved",
    "n", mnext_unsolved_cb, 0, NULL },
  { "/Game/Next",
    "<shift>n", mnext_cb, 0, NULL },
  { "/Game/sep2",
    NULL, NULL, 0, "<Separator>" },
  { "/Game/Reset",
    NULL, mreset_cb, 0, NULL },
  { "/Game/_Quit",
    "Q", mquit_cb, 0, NULL },
  { "/_Preferences",
    NULL, NULL, 0, "<Branch>" },
  { "/Preferences/Tile Style",
    NULL, NULL, 0, "<Branch>" },
  { "/Preferences/Tile Style/Separate",
    NULL, mset_tile_separate_cb, 0, NULL },
  { "/Preferences/Tile Style/Joined",
    NULL, mset_tile_joined_cb, 0, NULL },
  { "/Preferences/Tile Style/Merged",
    NULL, mset_tile_merged_cb, 0, NULL },
  { "/Preferences/Tile Colors",
    NULL, NULL, 0, "<Branch>" },
  { "/Preferences/Tile Colors/Tile #1",
    NULL, mcolorsel1_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #2",
    NULL, mcolorsel2_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #3",
    NULL, mcolorsel3_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #4",
    NULL, mcolorsel4_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #5",
    NULL, mcolorsel5_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #6",
    NULL, mcolorsel6_cb, 0, NULL },
  { "/Preferences/Tile Colors/Tile #7",
    NULL, mcolorsel7_cb, 0, NULL },
#if MAX_N_TILES != 7
#error Something is missing here...
#endif
  { "/Help",
    NULL, NULL, 0, "<LastBranch>" },
  { "/Help/About",
    NULL, mabout_cb, 0, NULL },
  { "/Help/Rules",
    NULL, mrules_cb, 0, NULL },
};


/* (Also the function below is basically from the GTK documentation.)
 */
void get_main_menu(GtkWidget  *window, GtkWidget **menubar)
{
  GtkItemFactory *item_factory;
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);
  
  accel_group = gtk_accel_group_new();
  
  /* This function initializes the item factory.
     Param 1: The type of menu - can be GTK_TYPE_MENU_BAR, GTK_TYPE_MENU,
              or GTK_TYPE_OPTION_MENU.
     Param 2: The path of the menu.
     Param 3: A pointer to a gtk_accel_group.  The item factory sets up
              the accelerator table while generating menus. */
  item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
				      accel_group);
  
  /* This function generates the menu items. Pass the item factory,
     the number of items in the array, the array itself, and any
     callback data for the the menu items. */
  gtk_item_factory_create_items(item_factory, nmenu_items, menu_items, NULL);
  
  /* Attach the new accelerator group to the window. */
  gtk_accel_group_attach(accel_group, GTK_OBJECT(window));
  
  if (menubar)
    /* Finally, return the actual menu bar created by the item factory. */ 
    *menubar = gtk_item_factory_get_widget(item_factory, "<main>");
}


static void glickomain(void)
{
  GtkWidget *menubar;

  mainwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_widget_set_name(mainwindow, "mainwindow");
  gtk_object_set_data(GTK_OBJECT(mainwindow), "mainwindow", mainwindow);
  gtk_window_set_title(GTK_WINDOW(mainwindow), "glickomania");
  gtk_window_set_policy(GTK_WINDOW(mainwindow), TRUE, TRUE, TRUE);

  gtk_signal_connect(GTK_OBJECT(mainwindow), "destroy",
		     GTK_SIGNAL_FUNC(mainwindow_destroy),
		     NULL);

  box1 = gtk_vbox_new(FALSE, 0);
  gtk_widget_set_name(box1, "box1");
  gtk_widget_ref(box1);
  gtk_object_set_data_full(GTK_OBJECT(mainwindow), "box1", box1,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(box1);
  gtk_container_add(GTK_CONTAINER (mainwindow), box1);

  get_main_menu(mainwindow, &menubar);
  gtk_box_pack_start(GTK_BOX(box1), menubar, FALSE, TRUE, 0);
  gtk_widget_show (menubar);

  init_board();
  build_colors();

  statusbar = gtk_statusbar_new();
  gtk_widget_set_name(statusbar, "statusbar");
  gtk_widget_ref(statusbar);
  gtk_object_set_data_full(GTK_OBJECT (mainwindow), "statusbar", statusbar,
			   (GtkDestroyNotify) gtk_widget_unref);
  gtk_widget_show(statusbar);
  gtk_box_pack_start(GTK_BOX(box1), statusbar, FALSE, FALSE, 0);
  statusbar_context_id = 
    gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "main");
  gtk_statusbar_push(GTK_STATUSBAR(statusbar), statusbar_context_id, "");

  first_problem();
  new_problem();

  gtk_widget_show(mainwindow);
}


struct option longopts[] = {
  { "version", no_argument, NULL, 'v' },
  { "help", no_argument, NULL, 'h' },
  { NULL, 0, NULL, 0 }
};

static char *progname;

int main(int argc, char *argv[])
{
  int c, h = 0, v = 0, lose = 0;

  progname = argv[0];

  while ((c = getopt_long(argc, argv, "ht", longopts, NULL)) != EOF)
    switch (c) {
    case 'v':
      v = 1;
      break;
    case 'h':
      h = 1;
      break;
    default:
      lose = 1;
      break;
    }
  
  if (lose || optind < argc) {
    /* Print error message and exit.  */
    if (optind < argc)
      fputs("Too many arguments\n", stderr);
    fprintf(stderr, "Try `%s --help' for more information\n",
	    progname);
    exit(1);
  }

  /* `help' should come first.  If `help' is requested, ignore the other
     options. */
  if (h) {
    /* Print help info and exit.  */
    printf("\
This is Glickomania, a puzzle game.\n\
Usage: %s [OPTION]\n\
  -h, --help          display this help and exit\n\
  -v, --version       display version information and exit\n\
\n\
Report bugs to cessu@iki.fi.\n",
	   progname);
    exit(0);
  }
  
  if (v) {
    /* Print version number.  */
    printf("glickomania %d.%d\n", PACKAGE, 
	   GLICKOMANIA_VERSION_MAJOR, GLICKOMANIA_VERSION_MINOR);
    /* xgettext: no-wrap */
    printf("\
Copyright (C) %s Free Software Foundation, Inc.\n\
Copyright (C) %s Kenneth Oksanen\n\
There is NO warranty; not even for MERCHANTABILITY or FITNESS FOR A\n\
PARTICULAR PURPOSE.  You may redistribute copies of GNU %s under the terms\n\
of the GNU General Public License.\n\
For more information about these matters, see the file named COPYING.\n",
	    "1992, 1993, 1997", "2002", PACKAGE);
    exit(0);
  }
  
  gtk_init(&argc, &argv);
  load();
  glickomain();
  gtk_main();

  exit(0);
}


#ifdef C_ALLOCA /* xmalloc() is only used by alloca.c.  */

char *xmalloc(unsigned int size)
{
  char *ptr = malloc(size);
  if (!ptr) {
    fprintf(stderr, "%s: virtual memory exhausted\n", progname);
    exit(1);
  }
  return ptr;
}
#endif /* C_ALLOCA */

/* glickomania.c ends here */
